% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fn.R
\name{fn}
\alias{fn}
\alias{fn_}
\title{Function Declarations with Quasiquotation}
\usage{
fn(..., ..env = parent.frame())

fn_(..., ..env = parent.frame())
}
\arguments{
\item{...}{Function declaration, which supports
\link[rlang:topic-inject]{quasiquotation}.}

\item{..env}{Environment in which to create the function (i.e., the
function's \link[base:environment]{enclosing environment}).}
}
\value{
A function whose enclosing environment is \code{..env}.
}
\description{
\code{fn()} enables you to create (anonymous) functions, of arbitrary call
signature. Use it in place of the usual \code{\link[=function]{function()}} invocation whenever you
want to:
\itemize{
\item \strong{Be concise}: The function declarations\preformatted{  fn(x, y = 1 ~ x + y)

  function(x, y = 1) x + y
}

are equivalent.
\item \strong{Enforce immutability}: By enabling Tidyverse
\link[rlang:topic-inject]{quasiquotation}, \code{fn()} allows you to \dQuote{burn
in} values at the point of function creation. This guards against changes
in a function's enclosing environment. (See \sQuote{Use Unquoting to Make
Robust Functions}.)
}

\code{fn_()} is a variant of \code{fn()} that does \emph{not} comprehend quasiquotation. It
is useful when you want unquoting (\code{`!!`}) or splicing (\code{`!!!`})
operators in the function body to be literally interpreted, rather than
immediately invoked. (See \sQuote{Quasiquotation} for a complementary way to
literally interpret unquoting and splicing operators in \code{fn()}.)
}
\section{Function Declarations}{
 A \strong{function declaration} is an expression
that specifies a function's arguments and body, as a comma-separated
expression of the form\preformatted{  arg1, arg2, ..., argN ~ body
}

or\preformatted{  arg1, arg2, ..., argN, ~body
}

(Note in the second form that the body is a one-sided formula. This
distinction is relevant for argument \link[rlang:topic-inject]{splicing}, see
\sQuote{Quasiquotation}.)
\itemize{
\item To the left of \code{~}, you write a conventional function-argument
declaration, just as in \verb{function(<arguments>)}: each of \code{arg1}, \code{arg2},
\dots, \code{argN} is either a bare argument (e.g., \code{x} or \code{...}) or an
argument with default value (e.g., \code{x = 1}).
\item To the right of \code{~}, you write the function body, i.e., an expression of
the arguments.
}
}

\section{Quasiquotation}{

All parts of a function declaration support Tidyverse
\link[rlang:topic-inject]{quasiquotation}:
\itemize{
\item To unquote values (of arguments or parts of the body), use \verb{!!}:\preformatted{  z <- 0
  fn(x, y = !!z ~ x + y)
  fn(x ~ x > !!z)
}
\item To unquote argument names (with default value), use \verb{:=} (definition
operator):\preformatted{  arg <- "y"
  fn(x, !!arg := 0 ~ x + !!as.name(arg))
}
\item To splice in a (formal) list of arguments, use \verb{!!!}:\preformatted{  # NB: Body is a one-sided formula
  fn(!!!alist(x, y = 0), ~ x + y)
}

Splicing allows you to treat a complete function declaration as a unit:\preformatted{  soma <- alist(x, y = 0, ~ x + y)
  fn(!!!soma)
}
\item To write literal unquoting operators, use \code{QUQ()}, \code{QUQS()}, which read
as \dQuote{quoted unquoting,} \dQuote{quoted unquote-splicing,} resp. (cf.
\code{fn_()}):\preformatted{  library(dplyr)

  my_summarise <- fn(df, ... ~ \{
    groups <- quos(...)
    df \%>\%
      group_by(QUQS(groups)) \%>\%
      summarise(a = mean(a))
  \})
}

(Source: \href{https://dplyr.tidyverse.org/articles/programming.html}{Programming with dplyr})
}
}

\section{Use Unquoting to Make Robust Functions}{

Functions in R are generally
\href{https://en.wikipedia.org/wiki/Pure_function}{impure}, i.e., the return
value of a function will \emph{not} in general be determined by the value of its
inputs alone. This is because, by design, a function may depend on objects
in its
\href{https://adv-r.hadley.nz/functions.html#lexical-scoping}{lexical scope}, and
these objects may mutate between function calls. Normally this isn't a
hazard.

However, if you are working interactively and sourcing files into the
global environment, or using a notebook interface like
\href{https://jupyter.org}{Jupyter} or
\href{https://bookdown.org/yihui/rmarkdown/notebook.html}{R Notebook}, it can be
tricky to ensure that you haven't unwittingly mutated an object that an
earlier function depends upon.

You can use unquoting to guard against such mutations.

\subsection{Example}{
Consider the following function:\preformatted{  a <- 1
  foo <- function(x) x + a
}

What is the value of \code{foo(1)}? It is not necessarily \code{2}, because the value
of \code{a} may have changed between the \emph{creation} of \code{foo()} and the \emph{calling}
of \code{foo(1)}:\preformatted{  foo(1)  #> [1] 2

  a <- 0

  foo(1)  #> [1] 1
}

In other words, \code{foo()} is impure because the value of \code{foo(x)} depends not
only on the value of \code{x} but also on the \emph{externally mutable} value of \code{a}.

With \code{fn()}, you can unquote \code{a} to \dQuote{burn in} its value at the point
of creation:\preformatted{  a <- 1
  foo <- fn(x ~ x + !!a)
}

Now \code{foo()} is a pure function, unaffected by changes to \code{a} in the lexical
scope:\preformatted{  foo(1)  #> [1] 2

  a <- 0

  foo(1)  #> [1] 2
}

}
}

\examples{
fn(x ~ x + 1)
fn(x, y ~ x + y)
fn(x, y = 2 ~ x + y)
fn(x, y = 1, ... ~ log(x + y, ...))

# to specify '...' in the middle, write '... = '
fn(x, ... = , y ~ log(x + y, ...))

# use one-sided formula for constant functions or commands
fn(~ NA)
fn(~ message("!"))

# unquoting is supported (using `!!` from rlang)
zero <- 0
fn(x = !!zero ~ x > !!zero)

# formals and function bodies can also be spliced in
f <- function(x, y) x + y
g <- function(y, x, ...) x - y
frankenstein <- fn(!!!formals(f), ~ !!body(g))
stopifnot(identical(frankenstein, function(x, y) x - y))

# mixing unquoting and literal unquoting is possible
# (Assume dplyr is available, which provides group_by() and `\%>\%`.)
summariser <- quote(mean)
my_summarise <- fn(df, ... ~ {
  groups <- quos(...)
  df \%>\%
    group_by(QUQS(groups)) \%>\%          # literal unquote-splice
    summarise(a = `!!`(summariser)(a))  # substitute `mean`
})
my_summarise

# Use fn_() with fn() as a concise way to force ("pin down") bindings
# For example, the 'x' is immutable in the function produced by call_upon():
call_upon <- fn_(x ~ fn(f ~ f(!!x)))
sapply(list(sin, cos), call_upon(0))  # [1] 0 1

# Return-value checking, as a functional transformation
enforce <- fn_(condition ~
  fn(x ~ {
    stopifnot(!!substitute(condition))
    x
  })
)
no_nan <- enforce(!is.nan(x))
log_strict <- fn(x ~ no_nan(log(x)))
log_strict(2)        # [1] 0.6931472
try(log_strict(-1))  # Error: !is.nan(x) is not TRUE

}
