% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/pairwise.slope.test.r
\name{pairwise.slope.test}
\alias{pairwise.slope.test}
\title{Pairwise Comparisons of Slopes}
\usage{
pairwise.slope.test(f1, f2, iter = 999, int.first = FALSE,
  angle.type = c("r", "deg", "rad"), RRPP = FALSE)
}
\arguments{
\item{f1}{A formula for the linear model from which groups are to be compared (e.g., y~x1*x2)}

\item{f2}{A right side formula for the covariate (e.g., ~ CS).}

\item{iter}{Number of iterations for permutation test}

\item{int.first}{A logical value to indicate if interactions of first main effects should precede subsequent main effects}

\item{angle.type}{A value specifying whether differences between slopes should be represented by vector
correlations (r), radians (rad) or degrees (deg)}

\item{RRPP}{a logical value indicating whether residual randomization should be used for significance testing}
}
\value{
Function returns a list with the following components:
  \item{ANOVA.table}{An ANOVA table assessing the linear model}
  \item{Slope.Dist}{A matrix of pairwise differences between slope magnitudes}
  \item{Prob.Dist}{Associated matrix of pairwise significance levels based on permutations}
  \item{Slope.cor}{A matrix of pairwise slope vector correlations (if vector correlation is chosen)}
  \item{Prob.cor}{Associated matrix of pairwise significance levels based on permutations}
  \item{Slope.angle}{A matrix of pairwise angular differences in slope (if "rad" or "deg" chosen)}
  \item{Prob.angle}{Associated matrix of pairwise significance levels based on permutations}
}
\description{
Function performs pairwise comparisons among slopes for groups as specified by a linear model.
}
\details{
The function performs pairwise comparisons to identify differences in slopes between groups. The function is
designed as a post-hoc test to MANCOVA, where the latter has identified significant shape variation explained by a
covariate*group interaction term.

As input the user provides a formula describing the linear model of how shape (y) varies as a function of a factor (a)
 or factorial interaction (a*b). A single covariate must also be added.  Shape data (y) can be
 in the form of a two-dimensional data matrix of dimension (n x [p x k]) or a 3D array (p x k x n).
 It is assumed that the landmarks have previously been aligned using Generalized Procrustes Analysis (GPA)
 [e.g., with \code{\link{gpagen}}]. From the data, the slopes for each group are estimated, and pairwise differences
 in slopes determined.

It is assumed that one has verified a significant group*covariate interaction [e.g., with \code{\link{procD.lm}}].
  To evaluate significance of the pairwise differences, two possible resampling procedures are provided. First, if
  RRPP=FALSE, the rows of the matrix of shape variables are randomized relative to the design matrix. This is
  analogous to a 'full' randomization. Second, if RRPP=TRUE, a residual randomization permutation procedure
  is utilized (Collyer et al. 2015). Here, residual shape values from a reduced model are
  obtained, and are randomized with respect to the linear model under consideration. These are then added to
  predicted values from the remaining effects to obtain pseudo-values from which SS are calculated. NOTE: for
  single-factor designs, the two approaches are identical.  However, when evaluating factorial models it has been
  shown that RRPP attains higher statistical power and thus has greater ability to identify patterns in data should
  they be present (see Anderson and terBraak 2003). Effect-sizes (Z-scores) are computed as standard deviates of the sampling
  distributions generated, which might be more intuitive for P-values than F-values (see Collyer et al. 2014).

Slopes can differ in two ways: the amount of shape change per covariate unit change and the direction of shape change
  associated with covariate change.  Tests statistics to compare these attributes between groups are the differences in length
  and direction between slope vectors, respectively.  These statistics are calculated with the exact same random permutations used
  to calculate random SS for ANOVA.

This test is essentially the same as procD.lm with post-hoc comparisons among slopes for appropriate
  models.  However, differences in slopes are calculated simultaneously with the same random permutations peformed for ANOVA,
  making it less so a post-hoc test and more so a simultaneous test of pairwise contrasts (see Collyer et al. 2014).
}
\examples{
### MANCOVA example for Goodall's F test (multivariate shape vs. factors)
data(plethodon)
Y.gpa<-gpagen(plethodon$land)    #GPA-alignment
y <- Y.gpa$coords

## Pairwise slope vector correlations
pairwise.slope.test(y~plethodon$site, ~ Y.gpa$Csize, iter=24, angle.type="r")

## Pairwise angular difference between slopes
pairwise.slope.test(y~plethodon$site, ~ Y.gpa$Csize, iter=24, angle.type="rad")

## Using RRPP
pairwise.slope.test(y~plethodon$site, ~Y.gpa$Csize, iter=24, angle.type="rad", RRPP=TRUE)
}
\author{
Michael Collyer
}
\references{
Anderson MJ. and C.J.F. terBraak. 2003. Permutation tests for multi-factorial analysis of variance.
   Journal of Statistical Computation and Simulation 73: 85-113.

Collyer, M.L., D.J. Sekora, and D.C. Adams. 2015. A method for analysis of phenotypic change for phenotypes described
by high-dimensional data. Heredity. 113: doi:10.1038/hdy.2014.75.
}
\keyword{analysis}

