#' Generate a citation for occurrence data
#'
#' If a `data.frame` was generated using [atlas_occurrences()],
#' and the `mint_doi` argument was set to `TRUE`, the DOI associated
#' with that dataset is appended to the resulting `data.frame` as an
#' attribute. This function simply formats that DOI as a citation that can be 
#' included in a scientific publication. Please also consider citing this 
#' package, using the information in `citation("galah")`.
#' @param data data.frame: occurrence data generated by 
#' [atlas_occurrences()]
#' @return A string containing the citation for that dataset.
#' @examples \dontrun{
#' atlas_citation(doi)
#' }
#' @export

atlas_citation <- function(data) {
  doi <- attributes(data)$doi
  if (is.null(doi)) {
    bullets <- c(
      "This data does not have a DOI attached.",
      i = "Did you set `atlas_occurrences(mint_doi = TRUE)`?",
      i = "`atlas_citation` extracts this citation info when present."
    )
    warn(bullets, call = caller_env())
    glue("Please consider citing R & galah. To do so, call:
       citation()
       citation('galah')")
  }else{
    current_date <- format(Sys.Date(), "%e %B %Y") |>
      trimws()
    if(grepl("10.26197/ala.", doi)){
      org_text <- "Atlas of Living Australia"
      description <- "Occurrence download"
    }else{
      org_text <- "GBIF.org"
      description <- "GBIF Occurrence Download"
    }
    glue("
       {org_text} ({current_date}) {description} {doi}
       
       Please consider citing R & galah, in addition to your dataset. To do so, call:
       citation()
       citation('galah')
       ")    
  }
}
