% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cptable-create.R
\name{cptable}
\alias{cptable}
\title{Create conditional probability tables (CPTs)}
\usage{
cptable(vpar, levels = NULL, values = NULL, normalize = TRUE, smooth = 0)
}
\arguments{
\item{vpar}{Specifications of the names in P(v|pa1,...pak). See
section 'details' for information about the form of the
argument.}

\item{levels}{See 'details' below.}

\item{values}{Probabilities; recycled if necessary. Regarding the
order, please see section 'details' and the examples.}

\item{normalize}{See 'details' below.}

\item{smooth}{See 'details' below.}
}
\value{
A \code{cptable} object (a numeric vector with various attributes).
}
\description{
Creates conditional probability tables of the form
p(v|pa(v)).
}
\details{
If \code{normalize=TRUE} then the probabilities are normalized to sum to one
for each configuration of the parents.

If \code{smooth} is non--zero then zero entries of \code{values} are
replaced with \code{smooth} \strong{before} normalization takes place.

Regarding the form of the argument \code{vpar}: To specify \eqn{P(a|b,c)}
one may write \code{~a|b:c}, \code{~a:b:c}, \code{~a|b+c}, \code{~a+b+c} or
\code{c("a","b","c")}. Internally, the last form is used. Notice that the
\code{+} and \code{:} operator is used as a separator only. The order of the
variables IS important so the operators DO NOT commute.

If \code{a} has levels \code{a1,a2} and likewise for \code{b} and \code{c}
then the order of \code{values} corresponds to the configurations
\code{(a1,b1,c1)}, \code{(a2,b1,c1)} \code{(a1,b2,c1)}, \code{(a2,b2,c1)}
etc. That is, the first variable varies fastest.  Hence the first two
elements in \code{values} will be the conditional probabilities of \code{a}
given \code{b=b1, c=c1}.
}
\examples{


## See the wet grass example at
## https://en.wikipedia.org/wiki/Bayesian_network

yn <- c("yes", "no")
p.R    <- cptable(~R, values=c(.2, .8), levels=yn)
p.S_R  <- cptable(~S:R, values=c(.01, .99, .4, .6), levels=yn)
p.G_SR <- cptable(~G:S:R, values=c(.99, .01, .8, .2, .9, .1, 0, 1), levels=yn)

# or
ssp <- list(R=yn, S=yn, G=yn) # state space
p.R    <- cptable(~R, values=c(.2, .8), levels=ssp)
p.S_R  <- cptable(~S:R, values=c(.01, .99, .4, .6), levels=ssp)
p.G_SR <- cptable(~G:S:R, values=c(.99, .01, .8, .2, .9, .1, 0, 1), levels=ssp)

# components above are "intermediate representations" and are turned into arrays with
wet.cpt <- compileCPT(p.R, p.S_R, p.G_SR)
wet.cpt
wet.cpt$S # etc

# A Bayesian network is created with:
wet.bn <- grain(wet.cpt)

# Can also create arrays directly
\dontrun{
ssp <- list(R=yn, S=yn, G=yn) # state space
p.R    <- c(.2, .8)
p.S_R  <- c(.01, .99, .4, .6)
p.G_SR <- c(.99, .01, .8, .2, .9, .1, 0, 1)
dim(p.R) <- 2
dimnames(p.R) <- ssp["R"]
dim(p.S_R) <- c(2, 2)
dimnames(p.S_R) <- ssp[c("S", "R")]
dim(p.G_SR) <- c(2, 2, 2)
dimnames(p.G_SR) <- ssp[c("G", "S", "R")]

# Arrays can be created (easier?) with parray() from gRbase
p.R    <- parray("R", levels=ssp, values=c(.2, .8))
p.S_R  <- parray(c("S", "R"), levels = ssp, values=c(.01, .99, .4, .6))
p.G_SR <- parray(~ G:S:R, levels = ssp, values=c(.99, .01, .8, .2, .9, .1, 0, 1))
}
}
\references{
Søren Højsgaard (2012). Graphical Independence
Networks with the gRain Package for R. Journal of Statistical
Software, 46(10), 1-26.
\url{http://www.jstatsoft.org/v46/i10/}.
}
\seealso{
\code{\link{andtable}}, \code{\link{ortable}},
\code{\link{extractCPT}}, \code{\link{compileCPT}},
\code{\link{extractPOT}}, \code{\link{compilePOT}},
\code{\link{grain}}, \code{\link[gRbase]{parray}}
}
\author{
Søren Højsgaard, \email{sorenh@math.aau.dk}
}
\keyword{models}
