% Generated by roxygen2 (4.0.1): do not edit by hand
\name{funreg}
\alias{funreg}
\title{Perform penalized functional regression}
\usage{
funreg(id, response, time, x, basis.method = 1, deg = 2, deg.penalty = 2,
  family = gaussian, other.covariates = NULL, num.bins = 35,
  preferred.num.eigenfunctions = 30, preferred.num.knots.for.beta = 35,
  se.method = 1, smoothing.method = 1, times.for.fit.grid = NULL)
}
\arguments{
\item{id}{An integer or string uniquely identifying the
subject to which each observation belongs}

\item{response}{The response, as a vector, one for each subject}

\item{time}{The time of observation, as a vector,
one for each observation (i.e., each
assessment on each person)}

\item{x}{The functional predictor(s), as a matrix,
one row for each observation (i.e., for each
assessment on each person)}

\item{basis.method}{An integer, either 1 or 2, describing how
the beta function should be internally
modeled. A value of 1 indicates that a
truncated power spline basis should be used,
and a value of 2 indicates that a B-spline
basis should be used.}

\item{deg}{An integer, either 1, 2, or 3, describing how
complicated the behavior of the beta function between knots
may be.  1, 2, or 3 represent linear, quadratic or cubic
function between knots.}

\item{deg.penalty}{Only relevant for B-splines.  The
difference order used to weight the smoothing
penalty (see Eilers and Marx, 1996)}

\item{family}{The response distribution.  For example,
this is  \code{family=gaussian} for normal linear
models, \code{family=binomial} for logistic regression models,
or \code{family=poisson} for count models.  See the \code{gam}
documentation in the \code{mgcv} package, or use \code{help(family)}
for details on \code{family} objects.}

\item{other.covariates}{Subject-level (time-invariant) covariates,
if any, as a matrix, one column per covariate. The default,
\code{NULL}, means that no subject-level covariates will be included
in the model.}

\item{num.bins}{The number of knots used in the spline basis for the
beta function. The default is based on the  Goldsmith et al. (2011)
sample code.}

\item{preferred.num.eigenfunctions}{The number of eigenfunctions to use in approximating the covariance function of x (see Goldsmith et al., 2011)}

\item{preferred.num.knots.for.beta}{number of knots to use in the spline
estimation.  The default, is based on the Goldsmith et al (2011) sample code.}

\item{se.method}{An integer, either 1 or 2, describing how
the standard errors should be calculated. A value
of 1 means that the uncertainty related to
selecting the smoothing parameter is ignored.
Option 2 means that a Bayesian approach is used
to try to take this uncertainty into account
(see the documentation for Wood's \code{mgcv} package).}

\item{smoothing.method}{An integer, either 1 or 2, describing
how the weight of the smoothing penalty should be
determined.  Option 1 means that the smoothing
weight should be estimated using an approach
similar to restricted maximum likelihood, and
Option 2 means an approach similar to generalized
cross-validation.  Option 1 is strongly
recommended (based both on our experience and on
remarks in the documentation for the gam function
in the mgcv package).}

\item{times.for.fit.grid}{Points at which to calculate
the estimated beta function.  The default, NULL, means that the
code will choose these times automatically.}
}
\value{
An object of type \code{funreg}.  This object
can be examined using \code{summary}, \code{print},
 or \code{fitted}.
}
\description{
Performs a penalized functional regression as in Goldsmith et al. (2012)
on irregularly measured data such as that found in ecological momentary
assessment (see, e.g., Shiffman, Stone, & Hufford, 2008).
}
\note{
This function mostly follows code by Jeff Goldsmith and co-workers:
the sample code from Goldsmith et al (2011), and the "pfr" function in
the "refund" R package.  However, this code is adapted here to allow
idiosyncratic measurement times and unequal numbers of observations
per subject to be handled easily, and also allows the use of a different
estimation method.  Also follows some sample code for penalized
B-splines from Eilers and Marx (1996) in implementing B-splines.
As the pfr function in refund also does, the function calls
the gam function in the mgcv package (Wood 2011) to do much of the
internal calculations.

In the example below, to fit a more complicated model, replace
 \code{x=SampleFunregData$x1} with \code{x=cbind(SampleFunregData$x1,
  SampleFunregData$x2),other.covariates=cbind(SampleFunregData$s1,
  SampleFunregData$s2, SampleFunregData$s3, SampleFunregData$s4)}.  This
  model will take longer to run, perhaps 10 or 20 seconds.  Then try
  \code{plot(complex.model)}.
}
\examples{
simple.model <- funreg(id=SampleFunregData$id,
                       response=SampleFunregData$y,
                       time=SampleFunregData$time,
                       x=SampleFunregData$x1,
                       family=binomial);
print(simple.model);
par(mfrow=c(2,2));
plot(x=simple.model$model.for.x[[1]]$bin.midpoints,
     y=simple.model$model.for.x[[1]]$mu.x.by.bin,
     xlab="Time t",ylab="X(t)",main="Smoothed mean x values");
# The smoothed average value of the predictor function x(t) at different times t.
# The ``[[1]]'' after model.for.x is there because model.for.x is a list with one entry.
# This is because more than one functional covariate is allowed.
plot(simple.model,type="correlations");
# The marginal correlation of x(t) with y at different times t.
# It appears that earlier time points are more strongly related to y.
plot(simple.model,type="coefficients");
# The functional regression coefficient of y on x(t).
# It also appears that earlier time points are more strongly related to y.
plot(simple.model$subject.info$response,
     simple.model$subject.info$fitted,
     main="Predictive Performance",
     xlab="True Y",
     ylab="Fitted Y");
}
\references{
Crainiceanu, C., Reiss, P., Goldsmith, J., Huang, L., Huo, L.,
   Scheipl, F. (2012). refund: Regression with Functional Data
   (version 0.1-6). R package Available online at cran.r-project.org.

 Eilers, P. H. C., and Marx, B. D. (1996). Flexible smoothing with
   B-splines and penalties. Statistical Science, 11, 89-121.

 Goldsmith, J., Bobb, J., Crainiceanu, C. M., Caffo, B., and Reich, D.
   (2011). Penalized functional regression. Journal of Computational
   and Graphical Statistics, 20(4), 830-851. The sample code can be
   found at www.jeffgoldsmith.com/Downloads/PFR_Web_Appendix.zip;
   in writing parts of this function I especially followed "PFR_Example.R",
   written on Jan. 15 2010, by Jeff Goldsmith.

 Ruppert, D., Wand, M., and Carroll, R. (2003). Semiparametric regression.
   Cambridge, UK: Cambridge University Press.

 Shiffman, S., Stone, A. A., and Hufford, M. R. (2008). Ecological
   momentary assessment. Annual Review of Clinical Psychology, 4, 1-32.

 Wood, S.N. (2006) Generalized Additive Models: An Introduction with
    R. Chapman and Hall/CRC.

 Wood, S.N. (2011) Fast stable restricted maximum likelihood and
   marginal likelihood estimation of semiparametric generalized linear
   models. Journal of the Royal Statistical Society (B) 73(1):3-36.
}
\seealso{
\code{\link{fitted.funreg}}, \code{link{plot.funreg}},
         \code{\link{print.funreg}}, \code{link{summary.funreg}}
}

