% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{sd3}
\alias{cent_cumulants}
\alias{cent_moments}
\alias{kurt5}
\alias{sd3}
\alias{skew4}
\alias{std_moments}
\title{Compute first K moments}
\usage{
sd3(v, na_rm = FALSE)

skew4(v, na_rm = FALSE)

kurt5(v, na_rm = FALSE)

cent_moments(v, max_order = 5L, used_df = 0L, na_rm = FALSE)

std_moments(v, max_order = 5L, used_df = 0L, na_rm = FALSE)

cent_cumulants(v, max_order = 5L, used_df = 0L, na_rm = FALSE)
}
\arguments{
\item{v}{a vector}

\item{na_rm}{whether to remove NA, false by default.}

\item{max_order}{the maximum order of the centered moment to be computed.}

\item{used_df}{the number of degrees of freedom consumed, used in the denominator
of the centered moments computation. These are subtracted from the number of
observations.}
}
\value{
a vector, filled out as follows:
\describe{
\item{sd3}{A vector of the (sample) standard devation, mean, and number of elements.}
\item{skew4}{A vector of the (sample) skewness, standard devation, mean, and number of elements.}
\item{kurt5}{A vector of the (sample) excess kurtosis, skewness, standard devation, mean, and number of elements.}
\item{cent_moments}{A vector of the (sample) \eqn{k}th centered moment, then \eqn{k-1}th centered moment, ..., 
 then the \emph{variance}, the mean, and number of elements.}
\item{std_moments}{A vector of the (sample) \eqn{k}th standardized (and centered) moment, then 
 \eqn{k-1}th, ..., then standard devation, mean, and number of elements.}
\item{cent_cumulants}{A vector of the (sample) \eqn{k}th (centered, but this is redundant) cumulant, then the \eqn{k-1}th, ...,
 then the \emph{variance} (which is the second cumulant), the mean, and number of elements.}
}
}
\description{
Compute the (standardized) 2nd through kth moments, the mean, and the number of elements.
}
\details{
Computes the number of elements, the mean, and the 2nd through kth
centered standardized moment, for \eqn{k=2,3,4}{k=2,3,4}. These
are computed via the numerically robust one-pass method of Bennett \emph{et. al.}
In general they will \emph{not} match exactly with the 'standard'
implementations, due to differences in roundoff.

These methods are reasonably fast, on par with the 'standard' implementations.
However, they will usually be faster than calling the various standard implementations
more than once.
}
\note{
The first centered (and standardized) moment is often defined to be identically 0. Instead \code{cent_moments}
and \code{std_moments} returns the mean. 
Similarly, the second standardized moments defined to be identically 1; \code{std_moments} instead returns the standard
deviation. The reason is that a user can always decide to ignore the results and fill in a 0 or 1 as they need, but 
could not efficiently compute the mean and standard deviation from scratch if we discard it.

The last minus two element of the output of \code{cent_moments} and \code{cent_cumulants} is the \emph{variance},
not the standard deviation. All other code return the standard deviation in that place.

The kurtosis is \emph{excess kurtosis}, with a 3 subtracted, and should be nearly zero
for Gaussian input.

'centered cumulants' is redundant. The intent was to avoid collision with existing code named 'cumulants'.

The moment computations provided by fromo are 
numerically robust, but will often \emph{not} provide the
same results as the 'standard' implementations,
due to differences in roundoff. We make every attempt to balance
speed and robustness. User assumes all risk from using
the fromo package.
}
\examples{
x <- rnorm(1e5)
sd3(x)[1] - sd(x)
skew4(x)[4] - length(x)
skew4(x)[3] - mean(x)
skew4(x)[2] - sd(x)
if (require(moments)) {
  skew4(x)[1] - skewness(x)
}
# check 'robustness'; only the mean should change:
kurt5(x + 1e12) - kurt5(x)
# check speed
if (require(microbenchmark) && require(moments)) {
  dumbk <- function(x) { c(kurtosis(x) - 3.0,skewness(x),sd(x),mean(x),length(x)) }
  set.seed(1234)
  x <- rnorm(1e6)
  print(kurt5(x) - dumbk(x))
  microbenchmark(dumbk(x),kurt5(x),times=10L)
}
y <- std_moments(x,6)
cml <- cent_cumulants(x,6)

}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
\references{
Terriberry, T. "Computing Higher-Order Moments Online."
\url{http://people.xiph.org/~tterribe/notes/homs.html}

J. Bennett, et. al., "Numerically Stable, Single-Pass, 
Parallel Statistics Algorithms," Proceedings of IEEE
International Conference on Cluster Computing, 2009.
\url{https://www.semanticscholar.org/paper/Numerically-stable-single-pass-parallel-statistics-Bennett-Grout/a83ed72a5ba86622d5eb6395299b46d51c901265}

Cook, J. D. "Accurately computing running variance."
\url{http://www.johndcook.com/standard_deviation.html}

Cook, J. D. "Comparing three methods of computing 
standard deviation."
\url{http://www.johndcook.com/blog/2008/09/26/comparing-three-methods-of-computing-standard-deviation}
}

