% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eim-class.R
\name{get_agg_opt}
\alias{get_agg_opt}
\title{Runs the EM algorithm \strong{over all possible group aggregating}, returning the one with higher likelihood while constraining the standard deviation of the probabilities.}
\usage{
get_agg_opt(
  object = NULL,
  X = NULL,
  W = NULL,
  json_path = NULL,
  sd_statistic = "maximum",
  sd_threshold = 0.05,
  method = "mult",
  nboot = 100,
  allow_mismatch = TRUE,
  seed = NULL,
  ...
)
}
\arguments{
\item{object}{An object of class \code{eim}, which can be created using the \link{eim} function. This parameter should not be used if either (i) \code{X} and \code{W} matrices or (ii) \code{json_path} is supplied. See \strong{Note} in \link{run_em}.}

\item{X}{A \verb{(b x c)} matrix representing candidate votes per ballot box.}

\item{W}{A \verb{(b x g)} matrix representing group votes per ballot box.}

\item{json_path}{A path to a JSON file containing \code{X} and \code{W} fields, stored as nested arrays. It may contain additional fields with other attributes, which will be added to the returned object.}

\item{sd_statistic}{String indicates the statistic for the standard deviation \verb{(g x c)} matrix for the stopping condition, i.e., the algorithm stops when the statistic is below the threshold. It can take the value \code{maximum}, in which case computes the maximum over the standard deviation matrix, or \code{average}, in which case computes the average.}

\item{sd_threshold}{Numeric with the value to use as a threshold for the statistic (\code{sc_statistic}) of the standard deviation of the estimated probabilities. Defaults to 0.05.}

\item{method}{An optional string specifying the method used for estimating the E-step. Valid
options are:
\itemize{
\item \code{mult}: The default method, using a single sum of Multinomial distributions.
\item \code{mvn_cdf}: Uses a Multivariate Normal CDF distribution to approximate the conditional probability.
\item \code{mvn_pdf}: Uses a Multivariate Normal PDF distribution to approximate the conditional probability.
\item \code{mcmc}: Uses MCMC to sample vote outcomes. This is used to estimate the conditional probability of the E-step.
\item \code{exact}: Solves the E-step using the Total Probability Law.
}}

\item{nboot}{Integer specifying how many times to run the
EM algorithm.}

\item{allow_mismatch}{Boolean, if \code{TRUE}, allows a mismatch between the voters and votes for each ballot-box, only works if \code{method} is \code{"mvn_cdf"}, \code{"mvn_pdf"}, \code{"mult"} and \code{"mcmc"}. If \code{FALSE}, throws an error if there is a mismatch. By default it is \code{TRUE}.}

\item{seed}{An optional integer indicating the random seed for the randomized algorithms. This argument is only applicable if \code{initial_prob = "random"} or \code{method} is either \code{"mcmc"} or \code{"mvn_cdf"}. Aditionally, it sets the random draws of the ballot boxes.}

\item{...}{Additional arguments passed to the \link{run_em} function that will execute the EM algorithm.}
}
\value{
It returns an eim object with the same attributes as the output of \link{run_em}, plus the attributes:
\itemize{
\item \strong{sd}: A \verb{(a x c)} matrix with the standard deviation of the estimated probabilities computed with bootstrapping. Note that \code{a} denotes the number of macro-groups of the resulting group aggregation, it should be between \code{1} and \code{g}.
\item \strong{nboot}: Number of samples used for the \link{bootstrap} method.
\item \strong{seed}: Random seed used (if specified).
\item \strong{sd_statistic}: The statistic used as input.
\item \strong{sd_threshold}: The threshold used as input.
\item \strong{group_agg}: Vector with the resulting group aggregation. See \strong{Examples} for more details.
}

Additionally, it will create the \code{W_agg} attribute with the aggregated groups, along with the attributes corresponding to running \link{run_em} with the aggregated groups.
}
\description{
This function estimates the voting probabilities (computed using \link{run_em}) by trying all group aggregations (of adjacent groups), choosing
the one that achieves the higher likelihood as long as the standard deviation (computed using \link{bootstrap}) of the estimated probabilities
is below a given threshold. See \strong{Details} for more informacion on adjacent groups.
}
\details{
Groups of consecutive column indices in the matrix \code{W} are considered adjacent. For example, consider the following seven groups defined by voters' age
ranges: 20-29, 30-39, 40-49, 50-59, 60-69, 70-79, and 80+. A possible group aggregation can be a macro-group composed of the three following age
ranges: 20-39, 40-59, and 60+. Since there are multiple group aggregations, the method evaluates all possible group aggregations (merging only adjacent groups).
}
\examples{
# Example 1: Using a simulated instance
simulations <- simulate_election(
    num_ballots = 20,
    num_candidates = 3,
    num_groups = 8,
    seed = 42
)

result <- get_agg_opt(
    X = simulations$X,
    W = simulations$W,
    sd_threshold = 0.05,
    seed = 42
)

result$group_agg # c(3,8)
# This means that the resulting group aggregation consists of
# two macro-groups: one that includes the original groups 1, 2, and 3;
# the remaining one with groups 4, 5, 6, 7 and 8.

\donttest{
# Example 2: Getting an unfeasible result
result2 <- get_agg_opt(
    X = simulations$X,
    W = simulations$W,
    sd_threshold = 0.001
)

result2$group_agg # Error
result2$X # Input candidates' vote matrix
result2$W # Input group-level voter matrix
}
}
