library(testthat)        # load testthat package
library(evprof)
library(dplyr)
library(tibble)
library(lubridate)
library(purrr)
library(ggplot2)

# Get the example `evmodel` and `sessions` included in the package
ev_model <- evprof::california_ev_model
sessions <- evprof::california_ev_sessions
temp_dir <- tempdir()

# Test exploration --------------------------------------------------------
test_that("Charging rates ratios calculation", {
  expect_true(is_tibble(get_charging_rates_distribution(sessions)))
})
test_that("Average daily sessions calculation", {
  expect_true(is.numeric(get_daily_avg_n_sessions(sessions, 2023, 1, 7)))
})

# Exploration plots
test_that("Statistic summary of sessions' features", {
  expect_true(is_tibble(
    summarise_sessions(sessions, mean, vars = c("Power", "Energy", "ConnectionHours"))
  ))
})

test_that("Statistic plot of sessions' features", {
  expect_true(is.ggplot(
    plot_histogram_grid(sessions, vars = c("Power", "Energy", "ConnectionHours"))
  ))
})

test_that("Points plot", {
  expect_true(is.ggplot(
    plot_points(sessions, log = FALSE)
  ))
  expect_true(is.ggplot(
    plot_points(sessions, log = TRUE)
  ))
})

test_that("Density 2D plots", {
  expect_true(is.ggplot(
    plot_density_2D(sessions, by = "wday", log = FALSE)
  ))
  expect_true(is.ggplot(
    plot_density_2D(sessions, by = "month", log = FALSE)
  ))
  expect_true(is.ggplot(
    plot_density_2D(sessions, by = "year", log = FALSE)
  ))
  expect_true(is.ggplot(
    plot_density_2D(sessions, by = "wday", log = TRUE)
  ))
  expect_true(is.ggplot(
    plot_density_2D(sessions, by = "month", log = TRUE)
  ))
  expect_true(is.ggplot(
    plot_density_2D(sessions, by = "year", log = TRUE)
  ))
})

test_that("Density 3D plots", {
  expect_true("plotly" %in% class(
    plot_density_3D(sessions, log = FALSE)
  ))
  expect_true("plotly" %in% class(
    plot_density_3D(sessions, log = TRUE)
  ))
})




# Test preprocessing ------------------------------------------------------
# Cut outlying sessions from threshold
test_that("The outliers are removed by cutting", {

  sessions2 <- cut_sessions(sessions, connection_start_min = 3, log = FALSE)
  expect_true(nrow(sessions2) < nrow(sessions))

  sessions2 <- cut_sessions(sessions, connection_start_max = 24, log = FALSE, start = 3)
  expect_true(nrow(sessions2) < nrow(sessions))

  sessions2 <- cut_sessions(sessions, connection_hours_max = 20, log = FALSE)
  expect_true(nrow(sessions2) < nrow(sessions))

  sessions2_log <- cut_sessions(sessions, connection_start_min = 1, log = TRUE)
  expect_true(nrow(sessions2_log) < nrow(sessions))

  sessions2_log <- cut_sessions(sessions, connection_hours_min = -2, log = TRUE)
  expect_true(nrow(sessions2_log) < nrow(sessions))

})


# kNN plot
test_that("kNN plots", {
  expect_true(is.ggplot(
    plot_kNNdist(sessions, log = FALSE)
  ))
  expect_true(is.ggplot(
    plot_kNNdist(sessions, log = TRUE)
  ))
})


# In the outliers detection function we depend on DBSCAN package
test_that("The outliers are detected properly with automatic MinPts and eps setting with log", {
  sessions_outliers <- sessions %>%
    head(1000) %>%
    detect_outliers(noise_th = 1, log = TRUE, MinPts = 200, eps = 0.66)
  expect_true(is.logical(sessions_outliers$Outlier))
})
test_that("The outliers are detected properly with automatic MinPts and eps setting", {
  sessions_outliers2 <- sessions %>%
    head(1000) %>%
    detect_outliers(noise_th = 1, log = FALSE, MinPts = 200, eps = 3.3)
  expect_true(is.logical(sessions_outliers2$Outlier))
})

# Outliers plots
test_that("Outliers plots", {
  sessions_outliers <- sessions %>%
    head(1000) %>%
    detect_outliers(noise_th = 1, log = TRUE, MinPts = 200, eps = 0.66)
  expect_true(is.ggplot(
    plot_outliers(sessions_outliers, log = TRUE)
  ))
  expect_true(is.ggplot(
    plot_outliers(sessions_outliers, log = FALSE)
  ))
})

# Remove outliers
test_that("The outliers are removed by filtering", {
  sessions_outliers <- sessions %>%
    head(1000) %>%
    detect_outliers(noise_th = 1, log = TRUE, MinPts = 200, eps = 0.66)
  sessions_outliers2 <- drop_outliers(sessions_outliers)
  expect_true(nrow(sessions_outliers2) < nrow(sessions_outliers))
})

# Disconnection day division lines
test_that("Disconnection day division lines plot", {
  expect_true(is.ggplot(
    sessions %>%
      head(1000) %>%
      plot_points() %>%
      plot_division_lines(n_lines = 1, division_hour = 10)
  ))
})


# Divisions by Disconnection day and Time-cycle
test_that("The divisions are done", {
  sessions_divided <- sessions %>%
    head(1000) %>%
    divide_by_disconnection(division_hour = 10) %>%
    divide_by_timecycle()
  expect_true(ncol(sessions_divided) > ncol(sessions))
})


# Test clustering ---------------------------------------------------------
# BIC plot
test_that("BIC plot is executed without errors", {
  skip_on_cran()
  expect_no_error(
    sessions %>%
      head(1000) %>%
      choose_k_GMM(k = 1:3)
  )
})

# Clustering iteration
test_that("Clustering iteration file is saved correctly",  {
  skip_on_cran()
  temp_file <- file.path(temp_dir, "iteration.pdf")
  sessions %>%
    head(1000) %>%
    save_clustering_iterations(2, 2, filename = temp_file)
  expect_true(file.exists(temp_file))
})

# In the clustering function we depend on MCLUST package
test_that("Clusers are found correctly with log", {
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  expect_equal(names(sessions_clusters), c("sessions", "models"))
  expect_true("Cluster" %in% names(sessions_clusters$sessions))
  expect_true(nrow(sessions_clusters$models) == 2) # Number of clusters == k
})

test_that("Clusers are found correctly without log", {
  sessions_clusters2 <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = FALSE)
  expect_equal(names(sessions_clusters2), c("sessions", "models"))
  expect_true("Cluster" %in% names(sessions_clusters2$sessions))
  expect_true(nrow(sessions_clusters2$models) == 2) # Number of clusters == k
})

test_that("Clusers are plotted correctly", {3
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  plot_clusters <- plot_bivarGMM(sessions_clusters$sessions, sessions_clusters$models, log = FALSE)
  plot_clusters_log <- plot_bivarGMM(sessions_clusters$sessions, sessions_clusters$models, log = TRUE)
  expect_true(is.ggplot(plot_clusters))
  expect_true(is.ggplot(plot_clusters_log))
})


# Test profiling ----------------------------------------------------------
test_that("Profiles are identified through cluster definitions", {
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  clusters_definition <- define_clusters(
    sessions_clusters$models,
    interpretations = c("Morning sessions", "Afternoon sessions"),
    profile_names = c("Morning", "Afternoon")
  )
  sessions_profiles <- set_profiles(list(sessions_clusters$sessions), list(clusters_definition))
  expect_true(is.data.frame(sessions_profiles))
})


# Test modelling ----------------------------------------------------------
test_that("Get the connection models", {
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  clusters_definition <- define_clusters(
    sessions_clusters$models,
    interpretations = c("Morning sessions", "Afternoon sessions"),
    profile_names = c("Morning", "Afternoon")
  )
  connection_GMM <- get_connection_models(list(sessions_clusters), list(clusters_definition))
  expect_true(is.data.frame(connection_GMM))
  expect_true(all.equal(c("profile", "ratio", "connection_models"), names(connection_GMM)))
  expect_true(all.equal(c("mu", "sigma", "ratio"), names(connection_GMM$connection_models[[1]])))
})

test_that("Tables and plot of connection GMM are generated without errors", {
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  clusters_definition <- define_clusters(
    sessions_clusters$models,
    interpretations = c("Morning sessions", "Afternoon sessions"),
    profile_names = c("Morning", "Afternoon")
  )
  connection_GMM <- get_connection_models(list(sessions_clusters), list(clusters_definition))
  expect_no_error(
    print_connection_models_table(connection_GMM, full_width = TRUE, label = "tab:conn", caption = "connection GMM")
  )
  expect_true(is.ggplot(
    plot_model_clusters(list(sessions_clusters), list(clusters_definition), connection_GMM)
  ))
})


test_that("Get and plot the energy models with `by_power = FALSE`", {
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  clusters_definition <- define_clusters(
    sessions_clusters$models,
    interpretations = c("Morning sessions", "Afternoon sessions"),
    profile_names = c("Morning", "Afternoon")
  )
  sessions_profiles <- set_profiles(list(sessions_clusters$sessions), list(clusters_definition))
  energy_GMM <- get_energy_models(sessions_profiles, log = TRUE, by_power = FALSE)
  expect_true(is.data.frame(energy_GMM))
  expect_true(all.equal(c("profile", "energy_models"), names(energy_GMM)))
  expect_true(all.equal(c("charging_rate", "energy_models", "mclust"), names(energy_GMM$energy_models[[1]])))
  expect_true(all.equal(c("mu", "sigma", "ratio"), names(energy_GMM$energy_models[[1]]$energy_models[[1]])))
})

test_that("Tables and plot of energy GMM are generated without errors", {
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  clusters_definition <- define_clusters(
    sessions_clusters$models,
    interpretations = c("Morning sessions", "Afternoon sessions"),
    profile_names = c("Morning", "Afternoon")
  )
  sessions_profiles <- set_profiles(list(sessions_clusters$sessions), list(clusters_definition))
  energy_GMM <- get_energy_models(sessions_profiles, log = TRUE, by_power = FALSE)
  expect_no_error(
    print_user_profile_energy_models_table(energy_GMM$energy_models[[1]], full_width = TRUE, label = "tab:en", caption = "energy GMM")
  )
  expect_true(is.ggplot(
    plot_energy_models(energy_GMM)
  ))
})


test_that("Get and plot the energy models with `by_power = TRUE`", {
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  clusters_definition <- define_clusters(
    sessions_clusters$models,
    interpretations = c("Morning sessions", "Afternoon sessions"),
    profile_names = c("Morning", "Afternoon")
  )
  sessions_profiles <- set_profiles(list(sessions_clusters$sessions), list(clusters_definition))
  sessions_profiles <- sessions_profiles %>%
    mutate(Power = round_to_interval(Power, 3.7)) %>%
    filter(Power < 11)
  sessions_profiles$Power[sessions_profiles$Power == 0] <- 3.7

  energy_GMM <- get_energy_models(sessions_profiles, log = TRUE, by_power = TRUE)
  expect_true(is.data.frame(energy_GMM))
  expect_true(all.equal(c("profile", "energy_models"), names(energy_GMM)))
  expect_true(all.equal(c("charging_rate", "energy_models", "mclust"), names(energy_GMM$energy_models[[1]])))
  expect_true(all.equal(c("mu", "sigma", "ratio"), names(energy_GMM$energy_models[[1]]$energy_models[[1]])))

  energy_plot <- plot_energy_models(energy_GMM)
  expect_true(is.ggplot(energy_plot))
})

test_that("Model file is created and saved correctly",  {
  skip_on_cran()
  sessions_clusters <- sessions %>%
    head(1000) %>%
    cluster_sessions(k = 2, seed = 123, log = TRUE)
  clusters_definition <- define_clusters(
    sessions_clusters$models,
    interpretations = c("Morning sessions", "Afternoon sessions"),
    profile_names = c("Morning", "Afternoon")
  )
  sessions_profiles <- set_profiles(list(sessions_clusters$sessions), list(clusters_definition))

  connection_GMM <- get_connection_models(list(sessions_clusters), list(clusters_definition))
  energy_GMM <- get_energy_models(sessions_profiles, log = TRUE, by_power = FALSE)

  evmodel <- get_ev_model(
    names = c("Weekday", "Weekend"),
    months_lst = list(1:12),
    wdays_lst = list(1:5, 6:7),
    connection_GMM = list(connection_GMM, connection_GMM),
    energy_GMM = list(energy_GMM, energy_GMM),
    connection_log = T, energy_log = T,
    data_tz = "America/Los_Angeles"
  )

  temp_model_file <- file.path(temp_dir, "model.json")
  save_ev_model(evmodel, file = temp_model_file)

  expect_true(file.exists(temp_model_file))
})

test_that("Model file is read correctly",  {
  skip_on_cran()
  temp_model_file <- file.path(temp_dir, "model.json")
  evmodel <- read_ev_model(file = temp_model_file)

  expect_true(class(evmodel) == "evmodel")
})

