#  File R/godfather.R in package ergm, part of the Statnet suite
#  of packages for network analysis, https://statnet.org .
#
#  This software is distributed under the GPL-3 license.  It is free,
#  open source, and has the attribution requirements (GPL Section 7) at
#  https://statnet.org/attribution
#
#  Copyright 2003-2019 Statnet Commons
#######################################################################
#=========================================================================
# This file contains the following 2 functions for computing changestat
# summaries of dynamic networks ??
#   <ergm.godfather>
#   <control.godfather>
#=========================================================================



#' A function to apply a given series of changes to a network.
#' 
#' Gives the network a series of proposals it can't refuse. Returns the
#' statistics of the network, and, optionally, the final network.
#' 
#' 
#' @param formula An \code{\link{ergm}}-style formula, with a
#'   \code{\link{network}} on its LHS.
#' @param changes Either a matrix with three columns: tail, head, and
#'   new value, describing the changes to be made; or a list of such
#'   matrices to apply these changes in a sequence. For binary network
#'   models, the third column may be omitted. In that case, the
#'   changes are treated as toggles. Note that if a list is passed, it
#'   must either be all of changes or all of toggles.
#' @template response
#' @param end.network Whether to return a network that
#'   results. Defaults to \code{FALSE}.
#' @param stats.start Whether to return the network statistics at
#'   \code{start} (before any changes are applied) as the first row of
#'   the statistics matrix.  Defaults to \code{FALSE}, to produce
#'   output similar to that of \code{\link[=simulate.ergm]{simulate}}
#'   for ERGMs when \code{output="stats"}, where initial network's
#'   statistics are not returned.
#' @param changes.only Whether to return network statistics or only
#'   their changes relative to the initial network.
#' @param verbose Whether to print progress messages.
#' @param control A control list generated by
#'   \code{\link{control.ergm.godfather}}.
#' @return If \code{end.network==FALSE} (the default), an
#'   \code{\link{mcmc}} object with the requested network statistics
#'   associed with the network series produced by applying the
#'   specified changes. Its \code{\link{mcmc}} attributes encode the
#'   timing information: so \code{\link{start}(out)} gives the time
#'   point associated with the first row returned, and
#'   \code{\link{end}(out)} out the last. The "thinning interval" is
#'   always 1.
#' 
#' If \code{end.network==TRUE}, return a \code{\link{network}} object,
#' representing the final network, with a matrix of statistics
#' described in the previous paragraph attached to it as an
#' \code{attr}-style attribute \code{"stats"}.
#' @seealso [tergm::tergm.godfather()], [simulate.ergm()],
#'   [simulate.formula()]
#' @examples
#' data(florentine)
#' ergm.godfather(flomarriage~edges+absdiff("wealth")+triangles,
#'                changes=list(cbind(1:2,2:3),
#'                             cbind(3,5),
#'                             cbind(3,5),
#'                             cbind(1:2,2:3)),
#'                stats.start=TRUE)
#' @export ergm.godfather
ergm.godfather <- function(formula, changes=NULL, response=NULL,
                           end.network=FALSE,
                           stats.start=FALSE,
                           changes.only=FALSE,
                           verbose=FALSE,
                           control=control.ergm.godfather()){
  check.control.class("ergm.godfather", "ergm.godfather")

  if(!is.list(changes)) changes <- list(changes)

  nw <- ergm.getnetwork(formula)

  ncols <- sapply(changes, ncol)
  if(!all_identical(ncols) || ncols[1]<2 || ncols[1]>3 || (!is.null(response)&&ncols[1]==2)) abort("Invalid format for list of changes. See help('ergm.godfather').")

  if(!is.directed(nw)) changes <- lapply(changes, function(x){
    x[,1:2] <- t(apply(x[,1:2,drop=FALSE], 1, sort))
  })

  m <- ergm_model(formula, nw, role="target", response=response, term.options=control$term.options)
  Clist <- ergm.Cprepare(nw, m, response=response)
  m$obs <- if(changes.only) numeric(nparam(m, canonical=TRUE))
           else summary(m, nw, response=response)

  changem <- changes %>% map(rbind, 0) %>% do.call(rbind, .) # 0s are sentinels indicating next iteration.
  
  if(end.network){
    p <- 1-summary(nw~density)
    newedges.m <- p*nrow(changem)
    newedges.sd <- sqrt(p*(1-p)*nrow(changem))
    maxedges <- Clist$nedges + newedges.m + newedges.sd*control$GF.init.maxedges.mul
  }

  
  if(verbose) message_print("Applying changes...\n")
  repeat{
    if(is.null(response)){
      z <- .C("Godfather_wrapper",
              as.integer(Clist$nedges), as.integer(Clist$tails), as.integer(Clist$heads),
              as.integer(Clist$n),
              as.integer(Clist$dir), as.integer(Clist$bipartite),
              as.integer(Clist$nterms), 
              as.character(Clist$fnamestring),
              as.character(Clist$snamestring),
              as.double(Clist$inputs),
              as.integer(nrow(changem) * if(ncol(changem)==3) +1 else -1), as.integer(changem[,1]),
              as.integer(changem[,2]), if(ncol(changem)==3) as.integer(changem[,3]) else integer(0),
              s = double((1+length(changes)) * Clist$nstats),
              if(end.network) as.integer(maxedges) else as.integer(0),
              newnwtails = if(end.network) integer(maxedges+1) else integer(0),
              newnwheads = if(end.network) integer(maxedges+1) else integer(0),
              as.integer(verbose),
              status = integer(1), # 0 = OK, TOO_MANY_EDGES = 1
              PACKAGE="ergm")
    }else{
      z <- .C("WtGodfather_wrapper",
              as.integer(Clist$nedges), as.integer(Clist$tails), as.integer(Clist$heads), as.double(Clist$weights),
              as.integer(Clist$n),
              as.integer(Clist$dir), as.integer(Clist$bipartite),
              as.integer(Clist$nterms), 
              as.character(Clist$fnamestring),
              as.character(Clist$snamestring),
              as.double(Clist$inputs),
              as.integer(nrow(changem)), as.integer(changem[,1]), as.integer(changem[,2]), as.double(changem[,3]),
              s = double((1+length(changes)) * Clist$nstats),
              if(end.network) as.integer(maxedges) else as.integer(0),
              newnwtails = if(end.network) integer(maxedges+1) else integer(0),
              newnwheads = if(end.network) integer(maxedges+1) else integer(0),
              newnwweights = if(end.network) double(maxedges+1) else double(0),
              as.integer(verbose),
              status = integer(1), # 0 = OK, TOO_MANY_EDGES = 1
              PACKAGE="ergm")
    }

    if(z$status==0) break;
    if(z$status==1){
      maxedges <- 5*maxedges
      if(verbose>0) message("Too many edges encountered in the simulation. Increasing capacity to ", maxedges, ".")
    }
  }

  stats <- matrix(z$s, ncol=Clist$nstats, byrow=TRUE)
  stats <- t(t(apply(stats,2,cumsum)) + m$obs)
  
  colnames(stats) <- param_names(m, canonical=TRUE)
  if(!stats.start) stats <- stats[-1,,drop=FALSE]
  #' @importFrom coda mcmc
  stats <- mcmc(stats)
  
  if(end.network){ 
    if(verbose) cat("Creating new network...\n")
    newnetwork <- as.network(pending_update_network(nw,z,response=response))
    attr(newnetwork,"stats")<-stats
    newnetwork
  }else stats
}




####################################################################
# The <control.godfather> function allows for tuning of the
# <ergm.godfather> function
#
# --PARAMETERS--
#   maxedges          : the maximum number of edges to make space
#                       for for the new network; this is ignored
#                       if 5*Clist$nedges is greater; this is also
#                       ignored if 'return_new_network' is FALSE;
#                       default=100000
#
#
# --RETURNED--
#   a list of the above parameters
#
####################################################################
#' Control parameters for [ergm.godfather()].
#'
#' Returns a list of its arguments.
#'
#' @param GF.init.maxedges.mul How much space
#'   is allocated for the edgelist of the final network. It is used
#'   adaptively, so should not be greater than \code{10}.
#' @template term_options
#' 
#' @export control.ergm.godfather
control.ergm.godfather<-function(GF.init.maxedges.mul=5,
                                 term.options=NULL
              ){
    control<-list()
    for(arg in names(formals(sys.function())))
      control[arg]<-list(get(arg))

    set.control.class("control.ergm.godfather")
}
