#' predict.tsd Estimate sex ratio according to constant incubation temperature
#' @title Estimate sex ratio according to constant incubation temperature
#' @author Marc Girondot
#' @return A data.frame with informations about sex-ratio
#' @param object A result file generated by tsd
#' @param temperatures A vector of temperatures
#' @param durations A vector of durations
#' @param SE The standard error for temperatures or durations
#' @param quantiles The quantiles to be returned, default=c(0.025, 0.5, 0.975)
#' @param replicates Number of replicates to estimate CI
#' @param progressbar Logical. Does a progression bar must be shown
#' @param ... Not used
#' @description Estimate sex ratio according to constant incubation temperature\cr
#' The data.frame has the following components:\cr
#' temperatures, SE, sexratio, CI.minus.sexratio, CI.plus.sexratio, range.CI\cr
#' Note that incubation duration is a very bad proxy for sex ratio. See Georges, 
#' A., Limpus, C. J. & Stoutjesdijk, R. 1994. Hatchling sex in the marine turtle 
#' Caretta caretta is determined by proportion of development at a temperature, 
#' not daily duration of exposure. J. Exp. Zool., 270, 432-444.
#' @family Functions for temperature-dependent sex determination
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' m <- c(10, 14, 7, 4, 3, 0, 0)
#' f <- c(0, 1, 2, 4, 15, 10, 13)
#' t <- c(25, 26, 27, 28, 29, 30, 31)
#' result <- tsd(males=m, females=f, temperatures=t)
#' plot(result)
#' predict(result, temperatures=c(25, 31))
#' predict(result, temperatures=c(25, 31), SE=c(1, 2))
#' d <- c(72, 70, 65, 63, 62, 60, 59)
#' result <- tsd(males=m, females=f, durations=d)
#' predict(result, durations=c(67, 68))
#' }
#' @method predict tsd
#' @export


predict.tsd <-
function(object, temperatures=NULL, durations=NULL, SE=NULL, 
	quantiles=c(0.025, 0.5, 0.975), replicates=1000, progressbar=FALSE, ...) {
  
  # durations=NULL; SE=NULL; quantiles=c(0.025, 0.5, 0.975); replicates=1000; progressbar=FALSE
  
  temper <- TRUE
  if (is.null(temperatures) & !is.null(durations)) {
    temperatures <- durations
    temper <- FALSE
  }
  
  if (is.null(temperatures)) temperatures <- object$temperatures
  
  if (!is.null(SE) & length(SE) != length(temperatures)) {
    warning("SE must be of the same length than temperatures or durations")
    return()
  }
   
  x <- object
	equation <- x$equation
	par <- c(x$par, x$fixed.parameters)
	res <- x$SE

    if (progressbar) pb<-txtProgressBar(min=1, max=length(temperatures), style=3)

    
    rep <- replicates-1
    df_par <- data.frame(P=unname(c(par["P"], rnorm(rep, par["P"], ifelse(is.na(res["P"]), 0, res["P"])))), 
                         S=unname(c(par["S"], rnorm(rep, par["S"], ifelse(is.na(res["S"]), 0, res["S"])))),
                         K=unname(ifelse(is.na(par["K"]), rep(NA, replicates), c(par["K"], rnorm(rep, par["K"], ifelse(is.na(res["K"]), 0, res["K"]))))),
                         K1=unname(ifelse(is.na(par["K1"]), rep(NA, replicates), c(par["K1"], rnorm(rep, par["K1"], ifelse(is.na(res["K1"]), 0, res["K1"]))))),
                         K2=unname(ifelse(is.na(par["K2"]), rep(NA, replicates), c(par["K2"], rnorm(rep, par["K2"], ifelse(is.na(res["K2"]), 0, res["K2"]))))))

    if (is.null(SE)) {
      SE <- rep(0, length(temperatures))
      replicates <- 1
    }
    
    df.out <- data.frame(temperatures=numeric(), 
                        SE=numeric(),
                        sexratio=numeric(), 
                        CI.minus.sexratio=numeric(), 
                        CI.plus.sexratio=numeric(), 
                        range.CI=numeric())
    
    for(i in 1:length(temperatures)) {
      if (progressbar) setTxtProgressBar(pb, i)
        temperatures_ec <- c(temperatures[i], rnorm(replicates-1, temperatures[i], SE[i]))
        out_tsd_plot <- apply(df_par, 1, function(par) {
        p <- getFromNamespace(".modelTSD", ns="embryogrowth")(par, temperatures_ec, equation)
        return(list(sr=p))
      }
      )
      
      out_tsd3_plot <- unlist(out_tsd_plot)
      outquant <- quantile(out_tsd3_plot, probs = quantiles)
      nm <- paste0("q", gsub("%", "", names(outquant)))
 #     outquant <- rbind(outquant, mean=out_tsd3_plot[1,], temperatures=temperatures_ec)
      df.out <- rbind(df.out, data.frame(temperatures=as.numeric(temperatures[i]), 
                                SE=as.numeric(SE[i]),
                                sexratio=as.numeric(out_tsd3_plot[1]),
                                matrix(outquant, nrow=1, dimnames =list(NULL, nm))
      )
                                )
      
    }

	if (!x$males.freq) {
	  df.out$sexratio <- 1-df.out$sexratio
	  df.out[, nm] <- 1-df.out[, nm]
	}
if (!temper) colnames(df.out)[1] <- "durations"
return(df.out)
}
