% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rank_effectsizes.R
\name{rank_biserial}
\alias{rank_biserial}
\alias{cliffs_delta}
\alias{rank_epsilon_squared}
\alias{kendalls_w}
\title{Effect size for non-parametric (rank sum) tests}
\usage{
rank_biserial(
  x,
  y = NULL,
  data = NULL,
  mu = 0,
  ci = 0.95,
  alternative = "two.sided",
  paired = FALSE,
  verbose = TRUE,
  ...,
  iterations
)

cliffs_delta(
  x,
  y = NULL,
  data = NULL,
  mu = 0,
  ci = 0.95,
  alternative = "two.sided",
  verbose = TRUE,
  ...
)

rank_epsilon_squared(
  x,
  groups,
  data = NULL,
  ci = 0.95,
  alternative = "greater",
  iterations = 200,
  ...
)

kendalls_w(
  x,
  groups,
  blocks,
  data = NULL,
  ci = 0.95,
  alternative = "greater",
  iterations = 200,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{Can be one of:
\itemize{
\item A numeric vector, or a character name of one in \code{data}.
\item A formula in to form of \code{DV ~ groups} (for \code{rank_biserial()} and
\code{rank_epsilon_squared()}) or \code{DV ~ groups | blocks} (for \code{kendalls_w()};
See details for the \code{blocks} and \code{groups} terminology used here).
\item A list of vectors (for \code{rank_epsilon_squared()}).
\item A matrix of \verb{blocks x groups} (for \code{kendalls_w()}). See details for the
\code{blocks} and \code{groups} terminology used here.
}}

\item{y}{An optional numeric vector of data values to compare to \code{x}, or a
character name of one in \code{data}. Ignored if \code{x} is not a vector.}

\item{data}{An optional data frame containing the variables.}

\item{mu}{a number indicating the value around which (a-)symmetry (for
one-sample or paired samples) or shift (for independent samples) is to be
estimated. See \link[stats:wilcox.test]{stats::wilcox.test}.}

\item{ci}{Confidence Interval (CI) level}

\item{alternative}{a character string specifying the alternative hypothesis;
Controls the type of CI returned: \code{"two.sided"} (two-sided CI; default for
rank-biserial correlation and Cliff's \emph{delta}), \code{"greater"} (default for
rank epsilon squared and Kendall's \emph{W}) or \code{"less"} (one-sided CI). Partial
matching is allowed (e.g., \code{"g"}, \code{"l"}, \code{"two"}...). See \emph{One-Sided CIs}
in \link{effectsize_CIs}.}

\item{paired}{If \code{TRUE}, the values of \code{x} and \code{y} are considered as paired.
This produces an effect size that is equivalent to the one-sample effect
size on \code{x - y}.}

\item{verbose}{Toggle warnings and messages on or off.}

\item{...}{Arguments passed to or from other methods. When \code{x} is a formula,
these can be \code{subset} and \code{na.action}.}

\item{iterations}{The number of bootstrap replicates for computing confidence
intervals. Only applies when \code{ci} is not \code{NULL}. (Deprecated for
\code{rank_biserial()}).}

\item{groups, blocks}{A factor vector giving the group / block for the
corresponding elements of \code{x}, or a character name of one in \code{data}.
Ignored if \code{x} is not a vector.}
}
\value{
A data frame with the effect size (\code{r_rank_biserial},
\code{rank_epsilon_squared} or \code{Kendalls_W}) and its CI (\code{CI_low} and
\code{CI_high}).
}
\description{
Compute the rank-biserial correlation (\eqn{r_{rb}}{r_rb}), Cliff's \emph{delta}
(\eqn{\delta}), rank epsilon squared (\eqn{\varepsilon^2}{\epsilon^2}), and
Kendall's \emph{W} effect sizes for non-parametric (rank sum) tests.
}
\details{
The rank-biserial correlation is appropriate for non-parametric tests of
differences - both for the one sample or paired samples case, that would
normally be tested with Wilcoxon's Signed Rank Test (giving the
\strong{matched-pairs} rank-biserial correlation) and for two independent samples
case, that would normally be tested with Mann-Whitney's \emph{U} Test (giving
\strong{Glass'} rank-biserial correlation). See \link[stats:wilcox.test]{stats::wilcox.test}. In both
cases, the correlation represents the difference between the proportion of
favorable and unfavorable pairs / signed ranks (Kerby, 2014). Values range
from \code{-1} (\emph{all} values of the second sample are larger than \emph{all} the values
of the first sample) to \code{+1} (\emph{all} values of the second sample are smaller
than \emph{all} the values of the first sample). Cliff's \emph{delta} is an alias to
the rank-biserial correlation in the two sample case.
\cr\cr
The rank epsilon squared is appropriate for non-parametric tests of
differences between 2 or more samples (a rank based ANOVA). See
\link[stats:kruskal.test]{stats::kruskal.test}. Values range from 0 to 1, with larger values
indicating larger differences between groups.
\cr\cr
Kendall's \emph{W} is appropriate for non-parametric tests of differences between
2 or more dependent samples (a rank based rmANOVA), where each \code{group} (e.g.,
experimental condition) was measured for each \code{block} (e.g., subject). This
measure is also common as a measure of reliability of the rankings of the
\code{groups} between raters (\code{blocks}). See \link[stats:friedman.test]{stats::friedman.test}. Values range
from 0 to 1, with larger values indicating larger differences between groups
/ higher agreement between raters.
\subsection{Ties}{

When tied values occur, they are each given the average of the ranks that
would have been given had no ties occurred. This results in an effect size of
reduced magnitude. A correction has been applied for Kendall's \emph{W}.
}
}
\section{Confidence Intervals}{
Confidence intervals for the rank-biserial correlation (and Cliff's \emph{delta})
are estimated using the normal approximation (via Fisher's transformation).
Confidence intervals for rank Epsilon squared, and Kendall's \emph{W} are
estimated using the bootstrap method (using the \code{{boot}} package).
}

\examples{
\donttest{
data(mtcars)
mtcars$am <- factor(mtcars$am)
mtcars$cyl <- factor(mtcars$cyl)

# Rank Biserial Correlation
# =========================

# Two Independent Samples ----------
(rb <- rank_biserial(mpg ~ am, data = mtcars))
# Same as:
# rank_biserial("mpg", "am", data = mtcars)
# rank_biserial(mtcars$mpg[mtcars$am=="0"], mtcars$mpg[mtcars$am=="1"])

# More options:
rank_biserial(mpg ~ am, data = mtcars, mu = -5)
print(rb, append_CLES = TRUE)


# One Sample ----------
rank_biserial(wt ~ 1, data = mtcars, mu = 3)
# same as:
# rank_biserial("wt", data = mtcars, mu = 3)
# rank_biserial(mtcars$wt, mu = 3)


# Paired Samples ----------
dat <- data.frame(Cond1 = c(1.83, 0.5, 1.62, 2.48, 1.68, 1.88, 1.55, 3.06, 1.3),
                  Cond2 = c(0.878, 0.647, 0.598, 2.05, 1.06, 1.29, 1.06, 3.14, 1.29))
(rb <- rank_biserial(Pair(Cond1, Cond2) ~ 1, data = dat, paired = TRUE))

# same as:
# rank_biserial(dat$Cond1, dat$Cond2, paired = TRUE)

interpret_rank_biserial(0.78)
interpret(rb, rules = "funder2019")


# Rank Epsilon Squared
# ====================

rank_epsilon_squared(mpg ~ cyl, data = mtcars)



# Kendall's W
# ===========
dat <- data.frame(cond = c("A", "B", "A", "B", "A", "B"),
                  ID = c("L", "L", "M", "M", "H", "H"),
                  y = c(44.56, 28.22, 24, 28.78, 24.56, 18.78))
(W <- kendalls_w(y ~ cond | ID, data = dat, verbose = FALSE))

interpret_kendalls_w(0.11)
interpret(W, rules = "landis1977")
}

}
\references{
\itemize{
\item Cureton, E. E. (1956). Rank-biserial correlation. Psychometrika, 21(3),
287-290.
\item Glass, G. V. (1965). A ranking variable analogue of biserial correlation:
Implications for short-cut item analysis. Journal of Educational Measurement,
2(1), 91-95.
\item Kendall, M.G. (1948) Rank correlation methods. London: Griffin.
\item Kerby, D. S. (2014). The simple difference formula: An approach to teaching
nonparametric correlation. Comprehensive Psychology, 3, 11-IT.
\item King, B. M., & Minium, E. W. (2008). Statistical reasoning in the
behavioral sciences. John Wiley & Sons Inc.
\item Cliff, N. (1993). Dominance statistics: Ordinal analyses to answer ordinal
questions. Psychological bulletin, 114(3), 494.
\item Tomczak, M., & Tomczak, E. (2014). The need to report effect size estimates
revisited. An overview of some recommended measures of effect size.
}
}
\seealso{
Other effect size indices: 
\code{\link{cles}()},
\code{\link{cohens_d}()},
\code{\link{effectsize.BFBayesFactor}()},
\code{\link{eta_squared}()},
\code{\link{phi}()}
}
\concept{effect size indices}
