% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PF.R
\name{PF_forward_filter}
\alias{PF_forward_filter}
\alias{PF_forward_filter.PF_EM}
\alias{PF_forward_filter.formula}
\alias{PF_forward_filter.data.frame}
\title{Forward Particle Filter}
\usage{
PF_forward_filter(x, N_fw, N_first, ...)

\method{PF_forward_filter}{PF_EM}(x, N_fw, N_first, seed, ...)

\method{PF_forward_filter}{formula}(x, N_fw, N_first, data,
  model = "logit", by, max_T, id, a_0, Q_0, Q, fixed_effects,
  control = PF_control(...), seed = NULL, trace = 0, G, theta, J, K,
  psi, phi, type = "RW", Fmat, ...)

\method{PF_forward_filter}{data.frame}(x, N_fw, N_first, formula,
  model = "logit", by, max_T, id, a_0, Q_0, Q, fixed_effects,
  control = PF_control(...), seed = NULL, trace = 0, fixed = NULL,
  random = NULL, G, theta, J, K, psi, phi, type = "RW", Fmat,
  order = 1, ...)
}
\arguments{
\item{x}{an \code{PF_EM} or \code{formula} object.}

\item{N_fw}{number of particles.}

\item{N_first}{number of time zero particles to draw.}

\item{...}{optional way to pass arguments to \code{control}.}

\item{seed}{\code{.GlobalEnv$.Random.seed} to set. Not \code{seed} as in
\code{\link{set.seed}} function. Can be used with the
\code{\link{.Random.seed}} returned by \code{\link{PF_EM}}.}

\item{data}{\code{data.frame} or environment containing the outcome and covariates.}

\item{model}{either \code{'logit'} for binary outcomes with the logistic
link function, \code{'cloglog'} for binary outcomes with the inverse cloglog
link function, or \code{'exponential'} for piecewise constant exponential distributed arrival times.}

\item{by}{interval length of the bins in which parameters are fixed.}

\item{max_T}{end of the last interval interval.}

\item{id}{vector of ids for each row of the in the design matrix.}

\item{a_0}{vector \eqn{a_0} for the initial coefficient vector for the first iteration (optional). Default is estimates from static model (see \code{\link{static_glm}}).}

\item{Q_0}{covariance matrix for the prior distribution.}

\item{Q}{initial covariance matrix for the state equation.}

\item{fixed_effects}{values for the fixed parameters.}

\item{control}{see \code{\link{PF_control}}.}

\item{trace}{argument to get progress information. Zero will yield no info and larger integer values will yield incrementally more information.}

\item{G, theta, J, K, psi, phi}{parameters for a restricted \code{type = "VAR"} model.
See the vignette mentioned in 'Details' of \code{\link{PF_EM}} and the
examples linked to in 'See Also'.}

\item{type}{type of state model. Either \code{"RW"} for a [R]andom [W]alk or
"VAR" for [V]ector [A]uto[R]egression.}

\item{Fmat}{starting value for \eqn{F} when \code{type = "VAR"}. See
'Details' in \code{\link{PF_EM}}.}

\item{formula}{\code{\link[survival]{coxph}} like formula with \code{\link[survival]{Surv}(tstart, tstop, event)} on the left hand site of \code{~}.}

\item{fixed}{two-sided \code{\link{formula}} to be used
with \code{random} instead of \code{formula}. It is of the form
\code{Surv(tstart, tstop, event) ~ x} or
\code{Surv(tstart, tstop, event) ~ - 1} for no fixed effects.}

\item{random}{one-sided \code{\link{formula}} to be used
with \code{fixed} instead of \code{formula}. It is of the form
\code{~ z}.}

\item{order}{order of the random walk.}
}
\value{
An object of class \code{PF_clouds}.
}
\description{
Functions to only use the forward particle filter. Useful for log-likelihood
evaluation though there is an \eqn{O(d^2)} variance of the estimate where \eqn{d} is the number of time
periods. The number of particles specified in the \code{control} argument
has no effect.

The function does not alter the \code{\link{.Random.seed}} to make sure the
same \code{rng.kind} is kept after the call. See \code{\link{PF_EM}} for
model details.
}
\section{Methods (by class)}{
\itemize{
\item \code{PF_EM}: Forward particle filter using the
estimates of an \code{\link{PF_EM}} call.

\item \code{formula}: Forward particle filter with formula input.

\item \code{data.frame}: Forward particle filter with \code{data.frame}
data input as \code{x} instead of \code{data}. Can be used with \code{fixed}
and \code{random} argument.
}}

\section{Warning}{

The function is still under development so the output and API may change.
}

\examples{
\dontrun{
# head-and-neck cancer study data. See Efron, B. (1988) doi:10.2307/2288857
is_censored <- c(
  6, 27, 34, 36, 42, 46, 48:51, 51 + c(15, 30:28, 33, 35:37, 39, 40, 42:45))
head_neck_cancer <- data.frame(
  id = 1:96,
  stop = c(
    1, 2, 2, rep(3, 6), 4, 4, rep(5, 8),
    rep(6, 7), 7, 8, 8, 8, 9, 9, 10, 10, 10, 11, 14, 14, 14, 15, 18, 18, 20,
    20, 37, 37, 38, 41, 45, 47, 47,
    2, 2, 3, rep(4, 4), rep(5, 5), rep(6, 5),
    7, 7, 7, 9, 10, 11, 12, 15, 16, 18, 18, 18, 21,
    21, 24, 25, 27, 36, 41, 44, 52, 54, 59, 59, 63, 67, 71, 76),
  event = !(1:96 \%in\% is_censored),
  group = factor(c(rep(1, 45 + 6), rep(2, 45))))

# fit model
set.seed(61364778)
ctrl <- PF_control(
  N_fw_n_bw = 500, N_smooth = 2500, N_first = 2000,
  n_max = 1, # set to one as an example
  n_threads = max(parallel::detectCores(logical = FALSE), 1),
  eps = .001, Q_tilde = as.matrix(.3^2), est_a_0 = FALSE)
pf_fit <- suppressWarnings(
  PF_EM(
    survival::Surv(stop, event) ~ ddFixed(group),
    data = head_neck_cancer, by = 1, Q_0 = 1, Q = 0.1^2, control = ctrl,
    max_T = 30))

# the log-likelihood in the final iteration
(end_log_like <- logLik(pf_fit))

# gives the same
fw_ps <- PF_forward_filter(
  survival::Surv(stop, event) ~ ddFixed(group), N_fw = 500, N_first = 2000,
  data = head_neck_cancer, by = 1, Q_0 = 1, Q = 0.1^2,
  a_0 = pf_fit$a_0, fixed_effects = -0.5370051,
  control = ctrl, max_T = 30, seed = pf_fit$seed)
all.equal(c(end_log_like), c(logLik(fw_ps)))

# will differ since we use different number of particles
fw_ps <- PF_forward_filter(
  survival::Surv(stop, event) ~ ddFixed(group), N_fw = 1000, N_first = 3000,
  data = head_neck_cancer, by = 1, Q_0 = 1, Q = 0.1^2,
  a_0 = pf_fit$a_0, fixed_effects = -0.5370051,
  control = ctrl, max_T = 30, seed = pf_fit$seed)
all.equal(c(end_log_like), c(logLik(fw_ps)))

# will differ since we use the final estimates
fw_ps <- PF_forward_filter(pf_fit, N_fw = 500, N_first = 2000)
all.equal(c(end_log_like), c(logLik(fw_ps)))
}
}
