% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dsrrec.R
\name{dsrrec}
\alias{dsrrec}
\title{Compute Directly Standardized Rates for Recurrent Events}
\usage{
dsrrec(data, event, fu, subgroup, ..., refdata, sig = 0.95, mp, decimals)
}
\arguments{
\item{data}{A data frame with counts and unit-times summarized by the standardization variables.}

\item{event}{A variable within the input data that corresponds to the event counts.}

\item{fu}{A variable within the input data that corresponds to the unit-time.}

\item{subgroup}{A variable within the input data frame for which rates are calculated by.}

\item{...}{Variables(s) within the input data that for which rates are to be standardized by. The input data and ref data should both be summarized by these.}

\item{refdata}{A data frame with population unit-times summarized by the standardization variables. The unit-time variable name must named pop.}

\item{sig}{The desired level of confidence in computing confidence intervals. The default is 0.95 for 95 percent CIs.}

\item{mp}{A constant to multiply rates by (e.g. mp=1000 for rates per 1000).}

\item{decimals}{Round estimates to a desired decimal place.}
}
\description{
Computes directly standardized rates for recurrent events by subgroup with confidence intervals.
}
\examples{
#An example of directly standardized rates for recurrent events

library(frailtypack)
library(dplyr)
library(dsr)
data(readmission)

#Make an individual level dataset with total event counts and total observation times
treadm <- as.data.frame(readmission \%>\%
                         group_by(id) \%>\%
                         filter(max(enum)==enum ) \%>\%
                         mutate(events=enum-1, time=t.stop) \%>\%
                         select(id, events, time, sex, dukes))

#Make the standard pop
tref <- as.data.frame(treadm \%>\%
                     group_by(sex) \%>\%
                     mutate(pop=sum(time)) \%>\%
                     select(sex, pop) \%>\%
                     distinct(sex, pop))

#Get directly standardized rates (age-adjusted) for readmissions by Dukes' tumor grade.
analysis <- dsrrec(data=treadm,
                  event=events,
                  fu=time,
                  refdata=tref,
                  subgroup=dukes,
                  sex,
                  mp=1000,
                  decimals=3)
}
\references{
Stukel, T. A., Glynn, R. J., Fisher, E. S., Sharp, S. M., Lu-Yao, G and Wennberg, J. E. (1994). Standardized rates of recurrent outcomes. Statistics in Medicine, 13, 1781-1791.

Fay, M.P., & Feuer, E.J. (1997). Confidence intervals for directly standardized rates: a method based on the gamma distribution. Statistics in Medicine, 16, 791-801.
}
