% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rows.R
\name{rows}
\alias{rows}
\alias{rows_insert}
\alias{rows_append}
\alias{rows_update}
\alias{rows_patch}
\alias{rows_upsert}
\alias{rows_delete}
\title{Manipulate individual rows}
\usage{
rows_insert(
  x,
  y,
  by = NULL,
  ...,
  conflict = c("error", "ignore"),
  copy = FALSE,
  in_place = FALSE
)

rows_append(x, y, ..., copy = FALSE, in_place = FALSE)

rows_update(
  x,
  y,
  by = NULL,
  ...,
  unmatched = c("error", "ignore"),
  copy = FALSE,
  in_place = FALSE
)

rows_patch(
  x,
  y,
  by = NULL,
  ...,
  unmatched = c("error", "ignore"),
  copy = FALSE,
  in_place = FALSE
)

rows_upsert(x, y, by = NULL, ..., copy = FALSE, in_place = FALSE)

rows_delete(
  x,
  y,
  by = NULL,
  ...,
  unmatched = c("error", "ignore"),
  copy = FALSE,
  in_place = FALSE
)
}
\arguments{
\item{x, y}{A pair of data frames or data frame extensions (e.g. a tibble).
\code{y} must have the same columns of \code{x} or a subset.}

\item{by}{An unnamed character vector giving the key columns. The key columns
must exist in both \code{x} and \code{y}. Keys typically uniquely identify each row,
but this is only enforced for the key values of \code{y} when \code{rows_update()},
\code{rows_patch()}, or \code{rows_upsert()} are used.

By default, we use the first column in \code{y}, since the first column is
a reasonable place to put an identifier variable.}

\item{...}{Other parameters passed onto methods.}

\item{conflict}{For \code{rows_insert()}, how should keys in \code{y} that conflict
with keys in \code{x} be handled? A conflict arises if there is a key in \code{y}
that already exists in \code{x}.

One of:
\itemize{
\item \code{"error"}, the default, will error if there are any keys in \code{y} that
conflict with keys in \code{x}.
\item \code{"ignore"} will ignore rows in \code{y} with keys that conflict with keys in
\code{x}.
}}

\item{copy}{If \code{x} and \code{y} are not from the same data source,
and \code{copy} is \code{TRUE}, then \code{y} will be copied into the
same src as \code{x}.  This allows you to join tables across srcs, but
it is a potentially expensive operation so you must opt into it.}

\item{in_place}{Should \code{x} be modified in place? This argument is only
relevant for mutable backends (e.g. databases, data.tables).

When \code{TRUE}, a modified version of \code{x} is returned invisibly;
when \code{FALSE}, a new object representing the resulting changes is returned.}

\item{unmatched}{For \code{rows_update()}, \code{rows_patch()}, and \code{rows_delete()},
how should keys in \code{y} that are unmatched by the keys in \code{x} be handled?

One of:
\itemize{
\item \code{"error"}, the default, will error if there are any keys in \code{y} that
are unmatched by the keys in \code{x}.
\item \code{"ignore"} will ignore rows in \code{y} with keys that are unmatched by the
keys in \code{x}.
}}
}
\value{
An object of the same type as \code{x}. The order of the rows and columns of \code{x}
is preserved as much as possible. The output has the following properties:
\itemize{
\item \code{rows_update()} and \code{rows_patch()} preserve the number of rows;
\code{rows_insert()}, \code{rows_append()}, and \code{rows_upsert()} return all existing
rows and potentially new rows; \code{rows_delete()} returns a subset of the
rows.
\item Columns are not added, removed, or relocated, though the data may be
updated.
\item Groups are taken from \code{x}.
\item Data frame attributes are taken from \code{x}.
}

If \code{in_place = TRUE}, the result will be returned invisibly.
}
\description{
These functions provide a framework for modifying rows in a table using a
second table of data. The two tables are matched \code{by} a set of key variables
whose values typically uniquely identify each row. The functions are inspired
by SQL's \code{INSERT}, \code{UPDATE}, and \code{DELETE}, and can optionally modify
\code{in_place} for selected backends.
\itemize{
\item \code{rows_insert()} adds new rows (like \code{INSERT}). By default, key values in
\code{y} must not exist in \code{x}.
\item \code{rows_append()} works like \code{rows_insert()} but ignores keys.
\item \code{rows_update()} modifies existing rows (like \code{UPDATE}). Key values in \code{y}
must be unique, and, by default, key values in \code{y} must exist in \code{x}.
\item \code{rows_patch()} works like \code{rows_update()} but only overwrites \code{NA} values.
\item \code{rows_upsert()} inserts or updates depending on whether or not the
key value in \code{y} already exists in \code{x}. Key values in \code{y} must be unique.
\item \code{rows_delete()} deletes rows (like \code{DELETE}). By default, key values in \code{y}
must exist in \code{x}.
}
}
\section{Methods}{

These function are \strong{generic}s, which means that packages can provide
implementations (methods) for other classes. See the documentation of
individual methods for extra arguments and differences in behaviour.

Methods available in currently loaded packages:
\itemize{
\item \code{rows_insert()}: \Sexpr[stage=render,results=rd]{dplyr:::methods_rd("rows_insert")}.
\item \code{rows_append()}: \Sexpr[stage=render,results=rd]{dplyr:::methods_rd("rows_append")}.
\item \code{rows_update()}: \Sexpr[stage=render,results=rd]{dplyr:::methods_rd("rows_update")}.
\item \code{rows_patch()}: \Sexpr[stage=render,results=rd]{dplyr:::methods_rd("rows_patch")}.
\item \code{rows_upsert()}: \Sexpr[stage=render,results=rd]{dplyr:::methods_rd("rows_upsert")}.
\item \code{rows_delete()}: \Sexpr[stage=render,results=rd]{dplyr:::methods_rd("rows_delete")}.
}
}

\examples{
data <- tibble(a = 1:3, b = letters[c(1:2, NA)], c = 0.5 + 0:2)
data

# Insert
rows_insert(data, tibble(a = 4, b = "z"))

# By default, if a key in `y` matches a key in `x`, then it can't be inserted
# and will throw an error. Alternatively, you can ignore rows in `y`
# containing keys that conflict with keys in `x` with `conflict = "ignore"`,
# or you can use `rows_append()` to ignore keys entirely.
try(rows_insert(data, tibble(a = 3, b = "z")))
rows_insert(data, tibble(a = 3, b = "z"), conflict = "ignore")
rows_append(data, tibble(a = 3, b = "z"))

# Update
rows_update(data, tibble(a = 2:3, b = "z"))
rows_update(data, tibble(b = "z", a = 2:3), by = "a")

# Variants: patch and upsert
rows_patch(data, tibble(a = 2:3, b = "z"))
rows_upsert(data, tibble(a = 2:4, b = "z"))

# Delete and truncate
rows_delete(data, tibble(a = 2:3))
rows_delete(data, tibble(a = 2:3, b = "b"))

# By default, for update, patch, and delete it is an error if a key in `y`
# doesn't exist in `x`. You can ignore rows in `y` that have unmatched keys
# with `unmatched = "ignore"`.
y <- tibble(a = 3:4, b = "z")
try(rows_update(data, y, by = "a"))
rows_update(data, y, by = "a", unmatched = "ignore")
rows_patch(data, y, by = "a", unmatched = "ignore")
rows_delete(data, y, by = "a", unmatched = "ignore")
}
