\name{crossplot}
\alias{crossplot}

\title{ Plot predicted effects for a DLNM }

\description{
Plot several graphs (3d, slices and overall effects) of predicted effects from distributed lag non-linear models (DLNM).
}

\usage{
crossplot(crosspred, type="3d", var=NULL, lag=NULL,
	ylim=NULL, title=NULL, label="var")
}

\arguments{
  \item{crosspred }{ an object of class \code{"crosspred"}.}
  \item{type }{ type of plot. See Details below.}
  \item{var }{ values of predictor for which specific effects must be plotted. Used only if \code{type="slices"}.}
  \item{lag }{ lag values for which specific effects must be plotted. Used only if \code{type="slices"}.}
  \item{ylim }{ numeric vectors of length 2, giving the coordinates ranges for the response axis (y-axis for \code{type} equal to \code{"overall"} or \code{"slices"}, z-axis for \code{"3d"}).}
  \item{title }{ the main title on top of the plot.}
  \item{label }{ label for predictor to be inserted in the plot.}
}

\details{
Different plots can be obtained choosing the following values for the argument \code{type}:

\bold{\code{"3d"}}: a 3-D plot generated by calling the function \code{\link[graphics::persp]{persp}}.

\bold{\code{"contour"}}: a contour/level plot generated by calling the function \code{\link[graphics::filled.contour]{filled.contour}}.

\bold{\code{"overall"}}: a plot of the overall effects (summed up all the single lag contributions).

\bold{\code{"slices"}}: a multiple plot of effects at specific values of predictor or lags, chosen by \code{var} and \code{lag}, respectively. Up to 4 plots for each dimension are allowed.

For a detailed overview of the options, see:

\code{vignette("dlnmOverview", package = "dlnm")}
}

\references{ 
Armstrong, B. Models for the relationship between ambient temperature and daily mortality. \emph{Epidemiology}. 2006, \bold{17}(6):624-31.
}

\author{Antonio Gasparrini, \email{antonio.gasparrini@lshtm.ac.uk}}

\note{
All the effects are reported versus a reference value. For continuous functions, this is specified by the centering point defined in the object \code{crossbasis} (see \code{\link{crossbasis}}). Exponentiated effects are automatically returned if \code{model} has link equal to \code{log} or \code{logit}.

The values in \code{var} and \code{lag} must match those specified in the object \code{crosspred} (see \code{\link{crosspred}}).

This function creates plots with default settings (i.e. perspective in 3-D plot, colours etc.). Refer to the original estimates stored in the \code{crosspred} object in order to personalize the output with generic plot commands.
}

\seealso{ \code{\link{crossbasis}}, \code{\link{crosspred}}}

\examples{
# Example 3. See crossbasis and crosspred for other examples

# load and prepare the dataset
initDB()
data <- readCity("chic", collapseAge=TRUE)
data$temp <- (data$tmpd-32)*5/9
data$pm10 <- with(data, pm10tmean+pm10mtrend)

### DLNM for the effect of temperature on mortality up to 30 days of lag
### space of predictor: 5df quadratic spline for temperature
### space of predictor: linear effect for PM10
### lag function: 5df natural cubic spline for temperature
### lag function: single strata at lag 0-1 for PM10

basis.pm <- crossbasis(data$pm10,vartype="lin", lagtype="strata",
	cen=FALSE, maxlag=1)
basis.temp <- crossbasis(data$temp, vartype="bs", vardf=5, vardegree=2,
	lagdf=5, cenvalue=21, maxlag=30)
summary(basis.pm)
summary(basis.temp)
model <- glm(death ~  basis.pm + basis.temp, family=quasipoisson(), data)
pred.temp <- crosspred(basis.temp, model, at=-26:33)

crossplot(pred.temp, label="Temperature",
	title="3D graph of temperature effect")
crossplot(pred.temp, "contour", label="Temperature",
	title="Contour graph of temperature effect")
crossplot(pred.temp, "overall", label="Temperature",
	title="Overall effect of temperature over 30 days of lag")
crossplot(pred.temp, "slices", var=c(-20,0,27,33),
	lag=c(0,5,15,28), label="Temperature")

### See the vignette 'dlnmOverview' for a detailed explanation of this example
}

\keyword{hplot}
\keyword{smooth}
\keyword{ts}