#' Create a worker object for use as a worker with master objects generated by \code{\link{SVDMaster}}
#' @description \code{SVDWorker} objects are worker objects at each site of
#' a distributed SVD model computation
#' @docType class
#' @seealso \code{SVDMaster} which goes hand-in-hand with this object
#' @importFrom R6 R6Class
#' @section Methods:
#'
#' \describe{
#'   \item{\code{SVDWorker$new(defn, data, stateful = TRUE)}}{Create a new SVD worker object with given definition defn, data and flag for preserving state between iterations}
#'   \item{\code{reset()}}{Initialize work matrix and set up starting values for iterating}
#'   \item{\code{dimX(...)}}{Return the dimensions of the matrix}
#'   \item{\code{updateV(arg, ...)}}{Return an updated value for the V vector, normalized by \code{arg}}
#'   \item{\code{updateU(arg, ...)}}{Return an updated value for the norm of the U vector}
#'   \item{\code{normU(arg, ...)}}{Normalize U vector by \code{arg}}
#'   \item{\code{fixU(arg, ...)}}{Construct the residual matrix using \code{arg}}
#'   \item{\code{getN(...)}}{Return the number of rows}
#'   \item{\code{getP(...)}}{Return the number of columns}
#'   \item{\code{kosher()}}{Check if inputs and state of object are sane. For future use}
#'   \item{\code{getStateful()}}{Returns \code{TRUE} if object is stateful, else \code{FALSE}}
#' }
#'
#' @export
#' @format An \code{\link{R6Class}} generator object
SVDWorker <- R6Class("SVDWorker",
                     private = list(
                         defn = NA,
                         stateful = TRUE,
                         x = NA,
                         u = NA,
                         p = NA,
                         workX = NA
                     ),
                     public = list(
                         initialize = function(defn, data, stateful = TRUE) {
                             private$x <- private$workX <- data
                             private$u <- rep(1, nrow(data))
                             private$stateful <- stateful
                             stopifnot(self$kosher())
                         },
                         reset = function(){
                             private$workX <- private$x
                             private$u <- rep(1, nrow(private$x))
                         },
                         dimX = function(...) dim(private$x),
                         updateV = function(arg, ...) {
                             t(private$workX) %*% private$u / arg
                         },
                         updateU = function(arg, ...) {
                             u <- private$u <- as.numeric(private$workX %*% arg)
                             sum(u^2)
                         },
                         normU = function(arg, ...) {
                             private$u <- private$u / arg
                             invisible(TRUE)
                         },
                         fixU = function(arg, ...) {
                             private$workX <- private$workX - private$u %*% t(arg)
                         },
                         getN = function() {
                             nrow(private$x)
                         },
                         getP = function() {
                             ncol(private$x)
                         },
                         getStateful = function() private$stateful,
                         kosher = function() {
                             TRUE
                         })
                     )

#' Create a master object to control worker objects generated by \code{\link{SVDWorker}}
#' @description \code{SVDMaster} objects instantiate and run a distributed SVD computation
#' @docType class
#' @seealso \code{SVDWorker} which goes hand-in-hand with this object
#' @importFrom R6 R6Class
#' @section Methods:
#'
#' \describe{
#'   \item{\code{SVDMaster$new(defn, debug = FALSE)}}{Create an SVD master object
#'         with the specified computation definition. The debug flag is used for debugging computations}
#'   \item{\code{kosher()}}{Check if inputs and state of object are sane. For future use}
#'   \item{\code{updateV(arg)}}{Return an updated value for the V vector}
#'   \item{\code{updateU(arg)}}{Return an updated value for the U vector}
#'   \item{\code{fixFit(v, d)}}{Construct the residual matrix using given the v vector and d so far}
#'   \item{\code{reset()}}{Initialize the computation}
#'   \item{\code{dimX(, ...)}}{Return the dimensions of the matrix}
#'   \item{\code{normU(arg, ...)}}{Normalize U vector by \code{arg}}
#'   \item{\code{addSite(name, url = NULL, worker = NULL)}}{Add a worker site
#'         for participating in the distributed computation. Exactly one of \code{url}
#'         or \code{worker} should be specified}
#'   \item{\code{run(k = private$k, thr = 1e-8, max.iter = 100)}}{Run the SVD computation until either the
#'         threshold is reached or the max.iter number of iterations are used up}
#'   \item{\code{summary()}}{Return the summary of results}
#' }
#' @export
#' @format An \code{\link{R6Class}} generator object

SVDMaster <- R6Class("SVDMaster",
                     private = list(
                         defn = NA,
                         dry_run = FALSE,
                         sites = list(),
                         dimX = NA,
                         mapFn = function(site, arg, method) {
                             payload <- list(objectId = site$instanceId,
                                             method = method,
                                             arg = arg)
                             q <- POST(.makeOpencpuURL(urlPrefix=site$url, fn="executeMethod"),
                                       body = toJSON(payload),
                                       add_headers("Content-Type" = "application/json"),
                                       config = getConfig()$sslConfig
                                       )
                             ## Should really examine result here.
                             .deSerialize(q)
                         },
                         result = list(),
                         debug = FALSE
                     ),
                     public = list(
                         initialize = function(defn, debug = FALSE) {
                             'Initialize the object with a dataset'
                             private$defn <- defn
                             private$debug <- debug
                             stopifnot(self$kosher())
                         },
                         kosher = function() {
                             ' Check for appropriateness'
                             TRUE
                         },
                         updateV = function(arg) { ## Here arg is a list of right size already
                             'Compute or Update VD'
                             sites <- private$sites
                             n <- length(sites)
                             if (private$dry_run) {
                                 mapFn <- function(x, arg) x$worker$updateV(arg)
                                 results <- Map(mapFn, sites, rep(list(arg), n))
                             } else {
                                 results <- Map(private$mapFn, sites, rep(arg, n), rep(list("updateV"), n))
                             }
                             vd <- Reduce(f = '+', results)
                             vd / sqrt(sum(vd^2))
                         },
                         updateU = function(arg) { ## arg is a single vector
                             'Compute/Update U'
                             sites <- private$sites
                             n <- length(sites)
                             if (private$dry_run) {
                                 mapFn <- function(x, arg) x$worker$updateU(arg)
                                 results <- Map(mapFn, sites, rep(list(arg), n))
                             } else {
                                 results <- Map(private$mapFn, sites, rep(list(arg), n), rep(list("updateU"), n))
                             }
                             sqrt(Reduce(f = '+', results))
                         },
                         fixFit = function(v, d) { ## arg is a single vector
                             'Compute/Update U'
                             result <- private$result
                             result$v <- cbind(result$v, v)
                             result$d <- c(result$d, d)

                             sites <- private$sites
                             n <- length(sites)
                             if (private$dry_run) {
                                 mapFn <- function(x, v) x$worker$fixU(v)
                                 Map(mapFn, sites, rep(list(v), n))
                             } else {
                                 Map(private$mapFn, sites, rep(list(v), n), rep(list("fixU"), n))
                             }
                             private$result <- result
                         },
                         reset = function() {
                             private$result <- list()
                             arg = "THISISUNUSED"
                             sites <- private$sites
                             n <- length(sites)
                             if (private$dry_run) {
                                 lapply(sites, function(x) x$worker$reset())
                             } else {
                                 Map(private$mapFn, sites, rep(list(arg), n), rep(list("reset"), n))
                             }
                         },
                         addSite = function(name, url = NULL, worker = NULL) {
                             ## critical section start
                             ## This is the time to cache "p" and check it
                             ## against all added sites
                             ## Only one of url/worker should be non-null
                             stopifnot(is.null(url) || is.null(worker))
                             n <- length(private$sites)
                             if (is.null(url)) {
                                 private$dry_run <- private$dry_run || TRUE
                                 private$sites[[n+1]] <- list(name = name, worker = worker)
                             } else {
                                 localhost <- (grepl("^http://localhost", url) ||
                                               grepl("^http://127.0.0.1", url))
                                 private$sites[[n+1]] <- list(name = name, url = url,
                                                              localhost = localhost,
                                                              dataFileName = if (localhost) paste0(name, ".rds") else NULL)
                             }
                             ## critical section end
                         },
                         run = function(thr = 1e-8, max.iter = 100) {
                             'Run Calculation'
                             dry_run <- private$dry_run
                             defn <- private$defn
                             debug <- private$debug
                             n <- length(sites)
                             stopifnot(n > 1)
                             if (debug) {
                                 print("run(): checking worker object creation")
                             }

                             if (dry_run) {
                                 ## Workers have already been created and passed
                                 sites <- private$sites
                                 pVals <- sapply(sites, function(x) x$worker$getP())
                             } else {
                                 ## Make remote call to instantiate workers
                                 instanceId <- generateId(object=list(Sys.time(), self))
                                 ## Augment each site with object instance ids
                                 private$sites <- sites <- lapply(private$sites,
                                                                  function(x) list(name = x$name,
                                                                                   url = x$url,
                                                                                   localhost = x$localhost,
                                                                                   dataFileName = x$dataFileName,
                                                                                   instanceId = if (x$localhost) x$name else instanceId))
                                 sitesOK <- sapply(sites,
                                                   function(x) {
                                                       payload <- if (is.null(x$dataFileName)) {
                                                                      list(defnId = defn$id, instanceId = x$instanceId)
                                                                  } else {
                                                                      list(defnId = defn$id, instanceId = x$instanceId,
                                                                           dataFileName = x$dataFileName)
                                                                  }
                                                       q <- POST(url = .makeOpencpuURL(urlPrefix=x$url, fn="createInstanceObject"),
                                                                 body = toJSON(payload),
                                                                 add_headers("Content-Type" = "application/json"),
                                                                 config = config$sslConfig
                                                                 )
                                                       .deSerialize(q)
                                                   })

                                 ## I am not checking the value of p here; I do it later below
                                 if (!all(sitesOK)) {
                                     warning("run():  Some sites did not respond successfully!")
                                     sites <- sites[which(sitesOK)]  ## Only use sites that created objects successfully.
                                 }
                                 ## stop if no sites
                                 if (debug) {
                                     print("run(): checking p")
                                 }

                                 pVals <- sapply(sites,
                                                 function(x) {
                                                     payload <- list(objectId = x$instanceId, method = "getP")
                                                     q <- POST(.makeOpencpuURL(urlPrefix=x$url, fn="executeMethod"),
                                                               body = toJSON(payload),
                                                               add_headers("Content-Type" = "application/json"),
                                                               config=getConfig()$sslConfig
                                                               )
                                                     .deSerialize(q)
                                                 })
                             }
                             if (debug) {
                                 print(pVals)
                             }
                             p <- pVals[1]
                             if (any(pVals != p)) {
                                 stop("run(): Heterogeneous sites! Stopping!")
                             }

                             returnCode <- 0
                             self$reset()
                             k <- private$defn$rank
                             for (j in seq_len(k)) {
                                 v <-  rep(1.0, p)
                                 vold <- rep(0.0, p)
                                 for (i in seq.int(max.iter)) {
                                     unorm <- self$updateU(v)
                                     v <- self$updateV(unorm) # computes vd
                                     discrepancy <- max(abs(v - vold))
                                     if (debug) {
                                         print(paste("Iteration:", i, "; Discrepancy: ", discrepancy))
                                     }
                                     if (discrepancy < thr) break
                                     vold <- v
                                 }
                                 self$fixFit(v, unorm)
                             }

                             if (!dry_run) {
                                 if (debug) {
                                     print("run(): checking worker object cleanup")
                                 }
                                 sitesOK <- sapply(sites,
                                                   function(x) {
                                                       payload <- list(instanceId = x$instanceId)
                                                       q <- POST(url = .makeOpencpuURL(urlPrefix=x$url, fn="destroyInstanceObject"),
                                                                 body = toJSON(payload),
                                                                 add_headers("Content-Type" = "application/json"),
                                                                 config=getConfig()$sslConfig
                                                                 )
                                                       .deSerialize(q)
                                                   })
                                 if (!all(sitesOK)) {
                                     warning("run():  Some sites did not clean up successfully!")
                                 }
                             }
                             return(private$result)
                         },
                         summary = function() {
                             'Return the summary'
                             result <- private$result
                             if (length(result) == 0) {
                                 stop ("Run the computation first using run()")
                                 result <- private$result
                             }
                             result
                         }
                     )
                     )

