////////////////////////////////////////////////////////////////////////////////
//
// DgOutLocFile.cpp: DgOutLocFile class implementation
//
// Version 6.1 - Kevin Sahr, 5/23/13
//
////////////////////////////////////////////////////////////////////////////////

#include <list>
#include <sstream>

#include "DgOutLocFile.h"
#include "DgLocList.h"
#include "DgCell.h"
#include "DgPolygon.h"
#include "DgLocation.h"
#include "DgOutAIGenFile.h"
#include "DgOutKMLfile.h"
#include "DgOutGeoJSONFile.h"
#include "DgOutShapefile.h"
#include "DgOutPtsText.h"
#include "DgGeoSphRF.h"

const string DgOutLocFile::defaultKMLColor = "ffffffff";
const int    DgOutLocFile::defaultKMLWidth = 4;
const string DgOutLocFile::defaultKMLName = "";
const string DgOutLocFile::defaultKMLDescription = "Generated by DGGRID " DGGRID_VERSION;

////////////////////////////////////////////////////////////////////////////////
DgOutLocFile::~DgOutLocFile (void) 
{ 
   // empty destructor
}

////////////////////////////////////////////////////////////////////////////////
DgOutLocFile* 
DgOutLocFile::makeOutLocFile (const string& type, const string& fileName, 
                    const DgRFBase& rf, bool isPointFile, int precision,
                    int shapefileIdLen, const string& kmlColor, int kmlWidth,
                    const string& kmlName, const string& kmlDesc,
                    DgReportLevel failLevelIn)
{
   DgOutLocFile* file = NULL;
   if (!type.compare("AIGEN"))
      file = new DgOutAIGenFile(rf, fileName, precision, isPointFile, 
                                 failLevelIn);
   else if (!type.compare("TEXT"))
      file = new DgOutPtsText(rf, fileName, precision, failLevelIn);
   else // must be KML, GEOJSON or SHAPEFILE
   {
      const DgGeoSphDegRF* geoRF = dynamic_cast<const DgGeoSphDegRF*>(&rf);
      if (geoRF == NULL)
         ::report("DgOutLocFile::makeOutLoc(): invalid RF type", failLevelIn);

      if (!type.compare("KML"))
         file = new DgOutKMLfile(*geoRF, fileName, precision, isPointFile, 
                                 kmlColor, kmlWidth, kmlName, kmlDesc, failLevelIn);
      else if (!type.compare("GEOJSON"))
         file = new DgOutGeoJSONFile(*geoRF, fileName, precision, isPointFile,
                                     failLevelIn);
      else if (!type.compare("SHAPEFILE"))
         file = new DgOutShapefile(*geoRF, fileName, precision, isPointFile, 
                                    shapefileIdLen, failLevelIn);
      else if (type.compare("NONE"))
         ::report("DgOutLocFile::makeOutLoc(): invalid file type " + type, 
                                 failLevelIn);
   }

   return file;

} // DgOutLocFile::makeOutLoc

////////////////////////////////////////////////////////////////////////////////
DgOutLocFile&
DgOutLocFile::insert (DgLocList& dlist)
//
// Output the drawlist dlist.
//
////////////////////////////////////////////////////////////////////////////////
{
   list<DgLocBase*>::iterator it;
   for (it = dlist.begin(); it != dlist.end(); it++)
   {
      if (DgLocList* d = dynamic_cast<DgLocList*>(*it))
      {
         this->insert(*d);
      }
      else if (DgPolygon* d = dynamic_cast<DgPolygon*>(*it))
      {
         this->insert(*d);
      }
      else if (DgCell* d = dynamic_cast<DgCell*>(*it))
      {
         this->insert(*d);
      }
      else if (DgLocation* d = dynamic_cast<DgLocation*>(*it))
      {
         this->insert(*d);
      }
      else if (DgLocVector* d = dynamic_cast<DgLocVector*>(*it))
      {
         this->insert(*d);
      }
      else
      {
         report("DgOutLocFile::insert() invalid location not inserted", Warning);
      }
   }

   //this->flush();

   return *this;

} // DgOutLocFile& DgOutLocFile::insert

////////////////////////////////////////////////////////////////////////////////
DgOutLocFile&
DgOutLocFile::insert (DgCell& cell)
//
// Put the cell.
//
////////////////////////////////////////////////////////////////////////////////
{
   rf().convert(&cell);

   if (!isPointFile() && cell.hasRegion())
      this->insert(cell.region(), &(cell.label()), &(cell.node()));
   else
      this->insert(cell.node(), &(cell.label()));

   return *this;

} // DgOutLocFile& DgOutLocFile::insert

////////////////////////////////////////////////////////////////////////////////
