% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seq_design.R
\name{IMSE}
\alias{IMSE}
\alias{IMSE.gp}
\alias{IMSE.dgp2}
\alias{IMSE.dgp3}
\title{Integrated Mean-Squared (prediction) Error for Sequential Design}
\usage{
IMSE(object, x_new, cores)

\method{IMSE}{gp}(object, x_new = NULL, cores = 1)

\method{IMSE}{dgp2}(object, x_new = NULL, cores = 1)

\method{IMSE}{dgp3}(object, x_new = NULL, cores = 1)
}
\arguments{
\item{object}{object of class \code{gp}, \code{dgp2}, or \code{dgp3}}

\item{x_new}{matrix of possible input locations, if object has been run 
through \code{predict} the previously stored \code{x_new} is used}

\item{cores}{number of cores to utilize in parallel, by default no 
parallelization is used}
}
\value{
list with elements:
\itemize{
  \item \code{value}: vector of IMSE values, indices correspond to \code{x_new}
  \item \code{time}: computation time in seconds
}
}
\description{
Acts on a \code{gp}, \code{dgp2}, or \code{dgp3} object.
    Current version requires squared exponential covariance
    (\code{cov = "exp2"}).  Calculates IMSE over the input locations 
    \code{x_new}.  Optionally utilizes SNOW parallelization.  User should 
    select the point with the lowest IMSE to add to the design.
}
\details{
Not yet implemented for Vecchia-approximated fits.

    All iterations in the object are used in the calculation, so samples
    should be burned-in.  Thinning the samples using \code{trim} will speed 
    up computation.  This function may be used in two ways:
    \itemize{
        \item Option 1: called on an object with only MCMC iterations, in 
        which case \code{x_new} must be specified
        \item Option 2: called on an object that has been predicted over, in 
        which case the \code{x_new} from \code{predict} is used
    }
    In Option 2, it is recommended to set \code{store_latent = TRUE} for 
    \code{dgp2} and \code{dgp3} objects so latent mappings do not have to 
    be re-calculated.  Through \code{predict}, the user may
    specify a mean mapping (\code{mean_map = TRUE}) or a full sample from 
    the MVN distribution over \code{w_new} (\code{mean_map = FALSE}).  When 
    the object has not yet been predicted over (Option 1), the mean mapping 
    is used.
    
    SNOW parallelization reduces computation time but requires more memory storage.
}
\examples{
# --------------------------------------------------------
# Example 1: toy step function, runs in less than 5 seconds
# --------------------------------------------------------

f <- function(x) {
    if (x <= 0.4) return(-1)
    if (x >= 0.6) return(1)
    if (x > 0.4 & x < 0.6) return(10*(x-0.5))
}

x <- seq(0.05, 0.95, length = 7)
y <- sapply(x, f)
x_new <- seq(0, 1, length = 100)

# Fit model and calculate IMSE
fit <- fit_one_layer(x, y, nmcmc = 100, cov = "exp2")
fit <- trim(fit, 50)
fit <- predict(fit, x_new, cores = 1, store_latent = TRUE)
imse <- IMSE(fit)

\donttest{
# --------------------------------------------------------
# Example 2: Higdon function
# --------------------------------------------------------

f <- function(x) {
    i <- which(x <= 0.48)
    x[i] <- 2 * sin(pi * x[i] * 4) + 0.4 * cos(pi * x[i] * 16)
    x[-i] <- 2 * x[-i] - 1
    return(x)
}

# Training data
x <- seq(0, 1, length = 30)
y <- f(x) + rnorm(30, 0, 0.05)

# Testing data
xx <- seq(0, 1, length = 100)
yy <- f(xx)

plot(xx, yy, type = "l")
points(x, y, col = 2)

# Conduct MCMC (can replace fit_three_layer with fit_one_layer/fit_two_layer)
fit <- fit_three_layer(x, y, D = 1, nmcmc = 2000, cov = "exp2")
plot(fit)
fit <- trim(fit, 1000, 2)

# Option 1 - calculate IMSE from only MCMC iterations
imse <- IMSE(fit, xx)

# Option 2 - calculate IMSE after predictions
fit <- predict(fit, xx, cores = 1, store_latent = TRUE)
imse <- IMSE(fit)

# Visualize fit
plot(fit)
par(new = TRUE) # overlay IMSE
plot(xx, imse$value, col = 2, type = 'l', lty = 2, 
     axes = FALSE, xlab = '', ylab = '')

# Select next design point
x_new <- xx[which.min(imse$value)]
}

}
\references{
Sauer, A, RB Gramacy, and D Higdon. 2020. "Active Learning for Deep Gaussian 
    Process Surrogates." \emph{Technometrics, to appear:} arXiv:2012.08015. 
    \cr\cr
Binois, M, J Huang, RB Gramacy, and M Ludkovski. 2019. "Replication or Exploration? 
    Sequential Design for Stochastic Simulation Experiments." \emph{Technometrics 
    61}, 7-23. Taylor & Francis. doi:10.1080/00401706.2018.1469433.
}
