library(testthat)

scrub_environment_code <- function(x) {
  # macos: 9
  # linux: 12
  # windows: 16
  m <- regexpr("(?<=<environment: 0x)[0-9a-f]+(?=>)", x, perl = TRUE)
  regmatches(x, m) <- strrep("0", max(0, 12L))
  x
}

test_documentation <- function() {
  skip_if_not_installed("here")

  cwd <- setwd(here::here())
  on.exit(setwd(cwd))

  dir <- tempfile("cnd_dir_")
  on.exit(unlink(dir, TRUE), add = TRUE)
  dir.create(dir, showWarnings = FALSE)
  # path to where the documentation will be sent
  path <- tempfile("cnd_documentation_", dir, ".R")

  # extra file to trigger cleanup
  temp <- tempfile("cnd_auto_generated_", dir, ".R")
  writeLines("# % Generated by cnd: do not edit by hand", temp)

  suppress_cnd_conditions(
    expect_message(
      expect_condition(
        expect_identical(cnd_document(package = "cnd", file = path), path),
        class = "cnd:cnd_generated_write"
      ),
      class = "cnd:cnd_generated_cleanup"
    )
  )

  is_ci_windows <-
    Sys.info()[["sysname"]] == "Windows" &&
    # GitHub should set CI to 'true'
    isTRUE(as.logical(Sys.getenv("CI", "false")))

  # no changes
  expect_no_condition(
    withCallingHandlers(
      cnd_document(package = "cnd", file = path),
      # line endings on CI Windows might be throwing off the check.  For now,
      # these are simply going to be muffled
      "cnd:cnd_generated_cleanup" = function(c) {
        if (is_ci_windows) tryInvokeRestart("muffleMessage")
      },
      "cnd:cnd_generated_write" = function(c) {
        if (is_ci_windows) tryInvokeRestart("muffleCondition")
      }
    )
  )

  skip_if_not_installed("roxygen2")
  parsed <- roxygen2::parse_text(readLines(path)[-1:-2], test_env())
  expect_failure(expect_identical(parsed, list()))
}

local_registry <- function(name = basename(tempfile(""))) {
  registrar$add(name)
  do <- as.call(list(function() registrar$remove(name)))
  do.call(on.exit, list(do), envir = parent.frame())
  registrar$get(name)
}
