#include <Matrix.h>

#include "test.h"
#include "lmm.h"
#include "util.h"
#include "lmer_common.h"
#include "lmer.h"
#include "blmer.h"
#include "Syms.h"

extern cholmod_common cholmodCommon;

struct _MERCache {
  double *weightedDenseDesignMatrix;
  double *weightedResponse;
  
  // there are more fields in lmm's definition, but so long as the first two are
  // the same we're safe
};

static int rotateSparseDesignMatrix_test(SEXP regression);
static int updateWeights_test(SEXP regression, MERCache *cache);
static int updateAugmentedDesignMatrixFactorizations_test(SEXP regression, MERCache *cache);
static int calculateJointMode_test(SEXP regression, MERCache *cache);
static int calculateDevianceAndCommonScale_test(SEXP regression);
static int newtonsMethodUpdateCommonScale_test(SEXP regression, MERCache *cache);

#define TEST_TOLERANCE 1.0e-10

SEXP bmer_lmmTest()
{
  SEXP regression = createTestRegression();
  MERCache *cache = createLMMCache(regression);
  
  char *errorMessage = NULL;
  
  if (!rotateSparseDesignMatrix_test(regression)) {
    errorMessage = "rotateSparseDesignMatrix_test() failed.";
    goto BLME_LMMTEST_CLEANUP;
  }
  
  if (!updateWeights_test(regression, cache)) {
    errorMessage = "updateWeights_test() failed.";
    goto BLME_LMMTEST_CLEANUP;
  }
  
  if (!updateAugmentedDesignMatrixFactorizations_test(regression, cache)) {
    errorMessage = "updateAugmentedDesignMatrixFactorizations_test() failed.";
    goto BLME_LMMTEST_CLEANUP;
  }
  
  if (!calculateJointMode_test(regression, cache)) {
    errorMessage = "calculateJointMode_test() failed.";
    goto BLME_LMMTEST_CLEANUP;
  }
  
  if (!calculateDevianceAndCommonScale_test(regression)) {
    errorMessage = "calculateDevianceAndCommonScale_test() failed.";
    goto BLME_LMMTEST_CLEANUP;
  }
  
  if (!newtonsMethodUpdateCommonScale_test(regression, cache)) {
    errorMessage = "newtonsMethodUpdateCommonScale_test() failed.";
    goto BLME_LMMTEST_CLEANUP;
  }
  
BLME_LMMTEST_CLEANUP:
  deleteLMMCache(cache);
  UNPROTECT(1); // gets rid of the regression object that we created
  if (errorMessage) error(errorMessage);
//  return (regression);
  return (ScalarLogical(TRUE));
}

// why any of these are the correct answer depends on how the regression is defined in lmer_test.c

static int rotateSparseDesignMatrix_test(SEXP regression)
{
  rotateSparseDesignMatrix(regression);

  double correctAnswer[] = { 2.20176740149696, 0.206434182925788, 4.43106133733892, 1.25621636731012, 1.23428008379303, 3.11717713427994, 0.445068234575872, 4.2777072218306, 2.00703888393513, 0.161180068508258, 0.323399086453827, -0.283229216377562, 2.08044501460982, -1.2854841348516, -0.132011807344598, -0.610068661349336, -0.526570730855164, 0.754305662588811, -2.60793673658099, -1.02250627936651, 1.48335141970849, 0.0191535474193714, 1.79658195843999, -0.36771788177241, -1.61165655081075, 1.0392449002406, -0.0966185864235233, 2.0184603475022, -0.676583972398935, -0.893605210024263, 0.556222285760795, -0.222535582282316, 3.39407452052623, -0.586486462930744, 1.40593391642668, 0.728526241799348, -0.177618437890794, 3.1125076056003, -0.540074272172659, 0.865637119057807, 0.388627052059946, -0.266225230996984, 2.04141771211829, -1.24253797651842, -0.246095703905913, 2.55264444644318, 0.297902743798491, 3.40183679882271, 1.17951827196514, -0.476248489733389, 3.22118160860958, 0.472180693849782, 3.88010733552207, 1.94955333281296, -0.469712357223793, 2.96998326649684, 0.40669692630898, 4.88795763768846, 2.10625218727239, 1.11770713980174, 0.734497096597873, -0.176061922571632, 2.23921302611568, -0.862801773505398, -0.30917064627218, 3.357027778483, 0.507593821717108, 5.37893414970158, 2.63246642930992, 1.4081040130742, 0.97062435948867, -0.114506966981015, 3.17887024311002, -0.301346367748858, 0.72497704004294, 4.17406233260838, 0.720582889897884, 5.99372662827258, 3.58491941832521, 1.45663969541499, 2.29161326011186, 0.229855696262851, 2.84147065704961, 0.73852744886955, -0.979001163892053, 0.697905225502172, -0.18560089309413, 2.45124724644405, -0.815479067160467, 0.0093845587353374, 0.101545380868205, -0.341063262424386, 1.28432048946974, -1.78042480200535, -0.987532186162473, -0.424476280345915, -0.478189486644893, 2.14878775309767, -1.92028235341408, 0.668457114298578, -0.265692056706531, -0.436796740568244, 1.86216912110906, -1.88770714268161, 0.134213666875734, -1.04906472441537, -0.64101064277242, 0.970924499729848, -2.91426509189513, -0.316340936590936, 3.22845449305315, 0.474076629429007, 5.45534535217983, 2.54761947807282, 1.6322869389236, 2.71821594231296, 0.341064833387653, 4.06659075714928, 1.57541330051348, 0.256750456511047, 1.05241853096799, -0.0931844117989426, 3.52976714540695, -0.0973186452735502, 1.11721057491588, 1.828360796516, 0.109092492239794, 3.89943282222508, 0.726777618485424, 0.876530701869446, 0.817537893646951, -0.154414390169033, 1.87848045320633, -0.92495387571006, -0.870832716276407, 5.24875975449483, 1.00074093377512, 5.15085204559396, 4.21743596113331, -0.690577122814986, 0.15926023992058, -0.32601783504597, 2.10183127064648, -1.42240671102824, 0.0522204846027738, 1.32358650235955, -0.0224948512956096, 1.94070103427665, -0.454680542523609, -1.26725868102471, 1.80320809852938, 0.102535548375727, 3.70983887789809, 0.633355109137175, 0.646550847817274, 2.38202375426602, 0.253424401902566, 2.56426235168438, 0.71425456702064, -1.43582913710769, 1.13843877056084, -0.0707601819018597, 4.11731788056324, 0.199325979880774, 1.82226479373823, -2.08705634759238, -0.911600016849243, 0.0529372647583186, -4.17572752837757, -0.561323506025293, -0.577071205476176, -0.517968772292611, 2.94682864276483, -1.75530667540217, 1.88151403537565, 1.97395477292803, 0.14704673201112, 3.00236096575854, 0.518424213256659, -0.462469909712604, 1.39250937452301, -0.00452765732037581, 2.12189620063572, -0.325758015503281, -1.09001630872935, -0.0692300586489609, -0.385581944709324, 1.92228763071015, -1.69162684369479, 0.028456355288035, 1.22775009921451, -0.047478012779532, 2.84242608741858, -0.200637312712442, 0.0308021913353568, 0.128394023909196, -0.334064210261482, 0.633993349074852, -2.00097032828693, -1.88362785772426, 1.79082072045858, 0.0993063384247144, 4.10053141930903, 0.76915563359815, 1.1813444182257, -0.830993978072337, -0.584162759379277, 0.424084291235243, -2.92706906285786, -1.25516590234186, 1.98529551619361, 0.150003099435748, 3.63687711258411, 0.766757274422318, 0.376255461862785, 1.80048987465719, 0.101826946803466, 3.63931224132809, 0.604465495924141, 0.554708174530773, 1.51253893443454, 0.026762309536745, 3.13668074437595, 0.161385720816118, 0.154768838232609, 1.43999978927847, 0.00785240559766139, 2.87023573689364, -0.00274939358676982, -0.133175973095009, 1.81452498362175, 0.105485696320344, 3.40045093390246, 0.527146381077432, 0.221620886005316, 0.389352838486563, -0.26603602899332, 1.33083550552297, -1.50878442362534, -1.1980955020153, 1.22249276369036, -0.04884852396385, 2.14308944891538, -0.467943881018372, -0.900470390365099, 2.45418648463828, 0.272236179962335, 2.75519219324808, 0.849694536581458, -1.24862552731283 };
  
  CHM_SP rotatedSparseDesignMatrix = A_SLOT(regression);
  R_CheckStack();
  
  int *indiciesForColumn = (int *) rotatedSparseDesignMatrix->p;
  
  int arrayLength = indiciesForColumn[rotatedSparseDesignMatrix->ncol];
  
  return (allApproximatelyEqual(correctAnswer, (double *) rotatedSparseDesignMatrix->x, arrayLength, TEST_TOLERANCE));
}

static int updateWeights_test(SEXP regression, MERCache *cache)
{
  updateWeights(regression, cache);
  
  double correctWeightedDenseDesignMatrix[] = { 0.0617565919738862, 0.050305918460172, 0.163964053735133, 0.161398546528127, 0.187806415235078, 0.108462035453558, 0.190717802468573, 0.0887467314199873, 0.175776717708043, 0.0935899702679429, 0.0346309094742158, 0.183339966218362, 0.163446001832697, 0.191662724575518, 0.162338416786167, 0.181317479326481, 0.118791602574563, 0.123691700131753, 0.148781716711936, 0.0609121886318083, 0.086926857003468, 0.151428673982027, 0.171183001019172, 0.183892613727555, 0.120405726350736, 0.17648911369551, 0.0476851130967194, 0.155916032874993, 0.117926340296253, 0.151410845937035, 0.188762997933986, 0.088186772491143, 0.119965764772369, 0.161803644951267, 0.17306785692037, 0.142703112760805, 0.179692429388008, 0.143363923942244, 0.139875199072492, 0.127969298594458, 0.118857906613404, 0.0693750623192124, 0.107825316763508, 0.103867832141938, 0.175674341952008, 0.174194859439429, 0.074878161760803, 0.141781735796163, 0.152605279559404, 0.140443880763893, 0.0311011120560344, 0.054620624998654, -0.113291592500548, -0.207332468585943, 0.00877547489635141, -0.0255652128811771, -0.103538454981554, -0.0384548743648465, -0.114161994318279, 0.0680165808400096, 0.039891751678079, 0.181902647884031, -0.0702022004811772, 0.237336737822596, -0.0453486331313244, 0.318738557144711, 0.0666119267740137, -0.0560056193575091, -0.123792829975069, -0.0710583652163356, -0.0926284400056561, -0.236801442390576, 0.197979473961869, 0.153007321217683, -0.0273716762113599, 0.0469703470827712, -0.017963111739502, 0.380647887738575, -0.0937914313923782, -0.00830904471755749, 0.0472174260091231, 0.0545208806739015, -0.0207089105125953, -0.35983517034293, -0.21869103358061, 0.051191730092359, -0.00198478885924802, -0.13485515500599, -0.0162010899943395, -0.104290974050514, 0.0287949301838588, -0.0988662899455768, 0.0394577175095761, 0.025802083315033, 0.0114694583524051, 0.00333694495262804, 0.0192690246319145, -0.0920177753667705, -0.0181857823035529, 0.0932737950469733, 0.0679920996200563, 0.00723255638791365, -0.0193073572971873, -0.147206508759744, -0.269987918481551, -0.0864539523280149, 0.2391759741751, 0.0685250951816365, -0.0385857363563435, -0.0397579817907967, -0.0145096618945798, 0.182787537399227, -0.0450748162477392, 0.240731988643007, 0.104980096729174, 0.235588031558688, -0.103736205721768, 0.00103541827364425, -0.131057607188515, 0.036319441693205, 0.010406678285168, -0.042729216075306, 0.249240463943642, 0.0421150111276617, 0.119989595556771, 0.137989634524794, -0.0370406810597324, -0.0960427027008518, 0.00549304463968454, -0.171152666821713, 0.108863166235617, -0.112945158759277, 0.194998028857357, -0.0810145342479297, 0.290459301032207, -0.0588678720194753, -0.174712583533973, 0.00363898749829418, 0.00384311819681759, -0.215011804495534, 0.12524662167165, -0.0776722576522294, 0.0361880319761723, 0.0513933636830509, 0.0242523187576267, -0.020692960233068, 0.0148022341734934, -0.151521107634412, -0.122574577581754, -0.156421498004242 };
  double correctWeightedResponse[] = { 0.633355044798475, 0.290343719516855, 0.858990677395523, -0.163090186894286, -0.452039442289309, 0.528925460706199, 2.11084082394415, 0.883239513316936, -0.21284653040441, 0.543288147830654, 0.0237173449700773, 2.0222049869005, 0.933781576699828, 3.20950971225159, 1.21317853868475, 1.54318167250863, 0.335435364133565, 0.537362440990296, -0.441119732974195, 0.358520942317532, 0.0848589597271782, 0.852466104265874, 2.3305139466043, 1.11787315881698, 1.15308833509839, 1.51168208391213, 0.0966213531504031, 0.831877459482254, 0.647813909115186, 0.054066783192161, 1.11449826170049, -0.00069306986109797, 1.50517224749665, -1.07790795679175, 2.06862365357347, 0.214310130080279, 0.208350426331088, 0.713146721059777, 0.827569431541108, 0.0335679788488684, 1.2595235873108, -0.0195392586745118, 0.796623050785554, 0.822408826599177, 1.09639550917565, 0.847204295830954, 0.377745709139824, 0.466148269197546, 0.207976197876662, -0.15157352412998 };
  double correctWeightedSparseDesignMatrix[] = { 0.135973651035652, 0.0127486716044105, 0.273647247021302, 0.0775796416268888, 0.0762249315163005, 0.156812458742999, 0.0223895663177865, 0.215193990697899, 0.100965934441635, 0.00810831138378138, 0.0530258251892084, -0.0464394104534901, 0.341118198168475, -0.210773189762469, -0.0216451910731217, -0.0984641952241435, -0.0849877506042773, 0.12174383757977, -0.42091719872148, -0.165031027305638, 0.278582912669316, 0.00359715907986722, 0.33740961729063, -0.0690597771935127, -0.302679439397899, 0.112718617214825, -0.0104794485461408, 0.218926317772384, -0.0733836748016421, -0.0969222399711355, 0.106081492024346, -0.0424414972239477, 0.64731043396934, -0.111853409387718, 0.268136626956931, 0.0646543227133795, -0.015763055802732, 0.276224876516878, -0.0479298263793521, 0.0768224649121947, 0.06831158762365, -0.0467961972757154, 0.358833704907215, -0.21840924714, -0.0432578950746319, 0.238901917847247, 0.0278807089348394, 0.318377804858212, 0.110391080003713, -0.0445720819943006, 0.111552448687767, 0.0163520468641842, 0.134371645886706, 0.0675148049838013, -0.0162665661219377, 0.544516631748631, 0.0745638007306002, 0.896157988170588, 0.386160204861871, 0.204920389253273, 0.120050613796646, -0.0287766173193111, 0.365990416370302, -0.141021500253617, -0.0505327060172186, 0.64341709049975, 0.0972868148480007, 1.0309411744441, 0.504545688195125, 0.269881051631523, 0.157569621813478, -0.0185888797306838, 0.516052762435139, -0.0489200922446117, 0.117691624886892, 0.756830460700165, 0.130654273242076, 1.08676740401039, 0.650008552519283, 0.264114237859539, 0.272224411649807, 0.027304926519956, 0.337542853019519, 0.0877308591965174, -0.116297117181099, 0.086325083873198, -0.0229572900127847, 0.303198939355942, -0.100867992238934, 0.00116079202496018, 0.0151080960897389, -0.0507439776908736, 0.191083407231621, -0.264894658518865, -0.146926733965544, -0.025855779258159, -0.0291275682122614, 0.130887364946405, -0.116968600937492, 0.0407171858384292, -0.0230957754202865, -0.0379693678069567, 0.16187250890692, -0.16409244885631, 0.0116667722284181, -0.158858480139539, -0.0970673916433943, 0.147025809530755, -0.441303298497789, -0.047903088554198, 0.552656528774668, 0.0811538601387113, 0.933862388982135, 0.436109147711403, 0.279419776729338, 0.499859834307837, 0.0627193036622084, 0.747816003292497, 0.289706869532577, 0.0472145125235594, 0.126717217646175, -0.0112199367872178, 0.425004176991688, -0.0117177221716314, 0.13451855075947, 0.322685776492725, 0.0192536372662355, 0.688207442709684, 0.128268337740226, 0.154698126699842, 0.0389843869194086, -0.00736326765897132, 0.0895755528611205, -0.0441065301724832, -0.0415257565639638, 0.818365798434759, 0.156031556329833, 0.803100416875055, 0.65756588396424, -0.10767204538354, 0.0187809772485371, -0.0384460901582786, 0.247861269667562, -0.16773921784439, 0.00615817063770194, 0.2004053519931, -0.00340596446389604, 0.293843185310706, -0.0688435655746093, -0.191876708915001, 0.340378966577248, 0.0193549175062074, 0.700280308444099, 0.11955400915754, 0.122044876350749, 0.210062986885956, 0.0223486800742856, 0.226134020615594, 0.0629878050026091, -0.12662113745027, 0.136573677756846, -0.00848879933728854, 0.493937188352719, 0.0239122936153988, 0.218609389598569, -0.337693324259126, -0.147500205463844, 0.00856544239164565, -0.675647934414837, -0.090824189271717, -0.0998724768222158, -0.0896437453723574, 0.510001317914872, -0.303787164549873, 0.325629601868061, 0.281689490545879, 0.0209840263792907, 0.428446255445281, 0.0739807489622967, -0.0659958956741971, 0.250223392453616, -0.000813585743334732, 0.381288683201417, -0.0585362491984011, -0.195867678588126, -0.00992509286266662, -0.05527854059481, 0.275586697684246, -0.242518262158118, 0.00407961475518732, 0.171731789538901, -0.00664099648910335, 0.397584914826519, -0.0280641840570226, 0.00430846264490201, 0.0164304931833799, -0.0427499626726733, 0.0811316841946604, -0.256062769419201, -0.241046535765955, 0.212853201953615, 0.0118033434986038, 0.487380580501563, 0.0914202284693827, 0.140412124539736, -0.0576502590156587, -0.0405263278365005, 0.029420874133044, -0.203065598648403, -0.0870772126959173, 0.214065117902749, 0.0161741317121676, 0.392147426694335, 0.0826758459953107, 0.0405698643593549, 0.187012980074152, 0.0105765442181085, 0.378007472994366, 0.0627845206662418, 0.0576163355599231, 0.265714281983581, 0.00470145111698363, 0.551034325681781, 0.0283513303048221, 0.0271889138111905, 0.25084056088617, 0.00136784868934601, 0.499980310746212, -0.000478930229391037, -0.0231985699139942, 0.135868295242648, 0.0078985750325256, 0.254619515088422, 0.0394717519939378, 0.0165945645518786, 0.0552031212777881, -0.0377190499749914, 0.188688168032211, -0.213918074523815, -0.169868059925306, 0.186558849962315, -0.00745454265556756, 0.32704676447254, -0.0714107067809211, -0.137416535656631, 0.344675474020897, 0.0382339055982479, 0.386949883870143, 0.119334398181378, -0.175361814676677 };
  
  CHM_SP rotatedSparseDesignMatrix = A_SLOT(regression);
  R_CheckStack();
  
  int *dims = DIMS_SLOT(regression);
  int numObservations  = dims[n_POS];
  int numUnmodeledCoef = dims[p_POS];
  
  int *indicesForColumn = (int *) rotatedSparseDesignMatrix->p;
  int arrayLength = indicesForColumn[rotatedSparseDesignMatrix->ncol];
  
  return (allApproximatelyEqual(correctWeightedDenseDesignMatrix, cache->weightedDenseDesignMatrix,
                                numObservations * numUnmodeledCoef, TEST_TOLERANCE) &&
          allApproximatelyEqual(correctWeightedResponse, cache->weightedResponse,
                                numObservations, TEST_TOLERANCE) &&
          allApproximatelyEqual(correctWeightedSparseDesignMatrix, Cx_SLOT(regression), arrayLength, TEST_TOLERANCE));
}

static int updateAugmentedDesignMatrixFactorizations_test(SEXP regression, MERCache *cache)
{
  updateAugmentedDesignMatrixFactorizations(regression, cache);
    
  double correctUpperLeftBlockLeftFactorizationValues[] =  { 1.11802518358661, -0.0002141773623994, -0.0103743890251717, 0.112175775126362, 0.000611701259321441, 1.00000009175111, 8.88853631700365e-06, -9.61096069059544e-05, -5.24091476176882e-07, 1.0002152496435, -0.00465438444975504, -2.53806387883136e-05, 1.49434440463776, -0.123695697626689, 0.121341873876496, 0.148610154626757, 0.0078624508320961, 0.087458489693987, 0.0165136442005272, 0.00886709678487256, -0.00708371837023432, -0.0360332718425236, -0.00153527331140769, 1.09301874426808, 0.111363295427212, 0.0546859834083894, -0.013763379390397, -0.0266724406983469, -0.00569893210222258, 0.00326293925792616, 0.0165955387202434, 0.0129333279583555, 0.000529829652003012, 1.08077643694642, 0.0238682472051938, -0.0118054663036574, 0.0125724898921121, -0.0113752346175748, 0.00142414264075875, 0.0145482539127655, -0.00554113019414657, 0.00105755543157494, 1.23367632664908, -0.13583688787858, 0.0710446258321875, 0.0385047413012036, 0.0252987951220653, -0.0510040619523389, 0.0744825871948285, 0.00409653388938489, -0.0014601068911009, -0.0454456018791, 0.00747082731502876, 1.13044686273235, -0.0372359511057397, 0.0163313918520155, 0.0104593890839883, 0.0573729017009881, 0.0300765701554647, 0.00173750291319356, 0.00311420663570389, 0.0502741330732691, 0.00301677036636861, 1.07625611421307, -0.0266905376762697, -0.000191383355038739, -0.00976708831276642, -0.0332918639629976, -0.00283961632832281, -0.000776047624604908, -0.00935127824113051, -0.00333927399718807, 1.59534784551812, -0.000720876059067456, -8.39944148761228e-05, -0.00694047026945949, -0.00383530110414246, 0.143131589001976, -9.62261563715187e-06, -0.000126663407066425, 0.00342190535741926, 0.360695294466859, 0.827250780468855, 0.131069246131821, 0.49310960802699, 0.0196852330476487, 0.0460918459924945, -0.000158032241698, 1.42522678448117, -4.24841600220265e-08, 0.00047949094306412, -0.00154925417665444, -1.34532344485164e-05, 0.0904101361316766, -6.4066027040981e-08, 0.00043821751731743, 1.05783039923595, 0.024095424282278, 0.407352153258517, 0.0113583238234561, 0.09014155656101, 0.000915440803418372, -0.00015528020537116, 1.16300818037506, -0.000779788006279697, -0.000294827194407741, 3.46246910567927e-06, 2.60767961239353e-09, 0.035580136356755, 0.000340657914623858, 0.422148558111065, 0.104615889173929, 0.18816562262877, 0.0141027699572739, 0.100360341148512, 0.0299722279036318, 2.737290464835e-05, 1.06397886488814, 0.000252651807293217, 0.000346390181292402, -0.000285851358630697, 0.00085164562736037, 0.00298181511540217, 0.0655879378354727, 0.06963437478599, 0.003878723396168, -0.00130787280125672, 0.0282223189065852, 0.00101762559477755, 0.000290846095002738, 1.35667137265052, -1.08369230349968e-05, 9.53476369812865e-05, 0.000285134331639584, 0.00160715131865885, 0.199375447091868, 0.557840686518341, 0.180806106149443, 0.229309472089971, -0.00164251803524425, 0.065780176185776, 0.0925383269703125, 1.01269147097472, 1.04712558367651e-06, 5.58651873536069e-06, -0.000131657405732353, -0.0264918575883484, 0.0841938308571651, 0.0459847264940497, 0.0667031802307761, 0.00486742863842052, -0.000585910182466902, -1.1287336786312e-05, 1.00811528346643, 2.18955046204251e-07, -0.000266897685110801, 0.0596112872664278, 0.00259317337077249, 0.0717214881817756, 0.00120632337311161, 0.0182764330089838, 9.24888394263492e-05, 4.1851840582522e-06, 1.00524340978596, -0.000401848376530717, 0.0407984856486992, -0.0024250103296907, 0.0365469016387285, 0.00132368240887922, 0.0149017355301842, -0.000630515549440507, -6.23543360801079e-05, 1.00695691055514, -0.00647668969213502, -0.00656502771749949, -0.00114837857450217, -0.00185289083592358, -0.00205251022275534, -0.000303890613354346, -4.83688587803921e-05, 2.19897401883313, -0.219053251398891, 0.236900625398117, -0.109066628100277, 0.174963144946136, -0.019747937378472, -0.0139369160768567, 1.86407758047341, -0.10253937213632, 0.480741167686214, 0.0042309486404125, 0.202900543430143, 0.0218884451090307, 1.52395681112607, -0.0271917022382678, 0.222887788935849, 0.0010337032539652, 0.0768924732035062, 1.17771557958903, 0.00934560258189148, 0.0670464877218782, 0.0203583362535387, 1.13745698088229, -0.00190679940750244, 0.00795164241849972, 1.10782901791163, 0.0112810581896278, 1.01169419110975 };
  double correctOffDiagonalBlockRightFactorization[] = { 0.0778998552371798, -6.67427923403457e-05, -0.00323221189642478, 0.282094028232767, -0.0845973344211915, -0.0218540538816995, 0.15805724104276, -0.0771677797707074, -0.00524129035496305, 0.280796442467815, 0.265517238995151, 0.116419460580719, 0.0736995024652554, 0.180458757145431, -0.00520224769456156, 0.00123558969071976, -0.00496400170240543, -0.0306645913999253, 0.451093951549468, 0.294066336255723, -0.214557305201791, -0.0192916527673542, -0.0556401109157742, -0.0196347365805808, -0.0170345510007762, 0.00149228013720205, -1.27855107057198e-06, -6.19175170169094e-05, -0.0872999568298233, 0.0812900319724784, 0.0280276365255666, -0.0862099159522169, 0.141842223087893, -0.0397799451354936, 0.356807608361328, 0.221217953078642, 0.0877060589516823, 0.0102091698539887, 0.230575038171117, 0.0612025737515461, 0.0387981580781825, 0.0243979388918357, 0.0317834542361428, 0.0404633812358011, 0.162623947906501, 0.426307070243975, 0.155318364187866, 0.0365013938425808, -0.000862573903665115, 0.0350199633885392, -0.00925388161150203, 7.92851150824531e-06, 0.000383960998922727, 0.108236093002336, 0.099335271624161, 0.138718906206799, 0.0633713036005644, -0.0332142049414317, 0.131220404904977, 0.0571546314737265, 0.0811101370724144, 0.107877898359283, 0.0243089636684658, 0.0299462991079491, 0.00287067844345891, 0.0157590558092529, 0.0249798293855021, -0.0143013323453459, 0.142629286613056, 0.194638060595372, 0.20394073838379, 0.0706578970098602, 0.227790047872929, 0.181674915802529, 0.0152590558588238 };
  double correctLowerRightBlockRightFactorization[] = { 0.577681597641182, 0, 0, -0.258693382460513, 0.506237479635821, 0, -0.142287523232981, -0.0305303008294976, 0.72482670034976 };
  
  int *dims = DIMS_SLOT(regression);
//  int numObservations  = dims[n_POS];
  int numUnmodeledCoef = dims[p_POS];
  int numModeledCoef   = dims[q_POS];

  SEXP upperLeftBlockLeftFactorizationExp = GET_SLOT(regression, lme4_LSym);
  int *ulfblf_IndicesForColumn = INTEGER(GET_SLOT(upperLeftBlockLeftFactorizationExp, install("p")));
  
  int numFactorizationNonZeroes = ulfblf_IndicesForColumn[numModeledCoef];
  
  double *ulfblf_values = REAL(GET_SLOT(upperLeftBlockLeftFactorizationExp, install("x")));
  double *offDiagonalBlockRightFactorization = RZX_SLOT(regression);
  double *lowerRightBlockRightFactorization  = RX_SLOT(regression);
  
  return(allApproximatelyEqual(ulfblf_values,
                               correctUpperLeftBlockLeftFactorizationValues,
                               numFactorizationNonZeroes, TEST_TOLERANCE) && 
         allApproximatelyEqual(offDiagonalBlockRightFactorization, correctOffDiagonalBlockRightFactorization,
                               numModeledCoef * numUnmodeledCoef, TEST_TOLERANCE) &&
         allApproximatelyEqual(lowerRightBlockRightFactorization, correctLowerRightBlockRightFactorization,
                               numUnmodeledCoef * numUnmodeledCoef, TEST_TOLERANCE));
}

//extern void calculateProjections(SEXP regression, MERCache *cache);
// extern void calculatePenalizedWeightedResidualSumOfSquaresFromProjections(SEXP regression, MERCache *cache);
// extern void rotateProjections(SEXP regression, MERCache *cache);

static int calculateJointMode_test(SEXP regression, MERCache *cache)
{
  calculateProjections(regression, cache);
  calculatePenalizedWeightedResidualSumOfSquaresFromProjections(regression, cache);
  rotateProjections(regression, cache);
  
  double correctModeledCoef[] = { -0.00231964142163336, 2.22198029460337e-06, 0.000107628965658346, 0.193723335728012, -0.265183671406125, -0.204410889696908, 0.008554735301422, 0.00155769498291329, 0.0367447612383233, -0.155334369948256, -0.377100041255745, 0.0767531433276913, 0.0563610252081127, 0.399320242668198, -0.0790506013889741, -0.0807488005279119, -0.0371839245659116, -0.0161373444482706, -0.272197559355291, 0.0722391297474915, 0.129524138500678, 0.13409961594224, 0.0120207928952145, 0.155537706597713, 0.213120670931068 };
  double correctUnmodeledCoef[] = { 5.33671607754357, 1.33018552831726, 4.59947659629086 };
  
  double *modeledCoef   = U_SLOT(regression);
  double *unmodeledCoef = FIXEF_SLOT(regression);
  
  int *dims = DIMS_SLOT(regression);
  int numUnmodeledCoef = dims[p_POS];
  int numModeledCoef   = dims[q_POS];
  
  return(allApproximatelyEqual(modeledCoef, correctModeledCoef, numModeledCoef, TEST_TOLERANCE) &&
         allApproximatelyEqual(unmodeledCoef, correctUnmodeledCoef, numUnmodeledCoef, TEST_TOLERANCE));
}

static int calculateDevianceAndCommonScale_test(SEXP regression)
{
  updateDeviance(regression);
  profileCommonScale(regression);
  
  double correctWeightedResidualSumOfSquares = 2.9906203297744;
  double correctPenaltySumOfSquares = 0.684162111054328;
  double correctDeviance = 20.7613927676771;
  double correctCommonScale = 0.271100809324824;

  double *deviances = DEV_SLOT(regression);
  
  return(fabs(deviances[wrss_POS] - correctWeightedResidualSumOfSquares) <= TEST_TOLERANCE &&
         fabs(deviances[usqr_POS] - correctPenaltySumOfSquares) <= TEST_TOLERANCE &&
         fabs(deviances[ML_POS] - correctDeviance) <= TEST_TOLERANCE &&
         fabs(deviances[sigmaML_POS] - correctCommonScale) <= TEST_TOLERANCE);
}

static int newtonsMethodUpdateCommonScale_test(SEXP regression, MERCache *cache)
{
  double priorModeledCoefSD = 1.5;
  int *dims = DIMS_SLOT(regression);
  int numUnmodeledCoefs = dims[p_POS];
  
  SEXP unmodeledCoefPrior = GET_SLOT(regression, blme_unmodeledCoefficientPriorSym);
  SET_SLOT(unmodeledCoefPrior, blme_prior_typeSym, ScalarInteger(PRIOR_TYPE_DIRECT));
  SET_SLOT(unmodeledCoefPrior, blme_prior_familiesSym, ScalarInteger(PRIOR_FAMILY_GAUSSIAN));
  SET_SLOT(unmodeledCoefPrior, blme_prior_scalesSym, ScalarInteger(PRIOR_SCALE_ABSOLUTE));
  double *hyperparameters = REAL(ALLOC_SLOT(unmodeledCoefPrior, blme_prior_hyperparametersSym, REALSXP, 2));
  hyperparameters[0] = 2.0 * ((double) numUnmodeledCoefs) * log(priorModeledCoefSD);
  hyperparameters[1] = 1.0 / priorModeledCoefSD;
  
  // now that we have set a prior, call a few updates to fill in the cache
  updateAugmentedDesignMatrixFactorizations(regression, cache);
  calculateProjections(regression, cache);
  calculatePenalizedWeightedResidualSumOfSquaresFromProjections(regression, cache);
  
  // perform the update
  double commonScale = performOneStepOfNewtonsMethodForCommonScale(regression, cache);
  
  double correctCommonScale = 0.277388800783761;
  double correctLowerRightBlockRightFactorization[] = { 0.606558822051857, 0, 0, -0.246377434547006, 0.544696601805623, 0, -0.135513458476428, -0.0220934004058438, 0.749597988699522 };
  double correctPenalizedWeightedResidualSumOfSquares = 5.2206764279238;
  
  
  
  int arrayLength = numUnmodeledCoefs * numUnmodeledCoefs;
  
  int firstTestSuiteResult = fabs(commonScale - correctCommonScale) <= TEST_TOLERANCE &&
    fabs(correctPenalizedWeightedResidualSumOfSquares - DEV_SLOT(regression)[pwrss_POS]) <= TEST_TOLERANCE &&
    allApproximatelyEqual(RX_SLOT(regression), correctLowerRightBlockRightFactorization, arrayLength, TEST_TOLERANCE);
  
  if (!firstTestSuiteResult) return(FALSE);
  
  double diagonalPriorSds[] = { 1.0, 1.0 / 1.5, 1.0 / 2.0 };
  hyperparameters = REAL(ALLOC_SLOT(unmodeledCoefPrior, blme_prior_hyperparametersSym, REALSXP, 1 + numUnmodeledCoefs));
  hyperparameters[0] = log(1.5) + log(2.0);
  Memcpy(hyperparameters + 1, diagonalPriorSds, numUnmodeledCoefs);
  
  updateAugmentedDesignMatrixFactorizations(regression, cache);
  calculateProjections(regression, cache);
  calculatePenalizedWeightedResidualSumOfSquaresFromProjections(regression, cache);
  
  commonScale = performOneStepOfNewtonsMethodForCommonScale(regression, cache);
  
  double correctCommonScale2 = 0.287957767760753;
  double correctLowerRightBlockRightFactorization2[] = { 0.645473240550702, 0, 0, -0.231523783002205, 0.553577979295315, 0, -0.127343596266682, -0.0146859400666481, 0.742192316542082 };
  double correctPenalizedWeightedResidualSumOfSquares2 = 5.86501307958162;
  
  return (fabs(commonScale - correctCommonScale2) <= TEST_TOLERANCE &&
          fabs(correctPenalizedWeightedResidualSumOfSquares2 - DEV_SLOT(regression)[pwrss_POS]) <= TEST_TOLERANCE &&
          allApproximatelyEqual(RX_SLOT(regression), correctLowerRightBlockRightFactorization2, arrayLength, TEST_TOLERANCE));
}
