% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/set.R
\name{set_calc_concentrations}
\alias{set_calc_concentrations}
\title{Calculate concentrations for the set using contained calibrators.}
\usage{
set_calc_concentrations(data, cal_names, cal_values, col_names = name,
  col_values = value, col_target = conc, col_real = real,
  col_recov = recovery, model_func = fit_linear,
  interpolate_func = interpolate_linear)
}
\arguments{
\item{data}{A tibble containing the data.}

\item{cal_names}{A vector of strings containing the names of the samples used
as calibrators.}

\item{cal_values}{A numeric vector with the known concentrations of those
samples (must be in the same order).}

\item{col_names}{The name of the column where the \code{cal_names} can be
found.}

\item{col_values}{The name of the column holding the raw values.}

\item{col_target}{The name of the column to created for the calculated
concentration.}

\item{col_real}{The name of the column to create for the known
concentrations.}

\item{col_recov}{The name of the column to create for the recovery of the
calibrators.}

\item{model_func}{A function generating a model to fit the calibrators,
e.g. \code{\link[=fit_linear]{fit_linear()}}, \code{\link[=fit_lnln]{fit_lnln()}}.}

\item{interpolate_func}{A function used to interpolate the concentrations of
the other samples, based on the model, e.g.
\code{\link[=interpolate_linear]{interpolate_linear()}}, \code{\link[=interpolate_lnln]{interpolate_lnln()}}.}
}
\value{
A tibble containing all original and additional columns.
}
\description{
If the data set is generated, for example by reading extinction rates or
relative light units from a plate, these raw values can be converted to
concentrations using data fields with known concentrations (calibrators).
}
\details{
If the data set contains samples with known concentrations (calibrators)
those can be used to interpolate the concentrations of the other samples.
}
\examples{
# generate data
library("tibble")

data <- tibble(
  name = c("CAL1", "CAL2", "CAL3", "A", "B", "C"),
  value = c(1, 5, 10, 2, 4, 6)
)

data

# the known concentration of the calibrators
cals <- c(1, 5, 10)
names(cals) <- c("CAL1", "CAL2", "CAL3")

set_calc_concentrations(
  data = data,
  cal_names = names(cals),
  cal_values = cals
)

# to set column names use notation like in dplyr / tidyverse
# set the name of the column holding the final concentration to "my_protein"
set_calc_concentrations(
  data = data,
  cal_names = names(cals),
  cal_values = cals,
  col_target = my_protein
)

\dontrun{
# notice that col_target is given a string
# this will fail
set_calc_concentrations(
  data = data,
  cal_names = names(cals),
  cal_values =  cals,
  col_target = "my_protein"
)
}

# simulate data which has to be transformed to get a good fit
cals <- exp(cals)
data$value <- exp(data$value)

# use ln-transformation on values and known concentrations prior to
# fitting a model

data <- set_calc_concentrations(
  data = data,
  cal_names = names(cals),
  cal_values = cals,
  model_func = fit_lnln,
  interpolate_func = interpolate_lnln
)

data

# inspect goodnes of fit
plot_lnln(data$real, data$value)

rm(cals, data)
}
\seealso{
Other set functions: \code{\link{set_calc_variability}},
  \code{\link{set_read}}, \code{\link{sets_read}}
}
