#' Review Experimental Effect
#'
#' This function visualizes or extracts the model-estimated effect 
#' of reinforcement learning parameters for each subject.
#'
#' @param data [data.frame]  
#'  Raw data. Must include the following columns:  
#'   - `"sub"`: Subject ID  
#'   - `"time_line"`: Trial index  
#'   - `"L_choice"` / `"R_choice"`: Left and right choice identifiers  
#'   - `"L_reward"` / `"R_reward"`: Left and right reward values  
#'
#' @param result [data.frame]  
#'  Output data generated by the `fit_p()` function. Each row represents 
#'   model fit results for a subject.
#'
#' @param model [function]  
#'   A model function to be applied in evaluating the experimental effect.
#'
#' @param model_name [character]  
#'   A character string specifying the name of the model to extract from the result.
#'
#' @param param_prefix [character]  
#'   A prefix string used to identify parameter columns in the `result` data (e.g., "param_").
#'
#' @param n_trials [integer]  
#'   Number of total trials in the experimental task.
#'
#' @return A list where each element contains the extracted results for one subject.
#' 
#' @export
#'
rev_e <- function(
  data, 
  result, 
  model,
  model_name,
  param_prefix, 
  n_trials
) {
  Subject <- "Subject"
  id <- unique(data[[Subject]])
  
  fit_model <- "fit_model"
  result <- result[result[[fit_model]] == model_name, ]
  
  res <- list()
  
  for (i in 1:length(id)) {
    
    params <- stats::na.omit(unlist(result[i, grep(param_prefix, names(result))]))
    
    binaryRL.env <- new.env()
    binaryRL.env$data <- data
    binaryRL.env$id <- id[i]
    binaryRL.env$n_params <- length(params)
    binaryRL.env$n_trials <- n_trials
    binaryRL.env$mode <- "review"
    
    
    obj_func <- model
    
    environment(obj_func) <- binaryRL.env
    res[[i]] <- obj_func(params = params)[[1]]
  }
  
  return(res)
}
