% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/InteractionTest.R
\name{diagnosticTest}
\alias{diagnosticTest}
\title{Testing Procedure for Bicluster Diagnostics}
\usage{
diagnosticTest(BCresult, data, number = 1:BCresult@Number, verbose = TRUE,
  statistics = c("F", "Tukey"), sampling = TRUE, samplingtypes = NULL,
  nSim = 1000, alpha = 0.05, save_F = FALSE)
}
\arguments{
\item{BCresult}{An object of class \code{biclust} containing the result of a biclustering algorithm}

\item{data}{data matrix, which \code{biclust} function was applied to}

\item{number}{Vector of bicluster numbers of which the diagnostics should be calculated. (default = all available biclusters)}

\item{verbose}{Boolean value to print progression of computed statistics.}

\item{statistics}{Vector select which statistics to compute. (default = \code{c("F","Tukey")})
\itemize{
\item \code{"F"} (Row and column F statistics of two-way ANOVA with one replicate for cell)
\item \code{"Tukey"} (Tukey's test for non-additivity)
\item \code{"ModTukey"} (\code{\link[additivityTests]{mtukey.test}})
\item \code{"Tusell"} (\code{\link[additivityTests]{tusell.test}})
\item \code{"Mandel"} (\code{\link[additivityTests]{mandel.test}})
\item \code{"LBI"} (\code{\link[additivityTests]{lbi.test}})
\item \code{"JandG"} (\code{\link[additivityTests]{johnson.graybill.test}})
}}

\item{sampling}{Boolean value to apply sampling methods to compute statistical significance (default=\code{TRUE}). 
If \code{FALSE} only the \code{"Theoretical"} p-values are computed. 
If \code{TRUE}, both the \code{"Theoretical"} and \code{samplingtypes} p-values are computed.}

\item{samplingtypes}{Vector of sampling methods for \code{sampling=TRUE}. (default=\code{NULL}=\code{c("Permutation","SemiparPerm")})
\itemize{
\item \code{"Permutation"} 
\item \code{"SemiparPerm"} 
\item \code{"SemiparBoot"} 
\item \code{"PermutationCor"} 
\item \code{"SamplingCor"}
\item \code{"NormSim"}  
}
See Details for more info.}

\item{nSim}{Number of permutations/bootstraps.}

\item{alpha}{Significance level (default=0.05)}

\item{save_F}{Option to save the permuted/bootstraped statistics. This is necessary for \code{\link{diagnosticPlot2}}}
}
\value{
Returns a list with \code{length(number)} elements. 
Each element corresponds with the requested biclusters and is a list containing:
\itemize{
\item \code{table}: a data frame where each row is \code{statistics} and \code{samplingtypes} (including Theoretical) combination. The data frame contains the \code{Method}, \code{Type} (p-value type), \code{StatVal} (statistical value), \code{CritVal} (critical value), \code{pVal} and \code{Sign} (0/1 significance indicator based on \code{alpha}).
\item \code{save_F}: if \code{save_F=TRUE}, a (\code{nSim} x number of permuted/bootstrapped p-values) matrix contained the sampled statistics.
}
}
\description{
Calculate the statistical value of the row, column and multiplicative effect based on discovered biclusters in the data. 
Additionally multiple sampling methods are available to compute the statistical significance through p-values.
}
\details{
Due to the uncertainty of discovering the true bicluster(s) in the data, it's often advisable to not rely on the theoretical p-values but instead retrieve the p-values through a sampling procedure.

Available p-values/sampling types for each statistical method:
\itemize{
\item \code{"F"}: \code{"Theoretical"} and \code{"Permutation"} for both row and column effect.
\item \code{"Tukey"}: \code{"Theoretical"}, \code{"SemiparPerm"} and \code{"SemiparBoot"}.
\item \code{"ModTukey"}: \code{"Theoretical"}, \code{"SemiparPerm"}, \code{"SemiparBoot"}, \code{"PermutationCor"} and \code{"SamplingCor"}.
\item \code{"Tusell"}: \code{"SemiparPerm"}, \code{"SemiparBoot"} and \code{"NormSim"}.
\item \code{"Mandel"}: \code{"Theoretical"}, \code{"SemiparPerm"} and \code{"SemiparBoot"}.
\item \code{"LBI"}: \code{"SemiparPerm"}, \code{"SemiparBoot"} and \code{"NormSim"}.
\item \code{"JandG"}: \code{"SemiparPerm"}, \code{"SemiparBoot"} and \code{"NormSim"}.
} 
More info on the sampling types can be found in the secion below.
If available, the \code{"Theoretical"} will always be computed.
By default when \code{sampling=TRUE}, a sampling method without replacement is chosen, namely \code{"Permutation"} and \code{"SemiparPerm"}.

When \code{save_F=TRUE}, the null distributions of the statistics can be visualised with \code{\link{diagnosticPlot2}}.

\emph{Disclaimer:} While their functionality did not change, some functions of the \code{\link{additivityTests}} package were altered in order to be able to return the permuted/bootstrapped statistics and p-values.
}
\section{Sampling Types}{
 
For each sampling type a permuted/bootstrapped BC is created as following:
\itemize{
\item \code{"Permutation"}: Sample a BC from the entire dataset with replacement.
\item \code{"SemiparPerm"}: A semi-parametric permutation procedure. Two-way ANOVA is applied on the original BC and the residual matrix extracted. A new residual matrix is created by sampling \emph{without replacement} from the original residual matrix. The sampled BC is then generated by adding this sampled residual matrix on top the mean, row and column effect of the ANOVA procedure of the original BC.
\item \code{"SemiparBoot"}: A semi-parametric bootstrapping procedure. Two-way ANOVA is applied on the original BC and the residual matrix extracted. A new residual matrix is created by sampling \emph{with replacement} from the original residual matrix. The sampled BC is then generated by adding this sampled residual matrix on top the mean, row and column effect of the ANOVA procedure of the original BC.
\item \code{"PermutationCor"}: See \code{correction=1} parameter of \code{\link[additivityTests]{mtukey.test}}. More info in Simecek and Simeckova (2012).
\item \code{"SamplingCor"}: See \code{correction=2} parameter of \code{\link[additivityTests]{mtukey.test}}. More info in Simecek and Simeckova (2012).
\item \code{"NormSim"}: Sample a BC from a standard normal distribution. This sampling procedure is used for some methods in the \code{\link{additivityTests}} package.
}
}

\examples{
\dontshow{
test <- matrix(rnorm(5000),100,50)
roweff <- sample(1:5,10,replace=TRUE)
coleff <- sample(1:5,10,replace=TRUE)
test[11:20,11:20] <- test[11:20,11:20] +
  matrix(coleff,nrow=10,ncol=10,byrow=TRUE) +
  matrix(roweff,nrow=10,ncol=10) +
  roweff \%*\% t(coleff)


#Apply Plaid Biclustering
res <- biclust(test, method=BCPlaid())

#Apply default diagnosticTest
out <- diagnosticTest(BCresult=res, data=test, save_F=TRUE, number=1,
                      statistics=c("F","Tukey"),
                      samplingtypes=c("Permutation","SemiparPerm"))
}

\dontrun{
#Random matrix with embedded bicluster (with multiplicative effect)
test <- matrix(rnorm(5000),100,50)
roweff <- sample(1:5,10,replace=TRUE)
coleff <- sample(1:5,10,replace=TRUE)
test[11:20,11:20] <- test[11:20,11:20] +
  matrix(coleff,nrow=10,ncol=10,byrow=TRUE) +
  matrix(roweff,nrow=10,ncol=10) +
  roweff \%*\% t(coleff)


#Apply Plaid Biclustering
res <- biclust(test, method=BCPlaid())

#Apply default diagnosticTest
out <- diagnosticTest(BCresult=res, data=test, save_F=TRUE, number=1,
                      statistics=c("F","Tukey","ModTukey","Tusell","Mandel","LBI","JandG"),
                      samplingtypes=c("Permutation","SemiparPerm","SemiparBoot",
                      "PermutationCor","SamplingCor","NormSim"))

out[[1]]$table
}

}
\references{
Tukey, J.W.: One Degree of Freedom for Non-additivity, \emph{Biometrics} \strong{5}, pp. 232-242, 1949.

Simecek, Petr, and Simeckova, Marie. "Modification of Tukey's additivity test." \emph{Journal of Statistical Planning and Inference}, \strong{2012}.
}
\author{
Ewoud De Troyer
}
