#' Accept Reserved Instances Exchange Quote
#' 
#' Accepts the Convertible Reserved Instance exchange quote described in
#' the GetReservedInstancesExchangeQuote call.
#' 
#' @param ReservedInstanceId List. The IDs of the Convertible Reserved Instances to exchange for another Convertible Reserved Instance...
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TargetConfiguration List. The configuration of the target Convertible Reserved Instance to exchange for your current Convertible...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ReservedInstanceId:
#' The IDs of the Convertible Reserved Instances to exchange for another
#' Convertible Reserved Instance of the same or higher
#' value.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TargetConfiguration:
#' The configuration of the
#' target Convertible Reserved Instance to exchange for your current
#' Convertible Reserved Instances.
#' @return A list object or a character vector
#' @export
ec2_accept_reserved_instances_exchange_quote <- function(ReservedInstanceId, DryRun = NULL, 
    TargetConfiguration = NULL, simplify = TRUE, others = list()) {
    ReservedInstanceId <- list_to_array("ReservedInstanceId", ReservedInstanceId)
    TargetConfiguration <- list_to_array("TargetConfiguration", TargetConfiguration)
    parameters <- c(others, list(DryRun = DryRun), ReservedInstanceId, TargetConfiguration)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AcceptReservedInstancesExchangeQuote", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Accept Transit Gateway Multicast Domain Associations
#' 
#' Accepts
#' a request to associate subnets with a transit gateway multicast
#' domain.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param TransitGatewayAttachmentId String. The ID of the transit gateway attachment.\[optional\]
#' @param SubnetIds List. The IDs of the subnets to associate with the transit gateway multicast domain.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the
#' transit gateway multicast domain.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' transit gateway attachment.
#' @section SubnetIds:
#' The IDs of the subnets
#' to associate with the transit gateway multicast
#' domain.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_accept_transit_gateway_multicast_domain_associations <- function(TransitGatewayMulticastDomainId = NULL, 
    TransitGatewayAttachmentId = NULL, SubnetIds = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    SubnetIds <- list_to_array("SubnetIds", SubnetIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun), SubnetIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AcceptTransitGatewayMulticastDomainAssociations", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Accept Transit Gateway Peering Attachment
#' 
#' Accepts a transit gateway peering attachment
#' request. The peering attachment must be in the `pendingAcceptance`
#' state.
#' 
#' @param TransitGatewayAttachmentId String. The ID of the transit gateway attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the transit
#' gateway attachment.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_accept_transit_gateway_peering_attachment <- function(TransitGatewayAttachmentId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AcceptTransitGatewayPeeringAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Accept Transit Gateway Vpc Attachment
#' 
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_accept_transit_gateway_vpc_attachment <- function(TransitGatewayAttachmentId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AcceptTransitGatewayVpcAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Accept Vpc Endpoint Connections
#' 
#' Accepts one or more interface VPC endpoint
#' connection requests to your VPC endpoint service.
#' 
#' @param ServiceId String. The ID of the VPC endpoint service.
#' @param VpcEndpointId List. The IDs of one or more interface VPC endpoints.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ServiceId:
#' The ID of the VPC endpoint
#' service.
#' @section VpcEndpointId:
#' The IDs of one or more interface VPC
#' endpoints.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_accept_vpc_endpoint_connections <- function(ServiceId, VpcEndpointId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    VpcEndpointId <- list_to_array("VpcEndpointId", VpcEndpointId)
    parameters <- c(others, list(ServiceId = ServiceId, DryRun = DryRun), VpcEndpointId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AcceptVpcEndpointConnections", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Accept Vpc Peering Connection
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VpcPeeringConnectionId String. The ID of the VPC peering connection. You must specify this parameter in the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section VpcPeeringConnectionId:
#' The ID of the VPC peering
#' connection. You must specify this parameter in the
#' request.
#' @return A list object or a character vector
#' @export
ec2_accept_vpc_peering_connection <- function(DryRun = NULL, VpcPeeringConnectionId = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun, VpcPeeringConnectionId = VpcPeeringConnectionId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AcceptVpcPeeringConnection", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Advertise Byoip Cidr
#' 
#' @param Cidr String. The address range, in CIDR notation.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Cidr:
#' The address range, in CIDR notation. This
#' must be the exact range that you provisioned. You can\'t advertise only
#' a portion of the provisioned range.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_advertise_byoip_cidr <- function(Cidr, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(Cidr = Cidr, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AdvertiseByoipCidr", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Allocate Address
#' 
#' @param Domain String.   Indicates whether the Elastic IP address is for use with instances in a VPC or instances in EC2-Classic....\[optional\]
#' @param Address String. \[EC2-VPC\] The Elastic IP address to recover or an IPv4 address from an address pool.\[optional\]
#' @param PublicIpv4Pool String. The ID of an address pool that you own.\[optional\]
#' @param NetworkBorderGroup String.   A unique set of Availability Zones, Local Zones, or Wavelength Zones from which AWS advertises...\[optional\]
#' @param CustomerOwnedIpv4Pool String. The ID of a customer-owned address pool.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to assign to the Elastic IP address.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Domain:
#' 
#' Indicates whether the Elastic IP address is for use with instances in a
#' VPC or instances in EC2-Classic.
#' 
#' Default: If the Region supports EC2-Classic, the default is `standard`.
#' Otherwise, the default is `vpc`.
#' 
#' @section Address:
#' \[EC2-VPC\] The Elastic IP address to recover or an
#' IPv4 address from an address pool.
#' @section PublicIpv4Pool:
#' The ID of an
#' address pool that you own. Use this parameter to let Amazon EC2 select
#' an address from the address pool. To specify a specific address from the
#' address pool, use the `Address` parameter instead.
#' @section NetworkBorderGroup:
#' 
#' A unique set of Availability Zones, Local Zones, or Wavelength Zones
#' from which AWS advertises IP addresses. Use this parameter to limit the
#' IP address to this location. IP addresses cannot move between network
#' border groups.
#' 
#' Use
#' [DescribeAvailabilityZones](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html)
#' to view the network border groups.
#' 
#' You cannot use a network border group with EC2 Classic. If you attempt
#' this operation on EC2 classic, you will receive an
#' `InvalidParameterCombination` error. For more information, see [Error
#' Codes](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/errors-overview.html).
#' 
#' @section CustomerOwnedIpv4Pool:
#' The ID of a customer-owned address pool. Use this
#' parameter to let Amazon EC2 select an address from the address pool.
#' Alternatively, specify a specific address from the address
#' pool.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to assign to the
#' Elastic IP address.
#' @return A list object or a character vector
#' @export
ec2_allocate_address <- function(Domain = NULL, Address = NULL, PublicIpv4Pool = NULL, NetworkBorderGroup = NULL, 
    CustomerOwnedIpv4Pool = NULL, DryRun = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(Domain = Domain, Address = Address, PublicIpv4Pool = PublicIpv4Pool, 
        NetworkBorderGroup = NetworkBorderGroup, CustomerOwnedIpv4Pool = CustomerOwnedIpv4Pool, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AllocateAddress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Allocate Hosts
#' 
#' Allocates a Dedicated Host to your account. At a
#' minimum, specify the supported instance type or instance family, the
#' Availability Zone in which to allocate the host, and the number of hosts
#' to allocate.
#' 
#' @param AvailabilityZone String. The Availability Zone in which to allocate the Dedicated Host.
#' @param Quantity Integer. The number of Dedicated Hosts to allocate to your account with these parameters.
#' @param AutoPlacement String.   Indicates whether the host accepts any untargeted instance launches that match its instance type...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param InstanceType String.   Specifies the instance type to be supported by the Dedicated Hosts.\[optional\]
#' @param InstanceFamily String.   Specifies the instance family to be supported by the Dedicated Hosts.\[optional\]
#' @param TagSpecification List. The tags to apply to the Dedicated Host during creation.\[optional\]
#' @param HostRecovery String.   Indicates whether to enable or disable host recovery for the Dedicated Host.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZone:
#' The Availability Zone in which
#' to allocate the Dedicated Host.
#' @section Quantity:
#' The number of
#' Dedicated Hosts to allocate to your account with these
#' parameters.
#' @section AutoPlacement:
#' 
#' Indicates whether the host accepts any untargeted instance launches that
#' match its instance type configuration, or if it only accepts Host
#' tenancy instance launches that specify its unique host ID. For more
#' information, see [Understanding auto-placement and
#' affinity](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding)
#' in the *Amazon EC2 User Guide*.
#' 
#' Default: `on`
#' 
#' @section ClientToken:
#' Unique, case-sensitive identifier that you provide
#' to ensure the idempotency of the request. For more information, see
#' [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section InstanceType:
#' 
#' Specifies the instance type to be supported by the Dedicated Hosts. If
#' you specify an instance type, the Dedicated Hosts support instances of
#' the specified instance type only.
#' 
#' If you want the Dedicated Hosts to support multiple instance types in a
#' specific instance family, omit this parameter and specify
#' **InstanceFamily** instead. You cannot specify **InstanceType** and
#' **InstanceFamily** in the same request.
#' 
#' @section InstanceFamily:
#' 
#' Specifies the instance family to be supported by the Dedicated Hosts. If
#' you specify an instance family, the Dedicated Hosts support multiple
#' instance types within that instance family.
#' 
#' If you want the Dedicated Hosts to support a specific instance type
#' only, omit this parameter and specify **InstanceType** instead. You
#' cannot specify **InstanceFamily** and **InstanceType** in the same
#' request.
#' 
#' @section TagSpecification:
#' The tags to apply to the Dedicated Host during
#' creation.
#' @section HostRecovery:
#' 
#' Indicates whether to enable or disable host recovery for the Dedicated
#' Host. Host recovery is disabled by default. For more information, see
#' [Host
#' recovery](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' Default: `off`
#' 
#' @return A list object or a character vector
#' @export
ec2_allocate_hosts <- function(AvailabilityZone, Quantity, AutoPlacement = NULL, ClientToken = NULL, 
    InstanceType = NULL, InstanceFamily = NULL, TagSpecification = NULL, HostRecovery = NULL, 
    simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(AvailabilityZone = AvailabilityZone, Quantity = Quantity, 
        AutoPlacement = AutoPlacement, ClientToken = ClientToken, InstanceType = InstanceType, 
        InstanceFamily = InstanceFamily, HostRecovery = HostRecovery), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AllocateHosts", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Apply Security Groups To Client Vpn Target Network
#' 
#' Applies a security group to the
#' association between the target network and the Client VPN endpoint. This
#' action replaces the existing security groups with the specified security
#' groups.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param VpcId String. The ID of the VPC in which the associated target network is located.
#' @param SecurityGroupId List. The IDs of the security groups to apply to the associated target network.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section VpcId:
#' The ID of the VPC in which the associated
#' target network is located.
#' @section SecurityGroupId:
#' The IDs of the security
#' groups to apply to the associated target network. Up to 5 security
#' groups can be applied to an associated target
#' network.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_apply_security_groups_to_client_vpn_target_network <- function(ClientVpnEndpointId, VpcId, 
    SecurityGroupId, DryRun = NULL, simplify = TRUE, others = list()) {
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, VpcId = VpcId, 
        DryRun = DryRun), SecurityGroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ApplySecurityGroupsToClientVpnTargetNetwork", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Assign Ipv6 Addresses
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param Ipv6AddressCount Integer. The number of additional IPv6 addresses to assign to the network interface.\[optional\]
#' @param Ipv6Addresses List. One or more specific IPv6 addresses to be assigned to the network interface.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section Ipv6AddressCount:
#' The number of additional IPv6 addresses
#' to assign to the network interface. The specified number of IPv6
#' addresses are assigned in addition to the existing IPv6 addresses that
#' are already assigned to the network interface. Amazon EC2 automatically
#' selects the IPv6 addresses from the subnet range. You can\'t use this
#' option if specifying specific IPv6 addresses.
#' @section Ipv6Addresses:
#' One or
#' more specific IPv6 addresses to be assigned to the network interface.
#' You can\'t use this option if you\'re specifying a number of IPv6
#' addresses.
#' @return A list object or a character vector
#' @export
ec2_assign_ipv6_addresses <- function(NetworkInterfaceId, Ipv6AddressCount = NULL, Ipv6Addresses = NULL, 
    simplify = TRUE, others = list()) {
    Ipv6Addresses <- list_to_array("Ipv6Addresses", Ipv6Addresses)
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, Ipv6AddressCount = Ipv6AddressCount), 
        Ipv6Addresses)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssignIpv6Addresses", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Assign Private Ip Addresses
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param AllowReassignment Logical. Indicates whether to allow an IP address that is already assigned to another network interface or...\[optional\]
#' @param PrivateIpAddress List.   One or more IP addresses to be assigned as a secondary private IP address to the network interface....\[optional\]
#' @param SecondaryPrivateIpAddressCount Integer. The number of secondary IP addresses to assign to the network interface.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section AllowReassignment:
#' Indicates whether to allow an IP address
#' that is already assigned to another network interface or instance to be
#' reassigned to the specified network interface.
#' @section PrivateIpAddress:
#' 
#' One or more IP addresses to be assigned as a secondary private IP
#' address to the network interface. You can\'t specify this parameter when
#' also specifying a number of secondary IP addresses.
#' 
#' If you don\'t specify an IP address, Amazon EC2 automatically selects an
#' IP address within the subnet range.
#' 
#' @section SecondaryPrivateIpAddressCount:
#' The number of secondary IP addresses to assign to
#' the network interface. You can\'t specify this parameter when also
#' specifying private IP addresses.
#' @return A list object or a character vector
#' @export
ec2_assign_private_ip_addresses <- function(NetworkInterfaceId, AllowReassignment = NULL, 
    PrivateIpAddress = NULL, SecondaryPrivateIpAddressCount = NULL, simplify = TRUE, others = list()) {
    PrivateIpAddress <- list_to_array("PrivateIpAddress", PrivateIpAddress)
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, AllowReassignment = AllowReassignment, 
        SecondaryPrivateIpAddressCount = SecondaryPrivateIpAddressCount), PrivateIpAddress)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssignPrivateIpAddresses", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Associate Address
#' 
#' @param AllocationId String. \[EC2-VPC\] The allocation ID. This is required for EC2-VPC.\[optional\]
#' @param InstanceId String. The ID of the instance.\[optional\]
#' @param PublicIp String. \[EC2-Classic\] The Elastic IP address to associate with the instance.\[optional\]
#' @param AllowReassociation Logical. \[EC2-VPC\] For a VPC in an EC2-Classic account, specify true to allow an Elastic IP address that...\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param NetworkInterfaceId String.   \[EC2-VPC\] The ID of the network interface.\[optional\]
#' @param PrivateIpAddress String. \[EC2-VPC\] The primary or secondary private IP address to associate with the Elastic IP address.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AllocationId:
#' \[EC2-VPC\] The
#' allocation ID. This is required for EC2-VPC.
#' @section InstanceId:
#' The ID
#' of the instance. The instance must have exactly one attached network
#' interface. For EC2-VPC, you can specify either the instance ID or the
#' network interface ID, but not both. For EC2-Classic, you must specify an
#' instance ID and the instance must be in the running
#' state.
#' @section PublicIp:
#' \[EC2-Classic\] The Elastic IP address to
#' associate with the instance. This is required for
#' EC2-Classic.
#' @section AllowReassociation:
#' \[EC2-VPC\] For a VPC in an EC2-Classic
#' account, specify true to allow an Elastic IP address that is already
#' associated with an instance or network interface to be reassociated with
#' the specified instance or network interface. Otherwise, the operation
#' fails. In a VPC in an EC2-VPC-only account, reassociation is automatic,
#' therefore you can specify false to ensure the operation fails if the
#' Elastic IP address is already associated with another
#' resource.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section NetworkInterfaceId:
#' 
#' \[EC2-VPC\] The ID of the network interface. If the instance has more
#' than one network interface, you must specify a network interface ID.
#' 
#' For EC2-VPC, you can specify either the instance ID or the network
#' interface ID, but not both.
#' 
#' @section PrivateIpAddress:
#' \[EC2-VPC\] The primary or secondary private IP
#' address to associate with the Elastic IP address. If no private IP
#' address is specified, the Elastic IP address is associated with the
#' primary private IP address.
#' @return A list object or a character vector
#' @export
ec2_associate_address <- function(AllocationId = NULL, InstanceId = NULL, PublicIp = NULL, 
    AllowReassociation = NULL, DryRun = NULL, NetworkInterfaceId = NULL, PrivateIpAddress = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(AllocationId = AllocationId, InstanceId = InstanceId, PublicIp = PublicIp, 
        AllowReassociation = AllowReassociation, DryRun = DryRun, NetworkInterfaceId = NetworkInterfaceId, 
        PrivateIpAddress = PrivateIpAddress))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateAddress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Associate Client Vpn Target Network
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param SubnetId String. The ID of the subnet to associate with the Client VPN endpoint.
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section SubnetId:
#' The ID of the subnet to associate with the
#' Client VPN endpoint.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_associate_client_vpn_target_network <- function(ClientVpnEndpointId, SubnetId, ClientToken = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, SubnetId = SubnetId, 
        ClientToken = ClientToken, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateClientVpnTargetNetwork", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Associate Dhcp Options
#' 
#' @param DhcpOptionsId String. The ID of the DHCP options set, or `default` to associate no DHCP options with the VPC.
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DhcpOptionsId:
#' The ID of
#' the DHCP options set, or `default` to associate no DHCP options with the
#' VPC.
#' @section VpcId:
#' The ID of the VPC.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_associate_dhcp_options <- function(DhcpOptionsId, VpcId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(DhcpOptionsId = DhcpOptionsId, VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateDhcpOptions", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Associate Enclave Certificate Iam Role
#' 
#' @param CertificateArn String. The ARN of the ACM certificate with which to associate the IAM role.\[optional\]
#' @param RoleArn String. The ARN of the IAM role to associate with the ACM certificate.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CertificateArn:
#' The ARN of
#' the ACM certificate with which to associate the IAM
#' role.
#' @section RoleArn:
#' The ARN of the IAM role to associate with the
#' ACM certificate. You can associate up to 16 IAM roles with an ACM
#' certificate.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_associate_enclave_certificate_iam_role <- function(CertificateArn = NULL, RoleArn = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(CertificateArn = CertificateArn, RoleArn = RoleArn, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateEnclaveCertificateIamRole", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Associate Iam Instance Profile
#' 
#' Associates an IAM instance profile with a running
#' or stopped instance. You cannot associate more than one IAM instance
#' profile with an instance.
#' 
#' @param IamInstanceProfile Object. The IAM instance profile.
#' @param InstanceId String. The ID of the instance.
#' @inheritParams CommonDoc
#' 
#' @section IamInstanceProfile:
#' The IAM instance
#' profile.
#' @section InstanceId:
#' The ID of the
#' instance.
#' @return A list object or a character vector
#' @export
ec2_associate_iam_instance_profile <- function(IamInstanceProfile, InstanceId, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(IamInstanceProfile = IamInstanceProfile, InstanceId = InstanceId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateIamInstanceProfile", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Associate Route Table
#' 
#' @param RouteTableId String. The ID of the route table.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param SubnetId String. The ID of the subnet.\[optional\]
#' @param GatewayId String. The ID of the internet gateway or virtual private gateway.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section RouteTableId:
#' The ID of the route
#' table.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section SubnetId:
#' The ID of the
#' subnet.
#' @section GatewayId:
#' The ID of the internet gateway or virtual
#' private gateway.
#' @return A list object or a character vector
#' @export
ec2_associate_route_table <- function(RouteTableId, DryRun = NULL, SubnetId = NULL, GatewayId = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(RouteTableId = RouteTableId, DryRun = DryRun, SubnetId = SubnetId, 
        GatewayId = GatewayId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateRouteTable", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Associate Subnet Cidr Block
#' 
#' Associates a CIDR block with your subnet. You can
#' only associate a single IPv6 CIDR block with your subnet. An IPv6 CIDR
#' block must have a prefix length of /64.
#' 
#' @param SubnetId String. The ID of your subnet.
#' @param Ipv6CidrBlock String. The IPv6 CIDR block for your subnet. The subnet must have a /64 prefix length.
#' @inheritParams CommonDoc
#' 
#' @section SubnetId:
#' The ID of your
#' subnet.
#' @section Ipv6CidrBlock:
#' The IPv6 CIDR block for your subnet. The
#' subnet must have a /64 prefix length.
#' @return A list object or a character vector
#' @export
ec2_associate_subnet_cidr_block <- function(SubnetId, Ipv6CidrBlock, simplify = TRUE, others = list()) {
    parameters <- c(others, list(SubnetId = SubnetId, Ipv6CidrBlock = Ipv6CidrBlock))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateSubnetCidrBlock", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Associate Transit Gateway Multicast Domain
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param TransitGatewayAttachmentId String. The ID of the transit gateway attachment to associate with the transit gateway multicast domain.\[optional\]
#' @param SubnetIds List. The IDs of the subnets to associate with the transit gateway multicast domain.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the
#' transit gateway multicast domain.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' transit gateway attachment to associate with the transit gateway
#' multicast domain.
#' @section SubnetIds:
#' The IDs of the subnets to
#' associate with the transit gateway multicast
#' domain.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_associate_transit_gateway_multicast_domain <- function(TransitGatewayMulticastDomainId = NULL, 
    TransitGatewayAttachmentId = NULL, SubnetIds = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    SubnetIds <- list_to_array("SubnetIds", SubnetIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun), SubnetIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateTransitGatewayMulticastDomain", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Associate Transit Gateway Route Table
#' 
#' Associates the specified attachment with the
#' specified transit gateway route table. You can associate only one route
#' table with an attachment.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_associate_transit_gateway_route_table <- function(TransitGatewayRouteTableId, TransitGatewayAttachmentId, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateTransitGatewayRouteTable", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Associate Vpc Cidr Block
#' 
#' @param VpcId String. The ID of the VPC.
#' @param AmazonProvidedIpv6CidrBlock Logical. Requests an Amazon-provided IPv6 CIDR block with a /56 prefix length for the VPC.\[optional\]
#' @param CidrBlock String. An IPv4 CIDR block to associate with the VPC.\[optional\]
#' @param Ipv6CidrBlockNetworkBorderGroup String.   The name of the location from which we advertise the IPV6 CIDR block.\[optional\]
#' @param Ipv6Pool String. The ID of an IPv6 address pool from which to allocate the IPv6 CIDR block.\[optional\]
#' @param Ipv6CidrBlock String.   An IPv6 CIDR block from the IPv6 address pool.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section AmazonProvidedIpv6CidrBlock:
#' Requests an Amazon-provided IPv6 CIDR block
#' with a /56 prefix length for the VPC. You cannot specify the range of
#' IPv6 addresses, or the size of the CIDR block.
#' @section CidrBlock:
#' An
#' IPv4 CIDR block to associate with the VPC.
#' @section Ipv6CidrBlockNetworkBorderGroup:
#' 
#' The name of the location from which we advertise the IPV6 CIDR block.
#' Use this parameter to limit the CIDR block to this location.
#' 
#' You must set `AmazonProvidedIpv6CidrBlock` to `true` to use this
#' parameter.
#' 
#' You can have one IPv6 CIDR block association per network border group.
#' 
#' @section Ipv6Pool:
#' The ID of an IPv6 address pool from which to
#' allocate the IPv6 CIDR block.
#' @section Ipv6CidrBlock:
#' 
#' An IPv6 CIDR block from the IPv6 address pool. You must also specify
#' `Ipv6Pool` in the request.
#' 
#' To let Amazon choose the IPv6 CIDR block for you, omit this parameter.
#' 
#' @return A list object or a character vector
#' @export
ec2_associate_vpc_cidr_block <- function(VpcId, AmazonProvidedIpv6CidrBlock = NULL, CidrBlock = NULL, 
    Ipv6CidrBlockNetworkBorderGroup = NULL, Ipv6Pool = NULL, Ipv6CidrBlock = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(VpcId = VpcId, AmazonProvidedIpv6CidrBlock = AmazonProvidedIpv6CidrBlock, 
        CidrBlock = CidrBlock, Ipv6CidrBlockNetworkBorderGroup = Ipv6CidrBlockNetworkBorderGroup, 
        Ipv6Pool = Ipv6Pool, Ipv6CidrBlock = Ipv6CidrBlock))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AssociateVpcCidrBlock", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Attach Classic Link Vpc
#' 
#' @param SecurityGroupId List. The ID of one or more of the VPC\'s security groups.
#' @param InstanceId String. The ID of an EC2-Classic instance to link to the ClassicLink-enabled VPC.
#' @param VpcId String. The ID of a ClassicLink-enabled VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SecurityGroupId:
#' The ID of one or more of the VPC\'s security
#' groups. You cannot specify security groups from a different
#' VPC.
#' @section InstanceId:
#' The ID of an EC2-Classic instance to link to
#' the ClassicLink-enabled VPC.
#' @section VpcId:
#' The ID of a
#' ClassicLink-enabled VPC.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_attach_classic_link_vpc <- function(SecurityGroupId, InstanceId, VpcId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    parameters <- c(others, list(InstanceId = InstanceId, VpcId = VpcId, DryRun = DryRun), 
        SecurityGroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AttachClassicLinkVpc", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Attach Internet Gateway
#' 
#' Attaches an internet gateway or a virtual private
#' gateway to a VPC, enabling connectivity between the internet and the
#' VPC. For more information about your VPC and internet gateway, see the
#' [Amazon Virtual Private Cloud User
#' Guide](https://docs.aws.amazon.com/vpc/latest/userguide/).
#' 
#' @param InternetGatewayId String. The ID of the internet gateway.
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InternetGatewayId:
#' The ID of the internet
#' gateway.
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_attach_internet_gateway <- function(InternetGatewayId, VpcId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InternetGatewayId = InternetGatewayId, VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AttachInternetGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Attach Network Interface
#' 
#' Attaches
#' a network interface to an instance.
#' 
#' @param DeviceIndex Integer. The index of the device for the network interface attachment.
#' @param InstanceId String. The ID of the instance.
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param NetworkCardIndex Integer. The index of the network card.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DeviceIndex:
#' The index of the device for
#' the network interface attachment.
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section NetworkCardIndex:
#' The index of the network
#' card. Some instance types support multiple network cards. The primary
#' network interface must be assigned to network card index 0. The default
#' is network card index 0.
#' @return A list object or a character vector
#' @export
ec2_attach_network_interface <- function(DeviceIndex, InstanceId, NetworkInterfaceId, DryRun = NULL, 
    NetworkCardIndex = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DeviceIndex = DeviceIndex, InstanceId = InstanceId, NetworkInterfaceId = NetworkInterfaceId, 
        DryRun = DryRun, NetworkCardIndex = NetworkCardIndex))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AttachNetworkInterface", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Attach Volume
#' 
#' @param Device String. The device name (for example, `/dev/sdh` or `xvdh`).
#' @param InstanceId String. The ID of the instance.
#' @param VolumeId String. The ID of the EBS volume. The volume and instance must be within the same Availability Zone.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Device:
#' The device name (for
#' example, `/dev/sdh` or `xvdh`).
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section VolumeId:
#' The ID of the EBS volume. The volume and
#' instance must be within the same Availability
#' Zone.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_attach_volume <- function(Device, InstanceId, VolumeId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Device = Device, InstanceId = InstanceId, VolumeId = VolumeId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AttachVolume", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Attach Vpn Gateway
#' 
#' @param VpcId String. The ID of the VPC.
#' @param VpnGatewayId String. The ID of the virtual private gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section VpnGatewayId:
#' The ID of the virtual private
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_attach_vpn_gateway <- function(VpcId, VpnGatewayId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId, VpnGatewayId = VpnGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AttachVpnGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Authorize Client Vpn Ingress
#' 
#' Adds an ingress authorization rule to a Client VPN
#' endpoint. Ingress authorization rules act as firewall rules that grant
#' access to networks. You must configure ingress authorization rules to
#' enable clients to access resources in AWS or on-premises
#' networks.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param TargetNetworkCidr String. The IPv4 address range, in CIDR notation, of the network for which access is being authorized.
#' @param AccessGroupId String. The ID of the group to grant access to, for example, the Active Directory group or identity provider...\[optional\]
#' @param AuthorizeAllGroups Logical. Indicates whether to grant access to all clients.\[optional\]
#' @param Description String. A brief description of the authorization rule.\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section TargetNetworkCidr:
#' The IPv4 address range, in CIDR notation,
#' of the network for which access is being
#' authorized.
#' @section AccessGroupId:
#' The ID of the group to grant access to,
#' for example, the Active Directory group or identity provider (IdP)
#' group. Required if `AuthorizeAllGroups` is `false` or not
#' specified.
#' @section AuthorizeAllGroups:
#' Indicates whether to grant access to all
#' clients. Specify `true` to grant all clients who successfully establish
#' a VPN connection access to the network. Must be set to `true` if
#' `AccessGroupId` is not specified.
#' @section Description:
#' A brief
#' description of the authorization rule.
#' @section ClientToken:
#' Unique,
#' case-sensitive identifier that you provide to ensure the idempotency of
#' the request. For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_authorize_client_vpn_ingress <- function(ClientVpnEndpointId, TargetNetworkCidr, AccessGroupId = NULL, 
    AuthorizeAllGroups = NULL, Description = NULL, ClientToken = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, TargetNetworkCidr = TargetNetworkCidr, 
        AccessGroupId = AccessGroupId, AuthorizeAllGroups = AuthorizeAllGroups, Description = Description, 
        ClientToken = ClientToken, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AuthorizeClientVpnIngress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Authorize Security Group Egress
#' 
#' @param GroupId String. The ID of the security group.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param IpPermissions List. The sets of IP permissions.\[optional\]
#' @param CidrIp String. Not supported. Use a set of IP permissions to specify the CIDR.\[optional\]
#' @param FromPort Integer. Not supported. Use a set of IP permissions to specify the port.\[optional\]
#' @param IpProtocol String. Not supported. Use a set of IP permissions to specify the protocol name or number.\[optional\]
#' @param ToPort Integer. Not supported. Use a set of IP permissions to specify the port.\[optional\]
#' @param SourceSecurityGroupName String. Not supported. Use a set of IP permissions to specify a destination security group.\[optional\]
#' @param SourceSecurityGroupOwnerId String. Not supported. Use a set of IP permissions to specify a destination security group.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GroupId:
#' The ID of
#' the security group.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section IpPermissions:
#' The sets of IP permissions.
#' You can\'t specify a destination security group and a CIDR IP address
#' range in the same set of permissions.
#' @section CidrIp:
#' Not supported.
#' Use a set of IP permissions to specify the CIDR.
#' @section FromPort:
#' Not
#' supported. Use a set of IP permissions to specify the
#' port.
#' @section IpProtocol:
#' Not supported. Use a set of IP permissions to
#' specify the protocol name or number.
#' @section ToPort:
#' Not supported.
#' Use a set of IP permissions to specify the port.
#' @section SourceSecurityGroupName:
#' Not
#' supported. Use a set of IP permissions to specify a destination security
#' group.
#' @section SourceSecurityGroupOwnerId:
#' Not supported. Use a set of IP permissions to
#' specify a destination security group.
#' @return A list object or a character vector
#' @export
ec2_authorize_security_group_egress <- function(GroupId, DryRun = NULL, IpPermissions = NULL, 
    CidrIp = NULL, FromPort = NULL, IpProtocol = NULL, ToPort = NULL, SourceSecurityGroupName = NULL, 
    SourceSecurityGroupOwnerId = NULL, simplify = TRUE, others = list()) {
    IpPermissions <- list_to_array("IpPermissions", IpPermissions)
    parameters <- c(others, list(GroupId = GroupId, DryRun = DryRun, CidrIp = CidrIp, FromPort = FromPort, 
        IpProtocol = IpProtocol, ToPort = ToPort, SourceSecurityGroupName = SourceSecurityGroupName, 
        SourceSecurityGroupOwnerId = SourceSecurityGroupOwnerId), IpPermissions)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AuthorizeSecurityGroupEgress", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Authorize Security Group Ingress
#' 
#' @param CidrIp String.   The IPv4 address range, in CIDR format.\[optional\]
#' @param FromPort Integer.   The start of port range for the TCP and UDP protocols, or an ICMP type number.\[optional\]
#' @param GroupId String. The ID of the security group.\[optional\]
#' @param GroupName String. \[EC2-Classic, default VPC\] The name of the security group.\[optional\]
#' @param IpPermissions List. The sets of IP permissions.\[optional\]
#' @param IpProtocol String.   The IP protocol name (`tcp`, `udp`, `icmp`) or number (see [Protocol Numbers](http://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml))....\[optional\]
#' @param SourceSecurityGroupName String. \[EC2-Classic, default VPC\] The name of the source security group.\[optional\]
#' @param SourceSecurityGroupOwnerId String. \[nondefault VPC\] The AWS account ID for the source security group, if the source security group...\[optional\]
#' @param ToPort Integer.   The end of port range for the TCP and UDP protocols, or an ICMP code number.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CidrIp:
#' 
#' The IPv4 address range, in CIDR format. You can\'t specify this
#' parameter when specifying a source security group. To specify an IPv6
#' address range, use a set of IP permissions.
#' 
#' Alternatively, use a set of IP permissions to specify multiple rules and
#' a description for the rule.
#' 
#' @section FromPort:
#' 
#' The start of port range for the TCP and UDP protocols, or an ICMP type
#' number. For the ICMP type number, use `-1` to specify all types. If you
#' specify all ICMP types, you must specify all codes.
#' 
#' Alternatively, use a set of IP permissions to specify multiple rules and
#' a description for the rule.
#' 
#' @section GroupId:
#' The ID of the security group. You must specify
#' either the security group ID or the security group name in the request.
#' For security groups in a nondefault VPC, you must specify the security
#' group ID.
#' @section GroupName:
#' \[EC2-Classic, default VPC\] The name of
#' the security group. You must specify either the security group ID or the
#' security group name in the request.
#' @section IpPermissions:
#' The sets of IP
#' permissions.
#' @section IpProtocol:
#' 
#' The IP protocol name (`tcp`, `udp`, `icmp`) or number (see [Protocol
#' Numbers](http://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)).
#' To specify `icmpv6`, use a set of IP permissions.
#' 
#' \[VPC only\] Use `-1` to specify all protocols. If you specify `-1` or a
#' protocol other than `tcp`, `udp`, or `icmp`, traffic on all ports is
#' allowed, regardless of any ports you specify.
#' 
#' Alternatively, use a set of IP permissions to specify multiple rules and
#' a description for the rule.
#' 
#' @section SourceSecurityGroupName:
#' \[EC2-Classic, default VPC\] The name of the source
#' security group. You can\'t specify this parameter in combination with
#' the following parameters: the CIDR IP address range, the start of the
#' port range, the IP protocol, and the end of the port range. Creates
#' rules that grant full ICMP, UDP, and TCP access. To create a rule with a
#' specific IP protocol and port range, use a set of IP permissions
#' instead. For EC2-VPC, the source security group must be in the same
#' VPC.
#' @section SourceSecurityGroupOwnerId:
#' \[nondefault VPC\] The AWS account ID for the
#' source security group, if the source security group is in a different
#' account. You can\'t specify this parameter in combination with the
#' following parameters: the CIDR IP address range, the IP protocol, the
#' start of the port range, and the end of the port range. Creates rules
#' that grant full ICMP, UDP, and TCP access. To create a rule with a
#' specific IP protocol and port range, use a set of IP permissions
#' instead.
#' @section ToPort:
#' 
#' The end of port range for the TCP and UDP protocols, or an ICMP code
#' number. For the ICMP code number, use `-1` to specify all codes. If you
#' specify all ICMP types, you must specify all codes.
#' 
#' Alternatively, use a set of IP permissions to specify multiple rules and
#' a description for the rule.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_authorize_security_group_ingress <- function(CidrIp = NULL, FromPort = NULL, GroupId = NULL, 
    GroupName = NULL, IpPermissions = NULL, IpProtocol = NULL, SourceSecurityGroupName = NULL, 
    SourceSecurityGroupOwnerId = NULL, ToPort = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    IpPermissions <- list_to_array("IpPermissions", IpPermissions)
    parameters <- c(others, list(CidrIp = CidrIp, FromPort = FromPort, GroupId = GroupId, 
        GroupName = GroupName, IpProtocol = IpProtocol, SourceSecurityGroupName = SourceSecurityGroupName, 
        SourceSecurityGroupOwnerId = SourceSecurityGroupOwnerId, ToPort = ToPort, DryRun = DryRun), 
        IpPermissions)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "AuthorizeSecurityGroupIngress", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Bundle Instance
#' 
#' @param InstanceId String.   The ID of the instance to bundle.  Type: String  Default: None  Required: Yes  
#' @param Storage Object. The bucket in which to store the AMI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' 
#' The ID of the instance to bundle.
#' 
#' Type: String
#' 
#' Default: None
#' 
#' Required: Yes
#' 
#' @section Storage:
#' The bucket in which to store the AMI. You can
#' specify a bucket that you already own or a new bucket that Amazon EC2
#' creates on your behalf. If you specify a bucket that belongs to someone
#' else, Amazon EC2 returns an error.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_bundle_instance <- function(InstanceId, Storage, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, Storage = Storage, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "BundleInstance", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Cancel Bundle Task
#' 
#' Cancels a bundling operation for an instance
#' store-backed Windows instance.
#' 
#' @param BundleId String. The ID of the bundle task.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section BundleId:
#' The ID of
#' the bundle task.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_cancel_bundle_task <- function(BundleId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(BundleId = BundleId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelBundleTask", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Cancel Capacity Reservation
#' 
#' @param CapacityReservationId String. The ID of the Capacity Reservation to be cancelled.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CapacityReservationId:
#' The ID of the Capacity
#' Reservation to be cancelled.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_cancel_capacity_reservation <- function(CapacityReservationId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(CapacityReservationId = CapacityReservationId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelCapacityReservation", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Cancel Conversion Task
#' 
#' @param ConversionTaskId String. The ID of the conversion task.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ReasonMessage String. The reason for canceling the conversion task.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ConversionTaskId:
#' The ID of the conversion
#' task.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ReasonMessage:
#' The reason for canceling
#' the conversion task.
#' @return A list object or a character vector
#' @export
ec2_cancel_conversion_task <- function(ConversionTaskId, DryRun = NULL, ReasonMessage = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(ConversionTaskId = ConversionTaskId, DryRun = DryRun, ReasonMessage = ReasonMessage))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelConversionTask", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Cancel Export Task
#' 
#' Cancels an active export task. The request removes
#' all artifacts of the export, including any partially-created Amazon S3
#' objects. If the export task is complete or is in the process of
#' transferring the final disk image, the command fails and returns an
#' error.
#' 
#' @param ExportTaskId String. The ID of the export task. This is the ID returned by `CreateInstanceExportTask`.
#' @inheritParams CommonDoc
#' 
#' @section ExportTaskId:
#' The ID of the export task. This
#' is the ID returned by
#' `CreateInstanceExportTask`.
#' @return A list object or a character vector
#' @export
ec2_cancel_export_task <- function(ExportTaskId, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ExportTaskId = ExportTaskId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelExportTask", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Cancel Import Task
#' 
#' Cancels an in-process import virtual machine
#' or import snapshot task.
#' 
#' @param CancelReason String. The reason for canceling the task.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ImportTaskId String. The ID of the import image or import snapshot task to be canceled.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CancelReason:
#' The reason for canceling
#' the task.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ImportTaskId:
#' The ID of the import image
#' or import snapshot task to be canceled.
#' @return A list object or a character vector
#' @export
ec2_cancel_import_task <- function(CancelReason = NULL, DryRun = NULL, ImportTaskId = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(CancelReason = CancelReason, DryRun = DryRun, ImportTaskId = ImportTaskId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelImportTask", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Cancel Reserved Instances Listing
#' 
#' @param ReservedInstancesListingId String. The ID of the Reserved Instance listing.
#' @inheritParams CommonDoc
#' 
#' @section ReservedInstancesListingId:
#' The ID of
#' the Reserved Instance listing.
#' @return A list object or a character vector
#' @export
ec2_cancel_reserved_instances_listing <- function(ReservedInstancesListingId, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(ReservedInstancesListingId = ReservedInstancesListingId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelReservedInstancesListing", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Cancel Spot Fleet Requests
#' 
#' @param SpotFleetRequestId List. The IDs of the Spot Fleet requests.
#' @param TerminateInstances Logical. Indicates whether to terminate instances for a Spot Fleet request if it is canceled successfully.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SpotFleetRequestId:
#' The IDs of the Spot
#' Fleet requests.
#' @section TerminateInstances:
#' Indicates whether to terminate
#' instances for a Spot Fleet request if it is canceled
#' successfully.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_cancel_spot_fleet_requests <- function(SpotFleetRequestId, TerminateInstances, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    SpotFleetRequestId <- list_to_array("SpotFleetRequestId", SpotFleetRequestId)
    parameters <- c(others, list(TerminateInstances = TerminateInstances, DryRun = DryRun), 
        SpotFleetRequestId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelSpotFleetRequests", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Cancel Spot Instance Requests
#' 
#' @param SpotInstanceRequestId List. One or more Spot Instance request IDs.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SpotInstanceRequestId:
#' One or more Spot Instance
#' request IDs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_cancel_spot_instance_requests <- function(SpotInstanceRequestId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    SpotInstanceRequestId <- list_to_array("SpotInstanceRequestId", SpotInstanceRequestId)
    parameters <- c(others, list(DryRun = DryRun), SpotInstanceRequestId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CancelSpotInstanceRequests", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Confirm Product Instance
#' 
#' Determines whether a product code is associated
#' with an instance. This action can only be used by the owner of the
#' product code. It is useful when a product code owner must verify whether
#' another user\'s instance is eligible for
#' support.
#' 
#' @param InstanceId String. The ID of the instance.
#' @param ProductCode String. The product code. This must be a product code that you own.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section ProductCode:
#' The product code. This must be a product
#' code that you own.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_confirm_product_instance <- function(InstanceId, ProductCode, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, ProductCode = ProductCode, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ConfirmProductInstance", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Copy Fpga Image
#' 
#' Copies the specified Amazon FPGA Image
#' (AFI) to the current Region.
#' 
#' @param SourceFpgaImageId String. The ID of the source AFI.
#' @param SourceRegion String. The Region that contains the source AFI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Description String. The description for the new AFI.\[optional\]
#' @param Name String. The name for the new AFI. The default is the name of the source AFI.\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SourceFpgaImageId:
#' The ID of the source
#' AFI.
#' @section SourceRegion:
#' The Region that contains the source
#' AFI.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Description:
#' The description for the new
#' AFI.
#' @section Name:
#' The name for the new AFI. The default is the
#' name of the source AFI.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_copy_fpga_image <- function(SourceFpgaImageId, SourceRegion, DryRun = NULL, Description = NULL, 
    Name = NULL, ClientToken = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(SourceFpgaImageId = SourceFpgaImageId, SourceRegion = SourceRegion, 
        DryRun = DryRun, Description = Description, Name = Name, ClientToken = ClientToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CopyFpgaImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Copy Image
#' 
#' @param Name String. The name of the new AMI in the destination Region.
#' @param SourceImageId String. The ID of the AMI to copy.
#' @param SourceRegion String. The name of the Region that contains the AMI to copy.
#' @param ClientToken String. Unique, case-sensitive identifier you provide to ensure idempotency of the request.\[optional\]
#' @param Description String. A description for the new AMI in the destination Region.\[optional\]
#' @param Encrypted Logical. Specifies whether the destination snapshots of the copied image should be encrypted.\[optional\]
#' @param KmsKeyId String.   The identifier of the symmetric AWS Key Management Service (AWS KMS) customer master key (CMK)...\[optional\]
#' @param DestinationOutpostArn String.   The Amazon Resource Name (ARN) of the Outpost to which to copy the AMI.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Name:
#' The
#' name of the new AMI in the destination Region.
#' @section SourceImageId:
#' The
#' ID of the AMI to copy.
#' @section SourceRegion:
#' The name of the Region that
#' contains the AMI to copy.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier you provide to ensure idempotency of the request. For more
#' information, see [Ensuring
#' idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html)
#' in the *Amazon EC2 API Reference*.
#' @section Description:
#' A description for
#' the new AMI in the destination Region.
#' @section Encrypted:
#' Specifies
#' whether the destination snapshots of the copied image should be
#' encrypted. You can encrypt a copy of an unencrypted snapshot, but you
#' cannot create an unencrypted copy of an encrypted snapshot. The default
#' CMK for EBS is used unless you specify a non-default AWS Key Management
#' Service (AWS KMS) CMK using `KmsKeyId`. For more information, see
#' [Amazon EBS
#' Encryption](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' @section KmsKeyId:
#' 
#' The identifier of the symmetric AWS Key Management Service (AWS KMS)
#' customer master key (CMK) to use when creating encrypted volumes. If
#' this parameter is not specified, your AWS managed CMK for EBS is used.
#' If you specify a CMK, you must also set the encrypted state to `true`.
#' 
#' You can specify a CMK using any of the following:
#' 
#' -   Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Key alias. For example, alias/ExampleAlias.
#' 
#' -   Key ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Alias ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
#' 
#' AWS authenticates the CMK asynchronously. Therefore, if you specify an
#' identifier that is not valid, the action can appear to complete, but
#' eventually fails.
#' 
#' The specified CMK must exist in the destination Region.
#' 
#' Amazon EBS does not support asymmetric CMKs.
#' 
#' @section DestinationOutpostArn:
#' 
#' The Amazon Resource Name (ARN) of the Outpost to which to copy the AMI.
#' Only specify this parameter when copying an AMI from an AWS Region to an
#' Outpost. The AMI must be in the Region of the destination Outpost. You
#' cannot copy an AMI from an Outpost to a Region, from one Outpost to
#' another, or within the same Outpost.
#' 
#' For more information, see [Copying AMIs from an AWS Region to an
#' Outpost](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshots-outposts.html#copy-amis)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_copy_image <- function(Name, SourceImageId, SourceRegion, ClientToken = NULL, Description = NULL, 
    Encrypted = NULL, KmsKeyId = NULL, DestinationOutpostArn = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Name = Name, SourceImageId = SourceImageId, SourceRegion = SourceRegion, 
        ClientToken = ClientToken, Description = Description, Encrypted = Encrypted, KmsKeyId = KmsKeyId, 
        DestinationOutpostArn = DestinationOutpostArn, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CopyImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Copy Snapshot
#' 
#' @param SourceRegion String. The ID of the Region that contains the snapshot to be copied.
#' @param SourceSnapshotId String. The ID of the EBS snapshot to copy.
#' @param Description String. A description for the EBS snapshot.\[optional\]
#' @param DestinationOutpostArn String.   The Amazon Resource Name (ARN) of the Outpost to which to copy the snapshot.\[optional\]
#' @param DestinationRegion String.   The destination Region to use in the `PresignedUrl` parameter of a snapshot copy operation.\[optional\]
#' @param Encrypted Logical. To encrypt a copy of an unencrypted snapshot if encryption by default is not enabled, enable encryption...\[optional\]
#' @param KmsKeyId String.   The identifier of the AWS Key Management Service (AWS KMS) customer master key (CMK) to use for...\[optional\]
#' @param PresignedUrl String.   When you copy an encrypted source snapshot using the Amazon EC2 Query API, you must supply a pre-signed...\[optional\]
#' @param TagSpecification List. The tags to apply to the new snapshot.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SourceRegion:
#' The ID of the Region that
#' contains the snapshot to be copied.
#' @section SourceSnapshotId:
#' The ID of the
#' EBS snapshot to copy.
#' @section Description:
#' A description for the EBS
#' snapshot.
#' @section DestinationOutpostArn:
#' 
#' The Amazon Resource Name (ARN) of the Outpost to which to copy the
#' snapshot. Only specify this parameter when copying a snapshot from an
#' AWS Region to an Outpost. The snapshot must be in the Region for the
#' destination Outpost. You cannot copy a snapshot from an Outpost to a
#' Region, from one Outpost to another, or within the same Outpost.
#' 
#' For more information, see [Copying snapshots from an AWS Region to an
#' Outpost](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshots-outposts.html#copy-snapshots)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' @section DestinationRegion:
#' 
#' The destination Region to use in the `PresignedUrl` parameter of a
#' snapshot copy operation. This parameter is only valid for specifying the
#' destination Region in a `PresignedUrl` parameter, where it is required.
#' 
#' The snapshot copy is sent to the regional endpoint that you sent the
#' HTTP request to (for example, `ec2.us-east-1.amazonaws.com`). With the
#' AWS CLI, this is specified using the `--region` parameter or the default
#' Region in your AWS configuration file.
#' 
#' @section Encrypted:
#' To encrypt a copy of an unencrypted snapshot if
#' encryption by default is not enabled, enable encryption using this
#' parameter. Otherwise, omit this parameter. Encrypted snapshots are
#' encrypted, even if you omit this parameter and encryption by default is
#' not enabled. You cannot set this parameter to false. For more
#' information, see [Amazon EBS
#' encryption](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' @section KmsKeyId:
#' 
#' The identifier of the AWS Key Management Service (AWS KMS) customer
#' master key (CMK) to use for Amazon EBS encryption. If this parameter is
#' not specified, your AWS managed CMK for EBS is used. If `KmsKeyId` is
#' specified, the encrypted state must be `true`.
#' 
#' You can specify the CMK using any of the following:
#' 
#' -   Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Key alias. For example, alias/ExampleAlias.
#' 
#' -   Key ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Alias ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
#' 
#' AWS authenticates the CMK asynchronously. Therefore, if you specify an
#' ID, alias, or ARN that is not valid, the action can appear to complete,
#' but eventually fails.
#' 
#' @section PresignedUrl:
#' 
#' When you copy an encrypted source snapshot using the Amazon EC2 Query
#' API, you must supply a pre-signed URL. This parameter is optional for
#' unencrypted snapshots. For more information, see [Query
#' requests](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html).
#' 
#' The `PresignedUrl` should use the snapshot source endpoint, the
#' `CopySnapshot` action, and include the `SourceRegion`,
#' `SourceSnapshotId`, and `DestinationRegion` parameters. The
#' `PresignedUrl` must be signed using AWS Signature Version 4. Because EBS
#' snapshots are stored in Amazon S3, the signing algorithm for this
#' parameter uses the same logic that is described in [Authenticating
#' Requests: Using Query Parameters (AWS Signature Version
#' 4)](https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html)
#' in the *Amazon Simple Storage Service API Reference*. An invalid or
#' improperly signed `PresignedUrl` will cause the copy operation to fail
#' asynchronously, and the snapshot will move to an `error` state.
#' 
#' @section TagSpecification:
#' The tags to apply to the new
#' snapshot.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_copy_snapshot <- function(SourceRegion, SourceSnapshotId, Description = NULL, DestinationOutpostArn = NULL, 
    DestinationRegion = NULL, Encrypted = NULL, KmsKeyId = NULL, PresignedUrl = NULL, TagSpecification = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(SourceRegion = SourceRegion, SourceSnapshotId = SourceSnapshotId, 
        Description = Description, DestinationOutpostArn = DestinationOutpostArn, DestinationRegion = DestinationRegion, 
        Encrypted = Encrypted, KmsKeyId = KmsKeyId, PresignedUrl = PresignedUrl, DryRun = DryRun), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CopySnapshot", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Capacity Reservation
#' 
#' @param InstanceType String. The instance type for which to reserve capacity.
#' @param InstancePlatform String. The type of operating system for which to reserve capacity.
#' @param InstanceCount Integer. The number of instances for which to reserve capacity.
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param AvailabilityZone String. The Availability Zone in which to create the Capacity Reservation.\[optional\]
#' @param AvailabilityZoneId String. The ID of the Availability Zone in which to create the Capacity Reservation.\[optional\]
#' @param Tenancy String.   Indicates the tenancy of the Capacity Reservation.\[optional\]
#' @param EbsOptimized Logical. Indicates whether the Capacity Reservation supports EBS-optimized instances.\[optional\]
#' @param EphemeralStorage Logical. Indicates whether the Capacity Reservation supports instances with temporary, block-level storage.\[optional\]
#' @param EndDate String.   The date and time at which the Capacity Reservation expires.\[optional\]
#' @param EndDateType String.   Indicates the way in which the Capacity Reservation ends.\[optional\]
#' @param InstanceMatchCriteria String.   Indicates the type of instance launches that the Capacity Reservation accepts.\[optional\]
#' @param TagSpecifications List. The tags to apply to the Capacity Reservation during launch.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceType:
#' The instance type for which
#' to reserve capacity. For more information, see [Instance
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html)
#' in the *Amazon EC2 User Guide*.
#' @section InstancePlatform:
#' The type of
#' operating system for which to reserve capacity.
#' @section InstanceCount:
#' The
#' number of instances for which to reserve
#' capacity.
#' @section ClientToken:
#' Unique, case-sensitive identifier that you
#' provide to ensure the idempotency of the request. For more information,
#' see [Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section AvailabilityZone:
#' The
#' Availability Zone in which to create the Capacity
#' Reservation.
#' @section AvailabilityZoneId:
#' The ID of the Availability Zone in
#' which to create the Capacity Reservation.
#' @section Tenancy:
#' 
#' Indicates the tenancy of the Capacity Reservation. A Capacity
#' Reservation can have one of the following tenancy settings:
#' 
#' -   `default` - The Capacity Reservation is created on hardware that is
#'     shared with other AWS accounts.
#' 
#' -   `dedicated` - The Capacity Reservation is created on single-tenant
#'     hardware that is dedicated to a single AWS account.
#' 
#' @section EbsOptimized:
#' Indicates whether the Capacity Reservation supports
#' EBS-optimized instances. This optimization provides dedicated throughput
#' to Amazon EBS and an optimized configuration stack to provide optimal
#' I/O performance. This optimization isn\'t available with all instance
#' types. Additional usage charges apply when using an EBS- optimized
#' instance.
#' @section EphemeralStorage:
#' Indicates whether the Capacity Reservation
#' supports instances with temporary, block-level
#' storage.
#' @section EndDate:
#' 
#' The date and time at which the Capacity Reservation expires. When a
#' Capacity Reservation expires, the reserved capacity is released and you
#' can no longer launch instances into it. The Capacity Reservation\'s
#' state changes to `expired` when it reaches its end date and time.
#' 
#' You must provide an `EndDate` value if `EndDateType` is `limited`. Omit
#' `EndDate` if `EndDateType` is `unlimited`.
#' 
#' If the `EndDateType` is `limited`, the Capacity Reservation is cancelled
#' within an hour from the specified time. For example, if you specify
#' 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end
#' between 13:30:55 and 14:30:55 on 5/31/2019.
#' 
#' @section EndDateType:
#' 
#' Indicates the way in which the Capacity Reservation ends. A Capacity
#' Reservation can have one of the following end types:
#' 
#' -   `unlimited` - The Capacity Reservation remains active until you
#'     explicitly cancel it. Do not provide an `EndDate` if the
#'     `EndDateType` is `unlimited`.
#' 
#' -   `limited` - The Capacity Reservation expires automatically at a
#'     specified date and time. You must provide an `EndDate` value if the
#'     `EndDateType` value is `limited`.
#' 
#' @section InstanceMatchCriteria:
#' 
#' Indicates the type of instance launches that the Capacity Reservation
#' accepts. The options include:
#' 
#' -   `open` - The Capacity Reservation automatically matches all
#'     instances that have matching attributes (instance type, platform,
#'     and Availability Zone). Instances that have matching attributes run
#'     in the Capacity Reservation automatically without specifying any
#'     additional parameters.
#' 
#' -   `targeted` - The Capacity Reservation only accepts instances that
#'     have matching attributes (instance type, platform, and Availability
#'     Zone), and explicitly target the Capacity Reservation. This ensures
#'     that only permitted instances can use the reserved capacity.
#' 
#' Default: `open`
#' 
#' @section TagSpecifications:
#' The tags to apply to the Capacity Reservation
#' during launch.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_capacity_reservation <- function(InstanceType, InstancePlatform, InstanceCount, 
    ClientToken = NULL, AvailabilityZone = NULL, AvailabilityZoneId = NULL, Tenancy = NULL, 
    EbsOptimized = NULL, EphemeralStorage = NULL, EndDate = NULL, EndDateType = NULL, InstanceMatchCriteria = NULL, 
    TagSpecifications = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecifications <- list_to_array("TagSpecifications", TagSpecifications)
    parameters <- c(others, list(InstanceType = InstanceType, InstancePlatform = InstancePlatform, 
        InstanceCount = InstanceCount, ClientToken = ClientToken, AvailabilityZone = AvailabilityZone, 
        AvailabilityZoneId = AvailabilityZoneId, Tenancy = Tenancy, EbsOptimized = EbsOptimized, 
        EphemeralStorage = EphemeralStorage, EndDate = EndDate, EndDateType = EndDateType, 
        InstanceMatchCriteria = InstanceMatchCriteria, DryRun = DryRun), TagSpecifications)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateCapacityReservation", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Carrier Gateway
#' 
#' Creates a carrier gateway. For more information
#' about carrier gateways, see [Carrier
#' gateways](https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#wavelength-carrier-gateway)
#' in the *AWS Wavelength Developer Guide*.
#' 
#' @param VpcId String. The ID of the VPC to associate with the carrier gateway.
#' @param TagSpecification List. The tags to associate with the carrier gateway.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the VPC to
#' associate with the carrier gateway.
#' @section TagSpecification:
#' The tags to
#' associate with the carrier gateway.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique,
#' case-sensitive identifier that you provide to ensure the idempotency of
#' the request. For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_create_carrier_gateway <- function(VpcId, TagSpecification = NULL, DryRun = NULL, ClientToken = NULL, 
    simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(VpcId = VpcId, DryRun = DryRun, ClientToken = ClientToken), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateCarrierGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Client Vpn Endpoint
#' 
#' Creates a
#' Client VPN endpoint. A Client VPN endpoint is the resource you create
#' and configure to enable and manage client VPN sessions. It is the
#' destination endpoint at which all client VPN sessions are
#' terminated.
#' 
#' @param ClientCidrBlock String. The IPv4 address range, in CIDR notation, from which to assign client IP addresses.
#' @param ServerCertificateArn String. The ARN of the server certificate.
#' @param Authentication List. Information about the authentication method to be used to authenticate clients.
#' @param ConnectionLogOptions Object.   Information about the client connection logging options.
#' @param DnsServers List. Information about the DNS servers to be used for DNS resolution.\[optional\]
#' @param TransportProtocol String.   The transport protocol to be used by the VPN session.  Default value: `udp`  \[optional\]
#' @param VpnPort Integer.   The port number to assign to the Client VPN endpoint for TCP and UDP traffic.\[optional\]
#' @param Description String. A brief description of the Client VPN endpoint.\[optional\]
#' @param SplitTunnel Logical.   Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param TagSpecification List. The tags to apply to the Client VPN endpoint during creation.\[optional\]
#' @param SecurityGroupId List. The IDs of one or more security groups to apply to the target network.\[optional\]
#' @param VpcId String. The ID of the VPC to associate with the Client VPN endpoint.\[optional\]
#' @param SelfServicePortal String.   Specify whether to enable the self-service portal for the Client VPN endpoint.\[optional\]
#' @param ClientConnectOptions Object. The options for managing connection authorization for new client connections.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientCidrBlock:
#' The
#' IPv4 address range, in CIDR notation, from which to assign client IP
#' addresses. The address range cannot overlap with the local CIDR of the
#' VPC in which the associated subnet is located, or the routes that you
#' add manually. The address range cannot be changed after the Client VPN
#' endpoint has been created. The CIDR block should be /22 or
#' greater.
#' @section ServerCertificateArn:
#' The ARN of the server certificate. For more
#' information, see the [AWS Certificate Manager User
#' Guide](https://docs.aws.amazon.com/acm/latest/userguide/).
#' @section Authentication:
#' Information
#' about the authentication method to be used to authenticate
#' clients.
#' @section ConnectionLogOptions:
#' 
#' Information about the client connection logging options.
#' 
#' If you enable client connection logging, data about client connections
#' is sent to a Cloudwatch Logs log stream. The following information is
#' logged:
#' 
#' -   Client connection requests
#' 
#' -   Client connection results (successful and unsuccessful)
#' 
#' -   Reasons for unsuccessful client connection requests
#' 
#' -   Client connection termination time
#' 
#' @section DnsServers:
#' Information about the DNS servers to be used for
#' DNS resolution. A Client VPN endpoint can have up to two DNS servers. If
#' no DNS server is specified, the DNS address configured on the device is
#' used for the DNS server.
#' @section TransportProtocol:
#' 
#' The transport protocol to be used by the VPN session.
#' 
#' Default value: `udp`
#' 
#' @section VpnPort:
#' 
#' The port number to assign to the Client VPN endpoint for TCP and UDP
#' traffic.
#' 
#' Valid Values: `443` \| `1194`
#' 
#' Default Value: `443`
#' 
#' @section Description:
#' A brief description of the Client VPN
#' endpoint.
#' @section SplitTunnel:
#' 
#' Indicates whether split-tunnel is enabled on the AWS Client VPN
#' endpoint.
#' 
#' By default, split-tunnel on a VPN endpoint is disabled.
#' 
#' For information about split-tunnel VPN endpoints, see [Split-Tunnel AWS
#' Client VPN
#' Endpoint](https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html)
#' in the *AWS Client VPN Administrator Guide*.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section TagSpecification:
#' The
#' tags to apply to the Client VPN endpoint during
#' creation.
#' @section SecurityGroupId:
#' The IDs of one or more security groups to
#' apply to the target network. You must also specify the ID of the VPC
#' that contains the security groups.
#' @section VpcId:
#' The ID of the VPC
#' to associate with the Client VPN endpoint. If no security group IDs are
#' specified in the request, the default security group for the VPC is
#' applied.
#' @section SelfServicePortal:
#' 
#' Specify whether to enable the self-service portal for the Client VPN
#' endpoint.
#' 
#' Default Value: `enabled`
#' 
#' @section ClientConnectOptions:
#' The options for managing connection authorization
#' for new client connections.
#' @return A list object or a character vector
#' @export
ec2_create_client_vpn_endpoint <- function(ClientCidrBlock, ServerCertificateArn, Authentication, 
    ConnectionLogOptions, DnsServers = NULL, TransportProtocol = NULL, VpnPort = NULL, Description = NULL, 
    SplitTunnel = NULL, DryRun = NULL, ClientToken = NULL, TagSpecification = NULL, SecurityGroupId = NULL, 
    VpcId = NULL, SelfServicePortal = NULL, ClientConnectOptions = NULL, simplify = TRUE, 
    others = list()) {
    Authentication <- list_to_array("Authentication", Authentication)
    DnsServers <- list_to_array("DnsServers", DnsServers)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    parameters <- c(others, list(ClientCidrBlock = ClientCidrBlock, ServerCertificateArn = ServerCertificateArn, 
        ConnectionLogOptions = ConnectionLogOptions, TransportProtocol = TransportProtocol, 
        VpnPort = VpnPort, Description = Description, SplitTunnel = SplitTunnel, DryRun = DryRun, 
        ClientToken = ClientToken, VpcId = VpcId, SelfServicePortal = SelfServicePortal, 
        ClientConnectOptions = ClientConnectOptions), Authentication, DnsServers, TagSpecification, 
        SecurityGroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateClientVpnEndpoint", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Client Vpn Route
#' 
#' Adds a route to a network to a Client
#' VPN endpoint. Each Client VPN endpoint has a route table that describes
#' the available destination network routes. Each route in the route table
#' specifies the path for tra_c to speci_c resources or
#' networks.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint to which to add the route.
#' @param DestinationCidrBlock String.   The IPv4 address range, in CIDR notation, of the route destination.
#' @param TargetVpcSubnetId String.   The ID of the subnet through which you want to route traffic.
#' @param Description String. A brief description of the route.\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint to which to add the route.
#' @section DestinationCidrBlock:
#' 
#' The IPv4 address range, in CIDR notation, of the route destination. For
#' example:
#' 
#' -   To add a route for Internet access, enter `0.0.0.0/0`
#' 
#' -   To add a route for a peered VPC, enter the peered VPC\'s IPv4 CIDR
#'     range
#' 
#' -   To add a route for an on-premises network, enter the AWS
#'     Site-to-Site VPN connection\'s IPv4 CIDR range
#' 
#' -   To add a route for the local network, enter the client CIDR range
#' 
#' @section TargetVpcSubnetId:
#' 
#' The ID of the subnet through which you want to route traffic. The
#' specified subnet must be an existing target network of the Client VPN
#' endpoint.
#' 
#' Alternatively, if you\'re adding a route for the local network, specify
#' `local`.
#' 
#' @section Description:
#' A brief description of the
#' route.
#' @section ClientToken:
#' Unique, case-sensitive identifier that you
#' provide to ensure the idempotency of the request. For more information,
#' see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_client_vpn_route <- function(ClientVpnEndpointId, DestinationCidrBlock, TargetVpcSubnetId, 
    Description = NULL, ClientToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, DestinationCidrBlock = DestinationCidrBlock, 
        TargetVpcSubnetId = TargetVpcSubnetId, Description = Description, ClientToken = ClientToken, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateClientVpnRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Customer Gateway
#' 
#' @param BgpAsn Integer.   For devices that support BGP, the customer gateway\'s BGP ASN.  Default: 65000  
#' @param Type String. The type of VPN connection that this customer gateway supports (`ipsec.1`).
#' @param IpAddress String. The Internet-routable IP address for the customer gateway\'s outside interface.\[optional\]
#' @param CertificateArn String. The Amazon Resource Name (ARN) for the customer gateway certificate.\[optional\]
#' @param TagSpecification List. The tags to apply to the customer gateway.\[optional\]
#' @param DeviceName String.   A name for the customer gateway device.  Length Constraints: Up to 255 characters.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section BgpAsn:
#' 
#' For devices that support BGP, the customer gateway\'s BGP ASN.
#' 
#' Default: 65000
#' 
#' @section Type:
#' The type of VPN connection that this customer
#' gateway supports (`ipsec.1`).
#' @section IpAddress:
#' The Internet-routable
#' IP address for the customer gateway\'s outside interface. The address
#' must be static.
#' @section CertificateArn:
#' The Amazon Resource Name (ARN) for
#' the customer gateway certificate.
#' @section TagSpecification:
#' The tags to apply
#' to the customer gateway.
#' @section DeviceName:
#' 
#' A name for the customer gateway device.
#' 
#' Length Constraints: Up to 255 characters.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_customer_gateway <- function(BgpAsn, Type, IpAddress = NULL, CertificateArn = NULL, 
    TagSpecification = NULL, DeviceName = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(BgpAsn = BgpAsn, Type = Type, IpAddress = IpAddress, CertificateArn = CertificateArn, 
        DeviceName = DeviceName, DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateCustomerGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Default Subnet
#' 
#' Creates a default subnet with a size `/20` IPv4
#' CIDR block in the specified Availability Zone in your default VPC. You
#' can have only one default subnet per Availability Zone. For more
#' information, see [Creating a Default
#' Subnet](https://docs.aws.amazon.com/vpc/latest/userguide/default-vpc.html#create-default-subnet)
#' in the *Amazon Virtual Private Cloud User Guide*.
#' 
#' @param AvailabilityZone String. The Availability Zone in which to create the default subnet.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZone:
#' The Availability Zone in
#' which to create the default subnet.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_default_subnet <- function(AvailabilityZone, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(AvailabilityZone = AvailabilityZone, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateDefaultSubnet", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Default Vpc
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_default_vpc <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateDefaultVpc", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Dhcp Options
#' 
#' @param DhcpConfiguration List. A DHCP configuration option.
#' @param TagSpecification List. The tags to assign to the DHCP option.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DhcpConfiguration:
#' A DHCP
#' configuration option.
#' @section TagSpecification:
#' The tags to assign to the DHCP
#' option.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_dhcp_options <- function(DhcpConfiguration, TagSpecification = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    DhcpConfiguration <- list_to_array("DhcpConfiguration", DhcpConfiguration)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(DryRun = DryRun), DhcpConfiguration, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateDhcpOptions", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Egress Only Internet Gateway
#' 
#' \[IPv6 only\] Creates an egress-only internet
#' gateway for your VPC. An egress-only internet gateway is used to enable
#' outbound communication over IPv6 from instances in your VPC to the
#' internet, and prevents hosts outside of your VPC from initiating an IPv6
#' connection with your instance.
#' 
#' @param VpcId String. The ID of the VPC for which to create the egress-only internet gateway.
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to assign to the egress-only internet gateway.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the VPC for which
#' to create the egress-only internet gateway.
#' @section ClientToken:
#' Unique,
#' case-sensitive identifier that you provide to ensure the idempotency of
#' the request. For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html).
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags
#' to assign to the egress-only internet gateway.
#' @return A list object or a character vector
#' @export
ec2_create_egress_only_internet_gateway <- function(VpcId, ClientToken = NULL, DryRun = NULL, 
    TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(VpcId = VpcId, ClientToken = ClientToken, DryRun = DryRun), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateEgressOnlyInternetGateway", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Fleet
#' 
#' @param LaunchTemplateConfigs List. The configuration for the EC2 Fleet.
#' @param TargetCapacitySpecification Object. The number of units to request.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param SpotOptions Object. Describes the configuration of Spot Instances in an EC2 Fleet.\[optional\]
#' @param OnDemandOptions Object. Describes the configuration of On-Demand Instances in an EC2 Fleet.\[optional\]
#' @param ExcessCapacityTerminationPolicy String. Indicates whether running instances should be terminated if the total target capacity of the EC2...\[optional\]
#' @param TerminateInstancesWithExpiration Logical. Indicates whether running instances should be terminated when the EC2 Fleet expires.\[optional\]
#' @param Type String.   The type of request.\[optional\]
#' @param ValidFrom String. The start date and time of the request, in UTC format (for example, *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z)....\[optional\]
#' @param ValidUntil String. The end date and time of the request, in UTC format (for example, *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z)....\[optional\]
#' @param ReplaceUnhealthyInstances Logical. Indicates whether EC2 Fleet should replace unhealthy Spot Instances.\[optional\]
#' @param TagSpecification List. The key-value pair for tagging the EC2 Fleet request on creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LaunchTemplateConfigs:
#' The
#' configuration for the EC2 Fleet.
#' @section TargetCapacitySpecification:
#' The number of units
#' to request.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section SpotOptions:
#' Describes
#' the configuration of Spot Instances in an EC2
#' Fleet.
#' @section OnDemandOptions:
#' Describes the configuration of On-Demand
#' Instances in an EC2 Fleet.
#' @section ExcessCapacityTerminationPolicy:
#' Indicates whether running
#' instances should be terminated if the total target capacity of the EC2
#' Fleet is decreased below the current size of the EC2
#' Fleet.
#' @section TerminateInstancesWithExpiration:
#' Indicates whether running instances should be
#' terminated when the EC2 Fleet expires.
#' @section Type:
#' 
#' The type of request. The default value is `maintain`.
#' 
#' -   `maintain` - The EC2 Fleet places an asynchronous request for your
#'     desired capacity, and continues to maintain your desired Spot
#'     capacity by replenishing interrupted Spot Instances.
#' 
#' -   `request` - The EC2 Fleet places an asynchronous one-time request
#'     for your desired capacity, but does submit Spot requests in
#'     alternative capacity pools if Spot capacity is unavailable, and does
#'     not maintain Spot capacity if Spot Instances are interrupted.
#' 
#' -   `instant` - The EC2 Fleet places a synchronous one-time request for
#'     your desired capacity, and returns errors for any instances that
#'     could not be launched.
#' 
#' For more information, see [EC2 Fleet request
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-configuration-strategies.html#ec2-fleet-request-type)
#' in the *Amazon EC2 User Guide*.
#' 
#' @section ValidFrom:
#' The start date and time of the request, in UTC
#' format (for example, *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z). The default is
#' to start fulfilling the request immediately.
#' @section ValidUntil:
#' The end
#' date and time of the request, in UTC format (for example,
#' *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z). At this point, no new EC2 Fleet
#' requests are placed or able to fulfill the request. If no value is
#' specified, the request remains until you cancel
#' it.
#' @section ReplaceUnhealthyInstances:
#' Indicates whether EC2 Fleet should replace
#' unhealthy Spot Instances. Supported only for fleets of type `maintain`.
#' For more information, see [EC2 Fleet health
#' checks](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks)
#' in the *Amazon EC2 User Guide*.
#' @section TagSpecification:
#' The key-value pair
#' for tagging the EC2 Fleet request on creation. The value for
#' `ResourceType` must be `fleet`, otherwise the fleet request fails. To
#' tag instances at launch, specify the tags in the [launch
#' template](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template).
#' For information about tagging after launch, see [Tagging your
#' resources](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources).
#' @return A list object or a character vector
#' @export
ec2_create_fleet <- function(LaunchTemplateConfigs, TargetCapacitySpecification, DryRun = NULL, 
    ClientToken = NULL, SpotOptions = NULL, OnDemandOptions = NULL, ExcessCapacityTerminationPolicy = NULL, 
    TerminateInstancesWithExpiration = NULL, Type = NULL, ValidFrom = NULL, ValidUntil = NULL, 
    ReplaceUnhealthyInstances = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    LaunchTemplateConfigs <- list_to_array("LaunchTemplateConfigs", LaunchTemplateConfigs)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(TargetCapacitySpecification = TargetCapacitySpecification, 
        DryRun = DryRun, ClientToken = ClientToken, SpotOptions = SpotOptions, OnDemandOptions = OnDemandOptions, 
        ExcessCapacityTerminationPolicy = ExcessCapacityTerminationPolicy, TerminateInstancesWithExpiration = TerminateInstancesWithExpiration, 
        Type = Type, ValidFrom = ValidFrom, ValidUntil = ValidUntil, ReplaceUnhealthyInstances = ReplaceUnhealthyInstances), 
        LaunchTemplateConfigs, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateFleet", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Flow Logs
#' 
#' @param ResourceId List.   The ID of the subnet, network interface, or VPC for which you want to create a flow log.
#' @param ResourceType String. The type of resource for which to create the flow log.
#' @param TrafficType String. The type of traffic to log.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param DeliverLogsPermissionArn String.   The ARN for the IAM role that permits Amazon EC2 to publish flow logs to a CloudWatch Logs log...\[optional\]
#' @param LogGroupName String.   The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs....\[optional\]
#' @param LogDestinationType String.   Specifies the type of destination to which the flow log data is to be published.\[optional\]
#' @param LogDestination String.   Specifies the destination to which the flow log data is to be published.\[optional\]
#' @param LogFormat String.   The fields to include in the flow log record, in the order in which they should appear.\[optional\]
#' @param TagSpecification List. The tags to apply to the flow logs.\[optional\]
#' @param MaxAggregationInterval Integer.   The maximum interval of time during which a flow of packets is captured and aggregated into a flow...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ResourceId:
#' 
#' The ID of the subnet, network interface, or VPC for which you want to
#' create a flow log.
#' 
#' Constraints: Maximum of 1000 resources
#' 
#' @section ResourceType:
#' The type of resource for which to create the flow
#' log. For example, if you specified a VPC ID for the `ResourceId`
#' property, specify `VPC` for this property.
#' @section TrafficType:
#' The type
#' of traffic to log. You can log traffic that the resource accepts or
#' rejects, or all traffic.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html).
#' @section DeliverLogsPermissionArn:
#' 
#' The ARN for the IAM role that permits Amazon EC2 to publish flow logs to
#' a CloudWatch Logs log group in your account.
#' 
#' If you specify `LogDestinationType` as `s3`, do not specify
#' `DeliverLogsPermissionArn` or `LogGroupName`.
#' 
#' @section LogGroupName:
#' 
#' The name of a new or existing CloudWatch Logs log group where Amazon EC2
#' publishes your flow logs.
#' 
#' If you specify `LogDestinationType` as `s3`, do not specify
#' `DeliverLogsPermissionArn` or `LogGroupName`.
#' 
#' @section LogDestinationType:
#' 
#' Specifies the type of destination to which the flow log data is to be
#' published. Flow log data can be published to CloudWatch Logs or Amazon
#' S3. To publish flow log data to CloudWatch Logs, specify
#' `cloud-watch-logs`. To publish flow log data to Amazon S3, specify `s3`.
#' 
#' If you specify `LogDestinationType` as `s3`, do not specify
#' `DeliverLogsPermissionArn` or `LogGroupName`.
#' 
#' Default: `cloud-watch-logs`
#' 
#' @section LogDestination:
#' 
#' Specifies the destination to which the flow log data is to be published.
#' Flow log data can be published to a CloudWatch Logs log group or an
#' Amazon S3 bucket. The value specified for this parameter depends on the
#' value specified for `LogDestinationType`.
#' 
#' If `LogDestinationType` is not specified or `cloud-watch-logs`, specify
#' the Amazon Resource Name (ARN) of the CloudWatch Logs log group. For
#' example, to publish to a log group called `my-logs`, specify
#' `arn:aws:logs:us-east-1:123456789012:log-group:my-logs`. Alternatively,
#' use `LogGroupName` instead.
#' 
#' If LogDestinationType is `s3`, specify the ARN of the Amazon S3 bucket.
#' You can also specify a subfolder in the bucket. To specify a subfolder
#' in the bucket, use the following ARN format:
#' `bucket_ARN/subfolder_name/`. For example, to specify a subfolder named
#' `my-logs` in a bucket named `my-bucket`, use the following ARN:
#' `arn:aws:s3:::my-bucket/my-logs/`. You cannot use `AWSLogs` as a
#' subfolder name. This is a reserved term.
#' 
#' @section LogFormat:
#' 
#' The fields to include in the flow log record, in the order in which they
#' should appear. For a list of available fields, see [Flow Log
#' Records](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records).
#' If you omit this parameter, the flow log is created using the default
#' format. If you specify this parameter, you must specify at least one
#' field.
#' 
#' Specify the fields using the `${field-id}` format, separated by spaces.
#' For the AWS CLI, use single quotation marks (\' \') to surround the
#' parameter value.
#' 
#' @section TagSpecification:
#' The tags to apply to the flow
#' logs.
#' @section MaxAggregationInterval:
#' 
#' The maximum interval of time during which a flow of packets is captured
#' and aggregated into a flow log record. You can specify 60 seconds (1
#' minute) or 600 seconds (10 minutes).
#' 
#' When a network interface is attached to a [Nitro-based
#' instance](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances),
#' the aggregation interval is always 60 seconds or less, regardless of the
#' value that you specify.
#' 
#' Default: 600
#' 
#' @return A list object or a character vector
#' @export
ec2_create_flow_logs <- function(ResourceId, ResourceType, TrafficType, DryRun = NULL, ClientToken = NULL, 
    DeliverLogsPermissionArn = NULL, LogGroupName = NULL, LogDestinationType = NULL, LogDestination = NULL, 
    LogFormat = NULL, TagSpecification = NULL, MaxAggregationInterval = NULL, simplify = TRUE, 
    others = list()) {
    ResourceId <- list_to_array("ResourceId", ResourceId)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(ResourceType = ResourceType, TrafficType = TrafficType, 
        DryRun = DryRun, ClientToken = ClientToken, DeliverLogsPermissionArn = DeliverLogsPermissionArn, 
        LogGroupName = LogGroupName, LogDestinationType = LogDestinationType, LogDestination = LogDestination, 
        LogFormat = LogFormat, MaxAggregationInterval = MaxAggregationInterval), ResourceId, 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateFlowLogs", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Fpga Image
#' 
#' @param InputStorageLocation Object. The location of the encrypted design checkpoint in Amazon S3. The input must be a tarball.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LogsStorageLocation Object. The location in Amazon S3 for the output logs.\[optional\]
#' @param Description String. A description for the AFI.\[optional\]
#' @param Name String. A name for the AFI.\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param TagSpecification List. The tags to apply to the FPGA image during creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InputStorageLocation:
#' The location of the encrypted design checkpoint in
#' Amazon S3. The input must be a tarball.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section LogsStorageLocation:
#' The
#' location in Amazon S3 for the output logs.
#' @section Description:
#' A
#' description for the AFI.
#' @section Name:
#' A name for the
#' AFI.
#' @section ClientToken:
#' Unique, case-sensitive identifier that you
#' provide to ensure the idempotency of the request. For more information,
#' see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html).
#' @section TagSpecification:
#' The
#' tags to apply to the FPGA image during creation.
#' @return A list object or a character vector
#' @export
ec2_create_fpga_image <- function(InputStorageLocation, DryRun = NULL, LogsStorageLocation = NULL, 
    Description = NULL, Name = NULL, ClientToken = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(InputStorageLocation = InputStorageLocation, DryRun = DryRun, 
        LogsStorageLocation = LogsStorageLocation, Description = Description, Name = Name, 
        ClientToken = ClientToken), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateFpgaImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Image
#' 
#' @param InstanceId String. The ID of the instance.
#' @param Name String.   A name for the new image.
#' @param BlockDeviceMapping List. The block device mappings.\[optional\]
#' @param Description String. A description for the new image.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param NoReboot Logical. By default, Amazon EC2 attempts to shut down and reboot the instance before creating the image.\[optional\]
#' @param TagSpecification List.   The tags to apply to the AMI and snapshots on creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The
#' ID of the instance.
#' @section Name:
#' 
#' A name for the new image.
#' 
#' Constraints: 3-128 alphanumeric characters, parentheses (()), square
#' brackets (\[\]), spaces ( ), periods (.), slashes (/), dashes (-),
#' single quotes (\'), at-signs (@), or underscores(_)
#' 
#' @section BlockDeviceMapping:
#' The block device mappings. This parameter cannot be
#' used to modify the encryption status of existing volumes or snapshots.
#' To create an AMI with encrypted snapshots, use the CopyImage
#' action.
#' @section Description:
#' A description for the new
#' image.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section NoReboot:
#' By default, Amazon EC2
#' attempts to shut down and reboot the instance before creating the image.
#' If the `No Reboot` option is set, Amazon EC2 doesn\'t shut down the
#' instance before creating the image. When this option is used, file
#' system integrity on the created image can\'t be
#' guaranteed.
#' @section TagSpecification:
#' 
#' The tags to apply to the AMI and snapshots on creation. You can tag the
#' AMI, the snapshots, or both.
#' 
#' -   To tag the AMI, the value for `ResourceType` must be `image`.
#' 
#' -   To tag the snapshots that are created of the root volume and of
#'     other EBS volumes that are attached to the instance, the value for
#'     `ResourceType` must be `snapshot`. The same tag is applied to all of
#'     the snapshots that are created.
#' 
#' If you specify other values for `ResourceType`, the request fails.
#' 
#' To tag an AMI or snapshot after it has been created, see
#' [CreateTags](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html).
#' 
#' @return A list object or a character vector
#' @export
ec2_create_image <- function(InstanceId, Name, BlockDeviceMapping = NULL, Description = NULL, 
    DryRun = NULL, NoReboot = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    BlockDeviceMapping <- list_to_array("BlockDeviceMapping", BlockDeviceMapping)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(InstanceId = InstanceId, Name = Name, Description = Description, 
        DryRun = DryRun, NoReboot = NoReboot), BlockDeviceMapping, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Instance Export Task
#' 
#' @param ExportToS3 Object. The format and location for an export instance task.
#' @param InstanceId String. The ID of the instance.
#' @param TargetEnvironment String. The target virtualization environment.
#' @param Description String. A description for the conversion task or the resource being exported.\[optional\]
#' @param TagSpecification List. The tags to apply to the export instance task during creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ExportToS3:
#' The format and location for an export instance
#' task.
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section TargetEnvironment:
#' The target virtualization
#' environment.
#' @section Description:
#' A description for the conversion task
#' or the resource being exported. The maximum length is 255
#' characters.
#' @section TagSpecification:
#' The tags to apply to the export instance
#' task during creation.
#' @return A list object or a character vector
#' @export
ec2_create_instance_export_task <- function(ExportToS3, InstanceId, TargetEnvironment, Description = NULL, 
    TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(ExportToS3 = ExportToS3, InstanceId = InstanceId, TargetEnvironment = TargetEnvironment, 
        Description = Description), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateInstanceExportTask", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Internet Gateway
#' 
#' @param TagSpecification List. The tags to assign to the internet gateway.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TagSpecification:
#' The tags to assign to the
#' internet gateway.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_internet_gateway <- function(TagSpecification = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateInternetGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Key Pair
#' 
#' @param KeyName String.   A unique name for the key pair.  Constraints: Up to 255 ASCII characters  
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to apply to the new key pair.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section KeyName:
#' 
#' A unique name for the key pair.
#' 
#' Constraints: Up to 255 ASCII characters
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to apply to the
#' new key pair.
#' @return A list object or a character vector
#' @export
ec2_create_key_pair <- function(KeyName, DryRun = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(KeyName = KeyName, DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateKeyPair", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Launch Template
#' 
#' Creates a launch template. A launch template
#' contains the parameters to launch an instance. When you launch an
#' instance using RunInstances, you can specify a launch template instead
#' of providing the launch parameters in the request. For more information,
#' see [Launching an instance from a launch
#' template](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html)in
#' the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' @param LaunchTemplateName String. A name for the launch template.
#' @param LaunchTemplateData Object. The information for the launch template.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String.   Unique, case-sensitive identifier you provide to ensure the idempotency of the request.\[optional\]
#' @param VersionDescription String. A description for the first version of the launch template.\[optional\]
#' @param TagSpecification List. The tags to apply to the launch template during creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LaunchTemplateName:
#' A name for the launch
#' template.
#' @section LaunchTemplateData:
#' The information for the launch
#' template.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' 
#' Unique, case-sensitive identifier you provide to ensure the idempotency
#' of the request. For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' 
#' Constraint: Maximum 128 ASCII characters.
#' 
#' @section VersionDescription:
#' A description for the first version of the launch
#' template.
#' @section TagSpecification:
#' The tags to apply to the launch template
#' during creation.
#' @return A list object or a character vector
#' @export
ec2_create_launch_template <- function(LaunchTemplateName, LaunchTemplateData, DryRun = NULL, 
    ClientToken = NULL, VersionDescription = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(LaunchTemplateName = LaunchTemplateName, LaunchTemplateData = LaunchTemplateData, 
        DryRun = DryRun, ClientToken = ClientToken, VersionDescription = VersionDescription), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateLaunchTemplate", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Launch Template Version
#' 
#' @param LaunchTemplateData Object. The information for the launch template.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String.   Unique, case-sensitive identifier you provide to ensure the idempotency of the request.\[optional\]
#' @param LaunchTemplateId String. The ID of the launch template.\[optional\]
#' @param LaunchTemplateName String. The name of the launch template.\[optional\]
#' @param SourceVersion String. The version number of the launch template version on which to base the new version.\[optional\]
#' @param VersionDescription String. A description for the version of the launch template.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LaunchTemplateData:
#' The information for the launch
#' template.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' 
#' Unique, case-sensitive identifier you provide to ensure the idempotency
#' of the request. For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' 
#' Constraint: Maximum 128 ASCII characters.
#' 
#' @section LaunchTemplateId:
#' The ID of the launch template. You must specify
#' either the launch template ID or launch template name in the
#' request.
#' @section LaunchTemplateName:
#' The name of the launch template. You must
#' specify either the launch template ID or launch template name in the
#' request.
#' @section SourceVersion:
#' The version number of the launch template
#' version on which to base the new version. The new version inherits the
#' same launch parameters as the source version, except for parameters that
#' you specify in `LaunchTemplateData`. Snapshots applied to the block
#' device mapping are ignored when creating a new version unless they are
#' explicitly included.
#' @section VersionDescription:
#' A description for the version
#' of the launch template.
#' @return A list object or a character vector
#' @export
ec2_create_launch_template_version <- function(LaunchTemplateData, DryRun = NULL, ClientToken = NULL, 
    LaunchTemplateId = NULL, LaunchTemplateName = NULL, SourceVersion = NULL, VersionDescription = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(LaunchTemplateData = LaunchTemplateData, DryRun = DryRun, 
        ClientToken = ClientToken, LaunchTemplateId = LaunchTemplateId, LaunchTemplateName = LaunchTemplateName, 
        SourceVersion = SourceVersion, VersionDescription = VersionDescription))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateLaunchTemplateVersion", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Local Gateway Route
#' 
#' Creates a static route for the specified local
#' gateway route table.
#' 
#' @param DestinationCidrBlock String. The CIDR range used for destination matches.
#' @param LocalGatewayRouteTableId String. The ID of the local gateway route table.
#' @param LocalGatewayVirtualInterfaceGroupId String. The ID of the virtual interface group.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DestinationCidrBlock:
#' The CIDR range used for
#' destination matches. Routing decisions are based on the most specific
#' match.
#' @section LocalGatewayRouteTableId:
#' The ID of the local gateway route
#' table.
#' @section LocalGatewayVirtualInterfaceGroupId:
#' The ID of the virtual interface
#' group.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_local_gateway_route <- function(DestinationCidrBlock, LocalGatewayRouteTableId, 
    LocalGatewayVirtualInterfaceGroupId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DestinationCidrBlock = DestinationCidrBlock, LocalGatewayRouteTableId = LocalGatewayRouteTableId, 
        LocalGatewayVirtualInterfaceGroupId = LocalGatewayVirtualInterfaceGroupId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateLocalGatewayRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Local Gateway Route Table Vpc Association
#' 
#' Associates the specified VPC
#' with the specified local gateway route table.
#' 
#' @param LocalGatewayRouteTableId String. The ID of the local gateway route table.
#' @param VpcId String. The ID of the VPC.
#' @param TagSpecification List. The tags to assign to the local gateway route table VPC association.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayRouteTableId:
#' The ID of the local gateway
#' route table.
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section TagSpecification:
#' The tags to assign to the local gateway route
#' table VPC association.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_local_gateway_route_table_vpc_association <- function(LocalGatewayRouteTableId, 
    VpcId, TagSpecification = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(LocalGatewayRouteTableId = LocalGatewayRouteTableId, VpcId = VpcId, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateLocalGatewayRouteTableVpcAssociation", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Managed Prefix List
#' 
#' @param PrefixListName String.   A name for the prefix list.
#' @param MaxEntries Integer. The maximum number of entries for the prefix list.
#' @param AddressFamily String.   The IP address type.  Valid Values: `IPv4` \| `IPv6`  
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Entry List. One or more entries for the prefix list.\[optional\]
#' @param TagSpecification List. The tags to apply to the prefix list during creation.\[optional\]
#' @param ClientToken String.   Unique, case-sensitive identifier you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PrefixListName:
#' 
#' A name for the prefix list.
#' 
#' Constraints: Up to 255 characters in length. The name cannot start with
#' `com.amazonaws`.
#' 
#' @section MaxEntries:
#' The maximum number of entries for the prefix
#' list.
#' @section AddressFamily:
#' 
#' The IP address type.
#' 
#' Valid Values: `IPv4` \| `IPv6`
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Entry:
#' One or more entries for the
#' prefix list.
#' @section TagSpecification:
#' The tags to apply to the prefix list
#' during creation.
#' @section ClientToken:
#' 
#' Unique, case-sensitive identifier you provide to ensure the idempotency
#' of the request. For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' 
#' Constraints: Up to 255 UTF-8 characters in length.
#' 
#' @return A list object or a character vector
#' @export
ec2_create_managed_prefix_list <- function(PrefixListName, MaxEntries, AddressFamily, DryRun = NULL, 
    Entry = NULL, TagSpecification = NULL, ClientToken = NULL, simplify = TRUE, others = list()) {
    Entry <- list_to_array("Entry", Entry)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(PrefixListName = PrefixListName, MaxEntries = MaxEntries, 
        AddressFamily = AddressFamily, DryRun = DryRun, ClientToken = ClientToken), Entry, 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateManagedPrefixList", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Nat Gateway
#' 
#' Creates a NAT gateway in the specified public
#' subnet. This action creates a network interface in the specified subnet
#' with a private IP address from the IP address range of the subnet.
#' Internet-bound traffic from a private subnet can be routed to the NAT
#' gateway, therefore enabling instances in the private subnet to connect
#' to the internet. For more information, see [NAT
#' Gateways](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html)
#' in the *Amazon Virtual Private Cloud User Guide*.
#' 
#' @param SubnetId String. The subnet in which to create the NAT gateway.
#' @param AllocationId String. The allocation ID of an Elastic IP address to associate with the NAT gateway.
#' @param ClientToken String.   Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to assign to the NAT gateway.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SubnetId:
#' The subnet in which to create the NAT
#' gateway.
#' @section AllocationId:
#' The allocation ID of an Elastic IP address
#' to associate with the NAT gateway. If the Elastic IP address is
#' associated with another resource, you must first disassociate
#' it.
#' @section ClientToken:
#' 
#' Unique, case-sensitive identifier that you provide to ensure the
#' idempotency of the request. For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' 
#' Constraint: Maximum 64 ASCII characters.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to assign to the
#' NAT gateway.
#' @return A list object or a character vector
#' @export
ec2_create_nat_gateway <- function(SubnetId, AllocationId, ClientToken = NULL, DryRun = NULL, 
    TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(SubnetId = SubnetId, AllocationId = AllocationId, ClientToken = ClientToken, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateNatGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Network Acl
#' 
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to assign to the network ACL.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to assign to the
#' network ACL.
#' @return A list object or a character vector
#' @export
ec2_create_network_acl <- function(VpcId, DryRun = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(VpcId = VpcId, DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateNetworkAcl", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Network Acl Entry
#' 
#' @param Egress Logical. Indicates whether this is an egress rule (rule is applied to traffic leaving the subnet).
#' @param NetworkAclId String. The ID of the network ACL.
#' @param Protocol String. The protocol number.
#' @param RuleAction String. Indicates whether to allow or deny the traffic that matches the rule.
#' @param RuleNumber Integer.   The rule number for the entry (for example, 100).
#' @param CidrBlock String. The IPv4 network range to allow or deny, in CIDR notation (for example `172.16.0.0/24`). We modify...\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Icmp Object. ICMP protocol: The ICMP or ICMPv6 type and code.\[optional\]
#' @param Ipv6CidrBlock String. The IPv6 network range to allow or deny, in CIDR notation (for example `2001:db8:1234:1a00::/64`).\[optional\]
#' @param PortRange Object. TCP or UDP protocols: The range of ports the rule applies to.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Egress:
#' Indicates whether this is an egress
#' rule (rule is applied to traffic leaving the
#' subnet).
#' @section NetworkAclId:
#' The ID of the network
#' ACL.
#' @section Protocol:
#' The protocol number. A value of \'-1\' means
#' all protocols. If you specify \'-1\' or a protocol number other than
#' \'6\' (TCP), \'17\' (UDP), or \'1\' (ICMP), traffic on all ports is
#' allowed, regardless of any ports or ICMP types or codes that you
#' specify. If you specify protocol \'58\' (ICMPv6) and specify an IPv4
#' CIDR block, traffic for all ICMP types and codes allowed, regardless of
#' any that you specify. If you specify protocol \'58\' (ICMPv6) and
#' specify an IPv6 CIDR block, you must specify an ICMP type and
#' code.
#' @section RuleAction:
#' Indicates whether to allow or deny the traffic
#' that matches the rule.
#' @section RuleNumber:
#' 
#' The rule number for the entry (for example, 100). ACL entries are
#' processed in ascending order by rule number.
#' 
#' Constraints: Positive integer from 1 to 32766. The range 32767 to 65535
#' is reserved for internal use.
#' 
#' @section CidrBlock:
#' The IPv4 network range to allow or deny, in CIDR
#' notation (for example `172.16.0.0/24`). We modify the specified CIDR
#' block to its canonical form; for example, if you specify
#' `100.68.0.18/18`, we modify it to
#' `100.68.0.0/18`.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Icmp:
#' ICMP protocol: The ICMP or
#' ICMPv6 type and code. Required if specifying protocol 1 (ICMP) or
#' protocol 58 (ICMPv6) with an IPv6 CIDR block.
#' @section Ipv6CidrBlock:
#' The
#' IPv6 network range to allow or deny, in CIDR notation (for example
#' `2001:db8:1234:1a00::/64`).
#' @section PortRange:
#' TCP or UDP protocols:
#' The range of ports the rule applies to. Required if specifying protocol
#' 6 (TCP) or 17 (UDP).
#' @return A list object or a character vector
#' @export
ec2_create_network_acl_entry <- function(Egress, NetworkAclId, Protocol, RuleAction, RuleNumber, 
    CidrBlock = NULL, DryRun = NULL, Icmp = NULL, Ipv6CidrBlock = NULL, PortRange = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(Egress = Egress, NetworkAclId = NetworkAclId, Protocol = Protocol, 
        RuleAction = RuleAction, RuleNumber = RuleNumber, CidrBlock = CidrBlock, DryRun = DryRun, 
        Icmp = Icmp, Ipv6CidrBlock = Ipv6CidrBlock, PortRange = PortRange))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateNetworkAclEntry", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Network Insights Path
#' 
#' @param Source String. The AWS resource that is the source of the path.
#' @param Destination String. The AWS resource that is the destination of the path.
#' @param Protocol String. The protocol.
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
#' @param SourceIp String. The IP address of the AWS resource that is the source of the path.\[optional\]
#' @param DestinationIp String. The IP address of the AWS resource that is the destination of the path.\[optional\]
#' @param DestinationPort Integer. The destination port.\[optional\]
#' @param TagSpecification List. The tags to add to the path.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Source:
#' The AWS resource that is the
#' source of the path.
#' @section Destination:
#' The AWS resource that is the
#' destination of the path.
#' @section Protocol:
#' The
#' protocol.
#' @section ClientToken:
#' Unique, case-sensitive identifier that you
#' provide to ensure the idempotency of the request. For more information,
#' see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section SourceIp:
#' The
#' IP address of the AWS resource that is the source of the
#' path.
#' @section DestinationIp:
#' The IP address of the AWS resource that is the
#' destination of the path.
#' @section DestinationPort:
#' The destination
#' port.
#' @section TagSpecification:
#' The tags to add to the
#' path.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_network_insights_path <- function(Source, Destination, Protocol, ClientToken, 
    SourceIp = NULL, DestinationIp = NULL, DestinationPort = NULL, TagSpecification = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(Source = Source, Destination = Destination, Protocol = Protocol, 
        ClientToken = ClientToken, SourceIp = SourceIp, DestinationIp = DestinationIp, DestinationPort = DestinationPort, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateNetworkInsightsPath", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Network Interface
#' 
#' @param SubnetId String. The ID of the subnet to associate with the network interface.
#' @param Description String. A description for the network interface.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param SecurityGroupId List. The IDs of one or more security groups.\[optional\]
#' @param Ipv6AddressCount Integer. The number of IPv6 addresses to assign to a network interface.\[optional\]
#' @param Ipv6Addresses List. One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet.\[optional\]
#' @param PrivateIpAddress String. The primary private IPv4 address of the network interface.\[optional\]
#' @param PrivateIpAddresses List. One or more private IPv4 addresses.\[optional\]
#' @param SecondaryPrivateIpAddressCount Integer.   The number of secondary private IPv4 addresses to assign to a network interface.\[optional\]
#' @param InterfaceType String. Indicates the type of network interface.\[optional\]
#' @param TagSpecification List. The tags to apply to the new network interface.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SubnetId:
#' The ID of the subnet to
#' associate with the network interface.
#' @section Description:
#' A description
#' for the network interface.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section SecurityGroupId:
#' The IDs of one or more
#' security groups.
#' @section Ipv6AddressCount:
#' The number of IPv6 addresses to
#' assign to a network interface. Amazon EC2 automatically selects the IPv6
#' addresses from the subnet range. You can\'t use this option if
#' specifying specific IPv6 addresses. If your subnet has the
#' `AssignIpv6AddressOnCreation` attribute set to `true`, you can specify
#' `0` to override this setting.
#' @section Ipv6Addresses:
#' One or more specific
#' IPv6 addresses from the IPv6 CIDR block range of your subnet. You can\'t
#' use this option if you\'re specifying a number of IPv6
#' addresses.
#' @section PrivateIpAddress:
#' The primary private IPv4 address of the
#' network interface. If you don\'t specify an IPv4 address, Amazon EC2
#' selects one for you from the subnet\'s IPv4 CIDR range. If you specify
#' an IP address, you cannot indicate any IP addresses specified in
#' `privateIpAddresses` as primary (only one IP address can be designated
#' as primary).
#' @section PrivateIpAddresses:
#' One or more private IPv4
#' addresses.
#' @section SecondaryPrivateIpAddressCount:
#' 
#' The number of secondary private IPv4 addresses to assign to a network
#' interface. When you specify a number of secondary IPv4 addresses, Amazon
#' EC2 selects these IP addresses within the subnet\'s IPv4 CIDR range. You
#' can\'t specify this option and specify more than one private IP address
#' using `privateIpAddresses`.
#' 
#' The number of IP addresses you can assign to a network interface varies
#' by instance type. For more information, see [IP Addresses Per ENI Per
#' Instance
#' Type](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI)
#' in the *Amazon Virtual Private Cloud User Guide*.
#' 
#' @section InterfaceType:
#' Indicates the type of network interface. To create
#' an Elastic Fabric Adapter (EFA), specify `efa`. For more information,
#' see [Elastic Fabric
#' Adapter](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html)
#' in the *Amazon Elastic Compute Cloud User
#' Guide*.
#' @section TagSpecification:
#' The tags to apply to the new network
#' interface.
#' @return A list object or a character vector
#' @export
ec2_create_network_interface <- function(SubnetId, Description = NULL, DryRun = NULL, SecurityGroupId = NULL, 
    Ipv6AddressCount = NULL, Ipv6Addresses = NULL, PrivateIpAddress = NULL, PrivateIpAddresses = NULL, 
    SecondaryPrivateIpAddressCount = NULL, InterfaceType = NULL, TagSpecification = NULL, 
    simplify = TRUE, others = list()) {
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    Ipv6Addresses <- list_to_array("Ipv6Addresses", Ipv6Addresses)
    PrivateIpAddresses <- list_to_array("PrivateIpAddresses", PrivateIpAddresses)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(SubnetId = SubnetId, Description = Description, DryRun = DryRun, 
        Ipv6AddressCount = Ipv6AddressCount, PrivateIpAddress = PrivateIpAddress, SecondaryPrivateIpAddressCount = SecondaryPrivateIpAddressCount, 
        InterfaceType = InterfaceType), SecurityGroupId, Ipv6Addresses, PrivateIpAddresses, 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateNetworkInterface", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Network Interface Permission
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param Permission String. The type of permission to grant.
#' @param AwsAccountId String. The AWS account ID.\[optional\]
#' @param AwsService String. The AWS service. Currently not supported.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section Permission:
#' The type of permission to
#' grant.
#' @section AwsAccountId:
#' The AWS account ID.
#' @section AwsService:
#' The
#' AWS service. Currently not supported.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_network_interface_permission <- function(NetworkInterfaceId, Permission, AwsAccountId = NULL, 
    AwsService = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, Permission = Permission, 
        AwsAccountId = AwsAccountId, AwsService = AwsService, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateNetworkInterfacePermission", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Placement Group
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param GroupName String.   A name for the placement group.\[optional\]
#' @param Strategy String. The placement strategy.\[optional\]
#' @param PartitionCount Integer. The number of partitions. Valid only when **Strategy** is set to `partition`.\[optional\]
#' @param TagSpecification List. The tags to apply to the new placement group.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section GroupName:
#' 
#' A name for the placement group. Must be unique within the scope of your
#' account for the Region.
#' 
#' Constraints: Up to 255 ASCII characters
#' 
#' @section Strategy:
#' The placement strategy.
#' @section PartitionCount:
#' The
#' number of partitions. Valid only when **Strategy** is set to
#' `partition`.
#' @section TagSpecification:
#' The tags to apply to the new placement
#' group.
#' @return A list object or a character vector
#' @export
ec2_create_placement_group <- function(DryRun = NULL, GroupName = NULL, Strategy = NULL, 
    PartitionCount = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(DryRun = DryRun, GroupName = GroupName, Strategy = Strategy, 
        PartitionCount = PartitionCount), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreatePlacementGroup", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Reserved Instances Listing
#' 
#' @param ClientToken String. Unique, case-sensitive identifier you provide to ensure idempotency of your listings.
#' @param InstanceCount Integer. The number of instances that are a part of a Reserved Instance account to be listed in the Reserved...
#' @param PriceSchedules List. A list specifying the price of the Standard Reserved Instance for each month remaining in the Reserved...
#' @param ReservedInstancesId String. The ID of the active Standard Reserved Instance.
#' @inheritParams CommonDoc
#' 
#' @section ClientToken:
#' Unique, case-sensitive identifier you provide
#' to ensure idempotency of your listings. This helps avoid duplicate
#' listings. For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section InstanceCount:
#' The
#' number of instances that are a part of a Reserved Instance account to be
#' listed in the Reserved Instance Marketplace. This number should be less
#' than or equal to the instance count associated with the Reserved
#' Instance ID specified in this call.
#' @section PriceSchedules:
#' A list
#' specifying the price of the Standard Reserved Instance for each month
#' remaining in the Reserved Instance term.
#' @section ReservedInstancesId:
#' The ID of
#' the active Standard Reserved Instance.
#' @return A list object or a character vector
#' @export
ec2_create_reserved_instances_listing <- function(ClientToken, InstanceCount, PriceSchedules, 
    ReservedInstancesId, simplify = TRUE, others = list()) {
    PriceSchedules <- list_to_array("PriceSchedules", PriceSchedules)
    parameters <- c(others, list(ClientToken = ClientToken, InstanceCount = InstanceCount, 
        ReservedInstancesId = ReservedInstancesId), PriceSchedules)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateReservedInstancesListing", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Route
#' 
#' @param RouteTableId String. The ID of the route table for the route.
#' @param DestinationCidrBlock String. The IPv4 CIDR address block used for the destination match.\[optional\]
#' @param DestinationIpv6CidrBlock String. The IPv6 CIDR block used for the destination match.\[optional\]
#' @param DestinationPrefixListId String. The ID of a prefix list used for the destination match.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VpcEndpointId String. The ID of a VPC endpoint. Supported for Gateway Load Balancer endpoints only.\[optional\]
#' @param EgressOnlyInternetGatewayId String. \[IPv6 traffic only\] The ID of an egress-only internet gateway.\[optional\]
#' @param GatewayId String. The ID of an internet gateway or virtual private gateway attached to your VPC.\[optional\]
#' @param InstanceId String. The ID of a NAT instance in your VPC.\[optional\]
#' @param NatGatewayId String. \[IPv4 traffic only\] The ID of a NAT gateway.\[optional\]
#' @param TransitGatewayId String. The ID of a transit gateway.\[optional\]
#' @param LocalGatewayId String. The ID of the local gateway.\[optional\]
#' @param CarrierGatewayId String.   The ID of the carrier gateway.\[optional\]
#' @param NetworkInterfaceId String. The ID of a network interface.\[optional\]
#' @param VpcPeeringConnectionId String. The ID of a VPC peering connection.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section RouteTableId:
#' The ID of the
#' route table for the route.
#' @section DestinationCidrBlock:
#' The IPv4 CIDR address
#' block used for the destination match. Routing decisions are based on the
#' most specific match. We modify the specified CIDR block to its canonical
#' form; for example, if you specify `100.68.0.18/18`, we modify it to
#' `100.68.0.0/18`.
#' @section DestinationIpv6CidrBlock:
#' The IPv6 CIDR block used for the
#' destination match. Routing decisions are based on the most specific
#' match.
#' @section DestinationPrefixListId:
#' The ID of a prefix list used for the
#' destination match.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section VpcEndpointId:
#' The ID of a VPC endpoint.
#' Supported for Gateway Load Balancer endpoints
#' only.
#' @section EgressOnlyInternetGatewayId:
#' \[IPv6 traffic only\] The ID of an egress-only
#' internet gateway.
#' @section GatewayId:
#' The ID of an internet gateway or
#' virtual private gateway attached to your VPC.
#' @section InstanceId:
#' The ID
#' of a NAT instance in your VPC. The operation fails if you specify an
#' instance ID unless exactly one network interface is
#' attached.
#' @section NatGatewayId:
#' \[IPv4 traffic only\] The ID of a NAT
#' gateway.
#' @section TransitGatewayId:
#' The ID of a transit
#' gateway.
#' @section LocalGatewayId:
#' The ID of the local
#' gateway.
#' @section CarrierGatewayId:
#' 
#' The ID of the carrier gateway.
#' 
#' You can only use this option when the VPC contains a subnet which is
#' associated with a Wavelength Zone.
#' 
#' @section NetworkInterfaceId:
#' The ID of a network
#' interface.
#' @section VpcPeeringConnectionId:
#' The ID of a VPC peering
#' connection.
#' @return A list object or a character vector
#' @export
ec2_create_route <- function(RouteTableId, DestinationCidrBlock = NULL, DestinationIpv6CidrBlock = NULL, 
    DestinationPrefixListId = NULL, DryRun = NULL, VpcEndpointId = NULL, EgressOnlyInternetGatewayId = NULL, 
    GatewayId = NULL, InstanceId = NULL, NatGatewayId = NULL, TransitGatewayId = NULL, LocalGatewayId = NULL, 
    CarrierGatewayId = NULL, NetworkInterfaceId = NULL, VpcPeeringConnectionId = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(RouteTableId = RouteTableId, DestinationCidrBlock = DestinationCidrBlock, 
        DestinationIpv6CidrBlock = DestinationIpv6CidrBlock, DestinationPrefixListId = DestinationPrefixListId, 
        DryRun = DryRun, VpcEndpointId = VpcEndpointId, EgressOnlyInternetGatewayId = EgressOnlyInternetGatewayId, 
        GatewayId = GatewayId, InstanceId = InstanceId, NatGatewayId = NatGatewayId, TransitGatewayId = TransitGatewayId, 
        LocalGatewayId = LocalGatewayId, CarrierGatewayId = CarrierGatewayId, NetworkInterfaceId = NetworkInterfaceId, 
        VpcPeeringConnectionId = VpcPeeringConnectionId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Route Table
#' 
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to assign to the route table.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to assign to the
#' route table.
#' @return A list object or a character vector
#' @export
ec2_create_route_table <- function(VpcId, DryRun = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(VpcId = VpcId, DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateRouteTable", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Security Group
#' 
#' @param GroupDescription String.   A description for the security group.
#' @param GroupName String.   The name of the security group.
#' @param VpcId String. \[EC2-VPC\] The ID of the VPC. Required for EC2-VPC.\[optional\]
#' @param TagSpecification List. The tags to assign to the security group.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GroupDescription:
#' 
#' A description for the security group. This is informational only.
#' 
#' Constraints: Up to 255 characters in length
#' 
#' Constraints for EC2-Classic: ASCII characters
#' 
#' Constraints for EC2-VPC: a-z, A-Z, 0-9, spaces, and
#' ._-:/()\#,@\[\]+=&;{}!\$\*
#' 
#' @section GroupName:
#' 
#' The name of the security group.
#' 
#' Constraints: Up to 255 characters in length. Cannot start with `sg-`.
#' 
#' Constraints for EC2-Classic: ASCII characters
#' 
#' Constraints for EC2-VPC: a-z, A-Z, 0-9, spaces, and
#' ._-:/()\#,@\[\]+=&;{}!\$\*
#' 
#' @section VpcId:
#' \[EC2-VPC\] The ID of the VPC. Required for
#' EC2-VPC.
#' @section TagSpecification:
#' The tags to assign to the security
#' group.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_security_group <- function(GroupDescription, GroupName, VpcId = NULL, TagSpecification = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(GroupDescription = GroupDescription, GroupName = GroupName, 
        VpcId = VpcId, DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateSecurityGroup", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Snapshot
#' 
#' @param VolumeId String. The ID of the EBS volume.
#' @param Description String. A description for the snapshot.\[optional\]
#' @param OutpostArn String.   The Amazon Resource Name (ARN) of the AWS Outpost on which to create a local snapshot.\[optional\]
#' @param TagSpecification List. The tags to apply to the snapshot during creation.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VolumeId:
#' The ID of the EBS
#' volume.
#' @section Description:
#' A description for the
#' snapshot.
#' @section OutpostArn:
#' 
#' The Amazon Resource Name (ARN) of the AWS Outpost on which to create a
#' local snapshot.
#' 
#' -   To create a snapshot of a volume in a Region, omit this parameter.
#'     The snapshot is created in the same Region as the volume.
#' 
#' -   To create a snapshot of a volume on an Outpost and store the
#'     snapshot in the Region, omit this parameter. The snapshot is created
#'     in the Region for the Outpost.
#' 
#' -   To create a snapshot of a volume on an Outpost and store the
#'     snapshot on an Outpost, specify the ARN of the destination Outpost.
#'     The snapshot must be created on the same Outpost as the volume.
#' 
#' For more information, see [Creating local snapshots from volumes on an
#' Outpost](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshots-outposts.html#create-snapshot)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' @section TagSpecification:
#' The tags to apply to the snapshot during
#' creation.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_snapshot <- function(VolumeId, Description = NULL, OutpostArn = NULL, TagSpecification = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(VolumeId = VolumeId, Description = Description, OutpostArn = OutpostArn, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateSnapshot", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Snapshots
#' 
#' @param InstanceSpecification Object. The instance to specify which volumes should be included in the snapshots.
#' @param Description String.  A description propagated to every snapshot specified by the instance.\[optional\]
#' @param OutpostArn String.   The Amazon Resource Name (ARN) of the AWS Outpost on which to create the local snapshots.\[optional\]
#' @param TagSpecification List. Tags to apply to every snapshot specified by the instance.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param CopyTagsFromSource String. Copies the tags from the specified volume to corresponding snapshot.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceSpecification:
#' The instance to specify
#' which volumes should be included in the snapshots.
#' @section Description:
#' 
#' A description propagated to every snapshot specified by the
#' instance.
#' @section OutpostArn:
#' 
#' The Amazon Resource Name (ARN) of the AWS Outpost on which to create the
#' local snapshots.
#' 
#' -   To create snapshots from an instance in a Region, omit this
#'     parameter. The snapshots are created in the same Region as the
#'     instance.
#' 
#' -   To create snapshots from an instance on an Outpost and store the
#'     snapshots in the Region, omit this parameter. The snapshots are
#'     created in the Region for the Outpost.
#' 
#' -   To create snapshots from an instance on an Outpost and store the
#'     snapshots on an Outpost, specify the ARN of the destination Outpost.
#'     The snapshots must be created on the same Outpost as the instance.
#' 
#' For more information, see [Creating multi-volume local snapshots from
#' instances on an
#' Outpost](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshots-outposts.html#create-multivol-snapshot)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' @section TagSpecification:
#' Tags to apply to every snapshot specified by the
#' instance.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section CopyTagsFromSource:
#' Copies the tags from the
#' specified volume to corresponding snapshot.
#' @return A list object or a character vector
#' @export
ec2_create_snapshots <- function(InstanceSpecification, Description = NULL, OutpostArn = NULL, 
    TagSpecification = NULL, DryRun = NULL, CopyTagsFromSource = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(InstanceSpecification = InstanceSpecification, Description = Description, 
        OutpostArn = OutpostArn, DryRun = DryRun, CopyTagsFromSource = CopyTagsFromSource), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateSnapshots", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Spot Datafeed Subscription
#' 
#' Creates a data feed for Spot Instances, enabling
#' you to view Spot Instance usage logs. You can create one data feed per
#' AWS account. For more information, see [Spot Instance data
#' feed](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-data-feeds.html)
#' in the *Amazon EC2 User Guide for Linux Instances*.
#' 
#' @param Bucket String. The name of the Amazon S3 bucket in which to store the Spot Instance data feed.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Prefix String. The prefix for the data feed file names.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Bucket:
#' The name
#' of the Amazon S3 bucket in which to store the Spot Instance data feed.
#' For more information about bucket names, see [Rules for bucket
#' naming](https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html#bucketnamingrules)
#' in the *Amazon S3 Developer Guide*.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section Prefix:
#' The prefix
#' for the data feed file names.
#' @return A list object or a character vector
#' @export
ec2_create_spot_datafeed_subscription <- function(Bucket, DryRun = NULL, Prefix = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Bucket = Bucket, DryRun = DryRun, Prefix = Prefix))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateSpotDatafeedSubscription", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Subnet
#' 
#' @param VpcId String. The ID of the VPC.
#' @param CidrBlock String. The IPv4 network range for the subnet, in CIDR notation.
#' @param TagSpecification List. The tags to assign to the subnet.\[optional\]
#' @param AvailabilityZone String.   The Availability Zone or Local Zone for the subnet.\[optional\]
#' @param AvailabilityZoneId String. The AZ ID or the Local Zone ID of the subnet.\[optional\]
#' @param Ipv6CidrBlock String. The IPv6 network range for the subnet, in CIDR notation.\[optional\]
#' @param OutpostArn String. The Amazon Resource Name (ARN) of the Outpost.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section CidrBlock:
#' The IPv4 network range for the subnet, in CIDR
#' notation. For example, `10.0.0.0/24`. We modify the specified CIDR block
#' to its canonical form; for example, if you specify `100.68.0.18/18`, we
#' modify it to `100.68.0.0/18`.
#' @section TagSpecification:
#' The tags to assign to
#' the subnet.
#' @section AvailabilityZone:
#' 
#' The Availability Zone or Local Zone for the subnet.
#' 
#' Default: AWS selects one for you. If you create more than one subnet in
#' your VPC, we do not necessarily select a different zone for each subnet.
#' 
#' To create a subnet in a Local Zone, set this value to the Local Zone ID,
#' for example `us-west-2-lax-1a`. For information about the Regions that
#' support Local Zones, see [Available
#' Regions](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#concepts-available-regions)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' To create a subnet in an Outpost, set this value to the Availability
#' Zone for the Outpost and specify the Outpost ARN.
#' 
#' @section AvailabilityZoneId:
#' The AZ ID or the Local Zone ID of the
#' subnet.
#' @section Ipv6CidrBlock:
#' The IPv6 network range for the subnet, in
#' CIDR notation. The subnet size must use a /64 prefix
#' length.
#' @section OutpostArn:
#' The Amazon Resource Name (ARN) of the
#' Outpost. If you specify an Outpost ARN, you must also specify the
#' Availability Zone of the Outpost subnet.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_subnet <- function(VpcId, CidrBlock, TagSpecification = NULL, AvailabilityZone = NULL, 
    AvailabilityZoneId = NULL, Ipv6CidrBlock = NULL, OutpostArn = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(VpcId = VpcId, CidrBlock = CidrBlock, AvailabilityZone = AvailabilityZone, 
        AvailabilityZoneId = AvailabilityZoneId, Ipv6CidrBlock = Ipv6CidrBlock, OutpostArn = OutpostArn, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateSubnet", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Tags
#' 
#' @param ResourceId List.   The IDs of the resources, separated by spaces.
#' @param Tag List. The tags.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ResourceId:
#' 
#' The IDs of the resources, separated by spaces.
#' 
#' Constraints: Up to 1000 resource IDs. We recommend breaking up this
#' request into smaller batches.
#' 
#' @section Tag:
#' The tags. The `value` parameter is required, but if
#' you don\'t want the tag to have a value, specify the parameter with no
#' value, and we set the value to an empty
#' string.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_tags <- function(ResourceId, Tag, DryRun = NULL, simplify = TRUE, others = list()) {
    ResourceId <- list_to_array("ResourceId", ResourceId)
    Tag <- list_to_array("Tag", Tag)
    parameters <- c(others, list(DryRun = DryRun), ResourceId, Tag)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTags", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Traffic Mirror Filter
#' 
#' @param Description String. The description of the Traffic Mirror filter.\[optional\]
#' @param TagSpecification List. The tags to assign to a Traffic Mirror filter.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Description:
#' The description of the
#' Traffic Mirror filter.
#' @section TagSpecification:
#' The tags to assign to a
#' Traffic Mirror filter.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_create_traffic_mirror_filter <- function(Description = NULL, TagSpecification = NULL, 
    DryRun = NULL, ClientToken = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(Description = Description, DryRun = DryRun, ClientToken = ClientToken), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTrafficMirrorFilter", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Traffic Mirror Filter Rule
#' 
#' @param TrafficMirrorFilterId String. The ID of the filter that this rule is associated with.
#' @param TrafficDirection String. The type of traffic (`ingress` \| `egress`).
#' @param RuleNumber Integer. The number of the Traffic Mirror rule.
#' @param RuleAction String. The action to take (`accept` \| `reject`) on the filtered traffic.
#' @param DestinationCidrBlock String. The destination CIDR block to assign to the Traffic Mirror rule.
#' @param SourceCidrBlock String. The source CIDR block to assign to the Traffic Mirror rule.
#' @param DestinationPortRange Object. The destination port range.\[optional\]
#' @param SourcePortRange Object. The source port range.\[optional\]
#' @param Protocol Integer.   The protocol, for example UDP, to assign to the Traffic Mirror rule.\[optional\]
#' @param Description String. The description of the Traffic Mirror rule.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorFilterId:
#' The
#' ID of the filter that this rule is associated
#' with.
#' @section TrafficDirection:
#' The type of traffic (`ingress` \|
#' `egress`).
#' @section RuleNumber:
#' The number of the Traffic Mirror rule.
#' This number must be unique for each Traffic Mirror rule in a given
#' direction. The rules are processed in ascending order by rule
#' number.
#' @section RuleAction:
#' The action to take (`accept` \| `reject`) on
#' the filtered traffic.
#' @section DestinationCidrBlock:
#' The destination CIDR block to
#' assign to the Traffic Mirror rule.
#' @section SourceCidrBlock:
#' The source CIDR
#' block to assign to the Traffic Mirror rule.
#' @section DestinationPortRange:
#' The
#' destination port range.
#' @section SourcePortRange:
#' The source port
#' range.
#' @section Protocol:
#' 
#' The protocol, for example UDP, to assign to the Traffic Mirror rule.
#' 
#' For information about the protocol value, see [Protocol
#' Numbers](https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)
#' on the Internet Assigned Numbers Authority (IANA) website.
#' 
#' @section Description:
#' The description of the Traffic Mirror
#' rule.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_create_traffic_mirror_filter_rule <- function(TrafficMirrorFilterId, TrafficDirection, 
    RuleNumber, RuleAction, DestinationCidrBlock, SourceCidrBlock, DestinationPortRange = NULL, 
    SourcePortRange = NULL, Protocol = NULL, Description = NULL, DryRun = NULL, ClientToken = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TrafficMirrorFilterId = TrafficMirrorFilterId, TrafficDirection = TrafficDirection, 
        RuleNumber = RuleNumber, RuleAction = RuleAction, DestinationCidrBlock = DestinationCidrBlock, 
        SourceCidrBlock = SourceCidrBlock, DestinationPortRange = DestinationPortRange, SourcePortRange = SourcePortRange, 
        Protocol = Protocol, Description = Description, DryRun = DryRun, ClientToken = ClientToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTrafficMirrorFilterRule", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Traffic Mirror Session
#' 
#' @param NetworkInterfaceId String. The ID of the source network interface.
#' @param TrafficMirrorTargetId String. The ID of the Traffic Mirror target.
#' @param TrafficMirrorFilterId String. The ID of the Traffic Mirror filter.
#' @param SessionNumber Integer.   The session number determines the order in which sessions are evaluated when an interface is used...
#' @param PacketLength Integer.   The number of bytes in each packet to mirror.\[optional\]
#' @param VirtualNetworkId Integer. The VXLAN ID for the Traffic Mirror session.\[optional\]
#' @param Description String. The description of the Traffic Mirror session.\[optional\]
#' @param TagSpecification List. The tags to assign to a Traffic Mirror session.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The
#' ID of the source network interface.
#' @section TrafficMirrorTargetId:
#' The ID of the
#' Traffic Mirror target.
#' @section TrafficMirrorFilterId:
#' The ID of the Traffic Mirror
#' filter.
#' @section SessionNumber:
#' 
#' The session number determines the order in which sessions are evaluated
#' when an interface is used by multiple sessions. The first session with a
#' matching filter is the one that mirrors the packets.
#' 
#' Valid values are 1-32766.
#' 
#' @section PacketLength:
#' 
#' The number of bytes in each packet to mirror. These are bytes after the
#' VXLAN header. Do not specify this parameter when you want to mirror the
#' entire packet. To mirror a subset of the packet, set this to the length
#' (in bytes) that you want to mirror. For example, if you set this value
#' to 100, then the first 100 bytes that meet the filter criteria are
#' copied to the target.
#' 
#' If you do not want to mirror the entire packet, use the `PacketLength`
#' parameter to specify the number of bytes in each packet to mirror.
#' 
#' @section VirtualNetworkId:
#' The VXLAN ID for the Traffic Mirror session. For
#' more information about the VXLAN protocol, see [RFC
#' 7348](https://tools.ietf.org/html/rfc7348). If you do not specify a
#' `VirtualNetworkId`, an account-wide unique id is chosen at
#' random.
#' @section Description:
#' The description of the Traffic Mirror
#' session.
#' @section TagSpecification:
#' The tags to assign to a Traffic Mirror
#' session.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_create_traffic_mirror_session <- function(NetworkInterfaceId, TrafficMirrorTargetId, 
    TrafficMirrorFilterId, SessionNumber, PacketLength = NULL, VirtualNetworkId = NULL, Description = NULL, 
    TagSpecification = NULL, DryRun = NULL, ClientToken = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, TrafficMirrorTargetId = TrafficMirrorTargetId, 
        TrafficMirrorFilterId = TrafficMirrorFilterId, SessionNumber = SessionNumber, PacketLength = PacketLength, 
        VirtualNetworkId = VirtualNetworkId, Description = Description, DryRun = DryRun, 
        ClientToken = ClientToken), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTrafficMirrorSession", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Traffic Mirror Target
#' 
#' @param NetworkInterfaceId String. The network interface ID that is associated with the target.\[optional\]
#' @param NetworkLoadBalancerArn String. The Amazon Resource Name (ARN) of the Network Load Balancer that is associated with the target.\[optional\]
#' @param Description String. The description of the Traffic Mirror target.\[optional\]
#' @param TagSpecification List. The tags to assign to the Traffic Mirror target.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The
#' network interface ID that is associated with the
#' target.
#' @section NetworkLoadBalancerArn:
#' The Amazon Resource Name (ARN) of the
#' Network Load Balancer that is associated with the
#' target.
#' @section Description:
#' The description of the Traffic Mirror
#' target.
#' @section TagSpecification:
#' The tags to assign to the Traffic Mirror
#' target.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' Unique, case-sensitive
#' identifier that you provide to ensure the idempotency of the request.
#' For more information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_create_traffic_mirror_target <- function(NetworkInterfaceId = NULL, NetworkLoadBalancerArn = NULL, 
    Description = NULL, TagSpecification = NULL, DryRun = NULL, ClientToken = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, NetworkLoadBalancerArn = NetworkLoadBalancerArn, 
        Description = Description, DryRun = DryRun, ClientToken = ClientToken), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTrafficMirrorTarget", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway
#' 
#' @param Description String. A description of the transit gateway.\[optional\]
#' @param Options Object. The transit gateway options.\[optional\]
#' @param TagSpecification List. The tags to apply to the transit gateway.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Description:
#' A
#' description of the transit gateway.
#' @section Options:
#' The transit
#' gateway options.
#' @section TagSpecification:
#' The tags to apply to the transit
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway <- function(Description = NULL, Options = NULL, TagSpecification = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(Description = Description, Options = Options, DryRun = DryRun), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Connect
#' 
#' @param TransportTransitGatewayAttachmentId String. The ID of the transit gateway attachment.
#' @param Options Object. The Connect attachment options.
#' @param TagSpecification List. The tags to apply to the Connect attachment.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransportTransitGatewayAttachmentId:
#' The ID of the transit
#' gateway attachment. You can specify a VPC attachment or a AWS Direct
#' Connect attachment.
#' @section Options:
#' The Connect attachment
#' options.
#' @section TagSpecification:
#' The tags to apply to the Connect
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_connect <- function(TransportTransitGatewayAttachmentId, Options, 
    TagSpecification = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(TransportTransitGatewayAttachmentId = TransportTransitGatewayAttachmentId, 
        Options = Options, DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayConnect", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Connect Peer
#' 
#' @param TransitGatewayAttachmentId String. The ID of the Connect attachment.
#' @param PeerAddress String. The peer IP address (GRE outer IP address) on the appliance side of the Connect peer.
#' @param InsideCidrBlocks List. The range of inside IP addresses that are used for BGP peering.
#' @param TransitGatewayAddress String. The peer IP address (GRE outer IP address) on the transit gateway side of the Connect peer, which...\[optional\]
#' @param BgpOptions Object. The BGP options for the Connect peer.\[optional\]
#' @param TagSpecification List. The tags to apply to the Connect peer.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the Connect
#' attachment.
#' @section PeerAddress:
#' The peer IP address (GRE outer IP
#' address) on the appliance side of the Connect
#' peer.
#' @section InsideCidrBlocks:
#' The range of inside IP addresses that are used
#' for BGP peering. You must specify a size /29 IPv4 CIDR block from the
#' `169.254.0.0/16` range. The first address from the range must be
#' configured on the appliance as the BGP IP address. You can also
#' optionally specify a size /125 IPv6 CIDR block from the `fd00::/8`
#' range.
#' @section TransitGatewayAddress:
#' The peer IP address (GRE outer IP address) on
#' the transit gateway side of the Connect peer, which must be specified
#' from a transit gateway CIDR block. If not specified, Amazon
#' automatically assigns the first available IP address from the transit
#' gateway CIDR block.
#' @section BgpOptions:
#' The BGP options for the Connect
#' peer.
#' @section TagSpecification:
#' The tags to apply to the Connect
#' peer.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_connect_peer <- function(TransitGatewayAttachmentId, PeerAddress, 
    InsideCidrBlocks, TransitGatewayAddress = NULL, BgpOptions = NULL, TagSpecification = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    InsideCidrBlocks <- list_to_array("InsideCidrBlocks", InsideCidrBlocks)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        PeerAddress = PeerAddress, TransitGatewayAddress = TransitGatewayAddress, BgpOptions = BgpOptions, 
        DryRun = DryRun), InsideCidrBlocks, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayConnectPeer", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Multicast Domain
#' 
#' @param TransitGatewayId String. The ID of the transit gateway.
#' @param Options Object. The options for the transit gateway multicast domain.\[optional\]
#' @param TagSpecification List. The tags for the transit gateway multicast domain.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayId:
#' The ID of the transit
#' gateway.
#' @section Options:
#' The options for the transit gateway
#' multicast domain.
#' @section TagSpecification:
#' The tags for the transit gateway
#' multicast domain.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_multicast_domain <- function(TransitGatewayId, Options = NULL, 
    TagSpecification = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(TransitGatewayId = TransitGatewayId, Options = Options, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayMulticastDomain", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Peering Attachment
#' 
#' @param TransitGatewayId String. The ID of the transit gateway.
#' @param PeerTransitGatewayId String. The ID of the peer transit gateway with which to create the peering attachment.
#' @param PeerAccountId String. The AWS account ID of the owner of the peer transit gateway.
#' @param PeerRegion String. The Region where the peer transit gateway is located.
#' @param TagSpecification List. The tags to apply to the transit gateway peering attachment.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayId:
#' The ID of the transit
#' gateway.
#' @section PeerTransitGatewayId:
#' The ID of the peer transit gateway with
#' which to create the peering attachment.
#' @section PeerAccountId:
#' The AWS
#' account ID of the owner of the peer transit
#' gateway.
#' @section PeerRegion:
#' The Region where the peer transit gateway
#' is located.
#' @section TagSpecification:
#' The tags to apply to the transit gateway
#' peering attachment.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_peering_attachment <- function(TransitGatewayId, PeerTransitGatewayId, 
    PeerAccountId, PeerRegion, TagSpecification = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(TransitGatewayId = TransitGatewayId, PeerTransitGatewayId = PeerTransitGatewayId, 
        PeerAccountId = PeerAccountId, PeerRegion = PeerRegion, DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayPeeringAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Prefix List Reference
#' 
#' Creates a reference (route) to a prefix list in a
#' specified transit gateway route table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param PrefixListId String. The ID of the prefix list that is used for destination matches.
#' @param TransitGatewayAttachmentId String. The ID of the attachment to which traffic is routed.\[optional\]
#' @param Blackhole Logical. Indicates whether to drop traffic that matches this route.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section PrefixListId:
#' The ID of the prefix list that
#' is used for destination matches.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment to which traffic is routed.
#' @section Blackhole:
#' Indicates
#' whether to drop traffic that matches this
#' route.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_prefix_list_reference <- function(TransitGatewayRouteTableId, 
    PrefixListId, TransitGatewayAttachmentId = NULL, Blackhole = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        PrefixListId = PrefixListId, TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        Blackhole = Blackhole, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayPrefixListReference", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Route
#' 
#' Creates a
#' static route for the specified transit gateway route
#' table.
#' 
#' @param DestinationCidrBlock String. The CIDR range used for destination matches.
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param TransitGatewayAttachmentId String. The ID of the attachment.\[optional\]
#' @param Blackhole Logical. Indicates whether to drop traffic that matches this route.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DestinationCidrBlock:
#' The CIDR range used for
#' destination matches. Routing decisions are based on the most specific
#' match.
#' @section TransitGatewayRouteTableId:
#' The ID of the transit gateway route
#' table.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section Blackhole:
#' Indicates whether to drop traffic that
#' matches this route.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_route <- function(DestinationCidrBlock, TransitGatewayRouteTableId, 
    TransitGatewayAttachmentId = NULL, Blackhole = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(DestinationCidrBlock = DestinationCidrBlock, TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, Blackhole = Blackhole, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Route Table
#' 
#' Creates a route table for the specified
#' transit gateway.
#' 
#' @param TransitGatewayId String. The ID of the transit gateway.
#' @param TagSpecifications List. The tags to apply to the transit gateway route table.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayId:
#' The ID of the transit
#' gateway.
#' @section TagSpecifications:
#' The tags to apply to the transit gateway
#' route table.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_route_table <- function(TransitGatewayId, TagSpecifications = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecifications <- list_to_array("TagSpecifications", TagSpecifications)
    parameters <- c(others, list(TransitGatewayId = TransitGatewayId, DryRun = DryRun), TagSpecifications)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayRouteTable", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Transit Gateway Vpc Attachment
#' 
#' @param TransitGatewayId String. The ID of the transit gateway.
#' @param VpcId String. The ID of the VPC.
#' @param SubnetIds List. The IDs of one or more subnets.
#' @param Options Object. The VPC attachment options.\[optional\]
#' @param TagSpecifications List. The tags to apply to the VPC attachment.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayId:
#' The ID of the transit
#' gateway.
#' @section VpcId:
#' The ID of the VPC.
#' @section SubnetIds:
#' The
#' IDs of one or more subnets. You can specify only one subnet per
#' Availability Zone. You must specify at least one subnet, but we
#' recommend that you specify two subnets for better availability. The
#' transit gateway uses one IP address from each specified
#' subnet.
#' @section Options:
#' The VPC attachment
#' options.
#' @section TagSpecifications:
#' The tags to apply to the VPC
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_transit_gateway_vpc_attachment <- function(TransitGatewayId, VpcId, SubnetIds, 
    Options = NULL, TagSpecifications = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    SubnetIds <- list_to_array("SubnetIds", SubnetIds)
    TagSpecifications <- list_to_array("TagSpecifications", TagSpecifications)
    parameters <- c(others, list(TransitGatewayId = TransitGatewayId, VpcId = VpcId, Options = Options, 
        DryRun = DryRun), SubnetIds, TagSpecifications)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateTransitGatewayVpcAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Volume
#' 
#' @param AvailabilityZone String. The Availability Zone in which to create the volume.
#' @param Encrypted Logical.   Indicates whether the volume should be encrypted.\[optional\]
#' @param Iops Integer.   The number of I/O operations per second (IOPS).\[optional\]
#' @param KmsKeyId String.   The identifier of the AWS Key Management Service (AWS KMS) customer master key (CMK) to use for...\[optional\]
#' @param OutpostArn String. The Amazon Resource Name (ARN) of the Outpost.\[optional\]
#' @param Size Integer.   The size of the volume, in GiBs.\[optional\]
#' @param SnapshotId String. The snapshot from which to create the volume.\[optional\]
#' @param VolumeType String.   The volume type.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to apply to the volume during creation.\[optional\]
#' @param MultiAttachEnabled Logical. Indicates whether to enable Amazon EBS Multi-Attach.\[optional\]
#' @param Throughput Integer.   The throughput to provision for a volume, with a maximum of 1,000 MiB/s.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZone:
#' The Availability Zone in
#' which to create the volume.
#' @section Encrypted:
#' 
#' Indicates whether the volume should be encrypted. The effect of setting
#' the encryption state to `true` depends on the volume origin (new or from
#' a snapshot), starting encryption state, ownership, and whether
#' encryption by default is enabled. For more information, see [Encryption
#' by
#' default](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-by-default)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' Encrypted Amazon EBS volumes must be attached to instances that support
#' Amazon EBS encryption. For more information, see [Supported instance
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances).
#' 
#' @section Iops:
#' 
#' The number of I/O operations per second (IOPS). For `gp3`, `io1`, and
#' `io2` volumes, this represents the number of IOPS that are provisioned
#' for the volume. For `gp2` volumes, this represents the baseline
#' performance of the volume and the rate at which the volume accumulates
#' I/O credits for bursting.
#' 
#' The following are the supported values for each volume type:
#' 
#' -   `gp3`: 3,000-16,000 IOPS
#' 
#' -   `io1`: 100-64,000 IOPS
#' 
#' -   `io2`: 100-64,000 IOPS
#' 
#' For `io1` and `io2` volumes, we guarantee 64,000 IOPS only for
#' [Instances built on the Nitro
#' System](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances).
#' Other instance families guarantee performance up to 32,000 IOPS.
#' 
#' This parameter is required for `io1` and `io2` volumes. The default for
#' `gp3` volumes is 3,000 IOPS. This parameter is not supported for `gp2`,
#' `st1`, `sc1`, or `standard` volumes.
#' 
#' @section KmsKeyId:
#' 
#' The identifier of the AWS Key Management Service (AWS KMS) customer
#' master key (CMK) to use for Amazon EBS encryption. If this parameter is
#' not specified, your AWS managed CMK for EBS is used. If `KmsKeyId` is
#' specified, the encrypted state must be `true`.
#' 
#' You can specify the CMK using any of the following:
#' 
#' -   Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Key alias. For example, alias/ExampleAlias.
#' 
#' -   Key ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Alias ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
#' 
#' AWS authenticates the CMK asynchronously. Therefore, if you specify an
#' ID, alias, or ARN that is not valid, the action can appear to complete,
#' but eventually fails.
#' 
#' @section OutpostArn:
#' The Amazon Resource Name (ARN) of the
#' Outpost.
#' @section Size:
#' 
#' The size of the volume, in GiBs. You must specify either a snapshot ID
#' or a volume size. If you specify a snapshot, the default is the snapshot
#' size. You can specify a volume size that is equal to or larger than the
#' snapshot size.
#' 
#' The following are the supported volumes sizes for each volume type:
#' 
#' -   `gp2` and `gp3`: 1-16,384
#' 
#' -   `io1` and `io2`: 4-16,384
#' 
#' -   `st1` and `sc1`: 125-16,384
#' 
#' -   `standard`: 1-1,024
#' 
#' @section SnapshotId:
#' The snapshot from which to create the volume. You
#' must specify either a snapshot ID or a volume size.
#' @section VolumeType:
#' 
#' The volume type. This parameter can be one of the following values:
#' 
#' -   General Purpose SSD: `gp2` \| `gp3`
#' 
#' -   Provisioned IOPS SSD: `io1` \| `io2`
#' 
#' -   Throughput Optimized HDD: `st1`
#' 
#' -   Cold HDD: `sc1`
#' 
#' -   Magnetic: `standard`
#' 
#' For more information, see [Amazon EBS volume
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' Default: `gp2`
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to apply to the
#' volume during creation.
#' @section MultiAttachEnabled:
#' Indicates whether to enable
#' Amazon EBS Multi-Attach. If you enable Multi-Attach, you can attach the
#' volume to up to 16 [Instances built on the Nitro
#' System](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances)
#' in the same Availability Zone. This parameter is supported with `io1`
#' and `io2` volumes only. For more information, see [Amazon EBS
#' Multi-Attach](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes-multi.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' @section Throughput:
#' 
#' The throughput to provision for a volume, with a maximum of 1,000 MiB/s.
#' 
#' This parameter is valid only for `gp3` volumes.
#' 
#' Valid Range: Minimum value of 125. Maximum value of 1000.
#' 
#' @return A list object or a character vector
#' @export
ec2_create_volume <- function(AvailabilityZone, Encrypted = NULL, Iops = NULL, KmsKeyId = NULL, 
    OutpostArn = NULL, Size = NULL, SnapshotId = NULL, VolumeType = NULL, DryRun = NULL, 
    TagSpecification = NULL, MultiAttachEnabled = NULL, Throughput = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(AvailabilityZone = AvailabilityZone, Encrypted = Encrypted, 
        Iops = Iops, KmsKeyId = KmsKeyId, OutpostArn = OutpostArn, Size = Size, SnapshotId = SnapshotId, 
        VolumeType = VolumeType, DryRun = DryRun, MultiAttachEnabled = MultiAttachEnabled, 
        Throughput = Throughput), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVolume", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Vpc
#' 
#' @param CidrBlock String. The IPv4 network range for the VPC, in CIDR notation.
#' @param AmazonProvidedIpv6CidrBlock Logical. Requests an Amazon-provided IPv6 CIDR block with a /56 prefix length for the VPC.\[optional\]
#' @param Ipv6Pool String. The ID of an IPv6 address pool from which to allocate the IPv6 CIDR block.\[optional\]
#' @param Ipv6CidrBlock String.   The IPv6 CIDR block from the IPv6 address pool.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceTenancy String.   The tenancy options for instances launched into the VPC.\[optional\]
#' @param Ipv6CidrBlockNetworkBorderGroup String.   The name of the location from which we advertise the IPV6 CIDR block.\[optional\]
#' @param TagSpecification List. The tags to assign to the VPC.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CidrBlock:
#' The IPv4 network range for the VPC, in CIDR
#' notation. For example, `10.0.0.0/16`. We modify the specified CIDR block
#' to its canonical form; for example, if you specify `100.68.0.18/18`, we
#' modify it to `100.68.0.0/18`.
#' @section AmazonProvidedIpv6CidrBlock:
#' Requests an
#' Amazon-provided IPv6 CIDR block with a /56 prefix length for the VPC.
#' You cannot specify the range of IP addresses, or the size of the CIDR
#' block.
#' @section Ipv6Pool:
#' The ID of an IPv6 address pool from which to
#' allocate the IPv6 CIDR block.
#' @section Ipv6CidrBlock:
#' 
#' The IPv6 CIDR block from the IPv6 address pool. You must also specify
#' `Ipv6Pool` in the request.
#' 
#' To let Amazon choose the IPv6 CIDR block for you, omit this parameter.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section InstanceTenancy:
#' 
#' The tenancy options for instances launched into the VPC. For `default`,
#' instances are launched with shared tenancy by default. You can launch
#' instances with any tenancy into a shared tenancy VPC. For `dedicated`,
#' instances are launched as dedicated tenancy instances by default. You
#' can only launch instances with a tenancy of `dedicated` or `host` into a
#' dedicated tenancy VPC.
#' 
#' **Important:** The `host` value cannot be used with this parameter. Use
#' the `default` or `dedicated` values only.
#' 
#' Default: `default`
#' 
#' @section Ipv6CidrBlockNetworkBorderGroup:
#' 
#' The name of the location from which we advertise the IPV6 CIDR block.
#' Use this parameter to limit the address to this location.
#' 
#' You must set `AmazonProvidedIpv6CidrBlock` to `true` to use this
#' parameter.
#' 
#' @section TagSpecification:
#' The tags to assign to the
#' VPC.
#' @return A list object or a character vector
#' @export
ec2_create_vpc <- function(CidrBlock, AmazonProvidedIpv6CidrBlock = NULL, Ipv6Pool = NULL, 
    Ipv6CidrBlock = NULL, DryRun = NULL, InstanceTenancy = NULL, Ipv6CidrBlockNetworkBorderGroup = NULL, 
    TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(CidrBlock = CidrBlock, AmazonProvidedIpv6CidrBlock = AmazonProvidedIpv6CidrBlock, 
        Ipv6Pool = Ipv6Pool, Ipv6CidrBlock = Ipv6CidrBlock, DryRun = DryRun, InstanceTenancy = InstanceTenancy, 
        Ipv6CidrBlockNetworkBorderGroup = Ipv6CidrBlockNetworkBorderGroup), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpc", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Vpc Endpoint
#' 
#' @param VpcId String. The ID of the VPC in which the endpoint will be used.
#' @param ServiceName String. The service name.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VpcEndpointType String.   The type of endpoint.  Default: Gateway  \[optional\]
#' @param PolicyDocument String. (Interface and gateway endpoints) A policy to attach to the endpoint that controls access to the...\[optional\]
#' @param RouteTableId List. (Gateway endpoint) One or more route table IDs.\[optional\]
#' @param SubnetId List. (Interface and Gateway Load Balancer endpoints) The ID of one or more subnets in which to create...\[optional\]
#' @param SecurityGroupId List. (Interface endpoint) The ID of one or more security groups to associate with the endpoint network...\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param PrivateDnsEnabled Logical.   (Interface endpoint) Indicates whether to associate a private hosted zone with the specified VPC.\[optional\]
#' @param TagSpecification List. The tags to associate with the endpoint.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the VPC in which the endpoint will be
#' used.
#' @section ServiceName:
#' The service name. To get a list of available
#' services, use the DescribeVpcEndpointServices request, or get the name
#' from the service provider.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section VpcEndpointType:
#' 
#' The type of endpoint.
#' 
#' Default: Gateway
#' 
#' @section PolicyDocument:
#' (Interface and gateway endpoints) A policy to
#' attach to the endpoint that controls access to the service. The policy
#' must be in valid JSON format. If this parameter is not specified, we
#' attach a default policy that allows full access to the
#' service.
#' @section RouteTableId:
#' (Gateway endpoint) One or more route table
#' IDs.
#' @section SubnetId:
#' (Interface and Gateway Load Balancer endpoints)
#' The ID of one or more subnets in which to create an endpoint network
#' interface. For a Gateway Load Balancer endpoint, you can specify one
#' subnet only.
#' @section SecurityGroupId:
#' (Interface endpoint) The ID of one or
#' more security groups to associate with the endpoint network
#' interface.
#' @section ClientToken:
#' Unique, case-sensitive identifier that
#' you provide to ensure the idempotency of the request. For more
#' information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section PrivateDnsEnabled:
#' 
#' (Interface endpoint) Indicates whether to associate a private hosted
#' zone with the specified VPC. The private hosted zone contains a record
#' set for the default public DNS name for the service for the Region (for
#' example, `kinesis.us-east-1.amazonaws.com`), which resolves to the
#' private IP addresses of the endpoint network interfaces in the VPC. This
#' enables you to make requests to the default public DNS name for the
#' service instead of the public DNS names that are automatically generated
#' by the VPC endpoint service.
#' 
#' To use a private hosted zone, you must set the following VPC attributes
#' to `true`: `enableDnsHostnames` and `enableDnsSupport`. Use
#' ModifyVpcAttribute to set the VPC attributes.
#' 
#' Default: `true`
#' 
#' @section TagSpecification:
#' The tags to associate with the
#' endpoint.
#' @return A list object or a character vector
#' @export
ec2_create_vpc_endpoint <- function(VpcId, ServiceName, DryRun = NULL, VpcEndpointType = NULL, 
    PolicyDocument = NULL, RouteTableId = NULL, SubnetId = NULL, SecurityGroupId = NULL, 
    ClientToken = NULL, PrivateDnsEnabled = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    RouteTableId <- list_to_array("RouteTableId", RouteTableId)
    SubnetId <- list_to_array("SubnetId", SubnetId)
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(VpcId = VpcId, ServiceName = ServiceName, DryRun = DryRun, 
        VpcEndpointType = VpcEndpointType, PolicyDocument = PolicyDocument, ClientToken = ClientToken, 
        PrivateDnsEnabled = PrivateDnsEnabled), RouteTableId, SubnetId, SecurityGroupId, 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpcEndpoint", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Vpc Endpoint Connection Notification
#' 
#' @param ConnectionNotificationArn String. The ARN of the SNS topic for the notifications.
#' @param ConnectionEvents List. One or more endpoint events for which to receive notifications.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ServiceId String. The ID of the endpoint service.\[optional\]
#' @param VpcEndpointId String. The ID of the endpoint.\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ConnectionNotificationArn:
#' The ARN of the SNS topic for the
#' notifications.
#' @section ConnectionEvents:
#' One or more endpoint events for which
#' to receive notifications. Valid values are `Accept`, `Connect`,
#' `Delete`, and `Reject`.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ServiceId:
#' The ID of the endpoint
#' service.
#' @section VpcEndpointId:
#' The ID of the
#' endpoint.
#' @section ClientToken:
#' Unique, case-sensitive identifier that you
#' provide to ensure the idempotency of the request. For more information,
#' see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_create_vpc_endpoint_connection_notification <- function(ConnectionNotificationArn, ConnectionEvents, 
    DryRun = NULL, ServiceId = NULL, VpcEndpointId = NULL, ClientToken = NULL, simplify = TRUE, 
    others = list()) {
    ConnectionEvents <- list_to_array("ConnectionEvents", ConnectionEvents)
    parameters <- c(others, list(ConnectionNotificationArn = ConnectionNotificationArn, DryRun = DryRun, 
        ServiceId = ServiceId, VpcEndpointId = VpcEndpointId, ClientToken = ClientToken), 
        ConnectionEvents)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpcEndpointConnectionNotification", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Vpc Endpoint Service Configuration
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param AcceptanceRequired Logical. Indicates whether requests from service consumers to create an endpoint to your service must be accepted....\[optional\]
#' @param PrivateDnsName String. (Interface endpoint configuration) The private DNS name to assign to the VPC endpoint service.\[optional\]
#' @param NetworkLoadBalancerArn List. The Amazon Resource Names (ARNs) of one or more Network Load Balancers for your service.\[optional\]
#' @param GatewayLoadBalancerArn List. The Amazon Resource Names (ARNs) of one or more Gateway Load Balancers.\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param TagSpecification List. The tags to associate with the service.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section AcceptanceRequired:
#' Indicates
#' whether requests from service consumers to create an endpoint to your
#' service must be accepted. To accept a request, use
#' AcceptVpcEndpointConnections.
#' @section PrivateDnsName:
#' (Interface endpoint
#' configuration) The private DNS name to assign to the VPC endpoint
#' service.
#' @section NetworkLoadBalancerArn:
#' The Amazon Resource Names (ARNs) of one or
#' more Network Load Balancers for your service.
#' @section GatewayLoadBalancerArn:
#' The
#' Amazon Resource Names (ARNs) of one or more Gateway Load
#' Balancers.
#' @section ClientToken:
#' Unique, case-sensitive identifier that
#' you provide to ensure the idempotency of the request. For more
#' information, see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html).
#' @section TagSpecification:
#' The
#' tags to associate with the service.
#' @return A list object or a character vector
#' @export
ec2_create_vpc_endpoint_service_configuration <- function(DryRun = NULL, AcceptanceRequired = NULL, 
    PrivateDnsName = NULL, NetworkLoadBalancerArn = NULL, GatewayLoadBalancerArn = NULL, 
    ClientToken = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    NetworkLoadBalancerArn <- list_to_array("NetworkLoadBalancerArn", NetworkLoadBalancerArn)
    GatewayLoadBalancerArn <- list_to_array("GatewayLoadBalancerArn", GatewayLoadBalancerArn)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(DryRun = DryRun, AcceptanceRequired = AcceptanceRequired, 
        PrivateDnsName = PrivateDnsName, ClientToken = ClientToken), NetworkLoadBalancerArn, 
        GatewayLoadBalancerArn, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpcEndpointServiceConfiguration", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Create Vpc Peering Connection
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param PeerOwnerId String.   The AWS account ID of the owner of the accepter VPC.  Default: Your AWS account ID  \[optional\]
#' @param PeerVpcId String. The ID of the VPC with which you are creating the VPC peering connection.\[optional\]
#' @param VpcId String. The ID of the requester VPC. You must specify this parameter in the request.\[optional\]
#' @param PeerRegion String.   The Region code for the accepter VPC, if the accepter VPC is located in a Region other than the...\[optional\]
#' @param TagSpecification List. The tags to assign to the peering connection.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section PeerOwnerId:
#' 
#' The AWS account ID of the owner of the accepter VPC.
#' 
#' Default: Your AWS account ID
#' 
#' @section PeerVpcId:
#' The ID of the VPC with which you are creating the
#' VPC peering connection. You must specify this parameter in the
#' request.
#' @section VpcId:
#' The ID of the requester VPC. You must
#' specify this parameter in the request.
#' @section PeerRegion:
#' 
#' The Region code for the accepter VPC, if the accepter VPC is located in
#' a Region other than the Region in which you make the request.
#' 
#' Default: The Region in which you make the request.
#' 
#' @section TagSpecification:
#' The tags to assign to the peering
#' connection.
#' @return A list object or a character vector
#' @export
ec2_create_vpc_peering_connection <- function(DryRun = NULL, PeerOwnerId = NULL, PeerVpcId = NULL, 
    VpcId = NULL, PeerRegion = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(DryRun = DryRun, PeerOwnerId = PeerOwnerId, PeerVpcId = PeerVpcId, 
        VpcId = VpcId, PeerRegion = PeerRegion), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpcPeeringConnection", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Vpn Connection
#' 
#' @param CustomerGatewayId String. The ID of the customer gateway.
#' @param Type String. The type of VPN connection (`ipsec.1`).
#' @param VpnGatewayId String. The ID of the virtual private gateway.\[optional\]
#' @param TransitGatewayId String. The ID of the transit gateway.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Options Object. The options for the VPN connection.\[optional\]
#' @param TagSpecification List. The tags to apply to the VPN connection.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CustomerGatewayId:
#' The ID of the customer
#' gateway.
#' @section Type:
#' The type of VPN connection
#' (`ipsec.1`).
#' @section VpnGatewayId:
#' The ID of the virtual private gateway.
#' If you specify a virtual private gateway, you cannot specify a transit
#' gateway.
#' @section TransitGatewayId:
#' The ID of the transit gateway. If you
#' specify a transit gateway, you cannot specify a virtual private
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Options:
#' The options for the VPN
#' connection.
#' @section TagSpecification:
#' The tags to apply to the VPN
#' connection.
#' @return A list object or a character vector
#' @export
ec2_create_vpn_connection <- function(CustomerGatewayId, Type, VpnGatewayId = NULL, TransitGatewayId = NULL, 
    DryRun = NULL, Options = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(CustomerGatewayId = CustomerGatewayId, Type = Type, VpnGatewayId = VpnGatewayId, 
        TransitGatewayId = TransitGatewayId, DryRun = DryRun, Options = Options), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpnConnection", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Vpn Connection Route
#' 
#' @param DestinationCidrBlock String. The CIDR block associated with the local subnet of the customer network.
#' @param VpnConnectionId String. The ID of the VPN connection.
#' @inheritParams CommonDoc
#' 
#' @section DestinationCidrBlock:
#' The CIDR block associated with the local
#' subnet of the customer network.
#' @section VpnConnectionId:
#' The ID of the VPN
#' connection.
#' @return A list object or a character vector
#' @export
ec2_create_vpn_connection_route <- function(DestinationCidrBlock, VpnConnectionId, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(DestinationCidrBlock = DestinationCidrBlock, VpnConnectionId = VpnConnectionId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpnConnectionRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Create Vpn Gateway
#' 
#' @param Type String. The type of VPN connection this virtual private gateway supports.
#' @param AvailabilityZone String. The Availability Zone for the virtual private gateway.\[optional\]
#' @param TagSpecification List. The tags to apply to the virtual private gateway.\[optional\]
#' @param AmazonSideAsn Integer.   A private Autonomous System Number (ASN) for the Amazon side of a BGP session.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Type:
#' The type of VPN connection this virtual
#' private gateway supports.
#' @section AvailabilityZone:
#' The Availability Zone for
#' the virtual private gateway.
#' @section TagSpecification:
#' The tags to apply to
#' the virtual private gateway.
#' @section AmazonSideAsn:
#' 
#' A private Autonomous System Number (ASN) for the Amazon side of a BGP
#' session. If you\'re using a 16-bit ASN, it must be in the 64512 to 65534
#' range. If you\'re using a 32-bit ASN, it must be in the 4200000000 to
#' 4294967294 range.
#' 
#' Default: 64512
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_create_vpn_gateway <- function(Type, AvailabilityZone = NULL, TagSpecification = NULL, 
    AmazonSideAsn = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(Type = Type, AvailabilityZone = AvailabilityZone, AmazonSideAsn = AmazonSideAsn, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "CreateVpnGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Carrier Gateway
#' 
#' @param CarrierGatewayId String. The ID of the carrier gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CarrierGatewayId:
#' The ID of the carrier
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_carrier_gateway <- function(CarrierGatewayId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(CarrierGatewayId = CarrierGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteCarrierGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Client Vpn Endpoint
#' 
#' Deletes the specified Client VPN endpoint. You must
#' disassociate all target networks before you can delete a Client VPN
#' endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN to be deleted.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN to
#' be deleted.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_client_vpn_endpoint <- function(ClientVpnEndpointId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteClientVpnEndpoint", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Client Vpn Route
#' 
#' Deletes a route from a Client VPN
#' endpoint. You can only delete routes that you manually added using the
#' **CreateClientVpnRoute** action. You cannot delete routes that were
#' automatically added when associating a subnet. To remove routes that
#' have been automatically added, disassociate the target subnet from the
#' Client VPN endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint from which the route is to be deleted.
#' @param DestinationCidrBlock String. The IPv4 address range, in CIDR notation, of the route to be deleted.
#' @param TargetVpcSubnetId String. The ID of the target subnet used by the route.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint from which the route is to be deleted.
#' @section DestinationCidrBlock:
#' The
#' IPv4 address range, in CIDR notation, of the route to be
#' deleted.
#' @section TargetVpcSubnetId:
#' The ID of the target subnet used by the
#' route.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_client_vpn_route <- function(ClientVpnEndpointId, DestinationCidrBlock, TargetVpcSubnetId = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, DestinationCidrBlock = DestinationCidrBlock, 
        TargetVpcSubnetId = TargetVpcSubnetId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteClientVpnRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Customer Gateway
#' 
#' Deletes the specified customer
#' gateway. You must delete the VPN connection before you can delete the
#' customer gateway.
#' 
#' @param CustomerGatewayId String. The ID of the customer gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CustomerGatewayId:
#' The ID of the customer
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_customer_gateway <- function(CustomerGatewayId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(CustomerGatewayId = CustomerGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteCustomerGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Dhcp Options
#' 
#' Deletes the specified set of DHCP
#' options. You must disassociate the set of DHCP options before you can
#' delete it. You can disassociate the set of DHCP options by associating
#' either a new set of options or the default set of options with the
#' VPC.
#' 
#' @param DhcpOptionsId String. The ID of the DHCP options set.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DhcpOptionsId:
#' The ID of the DHCP options
#' set.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_dhcp_options <- function(DhcpOptionsId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DhcpOptionsId = DhcpOptionsId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteDhcpOptions", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Egress Only Internet Gateway
#' 
#' Deletes an egress-only internet
#' gateway.
#' 
#' @param EgressOnlyInternetGatewayId String. The ID of the egress-only internet gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section EgressOnlyInternetGatewayId:
#' The ID of the egress-only
#' internet gateway.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_egress_only_internet_gateway <- function(EgressOnlyInternetGatewayId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(EgressOnlyInternetGatewayId = EgressOnlyInternetGatewayId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteEgressOnlyInternetGateway", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Fleets
#' 
#' @param FleetId List. The IDs of the EC2 Fleets.
#' @param TerminateInstances Logical.   Indicates whether to terminate the instances when the EC2 Fleet is deleted.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FleetId:
#' The IDs of the EC2
#' Fleets.
#' @section TerminateInstances:
#' 
#' Indicates whether to terminate the instances when the EC2 Fleet is
#' deleted. The default is to terminate the instances.
#' 
#' To let the instances continue to run after the EC2 Fleet is deleted,
#' specify `NoTerminateInstances`. Supported only for fleets of type
#' `maintain` and `request`.
#' 
#' For `instant` fleets, you cannot specify `NoTerminateInstances`. A
#' deleted `instant` fleet with running instances is not supported.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_fleets <- function(FleetId, TerminateInstances, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    FleetId <- list_to_array("FleetId", FleetId)
    parameters <- c(others, list(TerminateInstances = TerminateInstances, DryRun = DryRun), 
        FleetId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteFleets", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Flow Logs
#' 
#' Deletes one or more flow
#' logs.
#' 
#' @param FlowLogId List.   One or more flow log IDs.  Constraint: Maximum of 1000 flow log IDs.  
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FlowLogId:
#' 
#' One or more flow log IDs.
#' 
#' Constraint: Maximum of 1000 flow log IDs.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_flow_logs <- function(FlowLogId, DryRun = NULL, simplify = TRUE, others = list()) {
    FlowLogId <- list_to_array("FlowLogId", FlowLogId)
    parameters <- c(others, list(DryRun = DryRun), FlowLogId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteFlowLogs", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Fpga Image
#' 
#' Deletes the specified Amazon FPGA Image
#' (AFI).
#' 
#' @param FpgaImageId String. The ID of the AFI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FpgaImageId:
#' The ID of the
#' AFI.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_fpga_image <- function(FpgaImageId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(FpgaImageId = FpgaImageId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteFpgaImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Internet Gateway
#' 
#' Deletes the specified internet gateway. You
#' must detach the internet gateway from the VPC before you can delete
#' it.
#' 
#' @param InternetGatewayId String. The ID of the internet gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InternetGatewayId:
#' The ID of the internet
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_internet_gateway <- function(InternetGatewayId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InternetGatewayId = InternetGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteInternetGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Key Pair
#' 
#' Deletes the specified key pair, by removing the
#' public key from Amazon EC2.
#' 
#' @param KeyName String. The name of the key pair.\[optional\]
#' @param KeyPairId String. The ID of the key pair.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section KeyName:
#' The name of the key
#' pair.
#' @section KeyPairId:
#' The ID of the key
#' pair.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_key_pair <- function(KeyName = NULL, KeyPairId = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(KeyName = KeyName, KeyPairId = KeyPairId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteKeyPair", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Launch Template
#' 
#' Deletes a launch
#' template. Deleting a launch template deletes all of its
#' versions.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LaunchTemplateId String. The ID of the launch template.\[optional\]
#' @param LaunchTemplateName String. The name of the launch template.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section LaunchTemplateId:
#' The ID of the launch
#' template. You must specify either the launch template ID or launch
#' template name in the request.
#' @section LaunchTemplateName:
#' The name of the launch
#' template. You must specify either the launch template ID or launch
#' template name in the request.
#' @return A list object or a character vector
#' @export
ec2_delete_launch_template <- function(DryRun = NULL, LaunchTemplateId = NULL, LaunchTemplateName = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun, LaunchTemplateId = LaunchTemplateId, LaunchTemplateName = LaunchTemplateName))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteLaunchTemplate", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Launch Template Versions
#' 
#' Deletes one or more versions of a launch
#' template. You cannot delete the default version of a launch template;
#' you must first assign a different version as the default. If the default
#' version is the only version for the launch template, you must delete the
#' entire launch template using
#' DeleteLaunchTemplate.
#' 
#' @param LaunchTemplateVersion List. The version numbers of one or more launch template versions to delete.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LaunchTemplateId String. The ID of the launch template.\[optional\]
#' @param LaunchTemplateName String. The name of the launch template.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LaunchTemplateVersion:
#' The version numbers of
#' one or more launch template versions to
#' delete.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section LaunchTemplateId:
#' The ID of the launch
#' template. You must specify either the launch template ID or launch
#' template name in the request.
#' @section LaunchTemplateName:
#' The name of the launch
#' template. You must specify either the launch template ID or launch
#' template name in the request.
#' @return A list object or a character vector
#' @export
ec2_delete_launch_template_versions <- function(LaunchTemplateVersion, DryRun = NULL, LaunchTemplateId = NULL, 
    LaunchTemplateName = NULL, simplify = TRUE, others = list()) {
    LaunchTemplateVersion <- list_to_array("LaunchTemplateVersion", LaunchTemplateVersion)
    parameters <- c(others, list(DryRun = DryRun, LaunchTemplateId = LaunchTemplateId, LaunchTemplateName = LaunchTemplateName), 
        LaunchTemplateVersion)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteLaunchTemplateVersions", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Local Gateway Route
#' 
#' Deletes the specified route
#' from the specified local gateway route
#' table.
#' 
#' @param DestinationCidrBlock String. The CIDR range for the route. This must match the CIDR for the route exactly.
#' @param LocalGatewayRouteTableId String. The ID of the local gateway route table.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DestinationCidrBlock:
#' The CIDR range for the
#' route. This must match the CIDR for the route
#' exactly.
#' @section LocalGatewayRouteTableId:
#' The ID of the local gateway route
#' table.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_local_gateway_route <- function(DestinationCidrBlock, LocalGatewayRouteTableId, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DestinationCidrBlock = DestinationCidrBlock, LocalGatewayRouteTableId = LocalGatewayRouteTableId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteLocalGatewayRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Local Gateway Route Table Vpc Association
#' 
#' Deletes the specified association between a
#' VPC and local gateway route table.
#' 
#' @param LocalGatewayRouteTableVpcAssociationId String. The ID of the association.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayRouteTableVpcAssociationId:
#' The ID of the
#' association.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_local_gateway_route_table_vpc_association <- function(LocalGatewayRouteTableVpcAssociationId, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(LocalGatewayRouteTableVpcAssociationId = LocalGatewayRouteTableVpcAssociationId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteLocalGatewayRouteTableVpcAssociation", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Managed Prefix List
#' 
#' Deletes the
#' specified managed prefix list. You must first remove all references to
#' the prefix list in your resources.
#' 
#' @param PrefixListId String. The ID of the prefix list.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PrefixListId:
#' The ID of the prefix
#' list.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_managed_prefix_list <- function(PrefixListId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(PrefixListId = PrefixListId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteManagedPrefixList", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Nat Gateway
#' 
#' Deletes the
#' specified NAT gateway. Deleting a NAT gateway disassociates its Elastic
#' IP address, but does not release the address from your account. Deleting
#' a NAT gateway does not delete any NAT gateway routes in your route
#' tables.
#' 
#' @param NatGatewayId String. The ID of the NAT gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NatGatewayId:
#' The ID of the NAT
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_nat_gateway <- function(NatGatewayId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(NatGatewayId = NatGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteNatGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Network Acl
#' 
#' Deletes the specified network ACL. You
#' can\'t delete the ACL if it\'s associated with any subnets. You can\'t
#' delete the default network ACL.
#' 
#' @param NetworkAclId String. The ID of the network ACL.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkAclId:
#' The ID of the network
#' ACL.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_network_acl <- function(NetworkAclId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(NetworkAclId = NetworkAclId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteNetworkAcl", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Network Acl Entry
#' 
#' Deletes the
#' specified ingress or egress entry (rule) from the specified network
#' ACL.
#' 
#' @param Egress Logical. Indicates whether the rule is an egress rule.
#' @param NetworkAclId String. The ID of the network ACL.
#' @param RuleNumber Integer. The rule number of the entry to delete.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Egress:
#' Indicates whether the rule
#' is an egress rule.
#' @section NetworkAclId:
#' The ID of the network
#' ACL.
#' @section RuleNumber:
#' The rule number of the entry to
#' delete.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_network_acl_entry <- function(Egress, NetworkAclId, RuleNumber, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(Egress = Egress, NetworkAclId = NetworkAclId, RuleNumber = RuleNumber, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteNetworkAclEntry", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Network Insights Analysis
#' 
#' Deletes the specified network insights
#' analysis.
#' 
#' @param NetworkInsightsAnalysisId String. The ID of the network insights analysis.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInsightsAnalysisId:
#' The ID of the network
#' insights analysis.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_network_insights_analysis <- function(NetworkInsightsAnalysisId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(NetworkInsightsAnalysisId = NetworkInsightsAnalysisId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteNetworkInsightsAnalysis", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Network Insights Path
#' 
#' Deletes the specified
#' path.
#' 
#' @param NetworkInsightsPathId String. The ID of the path.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInsightsPathId:
#' The ID of the
#' path.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_network_insights_path <- function(NetworkInsightsPathId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(NetworkInsightsPathId = NetworkInsightsPathId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteNetworkInsightsPath", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Network Interface
#' 
#' Deletes the specified network interface. You
#' must detach the network interface before you can delete
#' it.
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_network_interface <- function(NetworkInterfaceId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteNetworkInterface", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Network Interface Permission
#' 
#' Deletes a permission for a network interface. By
#' default, you cannot delete the permission if the account for which
#' you\'re removing the permission has attached the network interface to an
#' instance. However, you can force delete the permission, regardless of
#' any attachment.
#' 
#' @param NetworkInterfacePermissionId String. The ID of the network interface permission.
#' @param Force Logical. Specify `true` to remove the permission even if the network interface is attached to an instance.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfacePermissionId:
#' The ID of the network
#' interface permission.
#' @section Force:
#' Specify `true` to remove the
#' permission even if the network interface is attached to an
#' instance.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_network_interface_permission <- function(NetworkInterfacePermissionId, Force = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(NetworkInterfacePermissionId = NetworkInterfacePermissionId, 
        Force = Force, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteNetworkInterfacePermission", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Placement Group
#' 
#' Deletes the specified placement
#' group. You must terminate all instances in the placement group before
#' you can delete the placement group. For more information, see [Placement
#' groups](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' @param GroupName String. The name of the placement group.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GroupName:
#' The name of the placement
#' group.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_placement_group <- function(GroupName, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(GroupName = GroupName, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeletePlacementGroup", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Queued Reserved Instances
#' 
#' Deletes the queued
#' purchases for the specified Reserved
#' Instances.
#' 
#' @param ReservedInstancesId List. The IDs of the Reserved Instances.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ReservedInstancesId:
#' The IDs of the Reserved
#' Instances.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_queued_reserved_instances <- function(ReservedInstancesId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    ReservedInstancesId <- list_to_array("ReservedInstancesId", ReservedInstancesId)
    parameters <- c(others, list(DryRun = DryRun), ReservedInstancesId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteQueuedReservedInstances", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Route
#' 
#' Deletes the specified route from the
#' specified route table.
#' 
#' @param RouteTableId String. The ID of the route table.
#' @param DestinationCidrBlock String. The IPv4 CIDR range for the route. The value you specify must match the CIDR for the route exactly.\[optional\]
#' @param DestinationIpv6CidrBlock String. The IPv6 CIDR range for the route. The value you specify must match the CIDR for the route exactly.\[optional\]
#' @param DestinationPrefixListId String. The ID of the prefix list for the route.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section RouteTableId:
#' The ID of the route
#' table.
#' @section DestinationCidrBlock:
#' The IPv4 CIDR range for the route. The value
#' you specify must match the CIDR for the route
#' exactly.
#' @section DestinationIpv6CidrBlock:
#' The IPv6 CIDR range for the route. The
#' value you specify must match the CIDR for the route
#' exactly.
#' @section DestinationPrefixListId:
#' The ID of the prefix list for the
#' route.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_route <- function(RouteTableId, DestinationCidrBlock = NULL, DestinationIpv6CidrBlock = NULL, 
    DestinationPrefixListId = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(RouteTableId = RouteTableId, DestinationCidrBlock = DestinationCidrBlock, 
        DestinationIpv6CidrBlock = DestinationIpv6CidrBlock, DestinationPrefixListId = DestinationPrefixListId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Route Table
#' 
#' Deletes the specified route
#' table. You must disassociate the route table from any subnets before you
#' can delete it. You can\'t delete the main route
#' table.
#' 
#' @param RouteTableId String. The ID of the route table.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section RouteTableId:
#' The ID of the route
#' table.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_route_table <- function(RouteTableId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(RouteTableId = RouteTableId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteRouteTable", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Security Group
#' 
#' @param GroupId String. The ID of the security group. Required for a nondefault VPC.\[optional\]
#' @param GroupName String. \[EC2-Classic, default VPC\] The name of the security group.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GroupId:
#' The ID of the security
#' group. Required for a nondefault VPC.
#' @section GroupName:
#' \[EC2-Classic,
#' default VPC\] The name of the security group. You can specify either the
#' security group name or the security group ID.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_security_group <- function(GroupId = NULL, GroupName = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(GroupId = GroupId, GroupName = GroupName, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteSecurityGroup", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Snapshot
#' 
#' @param SnapshotId String. The ID of the EBS snapshot.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SnapshotId:
#' The ID of
#' the EBS snapshot.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_snapshot <- function(SnapshotId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(SnapshotId = SnapshotId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteSnapshot", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Spot Datafeed Subscription
#' 
#' Deletes the data feed for Spot
#' Instances.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_spot_datafeed_subscription <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteSpotDatafeedSubscription", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Subnet
#' 
#' Deletes the specified subnet. You must
#' terminate all running instances in the subnet before you can delete the
#' subnet.
#' 
#' @param SubnetId String. The ID of the subnet.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SubnetId:
#' The ID of the
#' subnet.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_subnet <- function(SubnetId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(SubnetId = SubnetId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteSubnet", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Tags
#' 
#' @param ResourceId List.   The IDs of the resources, separated by spaces.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Tag List.   The tags to delete.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ResourceId:
#' 
#' The IDs of the resources, separated by spaces.
#' 
#' Constraints: Up to 1000 resource IDs. We recommend breaking up this
#' request into smaller batches.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Tag:
#' 
#' The tags to delete. Specify a tag key and an optional tag value to
#' delete specific tags. If you specify a tag key without a tag value, we
#' delete any tag with this key regardless of its value. If you specify a
#' tag key with an empty string as the tag value, we delete the tag only if
#' its value is an empty string.
#' 
#' If you omit this parameter, we delete all user-defined tags for the
#' specified resources. We do not delete AWS-generated tags (tags that have
#' the `aws:` prefix).
#' 
#' @return A list object or a character vector
#' @export
ec2_delete_tags <- function(ResourceId, DryRun = NULL, Tag = NULL, simplify = TRUE, others = list()) {
    ResourceId <- list_to_array("ResourceId", ResourceId)
    Tag <- list_to_array("Tag", Tag)
    parameters <- c(others, list(DryRun = DryRun), ResourceId, Tag)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTags", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Traffic Mirror Filter
#' 
#' @param TrafficMirrorFilterId String. The ID of the Traffic Mirror filter.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorFilterId:
#' The ID of the Traffic Mirror
#' filter.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_traffic_mirror_filter <- function(TrafficMirrorFilterId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(TrafficMirrorFilterId = TrafficMirrorFilterId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTrafficMirrorFilter", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Traffic Mirror Filter Rule
#' 
#' Deletes the specified Traffic Mirror
#' rule.
#' 
#' @param TrafficMirrorFilterRuleId String. The ID of the Traffic Mirror rule.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorFilterRuleId:
#' The ID of the Traffic
#' Mirror rule.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_traffic_mirror_filter_rule <- function(TrafficMirrorFilterRuleId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TrafficMirrorFilterRuleId = TrafficMirrorFilterRuleId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTrafficMirrorFilterRule", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Traffic Mirror Session
#' 
#' Deletes the specified Traffic Mirror
#' session.
#' 
#' @param TrafficMirrorSessionId String. The ID of the Traffic Mirror session.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorSessionId:
#' The ID of the Traffic
#' Mirror session.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_traffic_mirror_session <- function(TrafficMirrorSessionId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(TrafficMirrorSessionId = TrafficMirrorSessionId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTrafficMirrorSession", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Traffic Mirror Target
#' 
#' @param TrafficMirrorTargetId String. The ID of the Traffic Mirror target.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorTargetId:
#' The ID of the Traffic
#' Mirror target.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_traffic_mirror_target <- function(TrafficMirrorTargetId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(TrafficMirrorTargetId = TrafficMirrorTargetId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTrafficMirrorTarget", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway
#' 
#' Deletes the specified transit
#' gateway.
#' 
#' @param TransitGatewayId String. The ID of the transit gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayId:
#' The ID of the transit
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway <- function(TransitGatewayId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(TransitGatewayId = TransitGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Connect
#' 
#' Deletes the specified Connect attachment.
#' You must first delete any Connect peers for the
#' attachment.
#' 
#' @param TransitGatewayAttachmentId String. The ID of the Connect attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the Connect
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_connect <- function(TransitGatewayAttachmentId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayConnect", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Connect Peer
#' 
#' Deletes the specified Connect
#' peer.
#' 
#' @param TransitGatewayConnectPeerId String. The ID of the Connect peer.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayConnectPeerId:
#' The ID of the Connect
#' peer.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_connect_peer <- function(TransitGatewayConnectPeerId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayConnectPeerId = TransitGatewayConnectPeerId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayConnectPeer", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Multicast Domain
#' 
#' Deletes the specified transit gateway
#' multicast domain.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit
#' gateway multicast domain.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_multicast_domain <- function(TransitGatewayMulticastDomainId, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayMulticastDomain", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Peering Attachment
#' 
#' Deletes a transit gateway peering
#' attachment.
#' 
#' @param TransitGatewayAttachmentId String. The ID of the transit gateway peering attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the transit
#' gateway peering attachment.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_peering_attachment <- function(TransitGatewayAttachmentId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayPeeringAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Prefix List Reference
#' 
#' Deletes a reference (route) to a prefix
#' list in a specified transit gateway route
#' table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the route table.
#' @param PrefixListId String. The ID of the prefix list.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the route
#' table.
#' @section PrefixListId:
#' The ID of the prefix
#' list.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_prefix_list_reference <- function(TransitGatewayRouteTableId, 
    PrefixListId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        PrefixListId = PrefixListId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayPrefixListReference", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Route
#' 
#' Deletes the specified route from the
#' specified transit gateway route table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param DestinationCidrBlock String. The CIDR range for the route. This must match the CIDR for the route exactly.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section DestinationCidrBlock:
#' The CIDR range for the route.
#' This must match the CIDR for the route
#' exactly.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_route <- function(TransitGatewayRouteTableId, DestinationCidrBlock, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        DestinationCidrBlock = DestinationCidrBlock, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Route Table
#' 
#' Deletes the
#' specified transit gateway route table. You must disassociate the route
#' table from any transit gateway route tables before you can delete
#' it.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_route_table <- function(TransitGatewayRouteTableId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayRouteTable", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Transit Gateway Vpc Attachment
#' 
#' Deletes the specified VPC
#' attachment.
#' 
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_transit_gateway_vpc_attachment <- function(TransitGatewayAttachmentId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteTransitGatewayVpcAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Volume
#' 
#' @param VolumeId String. The ID of the volume.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VolumeId:
#' The ID of the
#' volume.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_volume <- function(VolumeId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VolumeId = VolumeId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVolume", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Vpc
#' 
#' Deletes the specified VPC. You must detach or
#' delete all gateways and resources that are associated with the VPC
#' before you can delete it. For example, you must terminate all instances
#' running in the VPC, delete all security groups associated with the VPC
#' (except the default one), delete all route tables associated with the
#' VPC (except the default one), and so on.
#' 
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_vpc <- function(VpcId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpc", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Vpc Endpoint Connection Notifications
#' 
#' Deletes one
#' or more VPC endpoint connection
#' notifications.
#' 
#' @param ConnectionNotificationId List. One or more notification IDs.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ConnectionNotificationId:
#' One or more notification
#' IDs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_vpc_endpoint_connection_notifications <- function(ConnectionNotificationId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    ConnectionNotificationId <- list_to_array("ConnectionNotificationId", ConnectionNotificationId)
    parameters <- c(others, list(DryRun = DryRun), ConnectionNotificationId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpcEndpointConnectionNotifications", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Vpc Endpoint Service Configurations
#' 
#' Deletes one or more VPC endpoint
#' service configurations in your account. Before you delete the endpoint
#' service configuration, you must reject any `Available` or
#' `PendingAcceptance` interface endpoint connections that are attached to
#' the service.
#' 
#' @param ServiceId List. The IDs of one or more services.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ServiceId:
#' The IDs of one or more
#' services.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_vpc_endpoint_service_configurations <- function(ServiceId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    ServiceId <- list_to_array("ServiceId", ServiceId)
    parameters <- c(others, list(DryRun = DryRun), ServiceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpcEndpointServiceConfigurations", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Delete Vpc Endpoints
#' 
#' @param VpcEndpointId List. One or more VPC endpoint IDs.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcEndpointId:
#' One or more VPC endpoint
#' IDs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_vpc_endpoints <- function(VpcEndpointId, DryRun = NULL, simplify = TRUE, others = list()) {
    VpcEndpointId <- list_to_array("VpcEndpointId", VpcEndpointId)
    parameters <- c(others, list(DryRun = DryRun), VpcEndpointId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpcEndpoints", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Vpc Peering Connection
#' 
#' Deletes a VPC peering connection. Either the owner
#' of the requester VPC or the owner of the accepter VPC can delete the VPC
#' peering connection if it\'s in the `active` state. The owner of the
#' requester VPC can delete a VPC peering connection in the
#' `pending-acceptance` state. You cannot delete a VPC peering connection
#' that\'s in the `failed` state.
#' 
#' @param VpcPeeringConnectionId String. The ID of the VPC peering connection.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcPeeringConnectionId:
#' The ID of the VPC peering
#' connection.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_vpc_peering_connection <- function(VpcPeeringConnectionId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(VpcPeeringConnectionId = VpcPeeringConnectionId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpcPeeringConnection", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Vpn Connection
#' 
#' @param VpnConnectionId String. The ID of the VPN connection.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpnConnectionId:
#' The ID of the VPN
#' connection.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_vpn_connection <- function(VpnConnectionId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpnConnectionId = VpnConnectionId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpnConnection", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Vpn Connection Route
#' 
#' Deletes the specified static route associated with
#' a VPN connection between an existing virtual private gateway and a VPN
#' customer gateway. The static route allows traffic to be routed from the
#' virtual private gateway to the VPN customer
#' gateway.
#' 
#' @param DestinationCidrBlock String. The CIDR block associated with the local subnet of the customer network.
#' @param VpnConnectionId String. The ID of the VPN connection.
#' @inheritParams CommonDoc
#' 
#' @section DestinationCidrBlock:
#' The CIDR block associated
#' with the local subnet of the customer network.
#' @section VpnConnectionId:
#' The
#' ID of the VPN connection.
#' @return A list object or a character vector
#' @export
ec2_delete_vpn_connection_route <- function(DestinationCidrBlock, VpnConnectionId, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(DestinationCidrBlock = DestinationCidrBlock, VpnConnectionId = VpnConnectionId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpnConnectionRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Delete Vpn Gateway
#' 
#' Deletes the specified virtual private
#' gateway. You must first detach the virtual private gateway from the VPC.
#' Note that you don\'t need to delete the virtual private gateway if you
#' plan to delete and recreate the VPN connection between your VPC and your
#' network.
#' 
#' @param VpnGatewayId String. The ID of the virtual private gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpnGatewayId:
#' The ID of the virtual
#' private gateway.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_delete_vpn_gateway <- function(VpnGatewayId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpnGatewayId = VpnGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeleteVpnGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Deprovision Byoip Cidr
#' 
#' @param Cidr String. The address range, in CIDR notation.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Cidr:
#' The address range, in CIDR
#' notation. The prefix must be the same prefix that you specified when you
#' provisioned the address range.
#' @section DryRun:
#' Checks whether you
#' have the required permissions for the action, without actually making
#' the request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_deprovision_byoip_cidr <- function(Cidr, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(Cidr = Cidr, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeprovisionByoipCidr", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Deregister Image
#' 
#' @param ImageId String. The ID of the AMI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ImageId:
#' The ID of the
#' AMI.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_deregister_image <- function(ImageId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ImageId = ImageId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeregisterImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Deregister Instance Event Notification Attributes
#' 
#' Deregisters tag keys to prevent tags that have the
#' specified tag keys from being included in scheduled event notifications
#' for resources in the Region.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceTagAttribute Object. Information about the tag keys to deregister.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section InstanceTagAttribute:
#' Information about the tag
#' keys to deregister.
#' @return A list object or a character vector
#' @export
ec2_deregister_instance_event_notification_attributes <- function(DryRun = NULL, InstanceTagAttribute = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun, InstanceTagAttribute = InstanceTagAttribute))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeregisterInstanceEventNotificationAttributes", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Deregister Transit Gateway Multicast Group Members
#' 
#' Deregisters the
#' specified members (network interfaces) from the transit gateway
#' multicast group.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param GroupIpAddress String. The IP address assigned to the transit gateway multicast group.\[optional\]
#' @param NetworkInterfaceIds List. The IDs of the group members\' network interfaces.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit gateway
#' multicast domain.
#' @section GroupIpAddress:
#' The IP address assigned to the
#' transit gateway multicast group.
#' @section NetworkInterfaceIds:
#' The IDs of the
#' group members\' network interfaces.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_deregister_transit_gateway_multicast_group_members <- function(TransitGatewayMulticastDomainId = NULL, 
    GroupIpAddress = NULL, NetworkInterfaceIds = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    NetworkInterfaceIds <- list_to_array("NetworkInterfaceIds", NetworkInterfaceIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        GroupIpAddress = GroupIpAddress, DryRun = DryRun), NetworkInterfaceIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeregisterTransitGatewayMulticastGroupMembers", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Deregister Transit Gateway Multicast Group Sources
#' 
#' Deregisters the specified sources
#' (network interfaces) from the transit gateway multicast
#' group.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param GroupIpAddress String. The IP address assigned to the transit gateway multicast group.\[optional\]
#' @param NetworkInterfaceIds List. The IDs of the group sources\' network interfaces.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of
#' the transit gateway multicast domain.
#' @section GroupIpAddress:
#' The IP address
#' assigned to the transit gateway multicast group.
#' @section NetworkInterfaceIds:
#' The
#' IDs of the group sources\' network
#' interfaces.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_deregister_transit_gateway_multicast_group_sources <- function(TransitGatewayMulticastDomainId = NULL, 
    GroupIpAddress = NULL, NetworkInterfaceIds = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    NetworkInterfaceIds <- list_to_array("NetworkInterfaceIds", NetworkInterfaceIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        GroupIpAddress = GroupIpAddress, DryRun = DryRun), NetworkInterfaceIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DeregisterTransitGatewayMulticastGroupSources", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Describe Account Attributes
#' 
#' @param AttributeName List. The account attribute names.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AttributeName:
#' The account attribute
#' names.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_account_attributes <- function(AttributeName = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    AttributeName <- list_to_array("AttributeName", AttributeName)
    parameters <- c(others, list(DryRun = DryRun), AttributeName)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeAccountAttributes", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Addresses
#' 
#' @param PublicIp List.   One or more Elastic IP addresses.  Default: Describes all your Elastic IP addresses.  \[optional\]
#' @param AllocationId List. \[EC2-VPC\] Information about the allocation IDs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `allocation-id` - \[EC2-VPC\] The allocation ID for the address.
#' 
#' -   `association-id` - \[EC2-VPC\] The association ID for the address.
#' 
#' -   `domain` - Indicates whether the address is for use in EC2-Classic
#'     (`standard`) or in a VPC (`vpc`).
#' 
#' -   `instance-id` - The ID of the instance the address is associated
#'     with, if any.
#' 
#' -   `network-border-group` - A unique set of Availability Zones, Local
#'     Zones, or Wavelength Zones from where AWS advertises IP addresses.
#' 
#' -   `network-interface-id` - \[EC2-VPC\] The ID of the network interface
#'     that the address is associated with, if any.
#' 
#' -   `network-interface-owner-id` - The AWS account ID of the owner.
#' 
#' -   `private-ip-address` - \[EC2-VPC\] The private IP address associated
#'     with the Elastic IP address.
#' 
#' -   `public-ip` - The Elastic IP address, or the carrier IP address.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section PublicIp:
#' 
#' One or more Elastic IP addresses.
#' 
#' Default: Describes all your Elastic IP addresses.
#' 
#' @section AllocationId:
#' \[EC2-VPC\] Information about the allocation
#' IDs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_addresses <- function(Filter = NULL, PublicIp = NULL, AllocationId = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    PublicIp <- list_to_array("PublicIp", PublicIp)
    AllocationId <- list_to_array("AllocationId", AllocationId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), Filter, PublicIp, AllocationId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeAddresses", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Addresses Attribute
#' 
#' Describes the attributes of the specified Elastic
#' IP addresses. For requirements, see [Using reverse DNS for email
#' applications](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html#Using_Elastic_Addressing_Reverse_DNS).
#' 
#' @param AllocationId List. \[EC2-VPC\] The allocation IDs.\[optional\]
#' @param Attribute String. The attribute of the IP address.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AllocationId:
#' \[EC2-VPC\] The allocation
#' IDs.
#' @section Attribute:
#' The attribute of the IP
#' address.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_addresses_attribute <- function(AllocationId = NULL, Attribute = NULL, NextToken = NULL, 
    MaxResults = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    AllocationId <- list_to_array("AllocationId", AllocationId)
    parameters <- c(others, list(Attribute = Attribute, NextToken = NextToken, MaxResults = MaxResults, 
        DryRun = DryRun), AllocationId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeAddressesAttribute", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Aggregate Id Format
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_aggregate_id_format <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeAggregateIdFormat", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Availability Zones
#' 
#' @param ZoneName List. The names of the Availability Zones, Local Zones, and Wavelength Zones.\[optional\]
#' @param ZoneId List. The IDs of the Availability Zones, Local Zones, and Wavelength Zones.\[optional\]
#' @param AllAvailabilityZones Logical.   Include all Availability Zones, Local Zones, and Wavelength Zones regardless of your opt-in status....\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `group-name` - For Availability Zones, use the Region name. For
#'     Local Zones, use the name of the group associated with the Local
#'     Zone (for example, `us-west-2-lax-1`) For Wavelength Zones, use the
#'     name of the group associated with the Wavelength Zone (for example,
#'     `us-east-1-wl1-bos-wlz-1`).
#' 
#' -   `message` - The Zone message.
#' 
#' -   `opt-in-status` - The opt-in status (`opted-in`, and `not-opted-in`
#'     \| `opt-in-not-required`).
#' 
#' -   `parent-zoneID` - The ID of the zone that handles some of the Local
#'     Zone and Wavelength Zone control plane operations, such as API
#'     calls.
#' 
#' -   `parent-zoneName` - The ID of the zone that handles some of the
#'     Local Zone and Wavelength Zone control plane operations, such as API
#'     calls.
#' 
#' -   `region-name` - The name of the Region for the Zone (for example,
#'     `us-east-1`).
#' 
#' -   `state` - The state of the Availability Zone, the Local Zone, or the
#'     Wavelength Zone (`available` \| `information` \| `impaired` \|
#'     `unavailable`).
#' 
#' -   `zone-id` - The ID of the Availability Zone (for example,
#'     `use1-az1`), the Local Zone (for example, `usw2-lax1-az1`), or the
#'     Wavelength Zone (for example, `us-east-1-wl1-bos-wlz-1`).
#' 
#' -   `zone-type` - The type of zone, for example, `local-zone`.
#' 
#' -   `zone-name` - The name of the Availability Zone (for example,
#'     `us-east-1a`), the Local Zone (for example, `us-west-2-lax-1a`), or
#'     the Wavelength Zone (for example, `us-east-1-wl1-bos-wlz-1`).
#' 
#' -   `zone-type` - The type of zone, for example, `local-zone`.
#' 
#' @section ZoneName:
#' The names of the Availability Zones, Local Zones,
#' and Wavelength Zones.
#' @section ZoneId:
#' The IDs of the Availability
#' Zones, Local Zones, and Wavelength Zones.
#' @section AllAvailabilityZones:
#' 
#' Include all Availability Zones, Local Zones, and Wavelength Zones
#' regardless of your opt-in status.
#' 
#' If you do not use this parameter, the results include only the zones for
#' the Regions where you have chosen the option to opt in.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_availability_zones <- function(Filter = NULL, ZoneName = NULL, ZoneId = NULL, 
    AllAvailabilityZones = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    ZoneName <- list_to_array("ZoneName", ZoneName)
    ZoneId <- list_to_array("ZoneId", ZoneId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(AllAvailabilityZones = AllAvailabilityZones, DryRun = DryRun), 
        Filter, ZoneName, ZoneId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeAvailabilityZones", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Bundle Tasks
#' 
#' @param BundleId List.   The bundle task IDs.  Default: Describes all your bundle tasks.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section BundleId:
#' 
#' The bundle task IDs.
#' 
#' Default: Describes all your bundle tasks.
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `bundle-id` - The ID of the bundle task.
#' 
#' -   `error-code` - If the task failed, the error code returned.
#' 
#' -   `error-message` - If the task failed, the error message returned.
#' 
#' -   `instance-id` - The ID of the instance.
#' 
#' -   `progress` - The level of task completion, as a percentage (for
#'     example, 20\%).
#' 
#' -   `s3-bucket` - The Amazon S3 bucket to store the AMI.
#' 
#' -   `s3-prefix` - The beginning of the AMI name.
#' 
#' -   `start-time` - The time the task started (for example,
#'     2013-09-15T17:15:20.000Z).
#' 
#' -   `state` - The state of the task (`pending` \| `waiting-for-shutdown`
#'     \| `bundling` \| `storing` \| `cancelling` \| `complete` \|
#'     `failed`).
#' 
#' -   `update-time` - The time of the most recent update for the task.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_bundle_tasks <- function(BundleId = NULL, Filter = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    BundleId <- list_to_array("BundleId", BundleId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), BundleId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeBundleTasks", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Byoip Cidrs
#' 
#' @param MaxResults Integer. The maximum number of results to return with a single call.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section MaxResults:
#' The maximum number of
#' results to return with a single call. To retrieve the remaining results,
#' make another call with the returned `nextToken`
#' value.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_byoip_cidrs <- function(MaxResults, DryRun = NULL, NextToken = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(MaxResults = MaxResults, DryRun = DryRun, NextToken = NextToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeByoipCidrs", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Capacity Reservations
#' 
#' Describes one or more of your Capacity
#' Reservations. The results describe only the Capacity Reservations in the
#' AWS Region that you\'re currently using.
#' 
#' @param CapacityReservationId List. The ID of the Capacity Reservation.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CapacityReservationId:
#' The ID of the Capacity
#' Reservation.
#' @section MaxResults:
#' The maximum number of results to
#' return for the request in a single page. The remaining results can be
#' seen by sending another request with the returned `nextToken` value.
#' This value can be between 5 and 500. If `maxResults` is given a larger
#' value than 500, you receive an error.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `instance-type` - The type of instance for which the Capacity
#'     Reservation reserves capacity.
#' 
#' -   `owner-id` - The ID of the AWS account that owns the Capacity
#'     Reservation.
#' 
#' -   `availability-zone-id` - The Availability Zone ID of the Capacity
#'     Reservation.
#' 
#' -   `instance-platform` - The type of operating system for which the
#'     Capacity Reservation reserves capacity.
#' 
#' -   `availability-zone` - The Availability Zone ID of the Capacity
#'     Reservation.
#' 
#' -   `tenancy` - Indicates the tenancy of the Capacity Reservation. A
#'     Capacity Reservation can have one of the following tenancy settings:
#' 
#'     -   `default` - The Capacity Reservation is created on hardware that
#'         is shared with other AWS accounts.
#' 
#'     -   `dedicated` - The Capacity Reservation is created on
#'         single-tenant hardware that is dedicated to a single AWS
#'         account.
#' 
#' -   `state` - The current state of the Capacity Reservation. A Capacity
#'     Reservation can be in one of the following states:
#' 
#'     -   `active`- The Capacity Reservation is active and the capacity is
#'         available for your use.
#' 
#'     -   `expired` - The Capacity Reservation expired automatically at
#'         the date and time specified in your request. The reserved
#'         capacity is no longer available for your use.
#' 
#'     -   `cancelled` - The Capacity Reservation was cancelled. The
#'         reserved capacity is no longer available for your use.
#' 
#'     -   `pending` - The Capacity Reservation request was successful but
#'         the capacity provisioning is still pending.
#' 
#'     -   `failed` - The Capacity Reservation request has failed. A
#'         request might fail due to invalid request parameters, capacity
#'         constraints, or instance limit constraints. Failed requests are
#'         retained for 60 minutes.
#' 
#' -   `end-date` - The date and time at which the Capacity Reservation
#'     expires. When a Capacity Reservation expires, the reserved capacity
#'     is released and you can no longer launch instances into it. The
#'     Capacity Reservation\'s state changes to expired when it reaches its
#'     end date and time.
#' 
#' -   `end-date-type` - Indicates the way in which the Capacity
#'     Reservation ends. A Capacity Reservation can have one of the
#'     following end types:
#' 
#'     -   `unlimited` - The Capacity Reservation remains active until you
#'         explicitly cancel it.
#' 
#'     -   `limited` - The Capacity Reservation expires automatically at a
#'         specified date and time.
#' 
#' -   `instance-match-criteria` - Indicates the type of instance launches
#'     that the Capacity Reservation accepts. The options include:
#' 
#'     -   `open` - The Capacity Reservation accepts all instances that
#'         have matching attributes (instance type, platform, and
#'         Availability Zone). Instances that have matching attributes
#'         launch into the Capacity Reservation automatically without
#'         specifying any additional parameters.
#' 
#'     -   `targeted` - The Capacity Reservation only accepts instances
#'         that have matching attributes (instance type, platform, and
#'         Availability Zone), and explicitly target the Capacity
#'         Reservation. This ensures that only permitted instances can use
#'         the reserved capacity.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_capacity_reservations <- function(CapacityReservationId = NULL, NextToken = NULL, 
    MaxResults = NULL, Filter = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    CapacityReservationId <- list_to_array("CapacityReservationId", CapacityReservationId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(NextToken = NextToken, MaxResults = MaxResults, DryRun = DryRun), 
        CapacityReservationId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeCapacityReservations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Carrier Gateways
#' 
#' Describes
#' one or more of your carrier gateways.
#' 
#' @param CarrierGatewayId List. One or more carrier gateway IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CarrierGatewayId:
#' One or more carrier gateway
#' IDs.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `carrier-gateway-id` - The ID of the carrier gateway.
#' 
#' -   `state` - The state of the carrier gateway (`pending` \| `failed` \|
#'     `available` \| `deleting` \| `deleted`).
#' 
#' -   `owner-id` - The AWS account ID of the owner of the carrier gateway.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC associated with the carrier gateway.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_carrier_gateways <- function(CarrierGatewayId = NULL, Filter = NULL, MaxResults = NULL, 
    NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    CarrierGatewayId <- list_to_array("CarrierGatewayId", CarrierGatewayId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        CarrierGatewayId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeCarrierGateways", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Classic Link Instances
#' 
#' Describes one
#' or more of your linked EC2-Classic instances. This request only returns
#' information about EC2-Classic instances linked to a VPC through
#' ClassicLink. You cannot use this request to return information about
#' other instances.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceId List. One or more instance IDs. Must be instances linked to a VPC through ClassicLink.\[optional\]
#' @param MaxResults Integer.   The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `group-id` - The ID of a VPC security group that\'s associated with
#'     the instance.
#' 
#' -   `instance-id` - The ID of the instance.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC to which the instance is linked.
#' 
#'     `vpc-id` - The ID of the VPC that the instance is linked to.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section InstanceId:
#' One or more instance IDs.
#' Must be instances linked to a VPC through
#' ClassicLink.
#' @section MaxResults:
#' 
#' The maximum number of results to return with a single call. To retrieve
#' the remaining results, make another call with the returned `nextToken`
#' value.
#' 
#' Constraint: If the value is greater than 1000, we return only 1000
#' items.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_classic_link_instances <- function(Filter = NULL, DryRun = NULL, InstanceId = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeClassicLinkInstances", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Client Vpn Authorization Rules
#' 
#' Describes the authorization rules
#' for a specified Client VPN endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `description` - The description of the authorization rule.
#' 
#' -   `destination-cidr` - The CIDR of the network to which the
#'     authorization rule applies.
#' 
#' -   `group-id` - The ID of the Active Directory group to which the
#'     authorization rule grants access.
#' 
#' @section MaxResults:
#' The maximum number of results to return for the
#' request in a single page. The remaining results can be seen by sending
#' another request with the nextToken value.
#' @return A list object or a character vector
#' @export
ec2_describe_client_vpn_authorization_rules <- function(ClientVpnEndpointId, DryRun = NULL, 
    NextToken = NULL, Filter = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, DryRun = DryRun, 
        NextToken = NextToken, MaxResults = MaxResults), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeClientVpnAuthorizationRules", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Client Vpn Connections
#' 
#' Describes
#' active client connections and connections that have been terminated
#' within the last 60 minutes for the specified Client VPN
#' endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of
#' the Client VPN endpoint.
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `connection-id` - The ID of the connection.
#' 
#' -   `username` - For Active Directory client authentication, the user
#'     name of the client who established the client connection.
#' 
#' @section MaxResults:
#' The maximum number of results to return for
#' the request in a single page. The remaining results can be seen by
#' sending another request with the nextToken
#' value.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_client_vpn_connections <- function(ClientVpnEndpointId, Filter = NULL, NextToken = NULL, 
    MaxResults = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, NextToken = NextToken, 
        MaxResults = MaxResults, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeClientVpnConnections", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Client Vpn Endpoints
#' 
#' Describes one or more Client VPN endpoints
#' in the account.
#' 
#' @param ClientVpnEndpointId List. The ID of the Client VPN endpoint.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section MaxResults:
#' The maximum number of results to return
#' for the request in a single page. The remaining results can be seen by
#' sending another request with the nextToken
#' value.
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `endpoint-id` - The ID of the Client VPN endpoint.
#' 
#' -   `transport-protocol` - The transport protocol (`tcp` \| `udp`).
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_client_vpn_endpoints <- function(ClientVpnEndpointId = NULL, MaxResults = NULL, 
    NextToken = NULL, Filter = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    ClientVpnEndpointId <- list_to_array("ClientVpnEndpointId", ClientVpnEndpointId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        ClientVpnEndpointId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeClientVpnEndpoints", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Client Vpn Routes
#' 
#' Describes the routes for the
#' specified Client VPN endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `destination-cidr` - The CIDR of the route destination.
#' 
#' -   `origin` - How the route was associated with the Client VPN endpoint
#'     (`associate` \| `add-route`).
#' 
#' -   `target-subnet` - The ID of the subnet through which traffic is
#'     routed.
#' 
#' @section MaxResults:
#' The maximum number of results to return for the
#' request in a single page. The remaining results can be seen by sending
#' another request with the nextToken value.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_client_vpn_routes <- function(ClientVpnEndpointId, Filter = NULL, MaxResults = NULL, 
    NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, MaxResults = MaxResults, 
        NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeClientVpnRoutes", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Client Vpn Target Networks
#' 
#' Describes the target
#' networks associated with the specified Client VPN
#' endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param AssociationIds List. The IDs of the target network associations.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of
#' the Client VPN endpoint.
#' @section AssociationIds:
#' The IDs of the target
#' network associations.
#' @section MaxResults:
#' The maximum number of results
#' to return for the request in a single page. The remaining results can be
#' seen by sending another request with the nextToken
#' value.
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `association-id` - The ID of the association.
#' 
#' -   `target-network-id` - The ID of the subnet specified as the target
#'     network.
#' 
#' -   `vpc-id` - The ID of the VPC in which the target network is located.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_client_vpn_target_networks <- function(ClientVpnEndpointId, AssociationIds = NULL, 
    MaxResults = NULL, NextToken = NULL, Filter = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    AssociationIds <- list_to_array("AssociationIds", AssociationIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, MaxResults = MaxResults, 
        NextToken = NextToken, DryRun = DryRun), AssociationIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeClientVpnTargetNetworks", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Coip Pools
#' 
#' Describes the specified customer-owned
#' address pools or all of your customer-owned address
#' pools.
#' 
#' @param PoolId List. The IDs of the address pools.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PoolId:
#' The IDs of the address
#' pools.
#' @section Filter:
#' 
#' The filters. The following are the possible values:
#' 
#' -   `coip-pool.pool-id`
#' 
#' ```{=html}
#' 
#' ```
#' -   `coip-pool.local-gateway-route-table-id`
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_coip_pools <- function(PoolId = NULL, Filter = NULL, MaxResults = NULL, NextToken = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    PoolId <- list_to_array("PoolId", PoolId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        PoolId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeCoipPools", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Conversion Tasks
#' 
#' @param ConversionTaskId List. The conversion task IDs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ConversionTaskId:
#' The conversion task
#' IDs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_conversion_tasks <- function(ConversionTaskId = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    ConversionTaskId <- list_to_array("ConversionTaskId", ConversionTaskId)
    parameters <- c(others, list(DryRun = DryRun), ConversionTaskId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeConversionTasks", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Customer Gateways
#' 
#' @param CustomerGatewayId List.   One or more customer gateway IDs.  Default: Describes all your customer gateways.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CustomerGatewayId:
#' 
#' One or more customer gateway IDs.
#' 
#' Default: Describes all your customer gateways.
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `bgp-asn` - The customer gateway\'s Border Gateway Protocol (BGP)
#'     Autonomous System Number (ASN).
#' 
#' -   `customer-gateway-id` - The ID of the customer gateway.
#' 
#' -   `ip-address` - The IP address of the customer gateway\'s
#'     Internet-routable external interface.
#' 
#' -   `state` - The state of the customer gateway (`pending` \|
#'     `available` \| `deleting` \| `deleted`).
#' 
#' -   `type` - The type of customer gateway. Currently, the only supported
#'     type is `ipsec.1`.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_customer_gateways <- function(CustomerGatewayId = NULL, Filter = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    CustomerGatewayId <- list_to_array("CustomerGatewayId", CustomerGatewayId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), CustomerGatewayId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeCustomerGateways", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Dhcp Options
#' 
#' @param DhcpOptionsId List.   The IDs of one or more DHCP options sets.  Default: Describes all your DHCP options sets.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DhcpOptionsId:
#' 
#' The IDs of one or more DHCP options sets.
#' 
#' Default: Describes all your DHCP options sets.
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `dhcp-options-id` - The ID of a DHCP options set.
#' 
#' -   `key` - The key for one of the options (for example, `domain-name`).
#' 
#' -   `value` - The value for one of the options.
#' 
#' -   `owner-id` - The ID of the AWS account that owns the DHCP options
#'     set.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_dhcp_options <- function(DhcpOptionsId = NULL, Filter = NULL, DryRun = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    DhcpOptionsId <- list_to_array("DhcpOptionsId", DhcpOptionsId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        DhcpOptionsId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeDhcpOptions", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Egress Only Internet Gateways
#' 
#' Describes one or more of your egress-only internet
#' gateways.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EgressOnlyInternetGatewayId List. One or more egress-only internet gateway IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section EgressOnlyInternetGatewayId:
#' One or
#' more egress-only internet gateway IDs.
#' @section MaxResults:
#' The maximum
#' number of results to return with a single call. To retrieve the
#' remaining results, make another call with the returned `nextToken`
#' value.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_egress_only_internet_gateways <- function(DryRun = NULL, EgressOnlyInternetGatewayId = NULL, 
    MaxResults = NULL, NextToken = NULL, Filter = NULL, simplify = TRUE, others = list()) {
    EgressOnlyInternetGatewayId <- list_to_array("EgressOnlyInternetGatewayId", EgressOnlyInternetGatewayId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        EgressOnlyInternetGatewayId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeEgressOnlyInternetGateways", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Elastic Gpus
#' 
#' Describes the Elastic Graphics accelerator
#' associated with your instances. For more information about Elastic
#' Graphics, see [Amazon Elastic
#' Graphics](https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html).
#' 
#' @param ElasticGpuId List. The Elastic Graphics accelerator IDs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ElasticGpuId:
#' The Elastic Graphics accelerator
#' IDs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `availability-zone` - The Availability Zone in which the Elastic
#'     Graphics accelerator resides.
#' 
#' -   `elastic-gpu-health` - The status of the Elastic Graphics
#'     accelerator (`OK` \| `IMPAIRED`).
#' 
#' -   `elastic-gpu-state` - The state of the Elastic Graphics accelerator
#'     (`ATTACHED`).
#' 
#' -   `elastic-gpu-type` - The type of Elastic Graphics accelerator; for
#'     example, `eg1.medium`.
#' 
#' -   `instance-id` - The ID of the instance to which the Elastic Graphics
#'     accelerator is associated.
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. To retrieve the remaining results, make another call with the
#' returned `NextToken` value. This value can be between 5 and
#' 1000.
#' @return A list object or a character vector
#' @export
ec2_describe_elastic_gpus <- function(ElasticGpuId = NULL, DryRun = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    ElasticGpuId <- list_to_array("ElasticGpuId", ElasticGpuId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        ElasticGpuId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeElasticGpus", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Export Image Tasks
#' 
#' Describes
#' the specified export image tasks or all of your export image
#' tasks.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ExportImageTaskId List. The IDs of the export image tasks.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' Filter tasks using the
#' `task-state` filter and one of the following values: `active`,
#' `completed`, `deleting`, or `deleted`.
#' @section ExportImageTaskId:
#' The IDs of
#' the export image tasks.
#' @section MaxResults:
#' The maximum number of
#' results to return in a single call.
#' @return A list object or a character vector
#' @export
ec2_describe_export_image_tasks <- function(DryRun = NULL, Filter = NULL, ExportImageTaskId = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    ExportImageTaskId <- list_to_array("ExportImageTaskId", ExportImageTaskId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, ExportImageTaskId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeExportImageTasks", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Export Tasks
#' 
#' Describes the specified export instance tasks
#' or all of your export instance tasks.
#' 
#' @param ExportTaskId List. The export task IDs.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ExportTaskId:
#' The export task
#' IDs.
#' @section Filter:
#' the filters for the export
#' tasks.
#' @return A list object or a character vector
#' @export
ec2_describe_export_tasks <- function(ExportTaskId = NULL, Filter = NULL, simplify = TRUE, 
    others = list()) {
    ExportTaskId <- list_to_array("ExportTaskId", ExportTaskId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(), ExportTaskId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeExportTasks", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Fast Snapshot Restores
#' 
#' Describes the
#' state of fast snapshot restores for your snapshots.
#' 
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters. The possible values are:
#' 
#' -   `availability-zone`: The Availability Zone of the snapshot.
#' 
#' -   `owner-id`: The ID of the AWS account that enabled fast snapshot
#'     restore on the snapshot.
#' 
#' -   `snapshot-id`: The ID of the snapshot.
#' 
#' -   `state`: The state of fast snapshot restores for the snapshot
#'     (`enabling` \| `optimizing` \| `enabled` \| `disabling` \|
#'     `disabled`).
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_fast_snapshot_restores <- function(Filter = NULL, MaxResults = NULL, NextToken = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeFastSnapshotRestores", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Fleet History
#' 
#' @param FleetId String. The ID of the EC2 Fleet.
#' @param StartTime String. The start date and time for the events, in UTC format (for example, *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z)....
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EventType String. The type of events to describe. By default, all events are described.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FleetId:
#' The ID of the EC2
#' Fleet.
#' @section StartTime:
#' The start date and time for the events, in
#' UTC format (for example,
#' *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z).
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section EventType:
#' The type
#' of events to describe. By default, all events are
#' described.
#' @section MaxResults:
#' The maximum number of results to return
#' in a single call. Specify a value between 1 and 1000. The default value
#' is 1000. To retrieve the remaining results, make another call with the
#' returned `NextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_fleet_history <- function(FleetId, StartTime, DryRun = NULL, EventType = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(FleetId = FleetId, StartTime = StartTime, DryRun = DryRun, 
        EventType = EventType, MaxResults = MaxResults, NextToken = NextToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeFleetHistory", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Fleet Instances
#' 
#' @param FleetId String. The ID of the EC2 Fleet.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FleetId:
#' The ID of the EC2
#' Fleet.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of
#' results to return in a single call. Specify a value between 1 and 1000.
#' The default value is 1000. To retrieve the remaining results, make
#' another call with the returned `NextToken`
#' value.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `instance-type` - The instance type.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_fleet_instances <- function(FleetId, DryRun = NULL, MaxResults = NULL, NextToken = NULL, 
    Filter = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(FleetId = FleetId, DryRun = DryRun, MaxResults = MaxResults, 
        NextToken = NextToken), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeFleetInstances", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Fleets
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @param FleetId List. The ID of the EC2 Fleets.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of
#' results to return in a single call. Specify a value between 1 and 1000.
#' The default value is 1000. To retrieve the remaining results, make
#' another call with the returned `NextToken`
#' value.
#' @section FleetId:
#' The ID of the EC2
#' Fleets.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `activity-status` - The progress of the EC2 Fleet ( `error` \|
#'     `pending-fulfillment` \| `pending-termination` \| `fulfilled`).
#' 
#' -   `excess-capacity-termination-policy` - Indicates whether to
#'     terminate running instances if the target capacity is decreased
#'     below the current EC2 Fleet size (`true` \| `false`).
#' 
#' -   `fleet-state` - The state of the EC2 Fleet (`submitted` \| `active`
#'     \| `deleted` \| `failed` \| `deleted-running` \|
#'     `deleted-terminating` \| `modifying`).
#' 
#' -   `replace-unhealthy-instances` - Indicates whether EC2 Fleet should
#'     replace unhealthy instances (`true` \| `false`).
#' 
#' -   `type` - The type of request (`instant` \| `request` \| `maintain`).
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_fleets <- function(DryRun = NULL, MaxResults = NULL, NextToken = NULL, FleetId = NULL, 
    Filter = NULL, simplify = TRUE, others = list()) {
    FleetId <- list_to_array("FleetId", FleetId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        FleetId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeFleets", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Flow Logs
#' 
#' Describes one or more flow logs. To view the
#' information in your flow logs (the log streams for the network
#' interfaces), you must use the CloudWatch Logs console or the CloudWatch
#' Logs API.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param FlowLogId List.   One or more flow log IDs.  Constraint: Maximum of 1000 flow log IDs.  \[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `deliver-log-status` - The status of the logs delivery (`SUCCESS` \|
#'     `FAILED`).
#' 
#' -   `log-destination-type` - The type of destination to which the flow
#'     log publishes data. Possible destination types include
#'     `cloud-watch-logs` and `s3`.
#' 
#' -   `flow-log-id` - The ID of the flow log.
#' 
#' -   `log-group-name` - The name of the log group.
#' 
#' -   `resource-id` - The ID of the VPC, subnet, or network interface.
#' 
#' -   `traffic-type` - The type of traffic (`ACCEPT` \| `REJECT` \|
#'     `ALL`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section FlowLogId:
#' 
#' One or more flow log IDs.
#' 
#' Constraint: Maximum of 1000 flow log IDs.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_flow_logs <- function(DryRun = NULL, Filter = NULL, FlowLogId = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    FlowLogId <- list_to_array("FlowLogId", FlowLogId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, FlowLogId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeFlowLogs", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Fpga Image Attribute
#' 
#' Describes the specified attribute of the
#' specified Amazon FPGA Image (AFI).
#' 
#' @param FpgaImageId String. The ID of the AFI.
#' @param Attribute String. The AFI attribute.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FpgaImageId:
#' The ID of the
#' AFI.
#' @section Attribute:
#' The AFI attribute.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_fpga_image_attribute <- function(FpgaImageId, Attribute, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(FpgaImageId = FpgaImageId, Attribute = Attribute, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeFpgaImageAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Fpga Images
#' 
#' Describes the
#' Amazon FPGA Images (AFIs) available to you. These include public AFIs,
#' private AFIs that you own, and AFIs owned by other AWS accounts for
#' which you have load permissions.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param FpgaImageId List. The AFI IDs.\[optional\]
#' @param Owner List. Filters the AFI by owner.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section FpgaImageId:
#' The AFI
#' IDs.
#' @section Owner:
#' Filters the AFI by owner. Specify an AWS
#' account ID, `self` (owner is the sender of the request), or an AWS owner
#' alias (valid values are `amazon` \|
#' `aws-marketplace`).
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `create-time` - The creation time of the AFI.
#' 
#' -   `fpga-image-id` - The FPGA image identifier (AFI ID).
#' 
#' -   `fpga-image-global-id` - The global FPGA image identifier (AGFI ID).
#' 
#' -   `name` - The name of the AFI.
#' 
#' -   `owner-id` - The AWS account ID of the AFI owner.
#' 
#' -   `product-code` - The product code.
#' 
#' -   `shell-version` - The version of the AWS Shell that was used to
#'     create the bitstream.
#' 
#' -   `state` - The state of the AFI (`pending` \| `failed` \| `available`
#'     \| `unavailable`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `update-time` - The time of the most recent update.
#' 
#' @section MaxResults:
#' The maximum number of results to return in
#' a single call.
#' @return A list object or a character vector
#' @export
ec2_describe_fpga_images <- function(DryRun = NULL, FpgaImageId = NULL, Owner = NULL, Filter = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    FpgaImageId <- list_to_array("FpgaImageId", FpgaImageId)
    Owner <- list_to_array("Owner", Owner)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        FpgaImageId, Owner, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeFpgaImages", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Host Reservation Offerings
#' 
#' @param MaxDuration Integer. This is the maximum duration of the reservation to purchase, specified in seconds.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @param MinDuration Integer. This is the minimum duration of the reservation you\'d like to purchase, specified in seconds.\[optional\]
#' @param OfferingId String. The ID of the reservation offering.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `instance-family` - The instance family of the offering (for
#'     example, `m4`).
#' 
#' -   `payment-option` - The payment option (`NoUpfront` \|
#'     `PartialUpfront` \| `AllUpfront`).
#' 
#' @section MaxDuration:
#' This is the maximum duration of the reservation to
#' purchase, specified in seconds. Reservations are available in one-year
#' and three-year terms. The number of seconds specified must be the number
#' of seconds in a year (365x24x60x60) times one of the supported durations
#' (1 or 3). For example, specify 94608000 for three
#' years.
#' @section MaxResults:
#' The maximum number of results to return for
#' the request in a single page. The remaining results can be seen by
#' sending another request with the returned `nextToken` value. This value
#' can be between 5 and 500. If `maxResults` is given a larger value than
#' 500, you receive an error.
#' @section MinDuration:
#' This is the minimum
#' duration of the reservation you\'d like to purchase, specified in
#' seconds. Reservations are available in one-year and three-year terms.
#' The number of seconds specified must be the number of seconds in a year
#' (365x24x60x60) times one of the supported durations (1 or 3). For
#' example, specify 31536000 for one year.
#' @section OfferingId:
#' The ID of
#' the reservation offering.
#' @return A list object or a character vector
#' @export
ec2_describe_host_reservation_offerings <- function(Filter = NULL, MaxDuration = NULL, MaxResults = NULL, 
    MinDuration = NULL, NextToken = NULL, OfferingId = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxDuration = MaxDuration, MaxResults = MaxResults, MinDuration = MinDuration, 
        NextToken = NextToken, OfferingId = OfferingId), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeHostReservationOfferings", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Host Reservations
#' 
#' Describes reservations that are associated with
#' Dedicated Hosts in your account.
#' 
#' @param HostReservationIdSet List. The host reservation IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `instance-family` - The instance family (for example, `m4`).
#' 
#' -   `payment-option` - The payment option (`NoUpfront` \|
#'     `PartialUpfront` \| `AllUpfront`).
#' 
#' -   `state` - The state of the reservation (`payment-pending` \|
#'     `payment-failed` \| `active` \| `retired`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section HostReservationIdSet:
#' The host reservation IDs.
#' @section MaxResults:
#' The
#' maximum number of results to return for the request in a single page.
#' The remaining results can be seen by sending another request with the
#' returned `nextToken` value. This value can be between 5 and 500. If
#' `maxResults` is given a larger value than 500, you receive an
#' error.
#' @return A list object or a character vector
#' @export
ec2_describe_host_reservations <- function(Filter = NULL, HostReservationIdSet = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    HostReservationIdSet <- list_to_array("HostReservationIdSet", HostReservationIdSet)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken), Filter, 
        HostReservationIdSet)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeHostReservations", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Hosts
#' 
#' @param HostId List. The IDs of the Dedicated Hosts. The IDs are used for targeted instance launches.\[optional\]
#' @param MaxResults Integer.   The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `auto-placement` - Whether auto-placement is enabled or disabled
#'     (`on` \| `off`).
#' 
#' -   `availability-zone` - The Availability Zone of the host.
#' 
#' -   `client-token` - The idempotency token that you provided when you
#'     allocated the host.
#' 
#' -   `host-reservation-id` - The ID of the reservation assigned to this
#'     host.
#' 
#' -   `instance-type` - The instance type size that the Dedicated Host is
#'     configured to support.
#' 
#' -   `state` - The allocation state of the Dedicated Host (`available` \|
#'     `under-assessment` \| `permanent-failure` \| `released` \|
#'     `released-permanent-failure`).
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section HostId:
#' The IDs of the Dedicated Hosts. The IDs are used
#' for targeted instance launches.
#' @section MaxResults:
#' 
#' The maximum number of results to return for the request in a single
#' page. The remaining results can be seen by sending another request with
#' the returned `nextToken` value. This value can be between 5 and 500. If
#' `maxResults` is given a larger value than 500, you receive an error.
#' 
#' You cannot specify this parameter and the host IDs parameter in the same
#' request.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_hosts <- function(Filter = NULL, HostId = NULL, MaxResults = NULL, NextToken = NULL, 
    simplify = TRUE, others = list()) {
    HostId <- list_to_array("HostId", HostId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken), Filter, 
        HostId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeHosts", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Iam Instance Profile Associations
#' 
#' Describes your IAM instance profile
#' associations.
#' 
#' @param AssociationId List. The IAM instance profile associations.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' The IAM instance profile
#' associations.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `instance-id` - The ID of the instance.
#' 
#' -   `state` - The state of the association (`associating` \|
#'     `associated` \| `disassociating`).
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. To retrieve the remaining results, make another call with the
#' returned `NextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_iam_instance_profile_associations <- function(AssociationId = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    AssociationId <- list_to_array("AssociationId", AssociationId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken), AssociationId, 
        Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeIamInstanceProfileAssociations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Id Format
#' 
#' @param Resource String. The type of resource: `bundle` \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \| `elastic-ip-allocation`...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Resource:
#' The type of resource: `bundle`
#' \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \|
#' `elastic-ip-allocation` \| `elastic-ip-association` \| `export-task` \|
#' `flow-log` \| `image` \| `import-task` \| `instance` \|
#' `internet-gateway` \| `network-acl` \| `network-acl-association` \|
#' `network-interface` \| `network-interface-attachment` \| `prefix-list`
#' \| `reservation` \| `route-table` \| `route-table-association` \|
#' `security-group` \| `snapshot` \| `subnet` \|
#' `subnet-cidr-block-association` \| `volume` \| `vpc` \|
#' `vpc-cidr-block-association` \| `vpc-endpoint` \|
#' `vpc-peering-connection` \| `vpn-connection` \| `vpn-gateway`
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_id_format <- function(Resource = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(Resource = Resource))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeIdFormat", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Identity Id Format
#' 
#' @param PrincipalArn String. The ARN of the principal, which can be an IAM role, IAM user, or the root user.
#' @param Resource String. The type of resource: `bundle` \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \| `elastic-ip-allocation`...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PrincipalArn:
#' The ARN of the principal, which can be an IAM role,
#' IAM user, or the root user.
#' @section Resource:
#' The type of resource:
#' `bundle` \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \|
#' `elastic-ip-allocation` \| `elastic-ip-association` \| `export-task` \|
#' `flow-log` \| `image` \| `import-task` \| `instance` \|
#' `internet-gateway` \| `network-acl` \| `network-acl-association` \|
#' `network-interface` \| `network-interface-attachment` \| `prefix-list`
#' \| `reservation` \| `route-table` \| `route-table-association` \|
#' `security-group` \| `snapshot` \| `subnet` \|
#' `subnet-cidr-block-association` \| `volume` \| `vpc` \|
#' `vpc-cidr-block-association` \| `vpc-endpoint` \|
#' `vpc-peering-connection` \| `vpn-connection` \| `vpn-gateway`
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_identity_id_format <- function(PrincipalArn, Resource = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(PrincipalArn = PrincipalArn, Resource = Resource))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeIdentityIdFormat", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Image Attribute
#' 
#' Describes the specified attribute of the specified
#' AMI. You can specify only one attribute at a time.
#' 
#' @param Attribute String.   The AMI attribute.
#' @param ImageId String. The ID of the AMI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' 
#' The AMI attribute.
#' 
#' **Note**: The `blockDeviceMapping` attribute is deprecated. Using this
#' attribute returns the `Client.AuthFailure` error. To get information
#' about the block device mappings for an AMI, use the DescribeImages
#' action.
#' 
#' @section ImageId:
#' The ID of the AMI.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_image_attribute <- function(Attribute, ImageId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, ImageId = ImageId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeImageAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Images
#' 
#' @param ExecutableBy List. Scopes the images by users with explicit launch permissions.\[optional\]
#' @param ImageId List.   The image IDs.  Default: Describes all images available to you.  \[optional\]
#' @param Owner List. Scopes the results to images with the specified owners.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ExecutableBy:
#' Scopes the
#' images by users with explicit launch permissions. Specify an AWS account
#' ID, `self` (the sender of the request), or `all` (public
#' AMIs).
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `architecture` - The image architecture (`i386` \| `x86_64` \|
#'     `arm64`).
#' 
#' -   `block-device-mapping.delete-on-termination` - A Boolean value that
#'     indicates whether the Amazon EBS volume is deleted on instance
#'     termination.
#' 
#' -   `block-device-mapping.device-name` - The device name specified in
#'     the block device mapping (for example, `/dev/sdh` or `xvdh`).
#' 
#' -   `block-device-mapping.snapshot-id` - The ID of the snapshot used for
#'     the EBS volume.
#' 
#' -   `block-device-mapping.volume-size` - The volume size of the EBS
#'     volume, in GiB.
#' 
#' -   `block-device-mapping.volume-type` - The volume type of the EBS
#'     volume (`gp2` \| `io1` \| `io2` \| `st1 `\| `sc1` \| `standard`).
#' 
#' -   `block-device-mapping.encrypted` - A Boolean that indicates whether
#'     the EBS volume is encrypted.
#' 
#' -   `description` - The description of the image (provided during image
#'     creation).
#' 
#' -   `ena-support` - A Boolean that indicates whether enhanced networking
#'     with ENA is enabled.
#' 
#' -   `hypervisor` - The hypervisor type (`ovm` \| `xen`).
#' 
#' -   `image-id` - The ID of the image.
#' 
#' -   `image-type` - The image type (`machine` \| `kernel` \| `ramdisk`).
#' 
#' -   `is-public` - A Boolean that indicates whether the image is public.
#' 
#' -   `kernel-id` - The kernel ID.
#' 
#' -   `manifest-location` - The location of the image manifest.
#' 
#' -   `name` - The name of the AMI (provided during image creation).
#' 
#' -   `owner-alias` - The owner alias (`amazon` \| `aws-marketplace`). The
#'     valid aliases are defined in an Amazon-maintained list. This is not
#'     the AWS account alias that can be set using the IAM console. We
#'     recommend that you use the **Owner** request parameter instead of
#'     this filter.
#' 
#' -   `owner-id` - The AWS account ID of the owner. We recommend that you
#'     use the **Owner** request parameter instead of this filter.
#' 
#' -   `platform` - The platform. To only list Windows-based AMIs, use
#'     `windows`.
#' 
#' -   `product-code` - The product code.
#' 
#' -   `product-code.type` - The type of the product code (`devpay` \|
#'     `marketplace`).
#' 
#' -   `ramdisk-id` - The RAM disk ID.
#' 
#' -   `root-device-name` - The device name of the root device volume (for
#'     example, `/dev/sda1`).
#' 
#' -   `root-device-type` - The type of the root device volume (`ebs` \|
#'     `instance-store`).
#' 
#' -   `state` - The state of the image (`available` \| `pending` \|
#'     `failed`).
#' 
#' -   `state-reason-code` - The reason code for the state change.
#' 
#' -   `state-reason-message` - The message for the state change.
#' 
#' -   `sriov-net-support` - A value of `simple` indicates that enhanced
#'     networking with the Intel 82599 VF interface is enabled.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `virtualization-type` - The virtualization type (`paravirtual` \|
#'     `hvm`).
#' 
#' @section ImageId:
#' 
#' The image IDs.
#' 
#' Default: Describes all images available to you.
#' 
#' @section Owner:
#' Scopes the results to images with the specified
#' owners. You can specify a combination of AWS account IDs, `self`,
#' `amazon`, and `aws-marketplace`. If you omit this parameter, the results
#' include all images for which you have launch permissions, regardless of
#' ownership.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_images <- function(ExecutableBy = NULL, Filter = NULL, ImageId = NULL, Owner = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    ExecutableBy <- list_to_array("ExecutableBy", ExecutableBy)
    ImageId <- list_to_array("ImageId", ImageId)
    Owner <- list_to_array("Owner", Owner)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), ExecutableBy, Filter, ImageId, Owner)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeImages", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Import Image Tasks
#' 
#' Displays details about an import virtual machine or
#' import snapshot tasks that are already
#' created.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Filters List. Filter tasks using the `task-state` filter and one of the following values: `active`, `completed`,...\[optional\]
#' @param ImportTaskId List. The IDs of the import image tasks.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filters:
#' Filter tasks using the
#' `task-state` filter and one of the following values: `active`,
#' `completed`, `deleting`, or `deleted`.
#' @section ImportTaskId:
#' The IDs of
#' the import image tasks.
#' @section MaxResults:
#' The maximum number of
#' results to return in a single call.
#' @return A list object or a character vector
#' @export
ec2_describe_import_image_tasks <- function(DryRun = NULL, Filters = NULL, ImportTaskId = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    Filters <- list_to_array("Filters", Filters)
    ImportTaskId <- list_to_array("ImportTaskId", ImportTaskId)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filters, ImportTaskId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeImportImageTasks", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Import Snapshot Tasks
#' 
#' Describes your import snapshot
#' tasks.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Filters List. The filters.\[optional\]
#' @param ImportTaskId List. A list of import snapshot task IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section Filters:
#' The
#' filters.
#' @section ImportTaskId:
#' A list of import snapshot task
#' IDs.
#' @section MaxResults:
#' The maximum number of results to return in a
#' single call. To retrieve the remaining results, make another call with
#' the returned `NextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_import_snapshot_tasks <- function(DryRun = NULL, Filters = NULL, ImportTaskId = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    Filters <- list_to_array("Filters", Filters)
    ImportTaskId <- list_to_array("ImportTaskId", ImportTaskId)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filters, ImportTaskId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeImportSnapshotTasks", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Instance Attribute
#' 
#' Describes the specified attribute of the
#' specified instance. You can specify only one attribute at a time. Valid
#' attribute values are: `instanceType` \| `kernel` \| `ramdisk` \|
#' `userData` \| `disableApiTermination` \|
#' `instanceInitiatedShutdownBehavior` \| `rootDeviceName` \|
#' `blockDeviceMapping` \| `productCodes` \| `sourceDestCheck` \|
#' `groupSet` \| `ebsOptimized` \| `sriovNetSupport` 
#' 
#' @param Attribute String.   The instance attribute.  Note: The `enaSupport` attribute is not supported at this time.  
#' @param InstanceId String. The ID of the instance.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' 
#' The instance attribute.
#' 
#' Note: The `enaSupport` attribute is not supported at this time.
#' 
#' @section InstanceId:
#' The ID of the instance.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_instance_attribute <- function(Attribute, InstanceId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, InstanceId = InstanceId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInstanceAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Instance Credit Specifications
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceId List.   The instance IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `instance-id` - The ID of the instance.
#' 
#' @section InstanceId:
#' 
#' The instance IDs.
#' 
#' Default: Describes all your instances.
#' 
#' Constraints: Maximum 1000 explicitly specified instance IDs.
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. To retrieve the remaining results, make another call with the
#' returned `NextToken` value. This value can be between 5 and 1000. You
#' cannot specify this parameter and the instance IDs parameter in the same
#' call.
#' @return A list object or a character vector
#' @export
ec2_describe_instance_credit_specifications <- function(DryRun = NULL, Filter = NULL, InstanceId = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInstanceCreditSpecifications", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Instance Event Notification Attributes
#' 
#' Describes the tag keys that are registered to
#' appear in scheduled event notifications for resources in the current
#' Region.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_instance_event_notification_attributes <- function(DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInstanceEventNotificationAttributes", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Describe Instance Status
#' 
#' @param InstanceId List.   The instance IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param IncludeAllInstances Logical.   When `true`, includes the health status for all instances.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `availability-zone` - The Availability Zone of the instance.
#' 
#' -   `event.code` - The code for the scheduled event (`instance-reboot`
#'     \| `system-reboot` \| `system-maintenance` \| `instance-retirement`
#'     \| `instance-stop`).
#' 
#' -   `event.description` - A description of the event.
#' 
#' -   `event.instance-event-id` - The ID of the event whose date and time
#'     you are modifying.
#' 
#' -   `event.not-after` - The latest end time for the scheduled event (for
#'     example, `2014-09-15T17:15:20.000Z`).
#' 
#' -   `event.not-before` - The earliest start time for the scheduled event
#'     (for example, `2014-09-15T17:15:20.000Z`).
#' 
#' -   `event.not-before-deadline` - The deadline for starting the event
#'     (for example, `2014-09-15T17:15:20.000Z`).
#' 
#' -   `instance-state-code` - The code for the instance state, as a 16-bit
#'     unsigned integer. The high byte is used for internal purposes and
#'     should be ignored. The low byte is set based on the state
#'     represented. The valid values are 0 (pending), 16 (running), 32
#'     (shutting-down), 48 (terminated), 64 (stopping), and 80 (stopped).
#' 
#' -   `instance-state-name` - The state of the instance (`pending` \|
#'     `running` \| `shutting-down` \| `terminated` \| `stopping` \|
#'     `stopped`).
#' 
#' -   `instance-status.reachability` - Filters on instance status where
#'     the name is `reachability` (`passed` \| `failed` \| `initializing`
#'     \| `insufficient-data`).
#' 
#' -   `instance-status.status` - The status of the instance (`ok` \|
#'     `impaired` \| `initializing` \| `insufficient-data` \|
#'     `not-applicable`).
#' 
#' -   `system-status.reachability` - Filters on system status where the
#'     name is `reachability` (`passed` \| `failed` \| `initializing` \|
#'     `insufficient-data`).
#' 
#' -   `system-status.status` - The system status of the instance (`ok` \|
#'     `impaired` \| `initializing` \| `insufficient-data` \|
#'     `not-applicable`).
#' 
#' @section InstanceId:
#' 
#' The instance IDs.
#' 
#' Default: Describes all your instances.
#' 
#' Constraints: Maximum 100 explicitly specified instance IDs.
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. To retrieve the remaining results, make another call with the
#' returned `NextToken` value. This value can be between 5 and 1000. You
#' cannot specify this parameter and the instance IDs parameter in the same
#' call.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section IncludeAllInstances:
#' 
#' When `true`, includes the health status for all instances. When `false`,
#' includes the health status for running instances only.
#' 
#' Default: `false`
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_instance_status <- function(Filter = NULL, InstanceId = NULL, MaxResults = NULL, 
    NextToken = NULL, DryRun = NULL, IncludeAllInstances = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun, 
        IncludeAllInstances = IncludeAllInstances), Filter, InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInstanceStatus", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Instance Type Offerings
#' 
#' Returns a list of all instance types offered. The
#' results can be filtered by location (Region or Availability Zone). If no
#' location is specified, the instance types offered in the current Region
#' are returned.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LocationType String. The location type.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section LocationType:
#' The location
#' type.
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `location` - This depends on the location type. For example, if the
#'     location type is `region` (default), the location is the Region code
#'     (for example, `us-east-2`.)
#' 
#' -   `instance-type` - The instance type. For example, `c5.2xlarge`.
#' 
#' @section MaxResults:
#' The maximum number of results to return for the
#' request in a single page. The remaining results can be seen by sending
#' another request with the next token value.
#' @return A list object or a character vector
#' @export
ec2_describe_instance_type_offerings <- function(DryRun = NULL, LocationType = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, LocationType = LocationType, MaxResults = MaxResults, 
        NextToken = NextToken), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInstanceTypeOfferings", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Instance Types
#' 
#' Describes the details of the instance
#' types that are offered in a location. The results can be filtered by the
#' attributes of the instance types.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceType List. The instance types.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section InstanceType:
#' The
#' instance types. For more information, see [Instance
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html)
#' in the *Amazon EC2 User Guide*.
#' @section Filter:
#' 
#' One or more filters. Filter names and values are case-sensitive.
#' 
#' -   `auto-recovery-supported` - Indicates whether auto recovery is
#'     supported (`true` \| `false`).
#' 
#' -   `bare-metal` - Indicates whether it is a bare metal instance type
#'     (`true` \| `false`).
#' 
#' -   `burstable-performance-supported` - Indicates whether it is a
#'     burstable performance instance type (`true` \| `false`).
#' 
#' -   `current-generation` - Indicates whether this instance type is the
#'     latest generation instance type of an instance family (`true` \|
#'     `false`).
#' 
#' -   `ebs-info.ebs-optimized-info.baseline-bandwidth-in-mbps` - The
#'     baseline bandwidth performance for an EBS-optimized instance type,
#'     in Mbps.
#' 
#' -   `ebs-info.ebs-optimized-info.baseline-iops` - The baseline
#'     input/output storage operations per second for an EBS-optimized
#'     instance type.
#' 
#' -   `ebs-info.ebs-optimized-info.baseline-throughput-in-mbps` - The
#'     baseline throughput performance for an EBS-optimized instance type,
#'     in MB/s.
#' 
#' -   `ebs-info.ebs-optimized-info.maximum-bandwidth-in-mbps` - The
#'     maximum bandwidth performance for an EBS-optimized instance type, in
#'     Mbps.
#' 
#' -   `ebs-info.ebs-optimized-info.maximum-iops` - The maximum
#'     input/output storage operations per second for an EBS-optimized
#'     instance type.
#' 
#' -   `ebs-info.ebs-optimized-info.maximum-throughput-in-mbps` - The
#'     maximum throughput performance for an EBS-optimized instance type,
#'     in MB/s.
#' 
#' -   `ebs-info.ebs-optimized-support` - Indicates whether the instance
#'     type is EBS-optimized (`supported` \| `unsupported` \| `default`).
#' 
#' -   `ebs-info.encryption-support` - Indicates whether EBS encryption is
#'     supported (`supported` \| `unsupported`).
#' 
#' -   `ebs-info.nvme-support` - Indicates whether non-volatile memory
#'     express (NVMe) is supported for EBS volumes (`required` \|
#'     `supported` \| `unsupported`).
#' 
#' -   `free-tier-eligible` - Indicates whether the instance type is
#'     eligible to use in the free tier (`true` \| `false`).
#' 
#' -   `hibernation-supported` - Indicates whether On-Demand hibernation is
#'     supported (`true` \| `false`).
#' 
#' -   `hypervisor` - The hypervisor (`nitro` \| `xen`).
#' 
#' -   `instance-storage-info.disk.count` - The number of local disks.
#' 
#' -   `instance-storage-info.disk.size-in-gb` - The storage size of each
#'     instance storage disk, in GB.
#' 
#' -   `instance-storage-info.disk.type` - The storage technology for the
#'     local instance storage disks (`hdd` \| `ssd`).
#' 
#' -   `instance-storage-info.nvme-support` - Indicates whether
#'     non-volatile memory express (NVMe) is supported for instance store
#'     (`required` \| `supported`) \| `unsupported`).
#' 
#' -   `instance-storage-info.total-size-in-gb` - The total amount of
#'     storage available from all local instance storage, in GB.
#' 
#' -   `instance-storage-supported` - Indicates whether the instance type
#'     has local instance storage (`true` \| `false`).
#' 
#' -   `instance-type` - The instance type (for example `c5.2xlarge` or
#'     c5\*).
#' 
#' -   `memory-info.size-in-mib` - The memory size.
#' 
#' -   `network-info.efa-supported` - Indicates whether the instance type
#'     supports Elastic Fabric Adapter (EFA) (`true` \| `false`).
#' 
#' -   `network-info.ena-support` - Indicates whether Elastic Network
#'     Adapter (ENA) is supported or required (`required` \| `supported` \|
#'     `unsupported`).
#' 
#' -   `network-info.ipv4-addresses-per-interface` - The maximum number of
#'     private IPv4 addresses per network interface.
#' 
#' -   `network-info.ipv6-addresses-per-interface` - The maximum number of
#'     private IPv6 addresses per network interface.
#' 
#' -   `network-info.ipv6-supported` - Indicates whether the instance type
#'     supports IPv6 (`true` \| `false`).
#' 
#' -   `network-info.maximum-network-interfaces` - The maximum number of
#'     network interfaces per instance.
#' 
#' -   `network-info.network-performance` - The network performance (for
#'     example, \'25 Gigabit\').
#' 
#' -   `processor-info.supported-architecture` - The CPU architecture
#'     (`arm64` \| `i386` \| `x86_64`).
#' 
#' -   `processor-info.sustained-clock-speed-in-ghz` - The CPU clock speed,
#'     in GHz.
#' 
#' -   `supported-root-device-type` - The root device type (`ebs` \|
#'     `instance-store`).
#' 
#' -   `supported-usage-class` - The usage class (`on-demand` \| `spot`).
#' 
#' -   `supported-virtualization-type` - The virtualization type (`hvm` \|
#'     `paravirtual`).
#' 
#' -   `vcpu-info.default-cores` - The default number of cores for the
#'     instance type.
#' 
#' -   `vcpu-info.default-threads-per-core` - The default number of threads
#'     per core for the instance type.
#' 
#' -   `vcpu-info.default-vcpus` - The default number of vCPUs for the
#'     instance type.
#' 
#' -   `vcpu-info.valid-cores` - The number of cores that can be configured
#'     for the instance type.
#' 
#' -   `vcpu-info.valid-threads-per-core` - The number of threads per core
#'     that can be configured for the instance type. For example, \'1\' or
#'     \'1,2\'.
#' 
#' @section MaxResults:
#' The maximum number of results to return for the
#' request in a single page. The remaining results can be seen by sending
#' another request with the next token value.
#' @return A list object or a character vector
#' @export
ec2_describe_instance_types <- function(DryRun = NULL, InstanceType = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    InstanceType <- list_to_array("InstanceType", InstanceType)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        InstanceType, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInstanceTypes", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Instances
#' 
#' @param InstanceId List.   The instance IDs.  Default: Describes all your instances.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `affinity` - The affinity setting for an instance running on a
#'     Dedicated Host (`default` \| `host`).
#' 
#' -   `architecture` - The instance architecture (`i386` \| `x86_64` \|
#'     `arm64`).
#' 
#' -   `availability-zone` - The Availability Zone of the instance.
#' 
#' -   `block-device-mapping.attach-time` - The attach time for an EBS
#'     volume mapped to the instance, for example,
#'     `2010-09-15T17:15:20.000Z`.
#' 
#' -   `block-device-mapping.delete-on-termination` - A Boolean that
#'     indicates whether the EBS volume is deleted on instance termination.
#' 
#' -   `block-device-mapping.device-name` - The device name specified in
#'     the block device mapping (for example, `/dev/sdh` or `xvdh`).
#' 
#' -   `block-device-mapping.status` - The status for the EBS volume
#'     (`attaching` \| `attached` \| `detaching` \| `detached`).
#' 
#' -   `block-device-mapping.volume-id` - The volume ID of the EBS volume.
#' 
#' -   `client-token` - The idempotency token you provided when you
#'     launched the instance.
#' 
#' -   `dns-name` - The public DNS name of the instance.
#' 
#' -   `group-id` - The ID of the security group for the instance.
#'     EC2-Classic only.
#' 
#' -   `group-name` - The name of the security group for the instance.
#'     EC2-Classic only.
#' 
#' -   `hibernation-options.configured` - A Boolean that indicates whether
#'     the instance is enabled for hibernation. A value of `true` means
#'     that the instance is enabled for hibernation.
#' 
#' -   `host-id` - The ID of the Dedicated Host on which the instance is
#'     running, if applicable.
#' 
#' -   `hypervisor` - The hypervisor type of the instance (`ovm` \| `xen`).
#'     The value `xen` is used for both Xen and Nitro hypervisors.
#' 
#' -   `iam-instance-profile.arn` - The instance profile associated with
#'     the instance. Specified as an ARN.
#' 
#' -   `image-id` - The ID of the image used to launch the instance.
#' 
#' -   `instance-id` - The ID of the instance.
#' 
#' -   `instance-lifecycle` - Indicates whether this is a Spot Instance or
#'     a Scheduled Instance (`spot` \| `scheduled`).
#' 
#' -   `instance-state-code` - The state of the instance, as a 16-bit
#'     unsigned integer. The high byte is used for internal purposes and
#'     should be ignored. The low byte is set based on the state
#'     represented. The valid values are: 0 (pending), 16 (running), 32
#'     (shutting-down), 48 (terminated), 64 (stopping), and 80 (stopped).
#' 
#' -   `instance-state-name` - The state of the instance (`pending` \|
#'     `running` \| `shutting-down` \| `terminated` \| `stopping` \|
#'     `stopped`).
#' 
#' -   `instance-type` - The type of instance (for example, `t2.micro`).
#' 
#' -   `instance.group-id` - The ID of the security group for the instance.
#' 
#' -   `instance.group-name` - The name of the security group for the
#'     instance.
#' 
#' -   `ip-address` - The public IPv4 address of the instance.
#' 
#' -   `kernel-id` - The kernel ID.
#' 
#' -   `key-name` - The name of the key pair used when the instance was
#'     launched.
#' 
#' -   `launch-index` - When launching multiple instances, this is the
#'     index for the instance in the launch group (for example, 0, 1, 2,
#'     and so on).
#' 
#' -   `launch-time` - The time when the instance was launched.
#' 
#' -   `metadata-options.http-tokens` - The metadata request authorization
#'     state (`optional` \| `required`)
#' 
#' -   `metadata-options.http-put-response-hop-limit` - The http metadata
#'     request put response hop limit (integer, possible values `1` to
#'     `64`)
#' 
#' -   `metadata-options.http-endpoint` - Enable or disable metadata access
#'     on http endpoint (`enabled` \| `disabled`)
#' 
#' -   `monitoring-state` - Indicates whether detailed monitoring is
#'     enabled (`disabled` \| `enabled`).
#' 
#' -   `network-interface.addresses.private-ip-address` - The private IPv4
#'     address associated with the network interface.
#' 
#' -   `network-interface.addresses.primary` - Specifies whether the IPv4
#'     address of the network interface is the primary private IPv4
#'     address.
#' 
#' -   `network-interface.addresses.association.public-ip` - The ID of the
#'     association of an Elastic IP address (IPv4) with a network
#'     interface.
#' 
#' -   `network-interface.addresses.association.ip-owner-id` - The owner ID
#'     of the private IPv4 address associated with the network interface.
#' 
#' -   `network-interface.association.public-ip` - The address of the
#'     Elastic IP address (IPv4) bound to the network interface.
#' 
#' -   `network-interface.association.ip-owner-id` - The owner of the
#'     Elastic IP address (IPv4) associated with the network interface.
#' 
#' -   `network-interface.association.allocation-id` - The allocation ID
#'     returned when you allocated the Elastic IP address (IPv4) for your
#'     network interface.
#' 
#' -   `network-interface.association.association-id` - The association ID
#'     returned when the network interface was associated with an IPv4
#'     address.
#' 
#' -   `network-interface.attachment.attachment-id` - The ID of the
#'     interface attachment.
#' 
#' -   `network-interface.attachment.instance-id` - The ID of the instance
#'     to which the network interface is attached.
#' 
#' -   `network-interface.attachment.instance-owner-id` - The owner ID of
#'     the instance to which the network interface is attached.
#' 
#' -   `network-interface.attachment.device-index` - The device index to
#'     which the network interface is attached.
#' 
#' -   `network-interface.attachment.status` - The status of the attachment
#'     (`attaching` \| `attached` \| `detaching` \| `detached`).
#' 
#' -   `network-interface.attachment.attach-time` - The time that the
#'     network interface was attached to an instance.
#' 
#' -   `network-interface.attachment.delete-on-termination` - Specifies
#'     whether the attachment is deleted when an instance is terminated.
#' 
#' -   `network-interface.availability-zone` - The Availability Zone for
#'     the network interface.
#' 
#' -   `network-interface.description` - The description of the network
#'     interface.
#' 
#' -   `network-interface.group-id` - The ID of a security group associated
#'     with the network interface.
#' 
#' -   `network-interface.group-name` - The name of a security group
#'     associated with the network interface.
#' 
#' -   `network-interface.ipv6-addresses.ipv6-address` - The IPv6 address
#'     associated with the network interface.
#' 
#' -   `network-interface.mac-address` - The MAC address of the network
#'     interface.
#' 
#' -   `network-interface.network-interface-id` - The ID of the network
#'     interface.
#' 
#' -   `network-interface.owner-id` - The ID of the owner of the network
#'     interface.
#' 
#' -   `network-interface.private-dns-name` - The private DNS name of the
#'     network interface.
#' 
#' -   `network-interface.requester-id` - The requester ID for the network
#'     interface.
#' 
#' -   `network-interface.requester-managed` - Indicates whether the
#'     network interface is being managed by AWS.
#' 
#' -   `network-interface.status` - The status of the network interface
#'     (`available`) \| `in-use`).
#' 
#' -   `network-interface.source-dest-check` - Whether the network
#'     interface performs source/destination checking. A value of `true`
#'     means that checking is enabled, and `false` means that checking is
#'     disabled. The value must be `false` for the network interface to
#'     perform network address translation (NAT) in your VPC.
#' 
#' -   `network-interface.subnet-id` - The ID of the subnet for the network
#'     interface.
#' 
#' -   `network-interface.vpc-id` - The ID of the VPC for the network
#'     interface.
#' 
#' -   `owner-id` - The AWS account ID of the instance owner.
#' 
#' -   `placement-group-name` - The name of the placement group for the
#'     instance.
#' 
#' -   `placement-partition-number` - The partition in which the instance
#'     is located.
#' 
#' -   `platform` - The platform. To list only Windows instances, use
#'     `windows`.
#' 
#' -   `private-dns-name` - The private IPv4 DNS name of the instance.
#' 
#' -   `private-ip-address` - The private IPv4 address of the instance.
#' 
#' -   `product-code` - The product code associated with the AMI used to
#'     launch the instance.
#' 
#' -   `product-code.type` - The type of product code (`devpay` \|
#'     `marketplace`).
#' 
#' -   `ramdisk-id` - The RAM disk ID.
#' 
#' -   `reason` - The reason for the current state of the instance (for
#'     example, shows \'User Initiated \[date\]\' when you stop or
#'     terminate the instance). Similar to the state-reason-code filter.
#' 
#' -   `requester-id` - The ID of the entity that launched the instance on
#'     your behalf (for example, AWS Management Console, Auto Scaling, and
#'     so on).
#' 
#' -   `reservation-id` - The ID of the instance\'s reservation. A
#'     reservation ID is created any time you launch an instance. A
#'     reservation ID has a one-to-one relationship with an instance launch
#'     request, but can be associated with more than one instance if you
#'     launch multiple instances using the same launch request. For
#'     example, if you launch one instance, you get one reservation ID. If
#'     you launch ten instances using the same launch request, you also get
#'     one reservation ID.
#' 
#' -   `root-device-name` - The device name of the root device volume (for
#'     example, `/dev/sda1`).
#' 
#' -   `root-device-type` - The type of the root device volume (`ebs` \|
#'     `instance-store`).
#' 
#' -   `source-dest-check` - Indicates whether the instance performs
#'     source/destination checking. A value of `true` means that checking
#'     is enabled, and `false` means that checking is disabled. The value
#'     must be `false` for the instance to perform network address
#'     translation (NAT) in your VPC.
#' 
#' -   `spot-instance-request-id` - The ID of the Spot Instance request.
#' 
#' -   `state-reason-code` - The reason code for the state change.
#' 
#' -   `state-reason-message` - A message that describes the state change.
#' 
#' -   `subnet-id` - The ID of the subnet for the instance.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources that have a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `tenancy` - The tenancy of an instance (`dedicated` \| `default` \|
#'     `host`).
#' 
#' -   `virtualization-type` - The virtualization type of the instance
#'     (`paravirtual` \| `hvm`).
#' 
#' -   `vpc-id` - The ID of the VPC that the instance is running in.
#' 
#' @section InstanceId:
#' 
#' The instance IDs.
#' 
#' Default: Describes all your instances.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of
#' results to return in a single call. To retrieve the remaining results,
#' make another call with the returned `NextToken` value. This value can be
#' between 5 and 1000. You cannot specify this parameter and the instance
#' IDs parameter in the same call.
#' @return A list object or a character vector
#' @export
ec2_describe_instances <- function(Filter = NULL, InstanceId = NULL, DryRun = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInstances", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Internet Gateways
#' 
#' Describes one or more of your internet
#' gateways.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InternetGatewayId List.   One or more internet gateway IDs.  Default: Describes all your internet gateways.  \[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `attachment.state` - The current state of the attachment between the
#'     gateway and the VPC (`available`). Present only if a VPC is
#'     attached.
#' 
#' -   `attachment.vpc-id` - The ID of an attached VPC.
#' 
#' -   `internet-gateway-id` - The ID of the Internet gateway.
#' 
#' -   `owner-id` - The ID of the AWS account that owns the internet
#'     gateway.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section InternetGatewayId:
#' 
#' One or more internet gateway IDs.
#' 
#' Default: Describes all your internet gateways.
#' 
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_internet_gateways <- function(Filter = NULL, DryRun = NULL, InternetGatewayId = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    InternetGatewayId <- list_to_array("InternetGatewayId", InternetGatewayId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, InternetGatewayId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeInternetGateways", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Ipv6 Pools
#' 
#' Describes your IPv6 address
#' pools.
#' 
#' @param PoolId List. The IDs of the IPv6 address pools.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PoolId:
#' The IDs of the
#' IPv6 address pools.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_ipv6_pools <- function(PoolId = NULL, NextToken = NULL, MaxResults = NULL, DryRun = NULL, 
    Filter = NULL, simplify = TRUE, others = list()) {
    PoolId <- list_to_array("PoolId", PoolId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(NextToken = NextToken, MaxResults = MaxResults, DryRun = DryRun), 
        PoolId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeIpv6Pools", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Key Pairs
#' 
#' @param KeyName List.   The key pair names.  Default: Describes all your key pairs.  \[optional\]
#' @param KeyPairId List. The IDs of the key pairs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `key-pair-id` - The ID of the key pair.
#' 
#' -   `fingerprint` - The fingerprint of the key pair.
#' 
#' -   `key-name` - The name of the key pair.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' @section KeyName:
#' 
#' The key pair names.
#' 
#' Default: Describes all your key pairs.
#' 
#' @section KeyPairId:
#' The IDs of the key
#' pairs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_key_pairs <- function(Filter = NULL, KeyName = NULL, KeyPairId = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    KeyName <- list_to_array("KeyName", KeyName)
    KeyPairId <- list_to_array("KeyPairId", KeyPairId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), Filter, KeyName, KeyPairId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeKeyPairs", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Launch Template Versions
#' 
#' Describes one or more versions of a specified
#' launch template. You can describe all versions, individual versions, or
#' a range of versions. You can also describe all the latest versions or
#' all the default versions of all the launch templates in your
#' account.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LaunchTemplateId String. The ID of the launch template.\[optional\]
#' @param LaunchTemplateName String. The name of the launch template.\[optional\]
#' @param LaunchTemplateVersion List.   One or more versions of the launch template.\[optional\]
#' @param MinVersion String. The version number after which to describe launch template versions.\[optional\]
#' @param MaxVersion String. The version number up to which to describe launch template versions.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section LaunchTemplateId:
#' The ID of the launch
#' template. To describe one or more versions of a specified launch
#' template, you must specify either the launch template ID or the launch
#' template name in the request. To describe all the latest or default
#' launch template versions in your account, you must omit this
#' parameter.
#' @section LaunchTemplateName:
#' The name of the launch template. To
#' describe one or more versions of a specified launch template, you must
#' specify either the launch template ID or the launch template name in the
#' request. To describe all the latest or default launch template versions
#' in your account, you must omit this parameter.
#' @section LaunchTemplateVersion:
#' 
#' One or more versions of the launch template. Valid values depend on
#' whether you are describing a specified launch template (by ID or name)
#' or all launch templates in your account.
#' 
#' To describe one or more versions of a specified launch template, valid
#' values are `$Latest`, `$Default`, and numbers.
#' 
#' To describe all launch templates in your account that are defined as the
#' latest version, the valid value is `$Latest`. To describe all launch
#' templates in your account that are defined as the default version, the
#' valid value is `$Default`. You can specify `$Latest` and `$Default` in
#' the same call. You cannot specify numbers.
#' 
#' @section MinVersion:
#' The version number after which to describe launch
#' template versions.
#' @section MaxVersion:
#' The version number up to which to
#' describe launch template versions.
#' @section MaxResults:
#' The maximum number
#' of results to return in a single call. To retrieve the remaining
#' results, make another call with the returned `NextToken` value. This
#' value can be between 1 and 200.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `create-time` - The time the launch template version was created.
#' 
#' -   `ebs-optimized` - A boolean that indicates whether the instance is
#'     optimized for Amazon EBS I/O.
#' 
#' -   `iam-instance-profile` - The ARN of the IAM instance profile.
#' 
#' -   `image-id` - The ID of the AMI.
#' 
#' -   `instance-type` - The instance type.
#' 
#' -   `is-default-version` - A boolean that indicates whether the launch
#'     template version is the default version.
#' 
#' -   `kernel-id` - The kernel ID.
#' 
#' -   `ram-disk-id` - The RAM disk ID.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_launch_template_versions <- function(DryRun = NULL, LaunchTemplateId = NULL, 
    LaunchTemplateName = NULL, LaunchTemplateVersion = NULL, MinVersion = NULL, MaxVersion = NULL, 
    NextToken = NULL, MaxResults = NULL, Filter = NULL, simplify = TRUE, others = list()) {
    LaunchTemplateVersion <- list_to_array("LaunchTemplateVersion", LaunchTemplateVersion)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, LaunchTemplateId = LaunchTemplateId, LaunchTemplateName = LaunchTemplateName, 
        MinVersion = MinVersion, MaxVersion = MaxVersion, NextToken = NextToken, MaxResults = MaxResults), 
        LaunchTemplateVersion, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLaunchTemplateVersions", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Launch Templates
#' 
#' Describes one or more launch
#' templates.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LaunchTemplateId List. One or more launch template IDs.\[optional\]
#' @param LaunchTemplateName List. One or more launch template names.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section LaunchTemplateId:
#' One or more launch template
#' IDs.
#' @section LaunchTemplateName:
#' One or more launch template
#' names.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `create-time` - The time the launch template was created.
#' 
#' -   `launch-template-name` - The name of the launch template.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section MaxResults:
#' The maximum number of results to return in
#' a single call. To retrieve the remaining results, make another call with
#' the returned `NextToken` value. This value can be between 1 and
#' 200.
#' @return A list object or a character vector
#' @export
ec2_describe_launch_templates <- function(DryRun = NULL, LaunchTemplateId = NULL, LaunchTemplateName = NULL, 
    Filter = NULL, NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    LaunchTemplateId <- list_to_array("LaunchTemplateId", LaunchTemplateId)
    LaunchTemplateName <- list_to_array("LaunchTemplateName", LaunchTemplateName)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        LaunchTemplateId, LaunchTemplateName, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLaunchTemplates", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Local Gateway Route Table Virtual Interface Group Associations
#' 
#' Describes the associations between
#' virtual interface groups and local gateway route
#' tables.
#' 
#' @param LocalGatewayRouteTableVirtualInterfaceGroupAssociationId List. The IDs of the associations.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayRouteTableVirtualInterfaceGroupAssociationId:
#' The IDs of the
#' associations.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `local-gateway-id` - The ID of a local gateway.
#' 
#' -   `local-gateway-route-table-id` - The ID of the local gateway route
#'     table.
#' 
#' -   `local-gateway-route-table-virtual-interface-group-association-id` -
#'     The ID of the association.
#' 
#' -   `local-gateway-route-table-virtual-interface-group-id` - The ID of
#'     the virtual interface group.
#' 
#' -   `state` - The state of the association.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_local_gateway_route_table_virtual_interface_group_associations <- function(LocalGatewayRouteTableVirtualInterfaceGroupAssociationId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    LocalGatewayRouteTableVirtualInterfaceGroupAssociationId <- list_to_array("LocalGatewayRouteTableVirtualInterfaceGroupAssociationId", 
        LocalGatewayRouteTableVirtualInterfaceGroupAssociationId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        LocalGatewayRouteTableVirtualInterfaceGroupAssociationId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Local Gateway Route Table Vpc Associations
#' 
#' Describes the specified associations between
#' VPCs and local gateway route tables.
#' 
#' @param LocalGatewayRouteTableVpcAssociationId List. The IDs of the associations.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayRouteTableVpcAssociationId:
#' The IDs of the
#' associations.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `local-gateway-id` - The ID of a local gateway.
#' 
#' -   `local-gateway-route-table-id` - The ID of the local gateway route
#'     table.
#' 
#' -   `local-gateway-route-table-vpc-association-id` - The ID of the
#'     association.
#' 
#' -   `state` - The state of the association.
#' 
#' -   `vpc-id` - The ID of the VPC.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_local_gateway_route_table_vpc_associations <- function(LocalGatewayRouteTableVpcAssociationId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    LocalGatewayRouteTableVpcAssociationId <- list_to_array("LocalGatewayRouteTableVpcAssociationId", 
        LocalGatewayRouteTableVpcAssociationId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        LocalGatewayRouteTableVpcAssociationId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLocalGatewayRouteTableVpcAssociations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Local Gateway Route Tables
#' 
#' Describes one
#' or more local gateway route tables. By default, all local gateway route
#' tables are described. Alternatively, you can filter the
#' results.
#' 
#' @param LocalGatewayRouteTableId List. The IDs of the local gateway route tables.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayRouteTableId:
#' The IDs of the local
#' gateway route tables.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `local-gateway-id` - The ID of a local gateway.
#' 
#' -   `local-gateway-route-table-id` - The ID of a local gateway route
#'     table.
#' 
#' -   `outpost-arn` - The Amazon Resource Name (ARN) of the Outpost.
#' 
#' -   `state` - The state of the local gateway route table.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_local_gateway_route_tables <- function(LocalGatewayRouteTableId = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    LocalGatewayRouteTableId <- list_to_array("LocalGatewayRouteTableId", LocalGatewayRouteTableId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        LocalGatewayRouteTableId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLocalGatewayRouteTables", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Local Gateway Virtual Interface Groups
#' 
#' Describes the specified local gateway
#' virtual interface groups.
#' 
#' @param LocalGatewayVirtualInterfaceGroupId List. The IDs of the virtual interface groups.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayVirtualInterfaceGroupId:
#' The IDs of the virtual
#' interface groups.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `local-gateway-id` - The ID of a local gateway.
#' 
#' -   `local-gateway-virtual-interface-id` - The ID of the virtual
#'     interface.
#' 
#' -   `local-gateway-virtual-interface-group-id` - The ID of the virtual
#'     interface group.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_local_gateway_virtual_interface_groups <- function(LocalGatewayVirtualInterfaceGroupId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    LocalGatewayVirtualInterfaceGroupId <- list_to_array("LocalGatewayVirtualInterfaceGroupId", 
        LocalGatewayVirtualInterfaceGroupId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        LocalGatewayVirtualInterfaceGroupId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLocalGatewayVirtualInterfaceGroups", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Local Gateway Virtual Interfaces
#' 
#' Describes the specified
#' local gateway virtual interfaces.
#' 
#' @param LocalGatewayVirtualInterfaceId List. The IDs of the virtual interfaces.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayVirtualInterfaceId:
#' The IDs of the virtual
#' interfaces.
#' @section Filter:
#' One or more
#' filters.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_local_gateway_virtual_interfaces <- function(LocalGatewayVirtualInterfaceId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    LocalGatewayVirtualInterfaceId <- list_to_array("LocalGatewayVirtualInterfaceId", LocalGatewayVirtualInterfaceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        LocalGatewayVirtualInterfaceId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLocalGatewayVirtualInterfaces", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Local Gateways
#' 
#' Describes one or
#' more local gateways. By default, all local gateways are described.
#' Alternatively, you can filter the results.
#' 
#' @param LocalGatewayId List.   One or more filters.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayId:
#' 
#' One or more filters.
#' 
#' -   `local-gateway-id` - The ID of a local gateway.
#' 
#' -   `local-gateway-route-table-id` - The ID of the local gateway route
#'     table.
#' 
#' -   `local-gateway-route-table-virtual-interface-group-association-id` -
#'     The ID of the association.
#' 
#' -   `local-gateway-route-table-virtual-interface-group-id` - The ID of
#'     the virtual interface group.
#' 
#' -   `outpost-arn` - The Amazon Resource Name (ARN) of the Outpost.
#' 
#' -   `state` - The state of the association.
#' 
#' @section Filter:
#' One or more filters.
#' @section MaxResults:
#' The
#' maximum number of results to return with a single call. To retrieve the
#' remaining results, make another call with the returned `nextToken`
#' value.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_local_gateways <- function(LocalGatewayId = NULL, Filter = NULL, MaxResults = NULL, 
    NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    LocalGatewayId <- list_to_array("LocalGatewayId", LocalGatewayId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        LocalGatewayId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeLocalGateways", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Managed Prefix Lists
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param PrefixListId List. One or more prefix list IDs.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `owner-id` - The ID of the prefix list owner.
#' 
#' -   `prefix-list-id` - The ID of the prefix list.
#' 
#' -   `prefix-list-name` - The name of the prefix list.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section PrefixListId:
#' One or more prefix list
#' IDs.
#' @return A list object or a character vector
#' @export
ec2_describe_managed_prefix_lists <- function(DryRun = NULL, Filter = NULL, MaxResults = NULL, 
    NextToken = NULL, PrefixListId = NULL, simplify = TRUE, others = list()) {
    PrefixListId <- list_to_array("PrefixListId", PrefixListId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, PrefixListId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeManagedPrefixLists", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Moving Addresses
#' 
#' Describes your Elastic IP addresses that are being
#' moved to the EC2-VPC platform, or that are being restored to the
#' EC2-Classic platform. This request does not return information about any
#' other Elastic IP addresses in your
#' account.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer.   The maximum number of results to return for the request in a single page.\[optional\]
#' @param PublicIp List. One or more Elastic IP addresses.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `moving-status` - The status of the Elastic IP address
#'     (`MovingToVpc` \| `RestoringToClassic`).
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' 
#' The maximum number of results to return for the request in a single
#' page. The remaining results of the initial request can be seen by
#' sending another request with the returned `NextToken` value. This value
#' can be between 5 and 1000; if `MaxResults` is given a value outside of
#' this range, an error is returned.
#' 
#' Default: If no value is provided, the default is 1000.
#' 
#' @section PublicIp:
#' One or more Elastic IP
#' addresses.
#' @return A list object or a character vector
#' @export
ec2_describe_moving_addresses <- function(Filter = NULL, DryRun = NULL, MaxResults = NULL, 
    NextToken = NULL, PublicIp = NULL, simplify = TRUE, others = list()) {
    PublicIp <- list_to_array("PublicIp", PublicIp)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, PublicIp)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeMovingAddresses", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Nat Gateways
#' 
#' Describes one or more of your NAT
#' gateways.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param NatGatewayId List. One or more NAT gateway IDs.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `nat-gateway-id` - The ID of the NAT gateway.
#' 
#' -   `state` - The state of the NAT gateway (`pending` \| `failed` \|
#'     `available` \| `deleting` \| `deleted`).
#' 
#' -   `subnet-id` - The ID of the subnet in which the NAT gateway resides.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC in which the NAT gateway resides.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section NatGatewayId:
#' One or more NAT
#' gateway IDs.
#' @return A list object or a character vector
#' @export
ec2_describe_nat_gateways <- function(DryRun = NULL, Filter = NULL, MaxResults = NULL, NatGatewayId = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    NatGatewayId <- list_to_array("NatGatewayId", NatGatewayId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, NatGatewayId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeNatGateways", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Network Acls
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param NetworkAclId List.   One or more network ACL IDs.  Default: Describes all your network ACLs.  \[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `association.association-id` - The ID of an association ID for the
#'     ACL.
#' 
#' -   `association.network-acl-id` - The ID of the network ACL involved in
#'     the association.
#' 
#' -   `association.subnet-id` - The ID of the subnet involved in the
#'     association.
#' 
#' -   `default` - Indicates whether the ACL is the default network ACL for
#'     the VPC.
#' 
#' -   `entry.cidr` - The IPv4 CIDR range specified in the entry.
#' 
#' -   `entry.icmp.code` - The ICMP code specified in the entry, if any.
#' 
#' -   `entry.icmp.type` - The ICMP type specified in the entry, if any.
#' 
#' -   `entry.ipv6-cidr` - The IPv6 CIDR range specified in the entry.
#' 
#' -   `entry.port-range.from` - The start of the port range specified in
#'     the entry.
#' 
#' -   `entry.port-range.to` - The end of the port range specified in the
#'     entry.
#' 
#' -   `entry.protocol` - The protocol specified in the entry (`tcp` \|
#'     `udp` \| `icmp` or a protocol number).
#' 
#' -   `entry.rule-action` - Allows or denies the matching traffic (`allow`
#'     \| `deny`).
#' 
#' -   `entry.rule-number` - The number of an entry (in other words, rule)
#'     in the set of ACL entries.
#' 
#' -   `network-acl-id` - The ID of the network ACL.
#' 
#' -   `owner-id` - The ID of the AWS account that owns the network ACL.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC for the network ACL.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section NetworkAclId:
#' 
#' One or more network ACL IDs.
#' 
#' Default: Describes all your network ACLs.
#' 
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_network_acls <- function(Filter = NULL, DryRun = NULL, NetworkAclId = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    NetworkAclId <- list_to_array("NetworkAclId", NetworkAclId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, NetworkAclId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeNetworkAcls", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Network Insights Analyses
#' 
#' Describes one or more of your network insights
#' analyses.
#' 
#' @param NetworkInsightsAnalysisId List. The ID of the network insights analyses. You must specify either analysis IDs or a path ID.\[optional\]
#' @param NetworkInsightsPathId String. The ID of the path. You must specify either a path ID or analysis IDs.\[optional\]
#' @param AnalysisStartTime String. The time when the network insights analyses started.\[optional\]
#' @param AnalysisEndTime String. The time when the network insights analyses ended.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInsightsAnalysisId:
#' The ID of the
#' network insights analyses. You must specify either analysis IDs or a
#' path ID.
#' @section NetworkInsightsPathId:
#' The ID of the path. You must specify either
#' a path ID or analysis IDs.
#' @section AnalysisStartTime:
#' The time when the network
#' insights analyses started.
#' @section AnalysisEndTime:
#' The time when the network
#' insights analyses ended.
#' @section Filter:
#' 
#' The filters. The following are possible values:
#' 
#' -   PathFound - A Boolean value that indicates whether a feasible path
#'     is found.
#' 
#' -   Status - The status of the analysis (running \| succeeded \|
#'     failed).
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you
#' have the required permissions for the action, without actually making
#' the request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_network_insights_analyses <- function(NetworkInsightsAnalysisId = NULL, NetworkInsightsPathId = NULL, 
    AnalysisStartTime = NULL, AnalysisEndTime = NULL, Filter = NULL, MaxResults = NULL, DryRun = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    NetworkInsightsAnalysisId <- list_to_array("NetworkInsightsAnalysisId", NetworkInsightsAnalysisId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(NetworkInsightsPathId = NetworkInsightsPathId, AnalysisStartTime = AnalysisStartTime, 
        AnalysisEndTime = AnalysisEndTime, MaxResults = MaxResults, DryRun = DryRun, NextToken = NextToken), 
        NetworkInsightsAnalysisId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeNetworkInsightsAnalyses", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Network Insights Paths
#' 
#' Describes one or more of your
#' paths.
#' 
#' @param NetworkInsightsPathId List. The IDs of the paths.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInsightsPathId:
#' The IDs of the
#' paths.
#' @section Filter:
#' 
#' The filters. The following are possible values:
#' 
#' -   Destination - The ID of the resource.
#' 
#' -   DestinationPort - The destination port.
#' 
#' -   Name - The path name.
#' 
#' -   Protocol - The protocol.
#' 
#' -   Source - The ID of the resource.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you
#' have the required permissions for the action, without actually making
#' the request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_network_insights_paths <- function(NetworkInsightsPathId = NULL, Filter = NULL, 
    MaxResults = NULL, DryRun = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    NetworkInsightsPathId <- list_to_array("NetworkInsightsPathId", NetworkInsightsPathId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, DryRun = DryRun, NextToken = NextToken), 
        NetworkInsightsPathId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeNetworkInsightsPaths", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Network Interface Attribute
#' 
#' Describes a network interface attribute. You
#' can specify only one attribute at a time.
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param Attribute String. The attribute of the network interface. This parameter is required.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section Attribute:
#' The attribute of the network interface.
#' This parameter is required.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_network_interface_attribute <- function(NetworkInterfaceId, Attribute = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, Attribute = Attribute, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeNetworkInterfaceAttribute", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Describe Network Interface Permissions
#' 
#' Describes
#' the permissions for your network interfaces.
#' 
#' @param NetworkInterfacePermissionId List. One or more network interface permission IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfacePermissionId:
#' One or more network
#' interface permission IDs.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `network-interface-permission.network-interface-permission-id` - The
#'     ID of the permission.
#' 
#' -   `network-interface-permission.network-interface-id` - The ID of the
#'     network interface.
#' 
#' -   `network-interface-permission.aws-account-id` - The AWS account ID.
#' 
#' -   `network-interface-permission.aws-service` - The AWS service.
#' 
#' -   `network-interface-permission.permission` - The type of permission
#'     (`INSTANCE-ATTACH` \| `EIP-ASSOCIATE`).
#' 
#' @section MaxResults:
#' The maximum number of results to return in
#' a single call. To retrieve the remaining results, make another call with
#' the returned `NextToken` value. If this parameter is not specified, up
#' to 50 results are returned by default.
#' @return A list object or a character vector
#' @export
ec2_describe_network_interface_permissions <- function(NetworkInterfacePermissionId = NULL, 
    Filter = NULL, NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    NetworkInterfacePermissionId <- list_to_array("NetworkInterfacePermissionId", NetworkInterfacePermissionId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(NextToken = NextToken, MaxResults = MaxResults), NetworkInterfacePermissionId, 
        Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeNetworkInterfacePermissions", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Network Interfaces
#' 
#' Describes one or more of your network
#' interfaces.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param NetworkInterfaceId List.   One or more network interface IDs.  Default: Describes all your network interfaces.  \[optional\]
#' @param MaxResults Integer. The maximum number of items to return for this request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `addresses.private-ip-address` - The private IPv4 addresses
#'     associated with the network interface.
#' 
#' -   `addresses.primary` - Whether the private IPv4 address is the
#'     primary IP address associated with the network interface.
#' 
#' -   `addresses.association.public-ip` - The association ID returned when
#'     the network interface was associated with the Elastic IP address
#'     (IPv4).
#' 
#' -   `addresses.association.owner-id` - The owner ID of the addresses
#'     associated with the network interface.
#' 
#' -   `association.association-id` - The association ID returned when the
#'     network interface was associated with an IPv4 address.
#' 
#' -   `association.allocation-id` - The allocation ID returned when you
#'     allocated the Elastic IP address (IPv4) for your network interface.
#' 
#' -   `association.ip-owner-id` - The owner of the Elastic IP address
#'     (IPv4) associated with the network interface.
#' 
#' -   `association.public-ip` - The address of the Elastic IP address
#'     (IPv4) bound to the network interface.
#' 
#' -   `association.public-dns-name` - The public DNS name for the network
#'     interface (IPv4).
#' 
#' -   `attachment.attachment-id` - The ID of the interface attachment.
#' 
#' -   `attachment.attach-time` - The time that the network interface was
#'     attached to an instance.
#' 
#' -   `attachment.delete-on-termination` - Indicates whether the
#'     attachment is deleted when an instance is terminated.
#' 
#' -   `attachment.device-index` - The device index to which the network
#'     interface is attached.
#' 
#' -   `attachment.instance-id` - The ID of the instance to which the
#'     network interface is attached.
#' 
#' -   `attachment.instance-owner-id` - The owner ID of the instance to
#'     which the network interface is attached.
#' 
#' -   `attachment.status` - The status of the attachment (`attaching` \|
#'     `attached` \| `detaching` \| `detached`).
#' 
#' -   `availability-zone` - The Availability Zone of the network
#'     interface.
#' 
#' -   `description` - The description of the network interface.
#' 
#' -   `group-id` - The ID of a security group associated with the network
#'     interface.
#' 
#' -   `group-name` - The name of a security group associated with the
#'     network interface.
#' 
#' -   `ipv6-addresses.ipv6-address` - An IPv6 address associated with the
#'     network interface.
#' 
#' -   `mac-address` - The MAC address of the network interface.
#' 
#' -   `network-interface-id` - The ID of the network interface.
#' 
#' -   `owner-id` - The AWS account ID of the network interface owner.
#' 
#' -   `private-ip-address` - The private IPv4 address or addresses of the
#'     network interface.
#' 
#' -   `private-dns-name` - The private DNS name of the network interface
#'     (IPv4).
#' 
#' -   `requester-id` - The alias or AWS account ID of the principal or
#'     service that created the network interface.
#' 
#' -   `requester-managed` - Indicates whether the network interface is
#'     being managed by an AWS service (for example, AWS Management
#'     Console, Auto Scaling, and so on).
#' 
#' -   `source-dest-check` - Indicates whether the network interface
#'     performs source/destination checking. A value of `true` means
#'     checking is enabled, and `false` means checking is disabled. The
#'     value must be `false` for the network interface to perform network
#'     address translation (NAT) in your VPC.
#' 
#' -   `status` - The status of the network interface. If the network
#'     interface is not attached to an instance, the status is `available`;
#'     if a network interface is attached to an instance the status is
#'     `in-use`.
#' 
#' -   `subnet-id` - The ID of the subnet for the network interface.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC for the network interface.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section NetworkInterfaceId:
#' 
#' One or more network interface IDs.
#' 
#' Default: Describes all your network interfaces.
#' 
#' @section MaxResults:
#' The maximum number of items to return for
#' this request. The request returns a token that you can specify in a
#' subsequent call to get the next set of results. You cannot specify this
#' parameter and the network interface IDs parameter in the same
#' request.
#' @return A list object or a character vector
#' @export
ec2_describe_network_interfaces <- function(Filter = NULL, DryRun = NULL, NetworkInterfaceId = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    NetworkInterfaceId <- list_to_array("NetworkInterfaceId", NetworkInterfaceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, NetworkInterfaceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeNetworkInterfaces", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Placement Groups
#' 
#' Describes the specified placement groups
#' or all of your placement groups. For more information, see [Placement
#' groups](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param GroupName List.   The names of the placement groups.\[optional\]
#' @param GroupId List. The IDs of the placement groups.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `group-name` - The name of the placement group.
#' 
#' -   `state` - The state of the placement group (`pending` \| `available`
#'     \| `deleting` \| `deleted`).
#' 
#' -   `strategy` - The strategy of the placement group (`cluster` \|
#'     `spread` \| `partition`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources that have a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section GroupName:
#' 
#' The names of the placement groups.
#' 
#' Default: Describes all your placement groups, or only those otherwise
#' specified.
#' 
#' @section GroupId:
#' The IDs of the placement
#' groups.
#' @return A list object or a character vector
#' @export
ec2_describe_placement_groups <- function(Filter = NULL, DryRun = NULL, GroupName = NULL, 
    GroupId = NULL, simplify = TRUE, others = list()) {
    GroupName <- list_to_array("GroupName", GroupName)
    GroupId <- list_to_array("GroupId", GroupId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), Filter, GroupName, GroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribePlacementGroups", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Prefix Lists
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param PrefixListId List. One or more prefix list IDs.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `prefix-list-id`: The ID of a prefix list.
#' 
#' -   `prefix-list-name`: The name of a prefix list.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section PrefixListId:
#' One or more prefix list
#' IDs.
#' @return A list object or a character vector
#' @export
ec2_describe_prefix_lists <- function(DryRun = NULL, Filter = NULL, MaxResults = NULL, NextToken = NULL, 
    PrefixListId = NULL, simplify = TRUE, others = list()) {
    PrefixListId <- list_to_array("PrefixListId", PrefixListId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, PrefixListId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribePrefixLists", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Principal Id Format
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Resource List. The type of resource: `bundle` \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \| `elastic-ip-allocation`...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Resource:
#' The type of resource:
#' `bundle` \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \|
#' `elastic-ip-allocation` \| `elastic-ip-association` \| `export-task` \|
#' `flow-log` \| `image` \| `import-task` \| `instance` \|
#' `internet-gateway` \| `network-acl` \| `network-acl-association` \|
#' `network-interface` \| `network-interface-attachment` \| `prefix-list`
#' \| `reservation` \| `route-table` \| `route-table-association` \|
#' `security-group` \| `snapshot` \| `subnet` \|
#' `subnet-cidr-block-association` \| `volume` \| `vpc` \|
#' `vpc-cidr-block-association` \| `vpc-endpoint` \|
#' `vpc-peering-connection` \| `vpn-connection` \| `vpn-gateway`
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. To retrieve the remaining results, make another call with the
#' returned NextToken value. 
#' @return A list object or a character vector
#' @export
ec2_describe_principal_id_format <- function(DryRun = NULL, Resource = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    Resource <- list_to_array("Resource", Resource)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Resource)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribePrincipalIdFormat", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Public Ipv4 Pools
#' 
#' Describes the specified IPv4 address
#' pools.
#' 
#' @param PoolId List. The IDs of the address pools.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PoolId:
#' The IDs of the address
#' pools.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_public_ipv4_pools <- function(PoolId = NULL, NextToken = NULL, MaxResults = NULL, 
    Filter = NULL, simplify = TRUE, others = list()) {
    PoolId <- list_to_array("PoolId", PoolId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(NextToken = NextToken, MaxResults = MaxResults), PoolId, 
        Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribePublicIpv4Pools", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Regions
#' 
#' @param RegionName List. The names of the Regions.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param AllRegions Logical. Indicates whether to display all Regions, including Regions that are disabled for your account.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `endpoint` - The endpoint of the Region (for example,
#'     `ec2.us-east-1.amazonaws.com`).
#' 
#' -   `opt-in-status` - The opt-in status of the Region
#'     (`opt-in-not-required` \| `opted-in` \| `not-opted-in`).
#' 
#' -   `region-name` - The name of the Region (for example, `us-east-1`).
#' 
#' @section RegionName:
#' The names of the Regions. You can specify any
#' Regions, whether they are enabled and disabled for your
#' account.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section AllRegions:
#' Indicates whether to
#' display all Regions, including Regions that are disabled for your
#' account.
#' @return A list object or a character vector
#' @export
ec2_describe_regions <- function(Filter = NULL, RegionName = NULL, DryRun = NULL, AllRegions = NULL, 
    simplify = TRUE, others = list()) {
    RegionName <- list_to_array("RegionName", RegionName)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, AllRegions = AllRegions), Filter, RegionName)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeRegions", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Reserved Instances
#' 
#' @param OfferingClass String. Describes whether the Reserved Instance is Standard or Convertible.\[optional\]
#' @param ReservedInstancesId List.   One or more Reserved Instance IDs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param OfferingType String. The Reserved Instance offering type.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `availability-zone` - The Availability Zone where the Reserved
#'     Instance can be used.
#' 
#' -   `duration` - The duration of the Reserved Instance (one year or
#'     three years), in seconds (`31536000` \| `94608000`).
#' 
#' -   `end` - The time when the Reserved Instance expires (for example,
#'     2015-08-07T11:54:42.000Z).
#' 
#' -   `fixed-price` - The purchase price of the Reserved Instance (for
#'     example, 9800.0).
#' 
#' -   `instance-type` - The instance type that is covered by the
#'     reservation.
#' 
#' -   `scope` - The scope of the Reserved Instance (`Region` or
#'     `Availability Zone`).
#' 
#' -   `product-description` - The Reserved Instance product platform
#'     description. Instances that include `(Amazon VPC)` in the product
#'     platform description will only be displayed to EC2-Classic account
#'     holders and are for use with Amazon VPC (`Linux/UNIX` \|
#'     `Linux/UNIX (Amazon VPC)` \| `SUSE Linux` \|
#'     `SUSE Linux (Amazon VPC)` \| `Red Hat Enterprise Linux` \|
#'     `Red Hat Enterprise Linux (Amazon VPC)` \| `Windows` \|
#'     `Windows (Amazon VPC)` \| `Windows with SQL Server Standard` \|
#'     `Windows with SQL Server Standard (Amazon VPC)` \|
#'     `Windows with SQL Server Web` \|
#'     `Windows with SQL Server Web (Amazon VPC)` \|
#'     `Windows with SQL Server Enterprise` \|
#'     `Windows with SQL Server Enterprise (Amazon VPC)`).
#' 
#' -   `reserved-instances-id` - The ID of the Reserved Instance.
#' 
#' -   `start` - The time at which the Reserved Instance purchase request
#'     was placed (for example, 2014-08-07T11:54:42.000Z).
#' 
#' -   `state` - The state of the Reserved Instance (`payment-pending` \|
#'     `active` \| `payment-failed` \| `retired`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `usage-price` - The usage price of the Reserved Instance, per hour
#'     (for example, 0.84).
#' 
#' @section OfferingClass:
#' Describes whether the Reserved Instance is Standard
#' or Convertible.
#' @section ReservedInstancesId:
#' 
#' One or more Reserved Instance IDs.
#' 
#' Default: Describes all your Reserved Instances, or only those otherwise
#' specified.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section OfferingType:
#' The Reserved Instance
#' offering type. If you are using tools that predate the 2011-11-01 API
#' version, you only have access to the `Medium Utilization` Reserved
#' Instance offering type.
#' @return A list object or a character vector
#' @export
ec2_describe_reserved_instances <- function(Filter = NULL, OfferingClass = NULL, ReservedInstancesId = NULL, 
    DryRun = NULL, OfferingType = NULL, simplify = TRUE, others = list()) {
    ReservedInstancesId <- list_to_array("ReservedInstancesId", ReservedInstancesId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(OfferingClass = OfferingClass, DryRun = DryRun, OfferingType = OfferingType), 
        Filter, ReservedInstancesId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeReservedInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Reserved Instances Listings
#' 
#' @param ReservedInstancesId String. One or more Reserved Instance IDs.\[optional\]
#' @param ReservedInstancesListingId String. One or more Reserved Instance listing IDs.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `reserved-instances-id` - The ID of the Reserved Instances.
#' 
#' -   `reserved-instances-listing-id` - The ID of the Reserved Instances
#'     listing.
#' 
#' -   `status` - The status of the Reserved Instance listing (`pending` \|
#'     `active` \| `cancelled` \| `closed`).
#' 
#' -   `status-message` - The reason for the status.
#' 
#' @section ReservedInstancesId:
#' One or more Reserved Instance
#' IDs.
#' @section ReservedInstancesListingId:
#' One or more Reserved Instance listing
#' IDs.
#' @return A list object or a character vector
#' @export
ec2_describe_reserved_instances_listings <- function(Filter = NULL, ReservedInstancesId = NULL, 
    ReservedInstancesListingId = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(ReservedInstancesId = ReservedInstancesId, ReservedInstancesListingId = ReservedInstancesListingId), 
        Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeReservedInstancesListings", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Describe Reserved Instances Modifications
#' 
#' @param ReservedInstancesModificationId List. IDs for the submitted modification request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `client-token` - The idempotency token for the modification request.
#' 
#' -   `create-date` - The time when the modification request was created.
#' 
#' -   `effective-date` - The time when the modification becomes effective.
#' 
#' -   `modification-result.reserved-instances-id` - The ID for the
#'     Reserved Instances created as part of the modification request. This
#'     ID is only available when the status of the modification is
#'     `fulfilled`.
#' 
#' -   `modification-result.target-configuration.availability-zone` - The
#'     Availability Zone for the new Reserved Instances.
#' 
#' -   `modification-result.target-configuration.instance-count ` - The
#'     number of new Reserved Instances.
#' 
#' -   `modification-result.target-configuration.instance-type` - The
#'     instance type of the new Reserved Instances.
#' 
#' -   `modification-result.target-configuration.platform` - The network
#'     platform of the new Reserved Instances (`EC2-Classic` \| `EC2-VPC`).
#' 
#' -   `reserved-instances-id` - The ID of the Reserved Instances modified.
#' 
#' -   `reserved-instances-modification-id` - The ID of the modification
#'     request.
#' 
#' -   `status` - The status of the Reserved Instances modification request
#'     (`processing` \| `fulfilled` \| `failed`).
#' 
#' -   `status-message` - The reason for the status.
#' 
#' -   `update-date` - The time when the modification request was last
#'     updated.
#' 
#' @section ReservedInstancesModificationId:
#' IDs for the submitted modification
#' request.
#' @return A list object or a character vector
#' @export
ec2_describe_reserved_instances_modifications <- function(Filter = NULL, ReservedInstancesModificationId = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    ReservedInstancesModificationId <- list_to_array("ReservedInstancesModificationId", ReservedInstancesModificationId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(NextToken = NextToken), Filter, ReservedInstancesModificationId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeReservedInstancesModifications", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Reserved Instances Offerings
#' 
#' @param AvailabilityZone String. The Availability Zone in which the Reserved Instance can be used.\[optional\]
#' @param IncludeMarketplace Logical. Include Reserved Instance Marketplace offerings in the response.\[optional\]
#' @param InstanceType String. The instance type that the reservation will cover (for example, `m1.small`). For more information,...\[optional\]
#' @param MaxDuration Integer.   The maximum duration (in seconds) to filter when searching for offerings.\[optional\]
#' @param MaxInstanceCount Integer.   The maximum number of instances to filter when searching for offerings.  Default: 20  \[optional\]
#' @param MinDuration Integer.   The minimum duration (in seconds) to filter when searching for offerings.\[optional\]
#' @param OfferingClass String. The offering class of the Reserved Instance. Can be `standard` or `convertible`.\[optional\]
#' @param ProductDescription String. The Reserved Instance product platform description.\[optional\]
#' @param ReservedInstancesOfferingId List. One or more Reserved Instances offering IDs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceTenancy String.   The tenancy of the instances covered by the reservation.\[optional\]
#' @param MaxResults Integer.   The maximum number of results to return for the request in a single page.\[optional\]
#' @param OfferingType String. The Reserved Instance offering type.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZone:
#' The Availability Zone in which the Reserved
#' Instance can be used.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `availability-zone` - The Availability Zone where the Reserved
#'     Instance can be used.
#' 
#' -   `duration` - The duration of the Reserved Instance (for example, one
#'     year or three years), in seconds (`31536000` \| `94608000`).
#' 
#' -   `fixed-price` - The purchase price of the Reserved Instance (for
#'     example, 9800.0).
#' 
#' -   `instance-type` - The instance type that is covered by the
#'     reservation.
#' 
#' -   `marketplace` - Set to `true` to show only Reserved Instance
#'     Marketplace offerings. When this filter is not used, which is the
#'     default behavior, all offerings from both AWS and the Reserved
#'     Instance Marketplace are listed.
#' 
#' -   `product-description` - The Reserved Instance product platform
#'     description. Instances that include `(Amazon VPC)` in the product
#'     platform description will only be displayed to EC2-Classic account
#'     holders and are for use with Amazon VPC. (`Linux/UNIX` \|
#'     `Linux/UNIX (Amazon VPC)` \| `SUSE Linux` \|
#'     `SUSE Linux (Amazon VPC)` \| `Red Hat Enterprise Linux` \|
#'     `Red Hat Enterprise Linux (Amazon VPC)` \| `Windows` \|
#'     `Windows (Amazon VPC)` \| `Windows with SQL Server Standard` \|
#'     `Windows with SQL Server Standard (Amazon VPC)` \|
#'     `Windows with SQL Server Web` \|
#'     ` Windows with SQL Server Web (Amazon VPC)` \|
#'     `Windows with SQL Server Enterprise` \|
#'     `Windows with SQL Server Enterprise (Amazon VPC)`)
#' 
#' -   `reserved-instances-offering-id` - The Reserved Instances offering
#'     ID.
#' 
#' -   `scope` - The scope of the Reserved Instance (`Availability Zone` or
#'     `Region`).
#' 
#' -   `usage-price` - The usage price of the Reserved Instance, per hour
#'     (for example, 0.84).
#' 
#' @section IncludeMarketplace:
#' Include Reserved Instance Marketplace offerings in
#' the response.
#' @section InstanceType:
#' The instance type that the reservation
#' will cover (for example, `m1.small`). For more information, see
#' [Instance
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html)
#' in the *Amazon EC2 User Guide*.
#' @section MaxDuration:
#' 
#' The maximum duration (in seconds) to filter when searching for
#' offerings.
#' 
#' Default: 94608000 (3 years)
#' 
#' @section MaxInstanceCount:
#' 
#' The maximum number of instances to filter when searching for offerings.
#' 
#' Default: 20
#' 
#' @section MinDuration:
#' 
#' The minimum duration (in seconds) to filter when searching for
#' offerings.
#' 
#' Default: 2592000 (1 month)
#' 
#' @section OfferingClass:
#' The offering class of the Reserved Instance. Can be
#' `standard` or `convertible`.
#' @section ProductDescription:
#' The Reserved Instance
#' product platform description. Instances that include `(Amazon VPC)` in
#' the description are for use with Amazon VPC.
#' @section ReservedInstancesOfferingId:
#' One or
#' more Reserved Instances offering IDs.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section InstanceTenancy:
#' 
#' The tenancy of the instances covered by the reservation. A Reserved
#' Instance with a tenancy of `dedicated` is applied to instances that run
#' in a VPC on single-tenant hardware (i.e., Dedicated Instances).
#' 
#' **Important:** The `host` value cannot be used with this parameter. Use
#' the `default` or `dedicated` values only.
#' 
#' Default: `default`
#' 
#' @section MaxResults:
#' 
#' The maximum number of results to return for the request in a single
#' page. The remaining results of the initial request can be seen by
#' sending another request with the returned `NextToken` value. The maximum
#' is 100.
#' 
#' Default: 100
#' 
#' @section OfferingType:
#' The Reserved Instance offering type. If you
#' are using tools that predate the 2011-11-01 API version, you only have
#' access to the `Medium Utilization` Reserved Instance offering type.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_reserved_instances_offerings <- function(AvailabilityZone = NULL, Filter = NULL, 
    IncludeMarketplace = NULL, InstanceType = NULL, MaxDuration = NULL, MaxInstanceCount = NULL, 
    MinDuration = NULL, OfferingClass = NULL, ProductDescription = NULL, ReservedInstancesOfferingId = NULL, 
    DryRun = NULL, InstanceTenancy = NULL, MaxResults = NULL, NextToken = NULL, OfferingType = NULL, 
    simplify = TRUE, others = list()) {
    ReservedInstancesOfferingId <- list_to_array("ReservedInstancesOfferingId", ReservedInstancesOfferingId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(AvailabilityZone = AvailabilityZone, IncludeMarketplace = IncludeMarketplace, 
        InstanceType = InstanceType, MaxDuration = MaxDuration, MaxInstanceCount = MaxInstanceCount, 
        MinDuration = MinDuration, OfferingClass = OfferingClass, ProductDescription = ProductDescription, 
        DryRun = DryRun, InstanceTenancy = InstanceTenancy, MaxResults = MaxResults, NextToken = NextToken, 
        OfferingType = OfferingType), Filter, ReservedInstancesOfferingId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeReservedInstancesOfferings", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Route Tables
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param RouteTableId List.   One or more route table IDs.  Default: Describes all your route tables.  \[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `association.route-table-association-id` - The ID of an association
#'     ID for the route table.
#' 
#' -   `association.route-table-id` - The ID of the route table involved in
#'     the association.
#' 
#' -   `association.subnet-id` - The ID of the subnet involved in the
#'     association.
#' 
#' -   `association.main` - Indicates whether the route table is the main
#'     route table for the VPC (`true` \| `false`). Route tables that do
#'     not have an association ID are not returned in the response.
#' 
#' -   `owner-id` - The ID of the AWS account that owns the route table.
#' 
#' -   `route-table-id` - The ID of the route table.
#' 
#' -   `route.destination-cidr-block` - The IPv4 CIDR range specified in a
#'     route in the table.
#' 
#' -   `route.destination-ipv6-cidr-block` - The IPv6 CIDR range specified
#'     in a route in the route table.
#' 
#' -   `route.destination-prefix-list-id` - The ID (prefix) of the AWS
#'     service specified in a route in the table.
#' 
#' -   `route.egress-only-internet-gateway-id` - The ID of an egress-only
#'     Internet gateway specified in a route in the route table.
#' 
#' -   `route.gateway-id` - The ID of a gateway specified in a route in the
#'     table.
#' 
#' -   `route.instance-id` - The ID of an instance specified in a route in
#'     the table.
#' 
#' -   `route.nat-gateway-id` - The ID of a NAT gateway.
#' 
#' -   `route.transit-gateway-id` - The ID of a transit gateway.
#' 
#' -   `route.origin` - Describes how the route was created.
#'     `CreateRouteTable` indicates that the route was automatically
#'     created when the route table was created; `CreateRoute` indicates
#'     that the route was manually added to the route table;
#'     `EnableVgwRoutePropagation` indicates that the route was propagated
#'     by route propagation.
#' 
#' -   `route.state` - The state of a route in the route table (`active` \|
#'     `blackhole`). The blackhole state indicates that the route\'s target
#'     isn\'t available (for example, the specified gateway isn\'t attached
#'     to the VPC, the specified NAT instance has been terminated, and so
#'     on).
#' 
#' -   `route.vpc-peering-connection-id` - The ID of a VPC peering
#'     connection specified in a route in the table.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC for the route table.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section RouteTableId:
#' 
#' One or more route table IDs.
#' 
#' Default: Describes all your route tables.
#' 
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_route_tables <- function(Filter = NULL, DryRun = NULL, RouteTableId = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    RouteTableId <- list_to_array("RouteTableId", RouteTableId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, RouteTableId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeRouteTables", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Scheduled Instance Availability
#' 
#' @param FirstSlotStartTimeRange Object. The time period for the first schedule to start.
#' @param Recurrence Object. The schedule recurrence.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @param MaxSlotDurationInHours Integer. The maximum available duration, in hours.\[optional\]
#' @param MinSlotDurationInHours Integer. The minimum available duration, in hours.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FirstSlotStartTimeRange:
#' The time period
#' for the first schedule to start.
#' @section Recurrence:
#' The schedule
#' recurrence.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `availability-zone` - The Availability Zone (for example,
#'     `us-west-2a`).
#' 
#' -   `instance-type` - The instance type (for example, `c4.large`).
#' 
#' -   `network-platform` - The network platform (`EC2-Classic` or
#'     `EC2-VPC`).
#' 
#' -   `platform` - The platform (`Linux/UNIX` or `Windows`).
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. This value can be between 5 and 300. The default value is 300. To
#' retrieve the remaining results, make another call with the returned
#' `NextToken` value.
#' @section MaxSlotDurationInHours:
#' The maximum available duration,
#' in hours. This value must be greater than `MinSlotDurationInHours` and
#' less than 1,720.
#' @section MinSlotDurationInHours:
#' The minimum available duration, in
#' hours. The minimum required duration is 1,200 hours per year. For
#' example, the minimum daily schedule is 4 hours, the minimum weekly
#' schedule is 24 hours, and the minimum monthly schedule is 100
#' hours.
#' @return A list object or a character vector
#' @export
ec2_describe_scheduled_instance_availability <- function(FirstSlotStartTimeRange, Recurrence, 
    DryRun = NULL, Filter = NULL, MaxResults = NULL, MaxSlotDurationInHours = NULL, MinSlotDurationInHours = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(FirstSlotStartTimeRange = FirstSlotStartTimeRange, Recurrence = Recurrence, 
        DryRun = DryRun, MaxResults = MaxResults, MaxSlotDurationInHours = MaxSlotDurationInHours, 
        MinSlotDurationInHours = MinSlotDurationInHours, NextToken = NextToken), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeScheduledInstanceAvailability", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Scheduled Instances
#' 
#' Describes the specified Scheduled Instances or all
#' your Scheduled Instances.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @param ScheduledInstanceId List. The Scheduled Instance IDs.\[optional\]
#' @param SlotStartTimeRange Object. The time period for the first schedule to start.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `availability-zone` - The Availability Zone (for example,
#'     `us-west-2a`).
#' 
#' -   `instance-type` - The instance type (for example, `c4.large`).
#' 
#' -   `network-platform` - The network platform (`EC2-Classic` or
#'     `EC2-VPC`).
#' 
#' -   `platform` - The platform (`Linux/UNIX` or `Windows`).
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. This value can be between 5 and 300. The default value is 100. To
#' retrieve the remaining results, make another call with the returned
#' `NextToken` value.
#' @section ScheduledInstanceId:
#' The Scheduled Instance
#' IDs.
#' @section SlotStartTimeRange:
#' The time period for the first schedule to
#' start.
#' @return A list object or a character vector
#' @export
ec2_describe_scheduled_instances <- function(DryRun = NULL, Filter = NULL, MaxResults = NULL, 
    NextToken = NULL, ScheduledInstanceId = NULL, SlotStartTimeRange = NULL, simplify = TRUE, 
    others = list()) {
    ScheduledInstanceId <- list_to_array("ScheduledInstanceId", ScheduledInstanceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken, 
        SlotStartTimeRange = SlotStartTimeRange), Filter, ScheduledInstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeScheduledInstances", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Security Group References
#' 
#' \[VPC only\] Describes the
#' VPCs on the other side of a VPC peering connection that are referencing
#' the security groups you\'ve specified in this
#' request.
#' 
#' @param GroupId List. The IDs of the security groups in your account.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GroupId:
#' The IDs of the security groups in your
#' account.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_security_group_references <- function(GroupId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    GroupId <- list_to_array("GroupId", GroupId)
    parameters <- c(others, list(DryRun = DryRun), GroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSecurityGroupReferences", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Describe Security Groups
#' 
#' @param GroupId List.   The IDs of the security groups.\[optional\]
#' @param GroupName List.   \[EC2-Classic and default VPC only\] The names of the security groups.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters. If using multiple filters for rules, the results include
#' security groups for which any combination of rules - not necessarily a
#' single rule - match all filters.
#' 
#' -   `description` - The description of the security group.
#' 
#' -   `egress.ip-permission.cidr` - An IPv4 CIDR block for an outbound
#'     security group rule.
#' 
#' -   `egress.ip-permission.from-port` - For an outbound rule, the start
#'     of port range for the TCP and UDP protocols, or an ICMP type number.
#' 
#' -   `egress.ip-permission.group-id` - The ID of a security group that
#'     has been referenced in an outbound security group rule.
#' 
#' -   `egress.ip-permission.group-name` - The name of a security group
#'     that is referenced in an outbound security group rule.
#' 
#' -   `egress.ip-permission.ipv6-cidr` - An IPv6 CIDR block for an
#'     outbound security group rule.
#' 
#' -   `egress.ip-permission.prefix-list-id` - The ID of a prefix list to
#'     which a security group rule allows outbound access.
#' 
#' -   `egress.ip-permission.protocol` - The IP protocol for an outbound
#'     security group rule (`tcp` \| `udp` \| `icmp`, a protocol number, or
#'     -1 for all protocols).
#' 
#' -   `egress.ip-permission.to-port` - For an outbound rule, the end of
#'     port range for the TCP and UDP protocols, or an ICMP code.
#' 
#' -   `egress.ip-permission.user-id` - The ID of an AWS account that has
#'     been referenced in an outbound security group rule.
#' 
#' -   `group-id` - The ID of the security group.
#' 
#' -   `group-name` - The name of the security group.
#' 
#' -   `ip-permission.cidr` - An IPv4 CIDR block for an inbound security
#'     group rule.
#' 
#' -   `ip-permission.from-port` - For an inbound rule, the start of port
#'     range for the TCP and UDP protocols, or an ICMP type number.
#' 
#' -   `ip-permission.group-id` - The ID of a security group that has been
#'     referenced in an inbound security group rule.
#' 
#' -   `ip-permission.group-name` - The name of a security group that is
#'     referenced in an inbound security group rule.
#' 
#' -   `ip-permission.ipv6-cidr` - An IPv6 CIDR block for an inbound
#'     security group rule.
#' 
#' -   `ip-permission.prefix-list-id` - The ID of a prefix list from which
#'     a security group rule allows inbound access.
#' 
#' -   `ip-permission.protocol` - The IP protocol for an inbound security
#'     group rule (`tcp` \| `udp` \| `icmp`, a protocol number, or -1 for
#'     all protocols).
#' 
#' -   `ip-permission.to-port` - For an inbound rule, the end of port range
#'     for the TCP and UDP protocols, or an ICMP code.
#' 
#' -   `ip-permission.user-id` - The ID of an AWS account that has been
#'     referenced in an inbound security group rule.
#' 
#' -   `owner-id` - The AWS account ID of the owner of the security group.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC specified when the security group was
#'     created.
#' 
#' @section GroupId:
#' 
#' The IDs of the security groups. Required for security groups in a
#' nondefault VPC.
#' 
#' Default: Describes all your security groups.
#' 
#' @section GroupName:
#' 
#' \[EC2-Classic and default VPC only\] The names of the security groups.
#' You can specify either the security group name or the security group ID.
#' For security groups in a nondefault VPC, use the `group-name` filter to
#' describe security groups by name.
#' 
#' Default: Describes all your security groups.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of results
#' to return in a single call. To retrieve the remaining results, make
#' another request with the returned `NextToken` value. This value can be
#' between 5 and 1000. If this parameter is not specified, then all results
#' are returned.
#' @return A list object or a character vector
#' @export
ec2_describe_security_groups <- function(Filter = NULL, GroupId = NULL, GroupName = NULL, 
    DryRun = NULL, NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    GroupId <- list_to_array("GroupId", GroupId)
    GroupName <- list_to_array("GroupName", GroupName)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, GroupId, GroupName)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSecurityGroups", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Snapshot Attribute
#' 
#' @param Attribute String. The snapshot attribute you would like to view.
#' @param SnapshotId String. The ID of the EBS snapshot.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' The snapshot attribute you would like
#' to view.
#' @section SnapshotId:
#' The ID of the EBS
#' snapshot.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_snapshot_attribute <- function(Attribute, SnapshotId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, SnapshotId = SnapshotId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSnapshotAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Snapshots
#' 
#' @param MaxResults Integer. The maximum number of snapshot results returned by `DescribeSnapshots` in paginated output.\[optional\]
#' @param Owner List. Scopes the results to snapshots with the specified owners.\[optional\]
#' @param RestorableBy List. The IDs of the AWS accounts that can create volumes from the snapshot.\[optional\]
#' @param SnapshotId List.   The snapshot IDs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `description` - A description of the snapshot.
#' 
#' -   `encrypted` - Indicates whether the snapshot is encrypted (`true` \|
#'     `false`)
#' 
#' -   `owner-alias` - The owner alias, from an Amazon-maintained list
#'     (`amazon`). This is not the user-configured AWS account alias set
#'     using the IAM console. We recommend that you use the related
#'     parameter instead of this filter.
#' 
#' -   `owner-id` - The AWS account ID of the owner. We recommend that you
#'     use the related parameter instead of this filter.
#' 
#' -   `progress` - The progress of the snapshot, as a percentage (for
#'     example, 80\%).
#' 
#' -   `snapshot-id` - The snapshot ID.
#' 
#' -   `start-time` - The time stamp when the snapshot was initiated.
#' 
#' -   `status` - The status of the snapshot (`pending` \| `completed` \|
#'     `error`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `volume-id` - The ID of the volume the snapshot is for.
#' 
#' -   `volume-size` - The size of the volume, in GiB.
#' 
#' @section MaxResults:
#' The maximum number of snapshot results returned by
#' `DescribeSnapshots` in paginated output. When this parameter is used,
#' `DescribeSnapshots` only returns `MaxResults` results in a single page
#' along with a `NextToken` response element. The remaining results of the
#' initial request can be seen by sending another `DescribeSnapshots`
#' request with the returned `NextToken` value. This value can be between 5
#' and 1,000; if `MaxResults` is given a value larger than 1,000, only
#' 1,000 results are returned. If this parameter is not used, then
#' `DescribeSnapshots` returns all results. You cannot specify this
#' parameter and the snapshot IDs parameter in the same
#' request.
#' @section Owner:
#' Scopes the results to snapshots with the
#' specified owners. You can specify a combination of AWS account IDs,
#' `self`, and `amazon`.
#' @section RestorableBy:
#' The IDs of the AWS accounts
#' that can create volumes from the snapshot.
#' @section SnapshotId:
#' 
#' The snapshot IDs.
#' 
#' Default: Describes the snapshots for which you have create volume
#' permissions.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_snapshots <- function(Filter = NULL, MaxResults = NULL, NextToken = NULL, Owner = NULL, 
    RestorableBy = NULL, SnapshotId = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Owner <- list_to_array("Owner", Owner)
    RestorableBy <- list_to_array("RestorableBy", RestorableBy)
    SnapshotId <- list_to_array("SnapshotId", SnapshotId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        Filter, Owner, RestorableBy, SnapshotId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSnapshots", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Spot Datafeed Subscription
#' 
#' Describes the data feed for Spot Instances. For
#' more information, see [Spot Instance data
#' feed](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-data-feeds.html)
#' in the *Amazon EC2 User Guide for Linux
#' Instances*.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_spot_datafeed_subscription <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSpotDatafeedSubscription", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Describe Spot Fleet Instances
#' 
#' Describes the running instances for the
#' specified Spot Fleet.
#' 
#' @param SpotFleetRequestId String. The ID of the Spot Fleet request.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SpotFleetRequestId:
#' The ID of the Spot Fleet
#' request.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of
#' results to return in a single call. Specify a value between 1 and 1000.
#' The default value is 1000. To retrieve the remaining results, make
#' another call with the returned `NextToken`
#' value.
#' @return A list object or a character vector
#' @export
ec2_describe_spot_fleet_instances <- function(SpotFleetRequestId, DryRun = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(SpotFleetRequestId = SpotFleetRequestId, DryRun = DryRun, 
        MaxResults = MaxResults, NextToken = NextToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSpotFleetInstances", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Spot Fleet Request History
#' 
#' @param SpotFleetRequestId String. The ID of the Spot Fleet request.
#' @param StartTime String. The starting date and time for the events, in UTC format (for example, *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z)....
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EventType String. The type of events to describe. By default, all events are described.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SpotFleetRequestId:
#' The ID of the Spot Fleet
#' request.
#' @section StartTime:
#' The starting date and time for the events,
#' in UTC format (for example,
#' *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z).
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section EventType:
#' The type
#' of events to describe. By default, all events are
#' described.
#' @section MaxResults:
#' The maximum number of results to return
#' in a single call. Specify a value between 1 and 1000. The default value
#' is 1000. To retrieve the remaining results, make another call with the
#' returned `NextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_spot_fleet_request_history <- function(SpotFleetRequestId, StartTime, DryRun = NULL, 
    EventType = NULL, MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(SpotFleetRequestId = SpotFleetRequestId, StartTime = StartTime, 
        DryRun = DryRun, EventType = EventType, MaxResults = MaxResults, NextToken = NextToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSpotFleetRequestHistory", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Spot Fleet Requests
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @param SpotFleetRequestId List. The IDs of the Spot Fleet requests.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of
#' results to return in a single call. Specify a value between 1 and 1000.
#' The default value is 1000. To retrieve the remaining results, make
#' another call with the returned `NextToken`
#' value.
#' @section SpotFleetRequestId:
#' The IDs of the Spot Fleet
#' requests.
#' @return A list object or a character vector
#' @export
ec2_describe_spot_fleet_requests <- function(DryRun = NULL, MaxResults = NULL, NextToken = NULL, 
    SpotFleetRequestId = NULL, simplify = TRUE, others = list()) {
    SpotFleetRequestId <- list_to_array("SpotFleetRequestId", SpotFleetRequestId)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        SpotFleetRequestId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSpotFleetRequests", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Spot Instance Requests
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param SpotInstanceRequestId List. One or more Spot Instance request IDs.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `availability-zone-group` - The Availability Zone group.
#' 
#' -   `create-time` - The time stamp when the Spot Instance request was
#'     created.
#' 
#' -   `fault-code` - The fault code related to the request.
#' 
#' -   `fault-message` - The fault message related to the request.
#' 
#' -   `instance-id` - The ID of the instance that fulfilled the request.
#' 
#' -   `launch-group` - The Spot Instance launch group.
#' 
#' -   `launch.block-device-mapping.delete-on-termination` - Indicates
#'     whether the EBS volume is deleted on instance termination.
#' 
#' -   `launch.block-device-mapping.device-name` - The device name for the
#'     volume in the block device mapping (for example, `/dev/sdh` or
#'     `xvdh`).
#' 
#' -   `launch.block-device-mapping.snapshot-id` - The ID of the snapshot
#'     for the EBS volume.
#' 
#' -   `launch.block-device-mapping.volume-size` - The size of the EBS
#'     volume, in GiB.
#' 
#' -   `launch.block-device-mapping.volume-type` - The type of EBS volume:
#'     `gp2` for General Purpose SSD, `io1` or `io2` for Provisioned IOPS
#'     SSD, `st1` for Throughput Optimized HDD, `sc1`for Cold HDD, or
#'     `standard` for Magnetic.
#' 
#' -   `launch.group-id` - The ID of the security group for the instance.
#' 
#' -   `launch.group-name` - The name of the security group for the
#'     instance.
#' 
#' -   `launch.image-id` - The ID of the AMI.
#' 
#' -   `launch.instance-type` - The type of instance (for example,
#'     `m3.medium`).
#' 
#' -   `launch.kernel-id` - The kernel ID.
#' 
#' -   `launch.key-name` - The name of the key pair the instance launched
#'     with.
#' 
#' -   `launch.monitoring-enabled` - Whether detailed monitoring is enabled
#'     for the Spot Instance.
#' 
#' -   `launch.ramdisk-id` - The RAM disk ID.
#' 
#' -   `launched-availability-zone` - The Availability Zone in which the
#'     request is launched.
#' 
#' -   `network-interface.addresses.primary` - Indicates whether the IP
#'     address is the primary private IP address.
#' 
#' -   `network-interface.delete-on-termination` - Indicates whether the
#'     network interface is deleted when the instance is terminated.
#' 
#' -   `network-interface.description` - A description of the network
#'     interface.
#' 
#' -   `network-interface.device-index` - The index of the device for the
#'     network interface attachment on the instance.
#' 
#' -   `network-interface.group-id` - The ID of the security group
#'     associated with the network interface.
#' 
#' -   `network-interface.network-interface-id` - The ID of the network
#'     interface.
#' 
#' -   `network-interface.private-ip-address` - The primary private IP
#'     address of the network interface.
#' 
#' -   `network-interface.subnet-id` - The ID of the subnet for the
#'     instance.
#' 
#' -   `product-description` - The product description associated with the
#'     instance (`Linux/UNIX` \| `Windows`).
#' 
#' -   `spot-instance-request-id` - The Spot Instance request ID.
#' 
#' -   `spot-price` - The maximum hourly price for any Spot Instance
#'     launched to fulfill the request.
#' 
#' -   `state` - The state of the Spot Instance request (`open` \| `active`
#'     \| `closed` \| `cancelled` \| `failed`). Spot request status
#'     information can help you track your Amazon EC2 Spot Instance
#'     requests. For more information, see [Spot request
#'     status](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html)
#'     in the *Amazon EC2 User Guide for Linux Instances*.
#' 
#' -   `status-code` - The short code describing the most recent evaluation
#'     of your Spot Instance request.
#' 
#' -   `status-message` - The message explaining the status of the Spot
#'     Instance request.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `type` - The type of Spot Instance request (`one-time` \|
#'     `persistent`).
#' 
#' -   `valid-from` - The start date of the request.
#' 
#' -   `valid-until` - The end date of the request.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section SpotInstanceRequestId:
#' One or more Spot Instance
#' request IDs.
#' @section MaxResults:
#' The maximum number of results to return in a
#' single call. Specify a value between 5 and 1000. To retrieve the
#' remaining results, make another call with the returned `NextToken`
#' value.
#' @return A list object or a character vector
#' @export
ec2_describe_spot_instance_requests <- function(Filter = NULL, DryRun = NULL, SpotInstanceRequestId = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    SpotInstanceRequestId <- list_to_array("SpotInstanceRequestId", SpotInstanceRequestId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, SpotInstanceRequestId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSpotInstanceRequests", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Spot Price History
#' 
#' @param AvailabilityZone String. Filters the results by the specified Availability Zone.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EndTime String. The date and time, up to the current date, from which to stop retrieving the price history data,...\[optional\]
#' @param InstanceType List. Filters the results by the specified instance types.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @param ProductDescription List. Filters the results by the specified basic product descriptions.\[optional\]
#' @param StartTime String. The date and time, up to the past 90 days, from which to start retrieving the price history data,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `availability-zone` - The Availability Zone for which prices should
#'     be returned.
#' 
#' -   `instance-type` - The type of instance (for example, `m3.medium`).
#' 
#' -   `product-description` - The product description for the Spot price
#'     (`Linux/UNIX` \| `Red Hat Enterprise Linux` \| `SUSE Linux` \|
#'     `Windows` \| `Linux/UNIX (Amazon VPC)` \|
#'     `Red Hat Enterprise Linux (Amazon VPC)` \| `SUSE Linux (Amazon VPC)`
#'     \| `Windows (Amazon VPC)`).
#' 
#' -   `spot-price` - The Spot price. The value must match exactly (or use
#'     wildcards; greater than or less than comparison is not supported).
#' 
#' -   `timestamp` - The time stamp of the Spot price history, in UTC
#'     format (for example, *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z). You can use
#'     wildcards (\* and ?). Greater than or less than comparison is not
#'     supported.
#' 
#' @section AvailabilityZone:
#' Filters the results by the specified Availability
#' Zone.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section EndTime:
#' The date and time, up to
#' the current date, from which to stop retrieving the price history data,
#' in UTC format (for example,
#' *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z).
#' @section InstanceType:
#' Filters the
#' results by the specified instance types.
#' @section MaxResults:
#' The maximum
#' number of results to return in a single call. Specify a value between 1
#' and 1000. The default value is 1000. To retrieve the remaining results,
#' make another call with the returned `NextToken`
#' value.
#' @section ProductDescription:
#' Filters the results by the specified basic
#' product descriptions.
#' @section StartTime:
#' The date and time, up to the
#' past 90 days, from which to start retrieving the price history data, in
#' UTC format (for example,
#' *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z).
#' @return A list object or a character vector
#' @export
ec2_describe_spot_price_history <- function(Filter = NULL, AvailabilityZone = NULL, DryRun = NULL, 
    EndTime = NULL, InstanceType = NULL, MaxResults = NULL, NextToken = NULL, ProductDescription = NULL, 
    StartTime = NULL, simplify = TRUE, others = list()) {
    InstanceType <- list_to_array("InstanceType", InstanceType)
    ProductDescription <- list_to_array("ProductDescription", ProductDescription)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(AvailabilityZone = AvailabilityZone, DryRun = DryRun, EndTime = EndTime, 
        MaxResults = MaxResults, NextToken = NextToken, StartTime = StartTime), Filter, InstanceType, 
        ProductDescription)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSpotPriceHistory", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Stale Security Groups
#' 
#' \[VPC only\] Describes the stale security group
#' rules for security groups in a specified VPC. Rules are stale when they
#' reference a deleted security group in a peer VPC, or a security group in
#' a peer VPC for which the VPC peering connection has been
#' deleted.
#' 
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of items to return for this request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of items
#' to return for this request. The request returns a token that you can
#' specify in a subsequent call to get the next set of
#' results.
#' @return A list object or a character vector
#' @export
ec2_describe_stale_security_groups <- function(VpcId, DryRun = NULL, MaxResults = NULL, NextToken = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId, DryRun = DryRun, MaxResults = MaxResults, 
        NextToken = NextToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeStaleSecurityGroups", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Subnets
#' 
#' @param SubnetId List.   One or more subnet IDs.  Default: Describes all your subnets.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `availability-zone` - The Availability Zone for the subnet. You can
#'     also use `availabilityZone` as the filter name.
#' 
#' -   `availability-zone-id` - The ID of the Availability Zone for the
#'     subnet. You can also use `availabilityZoneId` as the filter name.
#' 
#' -   `available-ip-address-count` - The number of IPv4 addresses in the
#'     subnet that are available.
#' 
#' -   `cidr-block` - The IPv4 CIDR block of the subnet. The CIDR block you
#'     specify must exactly match the subnet\'s CIDR block for information
#'     to be returned for the subnet. You can also use `cidr` or
#'     `cidrBlock` as the filter names.
#' 
#' -   `default-for-az` - Indicates whether this is the default subnet for
#'     the Availability Zone. You can also use `defaultForAz` as the filter
#'     name.
#' 
#' -   `ipv6-cidr-block-association.ipv6-cidr-block` - An IPv6 CIDR block
#'     associated with the subnet.
#' 
#' -   `ipv6-cidr-block-association.association-id` - An association ID for
#'     an IPv6 CIDR block associated with the subnet.
#' 
#' -   `ipv6-cidr-block-association.state` - The state of an IPv6 CIDR
#'     block associated with the subnet.
#' 
#' -   `owner-id` - The ID of the AWS account that owns the subnet.
#' 
#' -   `state` - The state of the subnet (`pending` \| `available`).
#' 
#' -   `subnet-arn` - The Amazon Resource Name (ARN) of the subnet.
#' 
#' -   `subnet-id` - The ID of the subnet.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC for the subnet.
#' 
#' @section SubnetId:
#' 
#' One or more subnet IDs.
#' 
#' Default: Describes all your subnets.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_subnets <- function(Filter = NULL, SubnetId = NULL, DryRun = NULL, NextToken = NULL, 
    MaxResults = NULL, simplify = TRUE, others = list()) {
    SubnetId <- list_to_array("SubnetId", SubnetId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, SubnetId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeSubnets", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Tags
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `key` - The tag key.
#' 
#' -   `resource-id` - The ID of the resource.
#' 
#' -   `resource-type` - The resource type (`customer-gateway` \|
#'     `dedicated-host` \| `dhcp-options` \| `elastic-ip` \| `fleet` \|
#'     `fpga-image` \| `host-reservation` \| `image` \| `instance` \|
#'     `internet-gateway` \| `key-pair` \| `launch-template` \|
#'     `natgateway` \| `network-acl` \| `network-interface` \|
#'     `placement-group` \| `reserved-instances` \| `route-table` \|
#'     `security-group` \| `snapshot` \| `spot-instances-request` \|
#'     `subnet` \| `volume` \| `vpc` \| `vpc-endpoint` \|
#'     `vpc-endpoint-service` \| `vpc-peering-connection` \|
#'     `vpn-connection` \| `vpn-gateway`).
#' 
#' -   `tag`:\ - The key/value combination of the tag. For example,
#'     specify \'tag:Owner\' for the filter name and \'TeamA\' for the
#'     filter value to find resources with the tag \'Owner=TeamA\'.
#' 
#' -   `value` - The tag value.
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. This value can be between 5 and 1000. To retrieve the remaining
#' results, make another call with the returned `NextToken`
#' value.
#' @return A list object or a character vector
#' @export
ec2_describe_tags <- function(DryRun = NULL, Filter = NULL, MaxResults = NULL, NextToken = NULL, 
    simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTags", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Traffic Mirror Filters
#' 
#' Describes one or more Traffic Mirror
#' filters.
#' 
#' @param TrafficMirrorFilterId List. The ID of the Traffic Mirror filter.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorFilterId:
#' The ID of the Traffic Mirror
#' filter.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `description`: The Traffic Mirror filter description.
#' 
#' -   `traffic-mirror-filter-id`: The ID of the Traffic Mirror filter.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_traffic_mirror_filters <- function(TrafficMirrorFilterId = NULL, DryRun = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    TrafficMirrorFilterId <- list_to_array("TrafficMirrorFilterId", TrafficMirrorFilterId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        TrafficMirrorFilterId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTrafficMirrorFilters", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Traffic Mirror Sessions
#' 
#' Describes one or more Traffic Mirror
#' sessions. By default, all Traffic Mirror sessions are described.
#' Alternatively, you can filter the
#' results.
#' 
#' @param TrafficMirrorSessionId List. The ID of the Traffic Mirror session.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorSessionId:
#' The ID of the Traffic Mirror
#' session.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `description`: The Traffic Mirror session description.
#' 
#' -   `network-interface-id`: The ID of the Traffic Mirror session network
#'     interface.
#' 
#' -   `owner-id`: The ID of the account that owns the Traffic Mirror
#'     session.
#' 
#' -   `packet-length`: The assigned number of packets to mirror.
#' 
#' -   `session-number`: The assigned session number.
#' 
#' -   `traffic-mirror-filter-id`: The ID of the Traffic Mirror filter.
#' 
#' -   `traffic-mirror-session-id`: The ID of the Traffic Mirror session.
#' 
#' -   `traffic-mirror-target-id`: The ID of the Traffic Mirror target.
#' 
#' -   `virtual-network-id`: The virtual network ID of the Traffic Mirror
#'     session.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_traffic_mirror_sessions <- function(TrafficMirrorSessionId = NULL, DryRun = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    TrafficMirrorSessionId <- list_to_array("TrafficMirrorSessionId", TrafficMirrorSessionId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        TrafficMirrorSessionId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTrafficMirrorSessions", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Traffic Mirror Targets
#' 
#' Information about one or more Traffic
#' Mirror targets.
#' 
#' @param TrafficMirrorTargetId List. The ID of the Traffic Mirror targets.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorTargetId:
#' The ID of the Traffic Mirror
#' targets.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `description`: The Traffic Mirror target description.
#' 
#' -   `network-interface-id`: The ID of the Traffic Mirror session network
#'     interface.
#' 
#' -   `network-load-balancer-arn`: The Amazon Resource Name (ARN) of the
#'     Network Load Balancer that is associated with the session.
#' 
#' -   `owner-id`: The ID of the account that owns the Traffic Mirror
#'     session.
#' 
#' -   `traffic-mirror-target-id`: The ID of the Traffic Mirror target.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_traffic_mirror_targets <- function(TrafficMirrorTargetId = NULL, DryRun = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    TrafficMirrorTargetId <- list_to_array("TrafficMirrorTargetId", TrafficMirrorTargetId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        TrafficMirrorTargetId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTrafficMirrorTargets", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateway Attachments
#' 
#' Describes one or more attachments
#' between resources and transit gateways. By default, all attachments are
#' described. Alternatively, you can filter the results by attachment ID,
#' attachment state, resource ID, or resource
#' owner.
#' 
#' @param TransitGatewayAttachmentIds List. The IDs of the attachments.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentIds:
#' The IDs of the
#' attachments.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `association.state` - The state of the association (`associating` \|
#'     `associated` \| `disassociating`).
#' 
#' -   `association.transit-gateway-route-table-id` - The ID of the route
#'     table for the transit gateway.
#' 
#' -   `resource-id` - The ID of the resource.
#' 
#' -   `resource-owner-id` - The ID of the AWS account that owns the
#'     resource.
#' 
#' -   `resource-type` - The resource type. Valid values are `vpc` \| `vpn`
#'     \| `direct-connect-gateway` \| `peering` \| `connect`.
#' 
#' -   `state` - The state of the attachment. Valid values are `available`
#'     \| `deleted` \| `deleting` \| `failed` \| `failing` \|
#'     `initiatingRequest` \| `modifying` \| `pendingAcceptance` \|
#'     `pending` \| `rollingBack` \| `rejected` \| `rejecting`.
#' 
#' -   `transit-gateway-attachment-id` - The ID of the attachment.
#' 
#' -   `transit-gateway-id` - The ID of the transit gateway.
#' 
#' -   `transit-gateway-owner-id` - The ID of the AWS account that owns the
#'     transit gateway.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateway_attachments <- function(TransitGatewayAttachmentIds = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayAttachmentIds <- list_to_array("TransitGatewayAttachmentIds", TransitGatewayAttachmentIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayAttachmentIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGatewayAttachments", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateway Connect Peers
#' 
#' Describes one or more Connect
#' peers.
#' 
#' @param TransitGatewayConnectPeerIds List. The IDs of the Connect peers.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayConnectPeerIds:
#' The IDs of the Connect
#' peers.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `state` - The state of the Connect peer (`pending` \| `available` \|
#'     `deleting` \| `deleted`).
#' 
#' -   `transit-gateway-attachment-id` - The ID of the attachment.
#' 
#' -   `transit-gateway-connect-peer-id` - The ID of the Connect peer.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateway_connect_peers <- function(TransitGatewayConnectPeerIds = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayConnectPeerIds <- list_to_array("TransitGatewayConnectPeerIds", TransitGatewayConnectPeerIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayConnectPeerIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGatewayConnectPeers", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateway Connects
#' 
#' Describes one or more Connect
#' attachments.
#' 
#' @param TransitGatewayAttachmentIds List. The IDs of the attachments.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentIds:
#' The IDs of the
#' attachments.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `options.protocol` - The tunnel protocol (`gre`).
#' 
#' -   `state` - The state of the attachment (`initiating` \|
#'     `initiatingRequest` \| `pendingAcceptance` \| `rollingBack` \|
#'     `pending` \| `available` \| `modifying` \| `deleting` \| `deleted`
#'     \| `failed` \| `rejected` \| `rejecting` \| `failing`).
#' 
#' -   `transit-gateway-attachment-id` - The ID of the Connect attachment.
#' 
#' -   `transit-gateway-id` - The ID of the transit gateway.
#' 
#' -   `transport-transit-gateway-attachment-id` - The ID of the transit
#'     gateway attachment from which the Connect attachment was created.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateway_connects <- function(TransitGatewayAttachmentIds = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayAttachmentIds <- list_to_array("TransitGatewayAttachmentIds", TransitGatewayAttachmentIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayAttachmentIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGatewayConnects", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateway Multicast Domains
#' 
#' Describes one or more transit gateway
#' multicast domains.
#' 
#' @param TransitGatewayMulticastDomainIds List. The ID of the transit gateway multicast domain.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainIds:
#' The ID of the transit
#' gateway multicast domain.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `state` - The state of the transit gateway multicast domain. Valid
#'     values are `pending` \| `available` \| `deleting` \| `deleted`.
#' 
#' -   `transit-gateway-id` - The ID of the transit gateway.
#' 
#' -   `transit-gateway-multicast-domain-id` - The ID of the transit
#'     gateway multicast domain.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateway_multicast_domains <- function(TransitGatewayMulticastDomainIds = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayMulticastDomainIds <- list_to_array("TransitGatewayMulticastDomainIds", 
        TransitGatewayMulticastDomainIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayMulticastDomainIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGatewayMulticastDomains", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateway Peering Attachments
#' 
#' Describes your transit gateway
#' peering attachments.
#' 
#' @param TransitGatewayAttachmentIds List. One or more IDs of the transit gateway peering attachments.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentIds:
#' One or more IDs of the
#' transit gateway peering attachments.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `transit-gateway-attachment-id` - The ID of the transit gateway
#'     attachment.
#' 
#' -   `local-owner-id` - The ID of your AWS account.
#' 
#' -   `remote-owner-id` - The ID of the AWS account in the remote Region
#'     that owns the transit gateway.
#' 
#' -   `state` - The state of the peering attachment. Valid values are
#'     `available` \| `deleted` \| `deleting` \| `failed` \| `failing` \|
#'     `initiatingRequest` \| `modifying` \| `pendingAcceptance` \|
#'     `pending` \| `rollingBack` \| `rejected` \| `rejecting`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources that have a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `transit-gateway-id` - The ID of the transit gateway.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateway_peering_attachments <- function(TransitGatewayAttachmentIds = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayAttachmentIds <- list_to_array("TransitGatewayAttachmentIds", TransitGatewayAttachmentIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayAttachmentIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGatewayPeeringAttachments", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateway Route Tables
#' 
#' Describes one or more transit
#' gateway route tables. By default, all transit gateway route tables are
#' described. Alternatively, you can filter the
#' results.
#' 
#' @param TransitGatewayRouteTableIds List. The IDs of the transit gateway route tables.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableIds:
#' The IDs of the transit
#' gateway route tables.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `default-association-route-table` - Indicates whether this is the
#'     default association route table for the transit gateway (`true` \|
#'     `false`).
#' 
#' -   `default-propagation-route-table` - Indicates whether this is the
#'     default propagation route table for the transit gateway (`true` \|
#'     `false`).
#' 
#' -   `state` - The state of the route table (`available` \| `deleting` \|
#'     `deleted` \| `pending`).
#' 
#' -   `transit-gateway-id` - The ID of the transit gateway.
#' 
#' -   `transit-gateway-route-table-id` - The ID of the transit gateway
#'     route table.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateway_route_tables <- function(TransitGatewayRouteTableIds = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayRouteTableIds <- list_to_array("TransitGatewayRouteTableIds", TransitGatewayRouteTableIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayRouteTableIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGatewayRouteTables", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateway Vpc Attachments
#' 
#' Describes one or more VPC attachments. By
#' default, all VPC attachments are described. Alternatively, you can
#' filter the results.
#' 
#' @param TransitGatewayAttachmentIds List. The IDs of the attachments.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentIds:
#' The IDs of the
#' attachments.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `state` - The state of the attachment. Valid values are `available`
#'     \| `deleted` \| `deleting` \| `failed` \| `failing` \|
#'     `initiatingRequest` \| `modifying` \| `pendingAcceptance` \|
#'     `pending` \| `rollingBack` \| `rejected` \| `rejecting`.
#' 
#' -   `transit-gateway-attachment-id` - The ID of the attachment.
#' 
#' -   `transit-gateway-id` - The ID of the transit gateway.
#' 
#' -   `vpc-id` - The ID of the VPC.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateway_vpc_attachments <- function(TransitGatewayAttachmentIds = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayAttachmentIds <- list_to_array("TransitGatewayAttachmentIds", TransitGatewayAttachmentIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayAttachmentIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGatewayVpcAttachments", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Transit Gateways
#' 
#' Describes one or more transit
#' gateways. By default, all transit gateways are described. Alternatively,
#' you can filter the results.
#' 
#' @param TransitGatewayIds List. The IDs of the transit gateways.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayIds:
#' The IDs of the transit
#' gateways.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `options.propagation-default-route-table-id` - The ID of the default
#'     propagation route table.
#' 
#' -   `options.amazon-side-asn` - The private ASN for the Amazon side of a
#'     BGP session.
#' 
#' -   `options.association-default-route-table-id` - The ID of the default
#'     association route table.
#' 
#' -   `options.auto-accept-shared-attachments` - Indicates whether there
#'     is automatic acceptance of attachment requests (`enable` \|
#'     `disable`).
#' 
#' -   `options.default-route-table-association` - Indicates whether
#'     resource attachments are automatically associated with the default
#'     association route table (`enable` \| `disable`).
#' 
#' -   `options.default-route-table-propagation` - Indicates whether
#'     resource attachments automatically propagate routes to the default
#'     propagation route table (`enable` \| `disable`).
#' 
#' -   `options.dns-support` - Indicates whether DNS support is enabled
#'     (`enable` \| `disable`).
#' 
#' -   `options.vpn-ecmp-support` - Indicates whether Equal Cost Multipath
#'     Protocol support is enabled (`enable` \| `disable`).
#' 
#' -   `owner-id` - The ID of the AWS account that owns the transit
#'     gateway.
#' 
#' -   `state` - The state of the transit gateway (`available` \| `deleted`
#'     \| `deleting` \| `modifying` \| `pending`).
#' 
#' -   `transit-gateway-id` - The ID of the transit gateway.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_transit_gateways <- function(TransitGatewayIds = NULL, Filter = NULL, MaxResults = NULL, 
    NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    TransitGatewayIds <- list_to_array("TransitGatewayIds", TransitGatewayIds)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        TransitGatewayIds, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeTransitGateways", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Volume Attribute
#' 
#' @param Attribute String. The attribute of the volume. This parameter is required.
#' @param VolumeId String. The ID of the volume.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' The attribute of the
#' volume. This parameter is required.
#' @section VolumeId:
#' The ID of the
#' volume.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_volume_attribute <- function(Attribute, VolumeId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, VolumeId = VolumeId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVolumeAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Volume Status
#' 
#' @param MaxResults Integer. The maximum number of volume results returned by `DescribeVolumeStatus` in paginated output.\[optional\]
#' @param VolumeId List.   The IDs of the volumes.  Default: Describes all your volumes.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `action.code` - The action code for the event (for example,
#'     `enable-volume-io`).
#' 
#' -   `action.description` - A description of the action.
#' 
#' -   `action.event-id` - The event ID associated with the action.
#' 
#' -   `availability-zone` - The Availability Zone of the instance.
#' 
#' -   `event.description` - A description of the event.
#' 
#' -   `event.event-id` - The event ID.
#' 
#' -   `event.event-type` - The event type (for `io-enabled`: `passed` \|
#'     `failed`; for `io-performance`: `io-performance:degraded` \|
#'     `io-performance:severely-degraded` \| `io-performance:stalled`).
#' 
#' -   `event.not-after` - The latest end time for the event.
#' 
#' -   `event.not-before` - The earliest start time for the event.
#' 
#' -   `volume-status.details-name` - The cause for `volume-status.status`
#'     (`io-enabled` \| `io-performance`).
#' 
#' -   `volume-status.details-status` - The status of
#'     `volume-status.details-name` (for `io-enabled`: `passed` \|
#'     `failed`; for `io-performance`: `normal` \| `degraded` \|
#'     `severely-degraded` \| `stalled`).
#' 
#' -   `volume-status.status` - The status of the volume (`ok` \|
#'     `impaired` \| `warning` \| `insufficient-data`).
#' 
#' @section MaxResults:
#' The maximum number of volume results returned by
#' `DescribeVolumeStatus` in paginated output. When this parameter is used,
#' the request only returns `MaxResults` results in a single page along
#' with a `NextToken` response element. The remaining results of the
#' initial request can be seen by sending another request with the returned
#' `NextToken` value. This value can be between 5 and 1,000; if
#' `MaxResults` is given a value larger than 1,000, only 1,000 results are
#' returned. If this parameter is not used, then `DescribeVolumeStatus`
#' returns all results. You cannot specify this parameter and the volume
#' IDs parameter in the same request.
#' @section VolumeId:
#' 
#' The IDs of the volumes.
#' 
#' Default: Describes all your volumes.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_volume_status <- function(Filter = NULL, MaxResults = NULL, NextToken = NULL, 
    VolumeId = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    VolumeId <- list_to_array("VolumeId", VolumeId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), 
        Filter, VolumeId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVolumeStatus", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Volumes
#' 
#' @param VolumeId List. The volume IDs.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of volume results returned by `DescribeVolumes` in paginated output.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `attachment.attach-time` - The time stamp when the attachment
#'     initiated.
#' 
#' -   `attachment.delete-on-termination` - Whether the volume is deleted
#'     on instance termination.
#' 
#' -   `attachment.device` - The device name specified in the block device
#'     mapping (for example, `/dev/sda1`).
#' 
#' -   `attachment.instance-id` - The ID of the instance the volume is
#'     attached to.
#' 
#' -   `attachment.status` - The attachment state (`attaching` \|
#'     `attached` \| `detaching`).
#' 
#' -   `availability-zone` - The Availability Zone in which the volume was
#'     created.
#' 
#' -   `create-time` - The time stamp when the volume was created.
#' 
#' -   `encrypted` - Indicates whether the volume is encrypted (`true` \|
#'     `false`)
#' 
#' -   `multi-attach-enabled` - Indicates whether the volume is enabled for
#'     Multi-Attach (`true` \| `false`)
#' 
#' -   `fast-restored` - Indicates whether the volume was created from a
#'     snapshot that is enabled for fast snapshot restore (`true` \|
#'     `false`).
#' 
#' -   `size` - The size of the volume, in GiB.
#' 
#' -   `snapshot-id` - The snapshot from which the volume was created.
#' 
#' -   `status` - The state of the volume (`creating` \| `available` \|
#'     `in-use` \| `deleting` \| `deleted` \| `error`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `volume-id` - The volume ID.
#' 
#' -   `volume-type` - The Amazon EBS volume type (`gp2` \| `gp3` \| `io1`
#'     \| `io2` \| `st1` \| `sc1`\| `standard`)
#' 
#' @section VolumeId:
#' The volume IDs.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section MaxResults:
#' The
#' maximum number of volume results returned by `DescribeVolumes` in
#' paginated output. When this parameter is used, `DescribeVolumes` only
#' returns `MaxResults` results in a single page along with a `NextToken`
#' response element. The remaining results of the initial request can be
#' seen by sending another `DescribeVolumes` request with the returned
#' `NextToken` value. This value can be between 5 and 500; if `MaxResults`
#' is given a value larger than 500, only 500 results are returned. If this
#' parameter is not used, then `DescribeVolumes` returns all results. You
#' cannot specify this parameter and the volume IDs parameter in the same
#' request.
#' @return A list object or a character vector
#' @export
ec2_describe_volumes <- function(Filter = NULL, VolumeId = NULL, DryRun = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    VolumeId <- list_to_array("VolumeId", VolumeId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter, VolumeId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVolumes", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Volumes Modifications
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VolumeId List. The IDs of the volumes.\[optional\]
#' @param MaxResults Integer. The maximum number of results (up to a limit of 500) to be returned in a paginated request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section VolumeId:
#' The IDs of the
#' volumes.
#' @section Filter:
#' 
#' The filters.
#' 
#' -   `modification-state` - The current modification state (modifying \|
#'     optimizing \| completed \| failed).
#' 
#' -   `original-iops` - The original IOPS rate of the volume.
#' 
#' -   `original-size` - The original size of the volume, in GiB.
#' 
#' -   `original-volume-type` - The original volume type of the volume
#'     (standard \| io1 \| io2 \| gp2 \| sc1 \| st1).
#' 
#' -   `originalMultiAttachEnabled` - Indicates whether Multi-Attach
#'     support was enabled (true \| false).
#' 
#' -   `start-time` - The modification start time.
#' 
#' -   `target-iops` - The target IOPS rate of the volume.
#' 
#' -   `target-size` - The target size of the volume, in GiB.
#' 
#' -   `target-volume-type` - The target volume type of the volume
#'     (standard \| io1 \| io2 \| gp2 \| sc1 \| st1).
#' 
#' -   `targetMultiAttachEnabled` - Indicates whether Multi-Attach support
#'     is to be enabled (true \| false).
#' 
#' -   `volume-id` - The ID of the volume.
#' 
#' @section MaxResults:
#' The maximum number of results (up
#' to a limit of 500) to be returned in a paginated
#' request.
#' @return A list object or a character vector
#' @export
ec2_describe_volumes_modifications <- function(DryRun = NULL, VolumeId = NULL, Filter = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    VolumeId <- list_to_array("VolumeId", VolumeId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        VolumeId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVolumesModifications", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Attribute
#' 
#' Describes the specified attribute of the specified
#' VPC. You can specify only one attribute at a
#' time.
#' 
#' @param Attribute String. The VPC attribute.
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' The VPC attribute.
#' @section VpcId:
#' The
#' ID of the VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_attribute <- function(Attribute, VpcId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Vpc Classic Link
#' 
#' Describes the ClassicLink status of one or
#' more VPCs.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VpcId List. One or more VPCs for which you want to describe the ClassicLink status.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `is-classic-link-enabled` - Whether the VPC is enabled for
#'     ClassicLink (`true` \| `false`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section VpcId:
#' One or more VPCs for which
#' you want to describe the ClassicLink status.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_classic_link <- function(Filter = NULL, DryRun = NULL, VpcId = NULL, simplify = TRUE, 
    others = list()) {
    VpcId <- list_to_array("VpcId", VpcId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), Filter, VpcId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcClassicLink", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Vpc Classic Link Dns Support
#' 
#' Describes the ClassicLink DNS support
#' status of one or more VPCs. If enabled, the DNS hostname of a linked
#' EC2-Classic instance resolves to its private IP address when addressed
#' from an instance in the VPC to which it\'s linked. Similarly, the DNS
#' hostname of an instance in a VPC resolves to its private IP address when
#' addressed from a linked EC2-Classic instance. For more information, see
#' [ClassicLink](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-classiclink.html)
#' in the *Amazon Elastic Compute Cloud User
#' Guide*.
#' 
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param VpcIds List. One or more VPC IDs.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section MaxResults:
#' The
#' maximum number of results to return with a single call. To retrieve the
#' remaining results, make another call with the returned `nextToken`
#' value.
#' @section VpcIds:
#' One or more VPC
#' IDs.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_classic_link_dns_support <- function(MaxResults = NULL, NextToken = NULL, 
    VpcIds = NULL, simplify = TRUE, others = list()) {
    VpcIds <- list_to_array("VpcIds", VpcIds)
    parameters <- c(others, list(MaxResults = MaxResults, NextToken = NextToken), VpcIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcClassicLinkDnsSupport", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Endpoint Connection Notifications
#' 
#' Describes the connection notifications for
#' VPC endpoints and VPC endpoint services.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ConnectionNotificationId String. The ID of the notification.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return in a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ConnectionNotificationId:
#' The ID of the
#' notification.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `connection-notification-arn` - The ARN of the SNS topic for the
#'     notification.
#' 
#' -   `connection-notification-id` - The ID of the notification.
#' 
#' -   `connection-notification-state` - The state of the notification
#'     (`Enabled` \| `Disabled`).
#' 
#' -   `connection-notification-type` - The type of notification (`Topic`).
#' 
#' -   `service-id` - The ID of the endpoint service.
#' 
#' -   `vpc-endpoint-id` - The ID of the VPC endpoint.
#' 
#' @section MaxResults:
#' The maximum number of results to return in a single
#' call. To retrieve the remaining results, make another request with the
#' returned `NextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_endpoint_connection_notifications <- function(DryRun = NULL, ConnectionNotificationId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, ConnectionNotificationId = ConnectionNotificationId, 
        MaxResults = MaxResults, NextToken = NextToken), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcEndpointConnectionNotifications", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Endpoint Connections
#' 
#' Describes
#' the VPC endpoint connections to your VPC endpoint services, including
#' any endpoints that are pending your
#' acceptance.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `service-id` - The ID of the service.
#' 
#' -   `vpc-endpoint-owner` - The AWS account number of the owner of the
#'     endpoint.
#' 
#' -   `vpc-endpoint-state` - The state of the endpoint
#'     (`pendingAcceptance` \| `pending` \| `available` \| `deleting` \|
#'     `deleted` \| `rejected` \| `failed`).
#' 
#' -   `vpc-endpoint-id` - The ID of the endpoint.
#' 
#' @section MaxResults:
#' The maximum number of results to return for the
#' request in a single page. The remaining results of the initial request
#' can be seen by sending another request with the returned `NextToken`
#' value. This value can be between 5 and 1,000; if `MaxResults` is given a
#' value larger than 1,000, only 1,000 results are
#' returned.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_endpoint_connections <- function(DryRun = NULL, Filter = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcEndpointConnections", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Endpoint Service Configurations
#' 
#' Describes the VPC endpoint service
#' configurations in your account (your
#' services).
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ServiceId List. The IDs of one or more services.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ServiceId:
#' The IDs of one or more
#' services.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `service-name` - The name of the service.
#' 
#' -   `service-id` - The ID of the service.
#' 
#' -   `service-state` - The state of the service (`Pending` \| `Available`
#'     \| `Deleting` \| `Deleted` \| `Failed`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section MaxResults:
#' The maximum number of results to return for the
#' request in a single page. The remaining results of the initial request
#' can be seen by sending another request with the returned `NextToken`
#' value. This value can be between 5 and 1,000; if `MaxResults` is given a
#' value larger than 1,000, only 1,000 results are
#' returned.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_endpoint_service_configurations <- function(DryRun = NULL, ServiceId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    ServiceId <- list_to_array("ServiceId", ServiceId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        ServiceId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcEndpointServiceConfigurations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Endpoint Service Permissions
#' 
#' Describes the principals (service
#' consumers) that are permitted to discover your VPC endpoint
#' service.
#' 
#' @param ServiceId String. The ID of the service.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ServiceId:
#' The ID of the
#' service.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `principal` - The ARN of the principal.
#' 
#' -   `principal-type` - The principal type (`All` \| `Service` \|
#'     `OrganizationUnit` \| `Account` \| `User` \| `Role`).
#' 
#' @section MaxResults:
#' The maximum number of results to return for the
#' request in a single page. The remaining results of the initial request
#' can be seen by sending another request with the returned `NextToken`
#' value. This value can be between 5 and 1,000; if `MaxResults` is given a
#' value larger than 1,000, only 1,000 results are
#' returned.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_endpoint_service_permissions <- function(ServiceId, DryRun = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(ServiceId = ServiceId, DryRun = DryRun, MaxResults = MaxResults, 
        NextToken = NextToken), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcEndpointServicePermissions", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Endpoint Services
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ServiceName List. One or more service names.\[optional\]
#' @param MaxResults Integer.   The maximum number of items to return for this request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ServiceName:
#' One or more service
#' names.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `service-name` - The name of the service.
#' 
#' -   `service-type` - The type of service (`Interface` \| `Gateway`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section MaxResults:
#' 
#' The maximum number of items to return for this request. The request
#' returns a token that you can specify in a subsequent call to get the
#' next set of results.
#' 
#' Constraint: If the value is greater than 1,000, we return only 1,000
#' items.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_endpoint_services <- function(DryRun = NULL, ServiceName = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    ServiceName <- list_to_array("ServiceName", ServiceName)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        ServiceName, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcEndpointServices", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Endpoints
#' 
#' Describes one or more of your VPC
#' endpoints.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VpcEndpointId List. One or more endpoint IDs.\[optional\]
#' @param MaxResults Integer.   The maximum number of items to return for this request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section VpcEndpointId:
#' One or
#' more endpoint IDs.
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `service-name` - The name of the service.
#' 
#' -   `vpc-id` - The ID of the VPC in which the endpoint resides.
#' 
#' -   `vpc-endpoint-id` - The ID of the endpoint.
#' 
#' -   `vpc-endpoint-state` - The state of the endpoint
#'     (`pendingAcceptance` \| `pending` \| `available` \| `deleting` \|
#'     `deleted` \| `rejected` \| `failed`).
#' 
#' -   `vpc-endpoint-type` - The type of VPC endpoint (`Interface` \|
#'     `Gateway` \| `GatewayLoadBalancer`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' @section MaxResults:
#' 
#' The maximum number of items to return for this request. The request
#' returns a token that you can specify in a subsequent call to get the
#' next set of results.
#' 
#' Constraint: If the value is greater than 1,000, we return only 1,000
#' items.
#' 
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_endpoints <- function(DryRun = NULL, VpcEndpointId = NULL, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    VpcEndpointId <- list_to_array("VpcEndpointId", VpcEndpointId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, MaxResults = MaxResults, NextToken = NextToken), 
        VpcEndpointId, Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcEndpoints", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Vpc Peering Connections
#' 
#' Describes one or more of your VPC peering
#' connections.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VpcPeeringConnectionId List.   One or more VPC peering connection IDs.  Default: Describes all your VPC peering connections.  \[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `accepter-vpc-info.cidr-block` - The IPv4 CIDR block of the accepter
#'     VPC.
#' 
#' -   `accepter-vpc-info.owner-id` - The AWS account ID of the owner of
#'     the accepter VPC.
#' 
#' -   `accepter-vpc-info.vpc-id` - The ID of the accepter VPC.
#' 
#' -   `expiration-time` - The expiration date and time for the VPC peering
#'     connection.
#' 
#' -   `requester-vpc-info.cidr-block` - The IPv4 CIDR block of the
#'     requester\'s VPC.
#' 
#' -   `requester-vpc-info.owner-id` - The AWS account ID of the owner of
#'     the requester VPC.
#' 
#' -   `requester-vpc-info.vpc-id` - The ID of the requester VPC.
#' 
#' -   `status-code` - The status of the VPC peering connection
#'     (`pending-acceptance` \| `failed` \| `expired` \| `provisioning` \|
#'     `active` \| `deleting` \| `deleted` \| `rejected`).
#' 
#' -   `status-message` - A message that provides more information about
#'     the status of the VPC peering connection, if applicable.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-peering-connection-id` - The ID of the VPC peering connection.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section VpcPeeringConnectionId:
#' 
#' One or more VPC peering connection IDs.
#' 
#' Default: Describes all your VPC peering connections.
#' 
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_vpc_peering_connections <- function(Filter = NULL, DryRun = NULL, VpcPeeringConnectionId = NULL, 
    NextToken = NULL, MaxResults = NULL, simplify = TRUE, others = list()) {
    VpcPeeringConnectionId <- list_to_array("VpcPeeringConnectionId", VpcPeeringConnectionId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, VpcPeeringConnectionId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcPeeringConnections", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Describe Vpcs
#' 
#' Describes one or more of your
#' VPCs.
#' 
#' @param VpcId List.   One or more VPC IDs.  Default: Describes all your VPCs.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `cidr` - The primary IPv4 CIDR block of the VPC. The CIDR block you
#'     specify must exactly match the VPC\'s CIDR block for information to
#'     be returned for the VPC. Must contain the slash followed by one or
#'     two digits (for example, `/28`).
#' 
#' -   `cidr-block-association.cidr-block` - An IPv4 CIDR block associated
#'     with the VPC.
#' 
#' -   `cidr-block-association.association-id` - The association ID for an
#'     IPv4 CIDR block associated with the VPC.
#' 
#' -   `cidr-block-association.state` - The state of an IPv4 CIDR block
#'     associated with the VPC.
#' 
#' -   `dhcp-options-id` - The ID of a set of DHCP options.
#' 
#' -   `ipv6-cidr-block-association.ipv6-cidr-block` - An IPv6 CIDR block
#'     associated with the VPC.
#' 
#' -   `ipv6-cidr-block-association.ipv6-pool` - The ID of the IPv6 address
#'     pool from which the IPv6 CIDR block is allocated.
#' 
#' -   `ipv6-cidr-block-association.association-id` - The association ID
#'     for an IPv6 CIDR block associated with the VPC.
#' 
#' -   `ipv6-cidr-block-association.state` - The state of an IPv6 CIDR
#'     block associated with the VPC.
#' 
#' -   `isDefault` - Indicates whether the VPC is the default VPC.
#' 
#' -   `owner-id` - The ID of the AWS account that owns the VPC.
#' 
#' -   `state` - The state of the VPC (`pending` \| `available`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `vpc-id` - The ID of the VPC.
#' 
#' @section VpcId:
#' 
#' One or more VPC IDs.
#' 
#' Default: Describes all your VPCs.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_describe_vpcs <- function(Filter = NULL, VpcId = NULL, DryRun = NULL, NextToken = NULL, 
    MaxResults = NULL, simplify = TRUE, others = list()) {
    VpcId <- list_to_array("VpcId", VpcId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun, NextToken = NextToken, MaxResults = MaxResults), 
        Filter, VpcId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpcs", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Describe Vpn Connections
#' 
#' @param VpnConnectionId List.   One or more VPN connection IDs.  Default: Describes your VPN connections.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `customer-gateway-configuration` - The configuration information for
#'     the customer gateway.
#' 
#' -   `customer-gateway-id` - The ID of a customer gateway associated with
#'     the VPN connection.
#' 
#' -   `state` - The state of the VPN connection (`pending` \| `available`
#'     \| `deleting` \| `deleted`).
#' 
#' -   `option.static-routes-only` - Indicates whether the connection has
#'     static routes only. Used for devices that do not support Border
#'     Gateway Protocol (BGP).
#' 
#' -   `route.destination-cidr-block` - The destination CIDR block. This
#'     corresponds to the subnet used in a customer data center.
#' 
#' -   `bgp-asn` - The BGP Autonomous System Number (ASN) associated with a
#'     BGP device.
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `type` - The type of VPN connection. Currently the only supported
#'     type is `ipsec.1`.
#' 
#' -   `vpn-connection-id` - The ID of the VPN connection.
#' 
#' -   `vpn-gateway-id` - The ID of a virtual private gateway associated
#'     with the VPN connection.
#' 
#' -   `transit-gateway-id` - The ID of a transit gateway associated with
#'     the VPN connection.
#' 
#' @section VpnConnectionId:
#' 
#' One or more VPN connection IDs.
#' 
#' Default: Describes your VPN connections.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_vpn_connections <- function(Filter = NULL, VpnConnectionId = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    VpnConnectionId <- list_to_array("VpnConnectionId", VpnConnectionId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), Filter, VpnConnectionId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpnConnections", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Describe Vpn Gateways
#' 
#' @param VpnGatewayId List.   One or more virtual private gateway IDs.  Default: Describes all your virtual private gateways.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Filter:
#' 
#' One or more filters.
#' 
#' -   `amazon-side-asn` - The Autonomous System Number (ASN) for the
#'     Amazon side of the gateway.
#' 
#' -   `attachment.state` - The current state of the attachment between the
#'     gateway and the VPC (`attaching` \| `attached` \| `detaching` \|
#'     `detached`).
#' 
#' -   `attachment.vpc-id` - The ID of an attached VPC.
#' 
#' -   `availability-zone` - The Availability Zone for the virtual private
#'     gateway (if applicable).
#' 
#' -   `state` - The state of the virtual private gateway (`pending` \|
#'     `available` \| `deleting` \| `deleted`).
#' 
#' -   `tag`:\ - The key/value combination of a tag assigned to the
#'     resource. Use the tag key in the filter name and the tag value as
#'     the filter value. For example, to find all resources that have a tag
#'     with the key `Owner` and the value `TeamA`, specify `tag:Owner` for
#'     the filter name and `TeamA` for the filter value.
#' 
#' -   `tag-key` - The key of a tag assigned to the resource. Use this
#'     filter to find all resources assigned a tag with a specific key,
#'     regardless of the tag value.
#' 
#' -   `type` - The type of virtual private gateway. Currently the only
#'     supported type is `ipsec.1`.
#' 
#' -   `vpn-gateway-id` - The ID of the virtual private gateway.
#' 
#' @section VpnGatewayId:
#' 
#' One or more virtual private gateway IDs.
#' 
#' Default: Describes all your virtual private gateways.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_describe_vpn_gateways <- function(Filter = NULL, VpnGatewayId = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    VpnGatewayId <- list_to_array("VpnGatewayId", VpnGatewayId)
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(DryRun = DryRun), Filter, VpnGatewayId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DescribeVpnGateways", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Detach Classic Link Vpc
#' 
#' Unlinks (detaches) a linked EC2-Classic instance
#' from a VPC. After the instance has been unlinked, the VPC security
#' groups are no longer associated with it. An instance is automatically
#' unlinked from a VPC when it\'s stopped.
#' 
#' @param InstanceId String. The ID of the instance to unlink from the VPC.
#' @param VpcId String. The ID of the VPC to which the instance is linked.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the instance to
#' unlink from the VPC.
#' @section VpcId:
#' The ID of the VPC to which the
#' instance is linked.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_detach_classic_link_vpc <- function(InstanceId, VpcId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DetachClassicLinkVpc", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Detach Internet Gateway
#' 
#' Detaches an
#' internet gateway from a VPC, disabling connectivity between the internet
#' and the VPC. The VPC must not contain any running instances with Elastic
#' IP addresses or public IPv4 addresses.
#' 
#' @param InternetGatewayId String. The ID of the internet gateway.
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InternetGatewayId:
#' The ID of the internet
#' gateway.
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_detach_internet_gateway <- function(InternetGatewayId, VpcId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InternetGatewayId = InternetGatewayId, VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DetachInternetGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Detach Network Interface
#' 
#' Detaches a
#' network interface from an instance.
#' 
#' @param AttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Force Logical.   Specifies whether to force a detachment.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Force:
#' 
#' Specifies whether to force a detachment.
#' 
#' -   Use the `Force` parameter only as a last resort to detach a network
#'     interface from a failed instance.
#' 
#' -   If you use the `Force` parameter to detach a network interface, you
#'     might not be able to attach a different network interface to the
#'     same index on the instance without first stopping and starting the
#'     instance.
#' 
#' -   If you force the detachment of a network interface, the [instance
#'     metadata](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html)
#'     might not get updated. This means that the attributes associated
#'     with the detached network interface might still be visible. The
#'     instance metadata will get updated when you stop and start the
#'     instance.
#' 
#' @return A list object or a character vector
#' @export
ec2_detach_network_interface <- function(AttachmentId, DryRun = NULL, Force = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(AttachmentId = AttachmentId, DryRun = DryRun, Force = Force))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DetachNetworkInterface", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Detach Volume
#' 
#' @param VolumeId String. The ID of the volume.
#' @param Device String. The device name.\[optional\]
#' @param Force Logical. Forces detachment if the previous detachment attempt did not occur cleanly (for example, logging into an instance, unmounting the volume, and detaching normally)....\[optional\]
#' @param InstanceId String. The ID of the instance.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VolumeId:
#' The ID of the volume.
#' @section Device:
#' The
#' device name.
#' @section Force:
#' Forces detachment if the previous
#' detachment attempt did not occur cleanly (for example, logging into an
#' instance, unmounting the volume, and detaching normally). This option
#' can lead to data loss or a corrupted file system. Use this option only
#' as a last resort to detach a volume from a failed instance. The instance
#' won\'t have an opportunity to flush file system caches or file system
#' metadata. If you use this option, you must perform file system check and
#' repair procedures.
#' @section InstanceId:
#' The ID of the instance. If you
#' are detaching a Multi-Attach enabled volume, you must specify an
#' instance ID.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_detach_volume <- function(VolumeId, Device = NULL, Force = NULL, InstanceId = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(VolumeId = VolumeId, Device = Device, Force = Force, InstanceId = InstanceId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DetachVolume", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Detach Vpn Gateway
#' 
#' @param VpcId String. The ID of the VPC.
#' @param VpnGatewayId String. The ID of the virtual private gateway.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section VpnGatewayId:
#' The ID of the virtual private
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_detach_vpn_gateway <- function(VpcId, VpnGatewayId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId, VpnGatewayId = VpnGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DetachVpnGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Disable Ebs Encryption By Default
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disable_ebs_encryption_by_default <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisableEbsEncryptionByDefault", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disable Fast Snapshot Restores
#' 
#' Disables fast snapshot restores for the specified
#' snapshots in the specified Availability
#' Zones.
#' 
#' @param AvailabilityZone List. One or more Availability Zones. For example, `us-east-2a`.
#' @param SourceSnapshotId List. The IDs of one or more snapshots. For example, `snap-1234567890abcdef0`.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZone:
#' One or more Availability
#' Zones. For example, `us-east-2a`.
#' @section SourceSnapshotId:
#' The IDs of one or
#' more snapshots. For example,
#' `snap-1234567890abcdef0`.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disable_fast_snapshot_restores <- function(AvailabilityZone, SourceSnapshotId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    AvailabilityZone <- list_to_array("AvailabilityZone", AvailabilityZone)
    SourceSnapshotId <- list_to_array("SourceSnapshotId", SourceSnapshotId)
    parameters <- c(others, list(DryRun = DryRun), AvailabilityZone, SourceSnapshotId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisableFastSnapshotRestores", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Disable Transit Gateway Route Table Propagation
#' 
#' Disables the specified resource attachment
#' from propagating routes to the specified propagation route
#' table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the propagation route table.
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the propagation
#' route table.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disable_transit_gateway_route_table_propagation <- function(TransitGatewayRouteTableId, 
    TransitGatewayAttachmentId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisableTransitGatewayRouteTablePropagation", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disable Vgw Route Propagation
#' 
#' Disables a virtual private gateway (VGW) from
#' propagating routes to a specified route table of a
#' VPC.
#' 
#' @param GatewayId String. The ID of the virtual private gateway.
#' @param RouteTableId String. The ID of the route table.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GatewayId:
#' The ID of the virtual
#' private gateway.
#' @section RouteTableId:
#' The ID of the route
#' table.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disable_vgw_route_propagation <- function(GatewayId, RouteTableId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(GatewayId = GatewayId, RouteTableId = RouteTableId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisableVgwRoutePropagation", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Disable Vpc Classic Link
#' 
#' Disables ClassicLink for a VPC. You cannot
#' disable ClassicLink for a VPC that has EC2-Classic instances linked to
#' it.
#' 
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disable_vpc_classic_link <- function(VpcId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisableVpcClassicLink", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Disable Vpc Classic Link Dns Support
#' 
#' @param VpcId String. The ID of the VPC.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @return A list object or a character vector
#' @export
ec2_disable_vpc_classic_link_dns_support <- function(VpcId = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisableVpcClassicLinkDnsSupport", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disassociate Address
#' 
#' @param AssociationId String. \[EC2-VPC\] The association ID. Required for EC2-VPC.\[optional\]
#' @param PublicIp String. \[EC2-Classic\] The Elastic IP address. Required for EC2-Classic.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' \[EC2-VPC\] The association ID. Required for
#' EC2-VPC.
#' @section PublicIp:
#' \[EC2-Classic\] The Elastic IP address.
#' Required for EC2-Classic.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disassociate_address <- function(AssociationId = NULL, PublicIp = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(AssociationId = AssociationId, PublicIp = PublicIp, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateAddress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Disassociate Client Vpn Target Network
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint from which to disassociate the target network.
#' @param AssociationId String. The ID of the target network association.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint from which to disassociate the target
#' network.
#' @section AssociationId:
#' The ID of the target network
#' association.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disassociate_client_vpn_target_network <- function(ClientVpnEndpointId, AssociationId, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, AssociationId = AssociationId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateClientVpnTargetNetwork", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disassociate Enclave Certificate Iam Role
#' 
#' Disassociates an IAM role from an AWS Certificate
#' Manager (ACM) certificate. Disassociating an IAM role from an ACM
#' certificate removes the Amazon S3 object that contains the certificate,
#' certificate chain, and encrypted private key from the Amazon S3 bucket.
#' It also revokes the IAM role\'s permission to use the AWS Key Management
#' Service (KMS) customer master key (CMK) used to encrypt the private key.
#' This effectively revokes the role\'s permission to use the certificate.
#' 
#' @param CertificateArn String. The ARN of the ACM certificate from which to disassociate the IAM role.\[optional\]
#' @param RoleArn String. The ARN of the IAM role to disassociate.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CertificateArn:
#' The ARN of the ACM
#' certificate from which to disassociate the IAM
#' role.
#' @section RoleArn:
#' The ARN of the IAM role to
#' disassociate.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disassociate_enclave_certificate_iam_role <- function(CertificateArn = NULL, RoleArn = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(CertificateArn = CertificateArn, RoleArn = RoleArn, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateEnclaveCertificateIamRole", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disassociate Iam Instance Profile
#' 
#' @param AssociationId String. The ID of the IAM instance profile association.
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' The ID of the IAM instance
#' profile association.
#' @return A list object or a character vector
#' @export
ec2_disassociate_iam_instance_profile <- function(AssociationId, simplify = TRUE, others = list()) {
    parameters <- c(others, list(AssociationId = AssociationId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateIamInstanceProfile", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disassociate Route Table
#' 
#' @param AssociationId String. The association ID representing the current association between the route table and subnet or gateway....
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' The association ID representing
#' the current association between the route table and subnet or
#' gateway.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disassociate_route_table <- function(AssociationId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(AssociationId = AssociationId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateRouteTable", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Disassociate Subnet Cidr Block
#' 
#' Disassociates a CIDR block from a subnet.
#' Currently, you can disassociate an IPv6 CIDR block only. You must detach
#' or delete all gateways and resources that are associated with the CIDR
#' block before you can disassociate it. 
#' 
#' @param AssociationId String. The association ID for the CIDR block.
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' The association ID for the
#' CIDR block.
#' @return A list object or a character vector
#' @export
ec2_disassociate_subnet_cidr_block <- function(AssociationId, simplify = TRUE, others = list()) {
    parameters <- c(others, list(AssociationId = AssociationId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateSubnetCidrBlock", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Disassociate Transit Gateway Multicast Domain
#' 
#' Disassociates
#' the specified subnets from the transit gateway multicast domain.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param TransitGatewayAttachmentId String. The ID of the attachment.\[optional\]
#' @param SubnetIds List. The IDs of the subnets;\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit gateway multicast
#' domain.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section SubnetIds:
#' The IDs of the
#' subnets;
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disassociate_transit_gateway_multicast_domain <- function(TransitGatewayMulticastDomainId = NULL, 
    TransitGatewayAttachmentId = NULL, SubnetIds = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    SubnetIds <- list_to_array("SubnetIds", SubnetIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun), SubnetIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateTransitGatewayMulticastDomain", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disassociate Transit Gateway Route Table
#' 
#' Disassociates a resource attachment from a transit
#' gateway route table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_disassociate_transit_gateway_route_table <- function(TransitGatewayRouteTableId, TransitGatewayAttachmentId, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateTransitGatewayRouteTable", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Disassociate Vpc Cidr Block
#' 
#' @param AssociationId String. The association ID for the CIDR block.
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' The association ID for the
#' CIDR block.
#' @return A list object or a character vector
#' @export
ec2_disassociate_vpc_cidr_block <- function(AssociationId, simplify = TRUE, others = list()) {
    parameters <- c(others, list(AssociationId = AssociationId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "DisassociateVpcCidrBlock", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Enable Ebs Encryption By Default
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_enable_ebs_encryption_by_default <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "EnableEbsEncryptionByDefault", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Enable Fast Snapshot Restores
#' 
#' @param AvailabilityZone List. One or more Availability Zones. For example, `us-east-2a`.
#' @param SourceSnapshotId List. The IDs of one or more snapshots.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZone:
#' One or more Availability
#' Zones. For example, `us-east-2a`.
#' @section SourceSnapshotId:
#' The IDs of one or
#' more snapshots. For example, `snap-1234567890abcdef0`. You can specify a
#' snapshot that was shared with you from another AWS
#' account.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_enable_fast_snapshot_restores <- function(AvailabilityZone, SourceSnapshotId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    AvailabilityZone <- list_to_array("AvailabilityZone", AvailabilityZone)
    SourceSnapshotId <- list_to_array("SourceSnapshotId", SourceSnapshotId)
    parameters <- c(others, list(DryRun = DryRun), AvailabilityZone, SourceSnapshotId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "EnableFastSnapshotRestores", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Enable Transit Gateway Route Table Propagation
#' 
#' Enables the specified attachment to propagate
#' routes to the specified propagation route
#' table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the propagation route table.
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the propagation
#' route table.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_enable_transit_gateway_route_table_propagation <- function(TransitGatewayRouteTableId, 
    TransitGatewayAttachmentId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "EnableTransitGatewayRouteTablePropagation", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Enable Vgw Route Propagation
#' 
#' Enables a virtual private gateway (VGW) to
#' propagate routes to the specified route table of a
#' VPC.
#' 
#' @param GatewayId String. The ID of the virtual private gateway that is attached to a VPC.
#' @param RouteTableId String. The ID of the route table.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GatewayId:
#' The ID of the virtual
#' private gateway that is attached to a VPC. The virtual private gateway
#' must be attached to the same VPC that the routing tables are associated
#' with. 
#' @section RouteTableId:
#' The ID of the route table. The routing table
#' must be associated with the same VPC that the virtual private gateway is
#' attached to. 
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_enable_vgw_route_propagation <- function(GatewayId, RouteTableId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(GatewayId = GatewayId, RouteTableId = RouteTableId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "EnableVgwRoutePropagation", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Enable Volume IO
#' 
#' Enables I/O operations for a volume that had
#' I/O operations disabled because the data on the volume was potentially
#' inconsistent.
#' 
#' @param VolumeId String. The ID of the volume.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VolumeId:
#' The ID of the
#' volume.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_enable_volume_io <- function(VolumeId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VolumeId = VolumeId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "EnableVolumeIO", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Enable Vpc Classic Link
#' 
#' Enables a VPC for ClassicLink. You can
#' then link EC2-Classic instances to your ClassicLink-enabled VPC to allow
#' communication over private IP addresses. You cannot enable your VPC for
#' ClassicLink if any of your VPC route tables have existing routes for
#' address ranges within the `10.0.0.0/8` IP address range, excluding local
#' routes for VPCs in the `10.0.0.0/16` and `10.1.0.0/16` IP address
#' ranges. For more information, see
#' [ClassicLink](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-classiclink.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' @param VpcId String. The ID of the VPC.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_enable_vpc_classic_link <- function(VpcId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "EnableVpcClassicLink", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Enable Vpc Classic Link Dns Support
#' 
#' @param VpcId String. The ID of the VPC.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @return A list object or a character vector
#' @export
ec2_enable_vpc_classic_link_dns_support <- function(VpcId = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "EnableVpcClassicLinkDnsSupport", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Export Client Vpn Client Certificate Revocation List
#' 
#' Downloads the client certificate revocation list
#' for the specified Client VPN endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_export_client_vpn_client_certificate_revocation_list <- function(ClientVpnEndpointId, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ExportClientVpnClientCertificateRevocationList", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Export Client Vpn Client Configuration
#' 
#' Downloads the
#' contents of the Client VPN endpoint configuration file for the specified
#' Client VPN endpoint. The Client VPN endpoint configuration file includes
#' the Client VPN endpoint and certificate information clients need to
#' establish a connection with the Client VPN
#' endpoint.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_export_client_vpn_client_configuration <- function(ClientVpnEndpointId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ExportClientVpnClientConfiguration", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Export Image
#' 
#' Exports an Amazon Machine Image (AMI) to a
#' VM file. For more information, see [Exporting a VM Directory from an
#' Amazon Machine Image
#' (AMI)](https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport_image.html)
#' in the *VM Import/Export User Guide*.
#' 
#' @param DiskImageFormat String. The disk image format.
#' @param ImageId String. The ID of the image.
#' @param S3ExportLocation Object. Information about the destination Amazon S3 bucket.
#' @param ClientToken String. Token to enable idempotency for export image requests.\[optional\]
#' @param Description String. A description of the image being exported. The maximum length is 255 characters.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param RoleName String. The name of the role that grants VM Import/Export permission to export images to your Amazon S3 bucket....\[optional\]
#' @param TagSpecification List. The tags to apply to the export image task during creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DiskImageFormat:
#' The disk image
#' format.
#' @section ImageId:
#' The ID of the
#' image.
#' @section S3ExportLocation:
#' Information about the destination Amazon S3
#' bucket. The bucket must exist and grant WRITE and READ_ACP permissions
#' to the AWS account
#' vm-import-export\@amazon.com.
#' @section ClientToken:
#' Token to enable
#' idempotency for export image requests.
#' @section Description:
#' A description
#' of the image being exported. The maximum length is 255
#' characters.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section RoleName:
#' The name of the role that
#' grants VM Import/Export permission to export images to your Amazon S3
#' bucket. If this parameter is not specified, the default role is named
#' \'vmimport\'.
#' @section TagSpecification:
#' The tags to apply to the export image
#' task during creation.
#' @return A list object or a character vector
#' @export
ec2_export_image <- function(DiskImageFormat, ImageId, S3ExportLocation, ClientToken = NULL, 
    Description = NULL, DryRun = NULL, RoleName = NULL, TagSpecification = NULL, simplify = TRUE, 
    others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(DiskImageFormat = DiskImageFormat, ImageId = ImageId, S3ExportLocation = S3ExportLocation, 
        ClientToken = ClientToken, Description = Description, DryRun = DryRun, RoleName = RoleName), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ExportImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Export Transit Gateway Routes
#' 
#' @param TransitGatewayRouteTableId String. The ID of the route table.
#' @param S3Bucket String. The name of the S3 bucket.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the route
#' table.
#' @section S3Bucket:
#' The name of the S3
#' bucket.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `attachment.transit-gateway-attachment-id` - The id of the transit
#'     gateway attachment.
#' 
#' -   `attachment.resource-id` - The resource id of the transit gateway
#'     attachment.
#' 
#' -   `route-search.exact-match` - The exact match of the specified
#'     filter.
#' 
#' -   `route-search.longest-prefix-match` - The longest prefix that
#'     matches the route.
#' 
#' -   `route-search.subnet-of-match` - The routes with a subnet that match
#'     the specified CIDR filter.
#' 
#' -   `route-search.supernet-of-match` - The routes with a CIDR that
#'     encompass the CIDR filter. For example, if you have 10.0.1.0/29 and
#'     10.0.1.0/31 routes in your route table and you specify
#'     supernet-of-match as 10.0.1.0/30, then the result returns
#'     10.0.1.0/29.
#' 
#' -   `state` - The state of the route (`active` \| `blackhole`).
#' 
#' -   `transit-gateway-route-destination-cidr-block` - The CIDR range.
#' 
#' -   `type` - The type of route (`propagated` \| `static`).
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_export_transit_gateway_routes <- function(TransitGatewayRouteTableId, S3Bucket, Filter = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        S3Bucket = S3Bucket, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ExportTransitGatewayRoutes", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Get Associated Enclave Certificate Iam Roles
#' 
#' Returns the IAM roles that are associated with the
#' specified AWS Certificate Manager (ACM) certificate. It also returns the
#' name of the Amazon S3 bucket and the Amazon S3 object key where the
#' certificate, certificate chain, and encrypted private key bundle are
#' stored, and the ARN of the AWS Key Management Service (KMS) customer
#' master key (CMK) that\'s used to encrypt the private
#' key.
#' 
#' @param CertificateArn String. The ARN of the ACM certificate for which to view the associated IAM roles, encryption keys, and Amazon...\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CertificateArn:
#' The ARN of the ACM
#' certificate for which to view the associated IAM roles, encryption keys,
#' and Amazon S3 object information.
#' @section DryRun:
#' Checks whether you
#' have the required permissions for the action, without actually making
#' the request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_associated_enclave_certificate_iam_roles <- function(CertificateArn = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(CertificateArn = CertificateArn, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetAssociatedEnclaveCertificateIamRoles", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Get Associated Ipv6 Pool Cidrs
#' 
#' Gets information about the IPv6 CIDR block
#' associations for a specified IPv6 address pool.
#' 
#' @param PoolId String. The ID of the IPv6 address pool.
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PoolId:
#' The ID of the IPv6 address
#' pool.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_associated_ipv6_pool_cidrs <- function(PoolId, NextToken = NULL, MaxResults = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(PoolId = PoolId, NextToken = NextToken, MaxResults = MaxResults, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetAssociatedIpv6PoolCidrs", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Get Capacity Reservation Usage
#' 
#' Gets
#' usage information about a Capacity Reservation. If the Capacity
#' Reservation is shared, it shows usage information for the Capacity
#' Reservation owner and each AWS account that is currently using the
#' shared capacity. If the Capacity Reservation is not shared, it shows
#' only the Capacity Reservation owner\'s
#' usage.
#' 
#' @param CapacityReservationId String. The ID of the Capacity Reservation.
#' @param MaxResults Integer.   The maximum number of results to return for the request in a single page.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CapacityReservationId:
#' The ID of
#' the Capacity Reservation.
#' @section MaxResults:
#' 
#' The maximum number of results to return for the request in a single
#' page. The remaining results can be seen by sending another request with
#' the returned `nextToken` value. This value can be between 5 and 500. If
#' `maxResults` is given a larger value than 500, you receive an error.
#' 
#' Valid range: Minimum value of 1. Maximum value of 1000.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_capacity_reservation_usage <- function(CapacityReservationId, NextToken = NULL, MaxResults = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(CapacityReservationId = CapacityReservationId, NextToken = NextToken, 
        MaxResults = MaxResults, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetCapacityReservationUsage", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Get Coip Pool Usage
#' 
#' Describes the allocations from the specified
#' customer-owned address pool.
#' 
#' @param PoolId String. The ID of the address pool.
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PoolId:
#' The ID of the address
#' pool.
#' @section Filter:
#' 
#' The filters. The following are the possible values:
#' 
#' -   `coip-address-usage.allocation-id`
#' 
#' ```{=html}
#' 
#' ```
#' -   `coip-address-usage.aws-account-id`
#' 
#' ```{=html}
#' 
#' ```
#' -   `coip-address-usage.aws-service`
#' 
#' ```{=html}
#' 
#' ```
#' -   `coip-address-usage.co-ip`
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_coip_pool_usage <- function(PoolId, Filter = NULL, MaxResults = NULL, NextToken = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(PoolId = PoolId, MaxResults = MaxResults, NextToken = NextToken, 
        DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetCoipPoolUsage", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Get Console Output
#' 
#' @param InstanceId String. The ID of the instance.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Latest Logical.   When enabled, retrieves the latest console output for the instance.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Latest:
#' 
#' When enabled, retrieves the latest console output for the instance.
#' 
#' Default: disabled (`false`)
#' 
#' @return A list object or a character vector
#' @export
ec2_get_console_output <- function(InstanceId, DryRun = NULL, Latest = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, DryRun = DryRun, Latest = Latest))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetConsoleOutput", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Get Console Screenshot
#' 
#' @param InstanceId String. The ID of the instance.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param WakeUp Logical. When set to `true`, acts as keystroke input and wakes up an instance that\'s in standby or \'sleep\'...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the instance.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section WakeUp:
#' When set
#' to `true`, acts as keystroke input and wakes up an instance that\'s in
#' standby or \'sleep\' mode.
#' @return A list object or a character vector
#' @export
ec2_get_console_screenshot <- function(InstanceId, DryRun = NULL, WakeUp = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, DryRun = DryRun, WakeUp = WakeUp))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetConsoleScreenshot", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Get Default Credit Specification
#' 
#' @param InstanceFamily String. The instance family.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceFamily:
#' The instance
#' family.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_default_credit_specification <- function(InstanceFamily, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InstanceFamily = InstanceFamily, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetDefaultCreditSpecification", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Get Ebs Default Kms Key Id
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_ebs_default_kms_key_id <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetEbsDefaultKmsKeyId", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Get Ebs Encryption By Default
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_ebs_encryption_by_default <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetEbsEncryptionByDefault", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Get Groups For Capacity Reservation
#' 
#' Lists the resource groups to which a Capacity
#' Reservation has been added.
#' 
#' @param CapacityReservationId String. The ID of the Capacity Reservation.
#' @param MaxResults Integer. The maximum number of results to return for the request in a single page.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CapacityReservationId:
#' The ID of the Capacity
#' Reservation.
#' @section MaxResults:
#' The maximum number of results to
#' return for the request in a single page. The remaining results can be
#' seen by sending another request with the returned `nextToken` value.
#' This value can be between 5 and 500. If `maxResults` is given a larger
#' value than 500, you receive an error.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_groups_for_capacity_reservation <- function(CapacityReservationId, NextToken = NULL, 
    MaxResults = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(CapacityReservationId = CapacityReservationId, NextToken = NextToken, 
        MaxResults = MaxResults, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetGroupsForCapacityReservation", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Get Host Reservation Purchase Preview
#' 
#' @param HostIdSet List. The IDs of the Dedicated Hosts with which the reservation is associated.
#' @param OfferingId String. The offering ID of the reservation.
#' @inheritParams CommonDoc
#' 
#' @section HostIdSet:
#' The IDs of
#' the Dedicated Hosts with which the reservation is
#' associated.
#' @section OfferingId:
#' The offering ID of the
#' reservation.
#' @return A list object or a character vector
#' @export
ec2_get_host_reservation_purchase_preview <- function(HostIdSet, OfferingId, simplify = TRUE, 
    others = list()) {
    HostIdSet <- list_to_array("HostIdSet", HostIdSet)
    parameters <- c(others, list(OfferingId = OfferingId), HostIdSet)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetHostReservationPurchasePreview", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Get Launch Template Data
#' 
#' @param InstanceId String. The ID of the instance.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_launch_template_data <- function(InstanceId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetLaunchTemplateData", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Get Managed Prefix List Associations
#' 
#' Gets information about the resources that are
#' associated with the specified managed prefix
#' list.
#' 
#' @param PrefixListId String. The ID of the prefix list.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PrefixListId:
#' The ID of the prefix
#' list.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section MaxResults:
#' The maximum number of
#' results to return with a single call. To retrieve the remaining results,
#' make another call with the returned `nextToken`
#' value.
#' @return A list object or a character vector
#' @export
ec2_get_managed_prefix_list_associations <- function(PrefixListId, DryRun = NULL, MaxResults = NULL, 
    NextToken = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(PrefixListId = PrefixListId, DryRun = DryRun, MaxResults = MaxResults, 
        NextToken = NextToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetManagedPrefixListAssociations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Get Managed Prefix List Entries
#' 
#' Gets information about the entries for a
#' specified managed prefix list.
#' 
#' @param PrefixListId String. The ID of the prefix list.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TargetVersion Integer. The version of the prefix list for which to return the entries. The default is the current version.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PrefixListId:
#' The ID of the prefix
#' list.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TargetVersion:
#' The version of the prefix
#' list for which to return the entries. The default is the current
#' version.
#' @section MaxResults:
#' The maximum number of results to return
#' with a single call. To retrieve the remaining results, make another call
#' with the returned `nextToken` value.
#' @return A list object or a character vector
#' @export
ec2_get_managed_prefix_list_entries <- function(PrefixListId, DryRun = NULL, TargetVersion = NULL, 
    MaxResults = NULL, NextToken = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(PrefixListId = PrefixListId, DryRun = DryRun, TargetVersion = TargetVersion, 
        MaxResults = MaxResults, NextToken = NextToken))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetManagedPrefixListEntries", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Get Password Data
#' 
#' @param InstanceId String. The ID of the Windows instance.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the Windows
#' instance.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_password_data <- function(InstanceId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetPasswordData", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Get Reserved Instances Exchange Quote
#' 
#' Returns a quote and exchange information for
#' exchanging one or more specified Convertible Reserved Instances for a
#' new Convertible Reserved Instance. If the exchange cannot be performed,
#' the reason is returned in the response. Use
#' AcceptReservedInstancesExchangeQuote to perform the
#' exchange.
#' 
#' @param ReservedInstanceId List. The IDs of the Convertible Reserved Instances to exchange.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TargetConfiguration List. The configuration of the target Convertible Reserved Instance to exchange for your current Convertible...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ReservedInstanceId:
#' The IDs of the Convertible
#' Reserved Instances to exchange.
#' @section DryRun:
#' Checks whether you
#' have the required permissions for the action, without actually making
#' the request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TargetConfiguration:
#' The configuration of the
#' target Convertible Reserved Instance to exchange for your current
#' Convertible Reserved Instances.
#' @return A list object or a character vector
#' @export
ec2_get_reserved_instances_exchange_quote <- function(ReservedInstanceId, DryRun = NULL, 
    TargetConfiguration = NULL, simplify = TRUE, others = list()) {
    ReservedInstanceId <- list_to_array("ReservedInstanceId", ReservedInstanceId)
    TargetConfiguration <- list_to_array("TargetConfiguration", TargetConfiguration)
    parameters <- c(others, list(DryRun = DryRun), ReservedInstanceId, TargetConfiguration)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetReservedInstancesExchangeQuote", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Get Transit Gateway Attachment Propagations
#' 
#' Lists the route tables to which the
#' specified resource attachment propagates
#' routes.
#' 
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `transit-gateway-route-table-id` - The ID of the transit gateway
#'     route table.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_transit_gateway_attachment_propagations <- function(TransitGatewayAttachmentId, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetTransitGatewayAttachmentPropagations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Get Transit Gateway Multicast Domain Associations
#' 
#' Gets information about the associations for
#' the transit gateway multicast domain.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit
#' gateway multicast domain.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `resource-id` - The ID of the resource.
#' 
#' -   `resource-type` - The type of resource. The valid value is: `vpc`.
#' 
#' -   `state` - The state of the subnet association. Valid values are
#'     `associated` \| `associating` \| `disassociated` \|
#'     `disassociating`.
#' 
#' -   `subnet-id` - The ID of the subnet.
#' 
#' -   `transit-gateway-attachment-id` - The id of the transit gateway
#'     attachment.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_transit_gateway_multicast_domain_associations <- function(TransitGatewayMulticastDomainId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetTransitGatewayMulticastDomainAssociations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Get Transit Gateway Prefix List References
#' 
#' Gets
#' information about the prefix list references in a specified transit
#' gateway route table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `attachment.resource-id` - The ID of the resource for the
#'     attachment.
#' 
#' -   `attachment.resource-type` - The type of resource for the
#'     attachment. Valid values are `vpc` \| `vpn` \|
#'     `direct-connect-gateway` \| `peering`.
#' 
#' -   `attachment.transit-gateway-attachment-id` - The ID of the
#'     attachment.
#' 
#' -   `is-blackhole` - Whether traffic matching the route is blocked
#'     (`true` \| `false`).
#' 
#' -   `prefix-list-id` - The ID of the prefix list.
#' 
#' -   `prefix-list-owner-id` - The ID of the owner of the prefix list.
#' 
#' -   `state` - The state of the prefix list reference (`pending` \|
#'     `available` \| `modifying` \| `deleting`).
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_transit_gateway_prefix_list_references <- function(TransitGatewayRouteTableId, Filter = NULL, 
    MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetTransitGatewayPrefixListReferences", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Get Transit Gateway Route Table Associations
#' 
#' Gets information about the
#' associations for the specified transit gateway route
#' table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `resource-id` - The ID of the resource.
#' 
#' -   `resource-type` - The resource type. Valid values are `vpc` \| `vpn`
#'     \| `direct-connect-gateway` \| `peering` \| `connect`.
#' 
#' -   `transit-gateway-attachment-id` - The ID of the attachment.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_transit_gateway_route_table_associations <- function(TransitGatewayRouteTableId, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetTransitGatewayRouteTableAssociations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Get Transit Gateway Route Table Propagations
#' 
#' Gets information about the route table
#' propagations for the specified transit gateway route
#' table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `resource-id` - The ID of the resource.
#' 
#' -   `resource-type` - The resource type. Valid values are `vpc` \| `vpn`
#'     \| `direct-connect-gateway` \| `peering` \| `connect`.
#' 
#' -   `transit-gateway-attachment-id` - The ID of the attachment.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_get_transit_gateway_route_table_propagations <- function(TransitGatewayRouteTableId, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "GetTransitGatewayRouteTablePropagations", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Import Client Vpn Client Certificate Revocation List
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint to which the client certificate revocation list applies.
#' @param CertificateRevocationList String. The client certificate revocation list file.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint to which the client certificate revocation list
#' applies.
#' @section CertificateRevocationList:
#' The client certificate revocation list
#' file. For more information, see [Generate a Client Certificate
#' Revocation
#' List](https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/cvpn-working-certificates.html#cvpn-working-certificates-generate)
#' in the *AWS Client VPN Administrator Guide*.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_import_client_vpn_client_certificate_revocation_list <- function(ClientVpnEndpointId, 
    CertificateRevocationList, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, CertificateRevocationList = CertificateRevocationList, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ImportClientVpnClientCertificateRevocationList", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Import Image
#' 
#' Import single or multi-volume disk images or EBS
#' snapshots into an Amazon Machine Image (AMI). For more information, see
#' [Importing a VM as an Image Using VM
#' Import/Export](https://docs.aws.amazon.com/vm-import/latest/userguide/vmimport-image-import.html)
#' in the *VM Import/Export User Guide*.
#' 
#' @param Architecture String.   The architecture of the virtual machine.  Valid values: `i386` \| `x86_64` \| `arm64`  \[optional\]
#' @param ClientData Object. The client-specific data.\[optional\]
#' @param ClientToken String. The token to enable idempotency for VM import requests.\[optional\]
#' @param Description String. A description string for the import image task.\[optional\]
#' @param DiskContainer List. Information about the disk containers.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Encrypted Logical. Specifies whether the destination AMI of the imported image should be encrypted.\[optional\]
#' @param Hypervisor String.   The target hypervisor platform.  Valid values: `xen`  \[optional\]
#' @param KmsKeyId String.   An identifier for the symmetric AWS Key Management Service (AWS KMS) customer master key (CMK)...\[optional\]
#' @param LicenseType String.   The license type to be used for the Amazon Machine Image (AMI) after importing.\[optional\]
#' @param Platform String.   The operating system of the virtual machine.  Valid values: `Windows` \| `Linux`  \[optional\]
#' @param RoleName String. The name of the role to use when not using the default role, \'vmimport\'.\[optional\]
#' @param LicenseSpecifications List. The ARNs of the license configurations.\[optional\]
#' @param TagSpecification List. The tags to apply to the import image task during creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Architecture:
#' 
#' The architecture of the virtual machine.
#' 
#' Valid values: `i386` \| `x86_64` \| `arm64`
#' 
#' @section ClientData:
#' The client-specific data.
#' @section ClientToken:
#' The
#' token to enable idempotency for VM import
#' requests.
#' @section Description:
#' A description string for the import image
#' task.
#' @section DiskContainer:
#' Information about the disk
#' containers.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Encrypted:
#' Specifies whether the
#' destination AMI of the imported image should be encrypted. The default
#' CMK for EBS is used unless you specify a non-default AWS Key Management
#' Service (AWS KMS) CMK using `KmsKeyId`. For more information, see
#' [Amazon EBS
#' Encryption](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' @section Hypervisor:
#' 
#' The target hypervisor platform.
#' 
#' Valid values: `xen`
#' 
#' @section KmsKeyId:
#' 
#' An identifier for the symmetric AWS Key Management Service (AWS KMS)
#' customer master key (CMK) to use when creating the encrypted AMI. This
#' parameter is only required if you want to use a non-default CMK; if this
#' parameter is not specified, the default CMK for EBS is used. If a
#' `KmsKeyId` is specified, the `Encrypted` flag must also be set.
#' 
#' The CMK identifier may be provided in any of the following formats:
#' 
#' -   Key ID
#' 
#' -   Key alias. The alias ARN contains the `arn:aws:kms` namespace,
#'     followed by the Region of the CMK, the AWS account ID of the CMK
#'     owner, the `alias` namespace, and then the CMK alias. For example,
#'     arn:aws:kms:*us-east-1*:*012345678910*:alias/*ExampleAlias*.
#' 
#' -   ARN using key ID. The ID ARN contains the `arn:aws:kms` namespace,
#'     followed by the Region of the CMK, the AWS account ID of the CMK
#'     owner, the `key` namespace, and then the CMK ID. For example,
#'     arn:aws:kms:*us-east-1*:*012345678910*:key/*abcd1234-a123-456a-a12b-a123b4cd56ef*.
#' 
#' -   ARN using key alias. The alias ARN contains the `arn:aws:kms`
#'     namespace, followed by the Region of the CMK, the AWS account ID of
#'     the CMK owner, the `alias` namespace, and then the CMK alias. For
#'     example,
#'     arn:aws:kms:*us-east-1*:*012345678910*:alias/*ExampleAlias*.
#' 
#' AWS parses `KmsKeyId` asynchronously, meaning that the action you call
#' may appear to complete even though you provided an invalid identifier.
#' This action will eventually report failure.
#' 
#' The specified CMK must exist in the Region that the AMI is being copied
#' to.
#' 
#' Amazon EBS does not support asymmetric CMKs.
#' 
#' @section LicenseType:
#' 
#' The license type to be used for the Amazon Machine Image (AMI) after
#' importing.
#' 
#' By default, we detect the source-system operating system (OS) and apply
#' the appropriate license. Specify `AWS` to replace the source-system
#' license with an AWS license, if appropriate. Specify `BYOL` to retain
#' the source-system license, if appropriate.
#' 
#' To use `BYOL`, you must have existing licenses with rights to use these
#' licenses in a third party cloud, such as AWS. For more information, see
#' [Prerequisites](https://docs.aws.amazon.com/vm-import/latest/userguide/vmimport-image-import.html#prerequisites-image)
#' in the VM Import/Export User Guide.
#' 
#' @section Platform:
#' 
#' The operating system of the virtual machine.
#' 
#' Valid values: `Windows` \| `Linux`
#' 
#' @section RoleName:
#' The name of the role to use when not using the
#' default role, \'vmimport\'.
#' @section LicenseSpecifications:
#' The ARNs of the license
#' configurations.
#' @section TagSpecification:
#' The tags to apply to the import
#' image task during creation.
#' @return A list object or a character vector
#' @export
ec2_import_image <- function(Architecture = NULL, ClientData = NULL, ClientToken = NULL, 
    Description = NULL, DiskContainer = NULL, DryRun = NULL, Encrypted = NULL, Hypervisor = NULL, 
    KmsKeyId = NULL, LicenseType = NULL, Platform = NULL, RoleName = NULL, LicenseSpecifications = NULL, 
    TagSpecification = NULL, simplify = TRUE, others = list()) {
    DiskContainer <- list_to_array("DiskContainer", DiskContainer)
    LicenseSpecifications <- list_to_array("LicenseSpecifications", LicenseSpecifications)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(Architecture = Architecture, ClientData = ClientData, ClientToken = ClientToken, 
        Description = Description, DryRun = DryRun, Encrypted = Encrypted, Hypervisor = Hypervisor, 
        KmsKeyId = KmsKeyId, LicenseType = LicenseType, Platform = Platform, RoleName = RoleName), 
        DiskContainer, LicenseSpecifications, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ImportImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Import Instance
#' 
#' @param Platform String. The instance operating system.
#' @param Description String. A description for the instance being imported.\[optional\]
#' @param DiskImage List. The disk image.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LaunchSpecification Object. The launch specification.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Platform:
#' The instance operating
#' system.
#' @section Description:
#' A description for the instance being
#' imported.
#' @section DiskImage:
#' The disk image.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section LaunchSpecification:
#' The launch
#' specification.
#' @return A list object or a character vector
#' @export
ec2_import_instance <- function(Platform, Description = NULL, DiskImage = NULL, DryRun = NULL, 
    LaunchSpecification = NULL, simplify = TRUE, others = list()) {
    DiskImage <- list_to_array("DiskImage", DiskImage)
    parameters <- c(others, list(Platform = Platform, Description = Description, DryRun = DryRun, 
        LaunchSpecification = LaunchSpecification), DiskImage)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ImportInstance", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Import Key Pair
#' 
#' @param KeyName String. A unique name for the key pair.
#' @param PublicKeyMaterial String. The public key.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to apply to the imported key pair.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section KeyName:
#' A unique name for the key
#' pair.
#' @section PublicKeyMaterial:
#' The public key. For API calls, the text must
#' be base64-encoded. For command line tools, base64 encoding is performed
#' for you.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to apply to the
#' imported key pair.
#' @return A list object or a character vector
#' @export
ec2_import_key_pair <- function(KeyName, PublicKeyMaterial, DryRun = NULL, TagSpecification = NULL, 
    simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(KeyName = KeyName, PublicKeyMaterial = PublicKeyMaterial, 
        DryRun = DryRun), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ImportKeyPair", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Import Snapshot
#' 
#' Imports a disk into an EBS
#' snapshot.
#' 
#' @param ClientData Object. The client-specific data.\[optional\]
#' @param ClientToken String. Token to enable idempotency for VM import requests.\[optional\]
#' @param Description String. The description string for the import snapshot task.\[optional\]
#' @param DiskContainer Object. Information about the disk container.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Encrypted Logical. Specifies whether the destination snapshot of the imported image should be encrypted.\[optional\]
#' @param KmsKeyId String.   An identifier for the symmetric AWS Key Management Service (AWS KMS) customer master key (CMK)...\[optional\]
#' @param RoleName String. The name of the role to use when not using the default role, \'vmimport\'.\[optional\]
#' @param TagSpecification List. The tags to apply to the import snapshot task during creation.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientData:
#' The client-specific
#' data.
#' @section ClientToken:
#' Token to enable idempotency for VM import
#' requests.
#' @section Description:
#' The description string for the import
#' snapshot task.
#' @section DiskContainer:
#' Information about the disk
#' container.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Encrypted:
#' Specifies whether the
#' destination snapshot of the imported image should be encrypted. The
#' default CMK for EBS is used unless you specify a non-default AWS Key
#' Management Service (AWS KMS) CMK using `KmsKeyId`. For more information,
#' see [Amazon EBS
#' Encryption](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' @section KmsKeyId:
#' 
#' An identifier for the symmetric AWS Key Management Service (AWS KMS)
#' customer master key (CMK) to use when creating the encrypted snapshot.
#' This parameter is only required if you want to use a non-default CMK; if
#' this parameter is not specified, the default CMK for EBS is used. If a
#' `KmsKeyId` is specified, the `Encrypted` flag must also be set.
#' 
#' The CMK identifier may be provided in any of the following formats:
#' 
#' -   Key ID
#' 
#' -   Key alias. The alias ARN contains the `arn:aws:kms` namespace,
#'     followed by the Region of the CMK, the AWS account ID of the CMK
#'     owner, the `alias` namespace, and then the CMK alias. For example,
#'     arn:aws:kms:*us-east-1*:*012345678910*:alias/*ExampleAlias*.
#' 
#' -   ARN using key ID. The ID ARN contains the `arn:aws:kms` namespace,
#'     followed by the Region of the CMK, the AWS account ID of the CMK
#'     owner, the `key` namespace, and then the CMK ID. For example,
#'     arn:aws:kms:*us-east-1*:*012345678910*:key/*abcd1234-a123-456a-a12b-a123b4cd56ef*.
#' 
#' -   ARN using key alias. The alias ARN contains the `arn:aws:kms`
#'     namespace, followed by the Region of the CMK, the AWS account ID of
#'     the CMK owner, the `alias` namespace, and then the CMK alias. For
#'     example,
#'     arn:aws:kms:*us-east-1*:*012345678910*:alias/*ExampleAlias*.
#' 
#' AWS parses `KmsKeyId` asynchronously, meaning that the action you call
#' may appear to complete even though you provided an invalid identifier.
#' This action will eventually report failure.
#' 
#' The specified CMK must exist in the Region that the snapshot is being
#' copied to.
#' 
#' Amazon EBS does not support asymmetric CMKs.
#' 
#' @section RoleName:
#' The name of the role to use when not using the
#' default role, \'vmimport\'.
#' @section TagSpecification:
#' The tags to apply to the
#' import snapshot task during creation.
#' @return A list object or a character vector
#' @export
ec2_import_snapshot <- function(ClientData = NULL, ClientToken = NULL, Description = NULL, 
    DiskContainer = NULL, DryRun = NULL, Encrypted = NULL, KmsKeyId = NULL, RoleName = NULL, 
    TagSpecification = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(ClientData = ClientData, ClientToken = ClientToken, Description = Description, 
        DiskContainer = DiskContainer, DryRun = DryRun, Encrypted = Encrypted, KmsKeyId = KmsKeyId, 
        RoleName = RoleName), TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ImportSnapshot", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Import Volume
#' 
#' @param AvailabilityZone String. The Availability Zone for the resulting EBS volume.
#' @param Image Object. The disk image.
#' @param Volume Object. The volume size.
#' @param Description String. A description of the volume.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZone:
#' The
#' Availability Zone for the resulting EBS volume.
#' @section Image:
#' The
#' disk image.
#' @section Volume:
#' The volume size.
#' @section Description:
#' A
#' description of the volume.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_import_volume <- function(AvailabilityZone, Image, Volume, Description = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(AvailabilityZone = AvailabilityZone, Image = Image, Volume = Volume, 
        Description = Description, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ImportVolume", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Address Attribute
#' 
#' Modifies an attribute of the specified Elastic IP
#' address. For requirements, see [Using reverse DNS for email
#' applications](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html#Using_Elastic_Addressing_Reverse_DNS).
#' 
#' @param AllocationId String. \[EC2-VPC\] The allocation ID.
#' @param DomainName String. The domain name to modify for the IP address.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AllocationId:
#' \[EC2-VPC\] The allocation
#' ID.
#' @section DomainName:
#' The domain name to modify for the IP
#' address.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_address_attribute <- function(AllocationId, DomainName = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(AllocationId = AllocationId, DomainName = DomainName, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyAddressAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Availability Zone Group
#' 
#' @param GroupName String. The name of the Availability Zone group, Local Zone group, or Wavelength Zone group.
#' @param OptInStatus String. Indicates whether you are opted in to the Local Zone group or Wavelength Zone group.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GroupName:
#' The name of the
#' Availability Zone group, Local Zone group, or Wavelength Zone
#' group.
#' @section OptInStatus:
#' Indicates whether you are opted in to the
#' Local Zone group or Wavelength Zone group. The only valid value is
#' `opted-in`. You must contact [AWS
#' Support](https://console.aws.amazon.com/support/home#/case/create\%3FissueType=customer-service\%26serviceCode=general-info\%26getting-started\%26categoryCode=using-aws\%26services)
#' to opt out of a Local Zone group, or Wavelength Zone
#' group.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_availability_zone_group <- function(GroupName, OptInStatus, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(GroupName = GroupName, OptInStatus = OptInStatus, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyAvailabilityZoneGroup", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Capacity Reservation
#' 
#' Modifies a Capacity Reservation\'s capacity and the
#' conditions under which it is to be released. You cannot change a
#' Capacity Reservation\'s instance type, EBS optimization, instance store
#' settings, platform, Availability Zone, or instance eligibility. If you
#' need to modify any of these attributes, we recommend that you cancel the
#' Capacity Reservation, and then create a new one with the required
#' attributes.
#' 
#' @param CapacityReservationId String. The ID of the Capacity Reservation.
#' @param InstanceCount Integer. The number of instances for which to reserve capacity.\[optional\]
#' @param EndDate String.   The date and time at which the Capacity Reservation expires.\[optional\]
#' @param EndDateType String.   Indicates the way in which the Capacity Reservation ends.\[optional\]
#' @param Accept Logical. Reserved. Capacity Reservations you have created are accepted by default.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CapacityReservationId:
#' The ID of the Capacity
#' Reservation.
#' @section InstanceCount:
#' The number of instances for which to
#' reserve capacity.
#' @section EndDate:
#' 
#' The date and time at which the Capacity Reservation expires. When a
#' Capacity Reservation expires, the reserved capacity is released and you
#' can no longer launch instances into it. The Capacity Reservation\'s
#' state changes to `expired` when it reaches its end date and time.
#' 
#' The Capacity Reservation is cancelled within an hour from the specified
#' time. For example, if you specify 5/31/2019, 13:30:55, the Capacity
#' Reservation is guaranteed to end between 13:30:55 and 14:30:55 on
#' 5/31/2019.
#' 
#' You must provide an `EndDate` value if `EndDateType` is `limited`. Omit
#' `EndDate` if `EndDateType` is `unlimited`.
#' 
#' @section EndDateType:
#' 
#' Indicates the way in which the Capacity Reservation ends. A Capacity
#' Reservation can have one of the following end types:
#' 
#' -   `unlimited` - The Capacity Reservation remains active until you
#'     explicitly cancel it. Do not provide an `EndDate` value if
#'     `EndDateType` is `unlimited`.
#' 
#' -   `limited` - The Capacity Reservation expires automatically at a
#'     specified date and time. You must provide an `EndDate` value if
#'     `EndDateType` is `limited`.
#' 
#' @section Accept:
#' Reserved. Capacity Reservations you have created
#' are accepted by default.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_capacity_reservation <- function(CapacityReservationId, InstanceCount = NULL, 
    EndDate = NULL, EndDateType = NULL, Accept = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(CapacityReservationId = CapacityReservationId, InstanceCount = InstanceCount, 
        EndDate = EndDate, EndDateType = EndDateType, Accept = Accept, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyCapacityReservation", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Client Vpn Endpoint
#' 
#' Modifies the specified Client VPN
#' endpoint. Modifying the DNS server resets existing client
#' connections.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint to modify.
#' @param ServerCertificateArn String. The ARN of the server certificate to be used.\[optional\]
#' @param ConnectionLogOptions Object.   Information about the client connection logging options.\[optional\]
#' @param DnsServers Object. Information about the DNS servers to be used by Client VPN connections.\[optional\]
#' @param VpnPort Integer.   The port number to assign to the Client VPN endpoint for TCP and UDP traffic.\[optional\]
#' @param Description String. A brief description of the Client VPN endpoint.\[optional\]
#' @param SplitTunnel Logical.   Indicates whether the VPN is split-tunnel.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param SecurityGroupId List. The IDs of one or more security groups to apply to the target network.\[optional\]
#' @param VpcId String. The ID of the VPC to associate with the Client VPN endpoint.\[optional\]
#' @param SelfServicePortal String. Specify whether to enable the self-service portal for the Client VPN endpoint.\[optional\]
#' @param ClientConnectOptions Object. The options for managing connection authorization for new client connections.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint to modify.
#' @section ServerCertificateArn:
#' The ARN of the server
#' certificate to be used. The server certificate must be provisioned in
#' AWS Certificate Manager (ACM).
#' @section ConnectionLogOptions:
#' 
#' Information about the client connection logging options.
#' 
#' If you enable client connection logging, data about client connections
#' is sent to a Cloudwatch Logs log stream. The following information is
#' logged:
#' 
#' -   Client connection requests
#' 
#' -   Client connection results (successful and unsuccessful)
#' 
#' -   Reasons for unsuccessful client connection requests
#' 
#' -   Client connection termination time
#' 
#' @section DnsServers:
#' Information about the DNS servers to be used by
#' Client VPN connections. A Client VPN endpoint can have up to two DNS
#' servers.
#' @section VpnPort:
#' 
#' The port number to assign to the Client VPN endpoint for TCP and UDP
#' traffic.
#' 
#' Valid Values: `443` \| `1194`
#' 
#' Default Value: `443`
#' 
#' @section Description:
#' A brief description of the Client VPN
#' endpoint.
#' @section SplitTunnel:
#' 
#' Indicates whether the VPN is split-tunnel.
#' 
#' For information about split-tunnel VPN endpoints, see [Split-Tunnel AWS
#' Client VPN
#' Endpoint](https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html)
#' in the *AWS Client VPN Administrator Guide*.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section SecurityGroupId:
#' The IDs of one or more
#' security groups to apply to the target network.
#' @section VpcId:
#' The
#' ID of the VPC to associate with the Client VPN
#' endpoint.
#' @section SelfServicePortal:
#' Specify whether to enable the self-service
#' portal for the Client VPN endpoint.
#' @section ClientConnectOptions:
#' The options for
#' managing connection authorization for new client
#' connections.
#' @return A list object or a character vector
#' @export
ec2_modify_client_vpn_endpoint <- function(ClientVpnEndpointId, ServerCertificateArn = NULL, 
    ConnectionLogOptions = NULL, DnsServers = NULL, VpnPort = NULL, Description = NULL, SplitTunnel = NULL, 
    DryRun = NULL, SecurityGroupId = NULL, VpcId = NULL, SelfServicePortal = NULL, ClientConnectOptions = NULL, 
    simplify = TRUE, others = list()) {
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, ServerCertificateArn = ServerCertificateArn, 
        ConnectionLogOptions = ConnectionLogOptions, DnsServers = DnsServers, VpnPort = VpnPort, 
        Description = Description, SplitTunnel = SplitTunnel, DryRun = DryRun, VpcId = VpcId, 
        SelfServicePortal = SelfServicePortal, ClientConnectOptions = ClientConnectOptions), 
        SecurityGroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyClientVpnEndpoint", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Default Credit Specification
#' 
#' @param InstanceFamily String. The instance family.
#' @param CpuCredits String.   The credit option for CPU usage of the instance family.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceFamily:
#' The instance
#' family.
#' @section CpuCredits:
#' 
#' The credit option for CPU usage of the instance family.
#' 
#' Valid Values: `standard` \| `unlimited`
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_default_credit_specification <- function(InstanceFamily, CpuCredits, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceFamily = InstanceFamily, CpuCredits = CpuCredits, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyDefaultCreditSpecification", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Ebs Default Kms Key Id
#' 
#' @param KmsKeyId String.   The identifier of the AWS Key Management Service (AWS KMS) customer master key (CMK) to use for...
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section KmsKeyId:
#' 
#' The identifier of the AWS Key Management Service (AWS KMS) customer
#' master key (CMK) to use for Amazon EBS encryption. If this parameter is
#' not specified, your AWS managed CMK for EBS is used. If `KmsKeyId` is
#' specified, the encrypted state must be `true`.
#' 
#' You can specify the CMK using any of the following:
#' 
#' -   Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Key alias. For example, alias/ExampleAlias.
#' 
#' -   Key ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
#' 
#' -   Alias ARN. For example,
#'     arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
#' 
#' AWS authenticates the CMK asynchronously. Therefore, if you specify an
#' ID, alias, or ARN that is not valid, the action can appear to complete,
#' but eventually fails.
#' 
#' Amazon EBS does not support asymmetric CMKs.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_ebs_default_kms_key_id <- function(KmsKeyId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(KmsKeyId = KmsKeyId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyEbsDefaultKmsKeyId", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Fleet
#' 
#' @param FleetId String. The ID of the EC2 Fleet.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ExcessCapacityTerminationPolicy String. Indicates whether running instances should be terminated if the total target capacity of the EC2...\[optional\]
#' @param LaunchTemplateConfig List. The launch template and overrides.\[optional\]
#' @param TargetCapacitySpecification Object. The size of the EC2 Fleet.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FleetId:
#' The ID of the EC2
#' Fleet.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ExcessCapacityTerminationPolicy:
#' Indicates whether running
#' instances should be terminated if the total target capacity of the EC2
#' Fleet is decreased below the current size of the EC2
#' Fleet.
#' @section LaunchTemplateConfig:
#' The launch template and
#' overrides.
#' @section TargetCapacitySpecification:
#' The size of the EC2
#' Fleet.
#' @return A list object or a character vector
#' @export
ec2_modify_fleet <- function(FleetId, DryRun = NULL, ExcessCapacityTerminationPolicy = NULL, 
    LaunchTemplateConfig = NULL, TargetCapacitySpecification = NULL, simplify = TRUE, others = list()) {
    LaunchTemplateConfig <- list_to_array("LaunchTemplateConfig", LaunchTemplateConfig)
    parameters <- c(others, list(FleetId = FleetId, DryRun = DryRun, ExcessCapacityTerminationPolicy = ExcessCapacityTerminationPolicy, 
        TargetCapacitySpecification = TargetCapacitySpecification), LaunchTemplateConfig)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyFleet", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Fpga Image Attribute
#' 
#' Modifies the specified attribute of the specified
#' Amazon FPGA Image (AFI).
#' 
#' @param FpgaImageId String. The ID of the AFI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Attribute String. The name of the attribute.\[optional\]
#' @param OperationType String. The operation type.\[optional\]
#' @param UserId List. The AWS account IDs. This parameter is valid only when modifying the `loadPermission` attribute.\[optional\]
#' @param UserGroup List. The user groups. This parameter is valid only when modifying the `loadPermission` attribute.\[optional\]
#' @param ProductCode List. The product codes.\[optional\]
#' @param LoadPermission Object. The load permission for the AFI.\[optional\]
#' @param Description String. A description for the AFI.\[optional\]
#' @param Name String. A name for the AFI.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FpgaImageId:
#' The ID of the AFI.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section Attribute:
#' The name
#' of the attribute.
#' @section OperationType:
#' The operation
#' type.
#' @section UserId:
#' The AWS account IDs. This parameter is valid
#' only when modifying the `loadPermission`
#' attribute.
#' @section UserGroup:
#' The user groups. This parameter is valid
#' only when modifying the `loadPermission`
#' attribute.
#' @section ProductCode:
#' The product codes. After you add a
#' product code to an AFI, it can\'t be removed. This parameter is valid
#' only when modifying the `productCodes`
#' attribute.
#' @section LoadPermission:
#' The load permission for the
#' AFI.
#' @section Description:
#' A description for the
#' AFI.
#' @section Name:
#' A name for the AFI.
#' @return A list object or a character vector
#' @export
ec2_modify_fpga_image_attribute <- function(FpgaImageId, DryRun = NULL, Attribute = NULL, 
    OperationType = NULL, UserId = NULL, UserGroup = NULL, ProductCode = NULL, LoadPermission = NULL, 
    Description = NULL, Name = NULL, simplify = TRUE, others = list()) {
    UserId <- list_to_array("UserId", UserId)
    UserGroup <- list_to_array("UserGroup", UserGroup)
    ProductCode <- list_to_array("ProductCode", ProductCode)
    parameters <- c(others, list(FpgaImageId = FpgaImageId, DryRun = DryRun, Attribute = Attribute, 
        OperationType = OperationType, LoadPermission = LoadPermission, Description = Description, 
        Name = Name), UserId, UserGroup, ProductCode)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyFpgaImageAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Hosts
#' 
#' @param HostId List. The IDs of the Dedicated Hosts to modify.
#' @param AutoPlacement String. Specify whether to enable or disable auto-placement.\[optional\]
#' @param HostRecovery String. Indicates whether to enable or disable host recovery for the Dedicated Host.\[optional\]
#' @param InstanceType String.   Specifies the instance type to be supported by the Dedicated Host.\[optional\]
#' @param InstanceFamily String.   Specifies the instance family to be supported by the Dedicated Host.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section HostId:
#' The IDs
#' of the Dedicated Hosts to modify.
#' @section AutoPlacement:
#' Specify whether to
#' enable or disable auto-placement.
#' @section HostRecovery:
#' Indicates whether
#' to enable or disable host recovery for the Dedicated Host. For more
#' information, see [Host
#' recovery](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html)
#' in the *Amazon EC2 User Guide*.
#' @section InstanceType:
#' 
#' Specifies the instance type to be supported by the Dedicated Host.
#' Specify this parameter to modify a Dedicated Host to support only a
#' specific instance type.
#' 
#' If you want to modify a Dedicated Host to support multiple instance
#' types in its current instance family, omit this parameter and specify
#' **InstanceFamily** instead. You cannot specify **InstanceType** and
#' **InstanceFamily** in the same request.
#' 
#' @section InstanceFamily:
#' 
#' Specifies the instance family to be supported by the Dedicated Host.
#' Specify this parameter to modify a Dedicated Host to support multiple
#' instance types within its current instance family.
#' 
#' If you want to modify a Dedicated Host to support a specific instance
#' type only, omit this parameter and specify **InstanceType** instead. You
#' cannot specify **InstanceFamily** and **InstanceType** in the same
#' request.
#' 
#' @return A list object or a character vector
#' @export
ec2_modify_hosts <- function(HostId, AutoPlacement = NULL, HostRecovery = NULL, InstanceType = NULL, 
    InstanceFamily = NULL, simplify = TRUE, others = list()) {
    HostId <- list_to_array("HostId", HostId)
    parameters <- c(others, list(AutoPlacement = AutoPlacement, HostRecovery = HostRecovery, 
        InstanceType = InstanceType, InstanceFamily = InstanceFamily), HostId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyHosts", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Id Format
#' 
#' @param Resource String.   The type of resource: `bundle` \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \|...
#' @param UseLongIds Logical. Indicate whether the resource should use longer IDs (17-character IDs).
#' @inheritParams CommonDoc
#' 
#' @section Resource:
#' 
#' The type of resource: `bundle` \| `conversion-task` \|
#' `customer-gateway` \| `dhcp-options` \| `elastic-ip-allocation` \|
#' `elastic-ip-association` \| `export-task` \| `flow-log` \| `image` \|
#' `import-task` \| `internet-gateway` \| `network-acl` \|
#' `network-acl-association` \| `network-interface` \|
#' `network-interface-attachment` \| `prefix-list` \| `route-table` \|
#' `route-table-association` \| `security-group` \| `subnet` \|
#' `subnet-cidr-block-association` \| `vpc` \| `vpc-cidr-block-association`
#' \| `vpc-endpoint` \| `vpc-peering-connection` \| `vpn-connection` \|
#' `vpn-gateway`.
#' 
#' Alternatively, use the `all-current` option to include all resource
#' types that are currently within their opt-in period for longer IDs.
#' 
#' @section UseLongIds:
#' Indicate whether the resource should use longer IDs
#' (17-character IDs).
#' @return A list object or a character vector
#' @export
ec2_modify_id_format <- function(Resource, UseLongIds, simplify = TRUE, others = list()) {
    parameters <- c(others, list(Resource = Resource, UseLongIds = UseLongIds))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyIdFormat", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Identity Id Format
#' 
#' @param PrincipalArn String. The ARN of the principal, which can be an IAM user, IAM role, or the root user.
#' @param Resource String.   The type of resource: `bundle` \| `conversion-task` \| `customer-gateway` \| `dhcp-options` \|...
#' @param UseLongIds Logical. Indicates whether the resource should use longer IDs (17-character IDs)
#' @inheritParams CommonDoc
#' 
#' @section PrincipalArn:
#' The ARN of the principal, which
#' can be an IAM user, IAM role, or the root user. Specify `all` to modify
#' the ID format for all IAM users, IAM roles, and the root user of the
#' account.
#' @section Resource:
#' 
#' The type of resource: `bundle` \| `conversion-task` \|
#' `customer-gateway` \| `dhcp-options` \| `elastic-ip-allocation` \|
#' `elastic-ip-association` \| `export-task` \| `flow-log` \| `image` \|
#' `import-task` \| `internet-gateway` \| `network-acl` \|
#' `network-acl-association` \| `network-interface` \|
#' `network-interface-attachment` \| `prefix-list` \| `route-table` \|
#' `route-table-association` \| `security-group` \| `subnet` \|
#' `subnet-cidr-block-association` \| `vpc` \| `vpc-cidr-block-association`
#' \| `vpc-endpoint` \| `vpc-peering-connection` \| `vpn-connection` \|
#' `vpn-gateway`.
#' 
#' Alternatively, use the `all-current` option to include all resource
#' types that are currently within their opt-in period for longer IDs.
#' 
#' @section UseLongIds:
#' Indicates whether the resource should use longer
#' IDs (17-character IDs)
#' @return A list object or a character vector
#' @export
ec2_modify_identity_id_format <- function(PrincipalArn, Resource, UseLongIds, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(PrincipalArn = PrincipalArn, Resource = Resource, UseLongIds = UseLongIds))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyIdentityIdFormat", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Image Attribute
#' 
#' @param ImageId String. The ID of the AMI.
#' @param Attribute String. The name of the attribute to modify.\[optional\]
#' @param Description Object. A new description for the AMI.\[optional\]
#' @param LaunchPermission Object. A new launch permission for the AMI.\[optional\]
#' @param OperationType String. The operation type.\[optional\]
#' @param ProductCode List. The DevPay product codes. After you add a product code to an AMI, it can\'t be removed.\[optional\]
#' @param UserGroup List. The user groups.\[optional\]
#' @param UserId List. The AWS account IDs.\[optional\]
#' @param Value String. The value of the attribute being modified.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ImageId:
#' The ID of the
#' AMI.
#' @section Attribute:
#' The name of the attribute to modify. The valid
#' values are `description`, `launchPermission`, and
#' `productCodes`.
#' @section Description:
#' A new description for the
#' AMI.
#' @section LaunchPermission:
#' A new launch permission for the
#' AMI.
#' @section OperationType:
#' The operation type. This parameter can be used
#' only when the `Attribute` parameter is
#' `launchPermission`.
#' @section ProductCode:
#' The DevPay product codes. After
#' you add a product code to an AMI, it can\'t be
#' removed.
#' @section UserGroup:
#' The user groups. This parameter can be used
#' only when the `Attribute` parameter is
#' `launchPermission`.
#' @section UserId:
#' The AWS account IDs. This
#' parameter can be used only when the `Attribute` parameter is
#' `launchPermission`.
#' @section Value:
#' The value of the attribute being
#' modified. This parameter can be used only when the `Attribute` parameter
#' is `description` or `productCodes`.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_image_attribute <- function(ImageId, Attribute = NULL, Description = NULL, LaunchPermission = NULL, 
    OperationType = NULL, ProductCode = NULL, UserGroup = NULL, UserId = NULL, Value = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    ProductCode <- list_to_array("ProductCode", ProductCode)
    UserGroup <- list_to_array("UserGroup", UserGroup)
    UserId <- list_to_array("UserId", UserId)
    parameters <- c(others, list(ImageId = ImageId, Attribute = Attribute, Description = Description, 
        LaunchPermission = LaunchPermission, OperationType = OperationType, Value = Value, 
        DryRun = DryRun), ProductCode, UserGroup, UserId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyImageAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Instance Attribute
#' 
#' @param InstanceId String. The ID of the instance.
#' @param SourceDestCheck Object. Specifies whether source/destination checking is enabled.\[optional\]
#' @param Attribute String. The name of the attribute.\[optional\]
#' @param BlockDeviceMapping List.   Modifies the `DeleteOnTermination` attribute for volumes that are currently attached.\[optional\]
#' @param DisableApiTermination Object. If the value is `true`, you can\'t terminate the instance using the Amazon EC2 console, CLI, or API;...\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EbsOptimized Object. Specifies whether the instance is optimized for Amazon EBS I/O.\[optional\]
#' @param EnaSupport Object.   Set to `true` to enable enhanced networking with ENA for the instance.\[optional\]
#' @param GroupId List. \[EC2-VPC\] Changes the security groups of the instance.\[optional\]
#' @param InstanceInitiatedShutdownBehavior Object. Specifies whether an instance stops or terminates when you initiate shutdown from the instance (using the operating system command for system shutdown)....\[optional\]
#' @param InstanceType Object. Changes the instance type to the specified value.\[optional\]
#' @param Kernel Object. Changes the instance\'s kernel to the specified value.\[optional\]
#' @param Ramdisk Object. Changes the instance\'s RAM disk to the specified value.\[optional\]
#' @param SriovNetSupport Object.   Set to `simple` to enable enhanced networking with the Intel 82599 Virtual Function interface for...\[optional\]
#' @param UserData Object. Changes the instance\'s user data to the specified value.\[optional\]
#' @param Value String. A new value for the attribute.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of
#' the instance.
#' @section SourceDestCheck:
#' Specifies whether source/destination
#' checking is enabled. A value of `true` means that checking is enabled,
#' and `false` means that checking is disabled. This value must be `false`
#' for a NAT instance to perform NAT.
#' @section Attribute:
#' The name of the
#' attribute.
#' @section BlockDeviceMapping:
#' 
#' Modifies the `DeleteOnTermination` attribute for volumes that are
#' currently attached. The volume must be owned by the caller. If no value
#' is specified for `DeleteOnTermination`, the default is `true` and the
#' volume is deleted when the instance is terminated.
#' 
#' To add instance store volumes to an Amazon EBS-backed instance, you must
#' add them when you launch the instance. For more information, see
#' [Updating the block device mapping when launching an
#' instance](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html#Using_OverridingAMIBDM)
#' in the *Amazon EC2 User Guide*.
#' 
#' @section DisableApiTermination:
#' If the value is `true`, you can\'t terminate the
#' instance using the Amazon EC2 console, CLI, or API; otherwise, you can.
#' You cannot use this parameter for Spot
#' Instances.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section EbsOptimized:
#' Specifies whether the
#' instance is optimized for Amazon EBS I/O. This optimization provides
#' dedicated throughput to Amazon EBS and an optimized configuration stack
#' to provide optimal EBS I/O performance. This optimization isn\'t
#' available with all instance types. Additional usage charges apply when
#' using an EBS Optimized instance.
#' @section EnaSupport:
#' 
#' Set to `true` to enable enhanced networking with ENA for the instance.
#' 
#' This option is supported only for HVM instances. Specifying this option
#' with a PV instance can make it unreachable.
#' 
#' @section GroupId:
#' \[EC2-VPC\] Changes the security groups of the
#' instance. You must specify at least one security group, even if it\'s
#' just the default security group for the VPC. You must specify the
#' security group ID, not the security group
#' name.
#' @section InstanceInitiatedShutdownBehavior:
#' Specifies whether an instance stops or
#' terminates when you initiate shutdown from the instance (using the
#' operating system command for system
#' shutdown).
#' @section InstanceType:
#' Changes the instance type to the
#' specified value. For more information, see [Instance
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html)
#' in the *Amazon EC2 User Guide*. If the instance type is not valid, the
#' error returned is
#' `InvalidInstanceAttributeValue`.
#' @section Kernel:
#' Changes the
#' instance\'s kernel to the specified value. We recommend that you use
#' PV-GRUB instead of kernels and RAM disks. For more information, see
#' [PV-GRUB](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html).
#' @section Ramdisk:
#' Changes
#' the instance\'s RAM disk to the specified value. We recommend that you
#' use PV-GRUB instead of kernels and RAM disks. For more information, see
#' [PV-GRUB](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html).
#' @section SriovNetSupport:
#' 
#' Set to `simple` to enable enhanced networking with the Intel 82599
#' Virtual Function interface for the instance.
#' 
#' There is no way to disable enhanced networking with the Intel 82599
#' Virtual Function interface at this time.
#' 
#' This option is supported only for HVM instances. Specifying this option
#' with a PV instance can make it unreachable.
#' 
#' @section UserData:
#' Changes the instance\'s user data to the specified
#' value. If you are using an AWS SDK or command line tool, base64-encoding
#' is performed for you, and you can load the text from a file. Otherwise,
#' you must provide base64-encoded text.
#' @section Value:
#' A new value
#' for the attribute. Use only with the `kernel`, `ramdisk`, `userData`,
#' `disableApiTermination`, or `instanceInitiatedShutdownBehavior`
#' attribute.
#' @return A list object or a character vector
#' @export
ec2_modify_instance_attribute <- function(InstanceId, SourceDestCheck = NULL, Attribute = NULL, 
    BlockDeviceMapping = NULL, DisableApiTermination = NULL, DryRun = NULL, EbsOptimized = NULL, 
    EnaSupport = NULL, GroupId = NULL, InstanceInitiatedShutdownBehavior = NULL, InstanceType = NULL, 
    Kernel = NULL, Ramdisk = NULL, SriovNetSupport = NULL, UserData = NULL, Value = NULL, 
    simplify = TRUE, others = list()) {
    BlockDeviceMapping <- list_to_array("BlockDeviceMapping", BlockDeviceMapping)
    GroupId <- list_to_array("GroupId", GroupId)
    parameters <- c(others, list(InstanceId = InstanceId, SourceDestCheck = SourceDestCheck, 
        Attribute = Attribute, DisableApiTermination = DisableApiTermination, DryRun = DryRun, 
        EbsOptimized = EbsOptimized, EnaSupport = EnaSupport, InstanceInitiatedShutdownBehavior = InstanceInitiatedShutdownBehavior, 
        InstanceType = InstanceType, Kernel = Kernel, Ramdisk = Ramdisk, SriovNetSupport = SriovNetSupport, 
        UserData = UserData, Value = Value), BlockDeviceMapping, GroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyInstanceAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Instance Capacity Reservation Attributes
#' 
#' Modifies the Capacity Reservation settings for a
#' stopped instance. Use this action to configure an instance to target a
#' specific Capacity Reservation, run in any `open` Capacity Reservation
#' with matching attributes, or run On-Demand Instance
#' capacity.
#' 
#' @param InstanceId String. The ID of the instance to be modified.
#' @param CapacityReservationSpecification Object. Information about the Capacity Reservation targeting option.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the instance to be
#' modified.
#' @section CapacityReservationSpecification:
#' Information about the Capacity Reservation
#' targeting option.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_instance_capacity_reservation_attributes <- function(InstanceId, CapacityReservationSpecification, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, CapacityReservationSpecification = CapacityReservationSpecification, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyInstanceCapacityReservationAttributes", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Instance Credit Specification
#' 
#' @param InstanceCreditSpecification List. Information about the credit option for CPU usage.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String. A unique, case-sensitive token that you provide to ensure idempotency of your modification request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceCreditSpecification:
#' Information about the
#' credit option for CPU usage.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' A unique, case-sensitive
#' token that you provide to ensure idempotency of your modification
#' request. For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_modify_instance_credit_specification <- function(InstanceCreditSpecification, DryRun = NULL, 
    ClientToken = NULL, simplify = TRUE, others = list()) {
    InstanceCreditSpecification <- list_to_array("InstanceCreditSpecification", InstanceCreditSpecification)
    parameters <- c(others, list(DryRun = DryRun, ClientToken = ClientToken), InstanceCreditSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyInstanceCreditSpecification", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Instance Event Start Time
#' 
#' Modifies the start time for a scheduled Amazon EC2
#' instance event.
#' 
#' @param InstanceId String. The ID of the instance with the scheduled event.
#' @param InstanceEventId String. The ID of the event whose date and time you are modifying.
#' @param NotBefore String. The new date and time when the event will take place.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The
#' ID of the instance with the scheduled event.
#' @section InstanceEventId:
#' The ID
#' of the event whose date and time you are
#' modifying.
#' @section NotBefore:
#' The new date and time when the event will
#' take place.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_instance_event_start_time <- function(InstanceId, InstanceEventId, NotBefore, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, InstanceEventId = InstanceEventId, 
        NotBefore = NotBefore, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyInstanceEventStartTime", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Instance Metadata Options
#' 
#' Modify the instance metadata
#' parameters on a running or stopped instance. When you modify the
#' parameters on a stopped instance, they are applied when the instance is
#' started. When you modify the parameters on a running instance, the API
#' responds with a state of _pending_. After the parameter modifications
#' are successfully applied to the instance, the state of the modifications
#' changes from _pending_ to _applied_ in subsequent describe-instances
#' API calls. For more information, see [Instance metadata and user
#' data](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' @param InstanceId String. The ID of the instance.
#' @param HttpTokens String.   The state of token usage for your instance metadata requests.\[optional\]
#' @param HttpPutResponseHopLimit Integer.   The desired HTTP PUT response hop limit for instance metadata requests.\[optional\]
#' @param HttpEndpoint String.   This parameter enables or disables the HTTP metadata endpoint on your instances.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section HttpTokens:
#' 
#' The state of token usage for your instance metadata requests. If the
#' parameter is not specified in the request, the default state is
#' `optional`.
#' 
#' If the state is `optional`, you can choose to retrieve instance metadata
#' with or without a signed token header on your request. If you retrieve
#' the IAM role credentials without a token, the version 1.0 role
#' credentials are returned. If you retrieve the IAM role credentials using
#' a valid signed token, the version 2.0 role credentials are returned.
#' 
#' If the state is `required`, you must send a signed token header with any
#' instance metadata retrieval requests. In this state, retrieving the IAM
#' role credential always returns the version 2.0 credentials; the version
#' 1.0 credentials are not available.
#' 
#' @section HttpPutResponseHopLimit:
#' 
#' The desired HTTP PUT response hop limit for instance metadata requests.
#' The larger the number, the further instance metadata requests can
#' travel. If no parameter is specified, the existing state is maintained.
#' 
#' Possible values: Integers from 1 to 64
#' 
#' @section HttpEndpoint:
#' 
#' This parameter enables or disables the HTTP metadata endpoint on your
#' instances. If the parameter is not specified, the existing state is
#' maintained.
#' 
#' If you specify a value of `disabled`, you will not be able to access
#' your instance metadata.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_instance_metadata_options <- function(InstanceId, HttpTokens = NULL, HttpPutResponseHopLimit = NULL, 
    HttpEndpoint = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, HttpTokens = HttpTokens, HttpPutResponseHopLimit = HttpPutResponseHopLimit, 
        HttpEndpoint = HttpEndpoint, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyInstanceMetadataOptions", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Instance Placement
#' 
#' @param InstanceId String. The ID of the instance that you are modifying.
#' @param Affinity String. The affinity setting for the instance.\[optional\]
#' @param GroupName String.   The name of the placement group in which to place the instance.\[optional\]
#' @param HostId String. The ID of the Dedicated Host with which to associate the instance.\[optional\]
#' @param Tenancy String. The tenancy for the instance.\[optional\]
#' @param PartitionNumber Integer. Reserved for future use.\[optional\]
#' @param HostResourceGroupArn String. The ARN of the host resource group in which to place the instance.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the instance that
#' you are modifying.
#' @section Affinity:
#' The affinity setting for the
#' instance.
#' @section GroupName:
#' 
#' The name of the placement group in which to place the instance. For
#' spread placement groups, the instance must have a tenancy of `default`.
#' For cluster and partition placement groups, the instance must have a
#' tenancy of `default` or `dedicated`.
#' 
#' To remove an instance from a placement group, specify an empty string
#' (\'\').
#' 
#' @section HostId:
#' The ID of the Dedicated Host with which to
#' associate the instance.
#' @section Tenancy:
#' The tenancy for the
#' instance.
#' @section PartitionNumber:
#' Reserved for future
#' use.
#' @section HostResourceGroupArn:
#' The ARN of the host resource group in which to
#' place the instance.
#' @return A list object or a character vector
#' @export
ec2_modify_instance_placement <- function(InstanceId, Affinity = NULL, GroupName = NULL, 
    HostId = NULL, Tenancy = NULL, PartitionNumber = NULL, HostResourceGroupArn = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, Affinity = Affinity, GroupName = GroupName, 
        HostId = HostId, Tenancy = Tenancy, PartitionNumber = PartitionNumber, HostResourceGroupArn = HostResourceGroupArn))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyInstancePlacement", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Launch Template
#' 
#' Modifies a launch template. You can specify which
#' version of the launch template to set as the default version. When
#' launching an instance, the default version applies when a launch
#' template version is not specified.
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ClientToken String.   Unique, case-sensitive identifier you provide to ensure the idempotency of the request.\[optional\]
#' @param LaunchTemplateId String. The ID of the launch template.\[optional\]
#' @param LaunchTemplateName String. The name of the launch template.\[optional\]
#' @param SetDefaultVersion String. The version number of the launch template to set as the default version.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ClientToken:
#' 
#' Unique, case-sensitive identifier you provide to ensure the idempotency
#' of the request. For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' 
#' Constraint: Maximum 128 ASCII characters.
#' 
#' @section LaunchTemplateId:
#' The ID of the launch template. You must specify
#' either the launch template ID or launch template name in the
#' request.
#' @section LaunchTemplateName:
#' The name of the launch template. You must
#' specify either the launch template ID or launch template name in the
#' request.
#' @section SetDefaultVersion:
#' The version number of the launch template
#' to set as the default version.
#' @return A list object or a character vector
#' @export
ec2_modify_launch_template <- function(DryRun = NULL, ClientToken = NULL, LaunchTemplateId = NULL, 
    LaunchTemplateName = NULL, SetDefaultVersion = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun, ClientToken = ClientToken, LaunchTemplateId = LaunchTemplateId, 
        LaunchTemplateName = LaunchTemplateName, SetDefaultVersion = SetDefaultVersion))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyLaunchTemplate", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Managed Prefix List
#' 
#' @param PrefixListId String. The ID of the prefix list.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param CurrentVersion Integer. The current version of the prefix list.\[optional\]
#' @param PrefixListName String. A name for the prefix list.\[optional\]
#' @param AddEntry List. One or more entries to add to the prefix list.\[optional\]
#' @param RemoveEntry List. One or more entries to remove from the prefix list.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PrefixListId:
#' The ID of the prefix
#' list.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section CurrentVersion:
#' The current version of the
#' prefix list.
#' @section PrefixListName:
#' A name for the prefix
#' list.
#' @section AddEntry:
#' One or more entries to add to the prefix
#' list.
#' @section RemoveEntry:
#' One or more entries to remove from the prefix
#' list.
#' @return A list object or a character vector
#' @export
ec2_modify_managed_prefix_list <- function(PrefixListId, DryRun = NULL, CurrentVersion = NULL, 
    PrefixListName = NULL, AddEntry = NULL, RemoveEntry = NULL, simplify = TRUE, others = list()) {
    AddEntry <- list_to_array("AddEntry", AddEntry)
    RemoveEntry <- list_to_array("RemoveEntry", RemoveEntry)
    parameters <- c(others, list(PrefixListId = PrefixListId, DryRun = DryRun, CurrentVersion = CurrentVersion, 
        PrefixListName = PrefixListName), AddEntry, RemoveEntry)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyManagedPrefixList", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Network Interface Attribute
#' 
#' Modifies the specified network interface attribute.
#' You can specify only one attribute at a time. You can use this action to
#' attach and detach security groups from an existing EC2
#' instance.
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param Attachment Object. Information about the interface attachment.\[optional\]
#' @param Description Object. A description for the network interface.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param SecurityGroupId List. Changes the security groups for the network interface.\[optional\]
#' @param SourceDestCheck Object. Indicates whether source/destination checking is enabled.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section Attachment:
#' Information about the interface
#' attachment. If modifying the \'delete on termination\' attribute, you
#' must specify the ID of the interface attachment.
#' @section Description:
#' A
#' description for the network interface.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section SecurityGroupId:
#' Changes
#' the security groups for the network interface. The new set of groups you
#' specify replaces the current set. You must specify at least one group,
#' even if it\'s just the default security group in the VPC. You must
#' specify the ID of the security group, not the
#' name.
#' @section SourceDestCheck:
#' Indicates whether source/destination checking
#' is enabled. A value of `true` means checking is enabled, and `false`
#' means checking is disabled. This value must be `false` for a NAT
#' instance to perform NAT. For more information, see [NAT
#' Instances](https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_NAT_Instance.html)
#' in the *Amazon Virtual Private Cloud User
#' Guide*.
#' @return A list object or a character vector
#' @export
ec2_modify_network_interface_attribute <- function(NetworkInterfaceId, Attachment = NULL, 
    Description = NULL, DryRun = NULL, SecurityGroupId = NULL, SourceDestCheck = NULL, simplify = TRUE, 
    others = list()) {
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, Attachment = Attachment, 
        Description = Description, DryRun = DryRun, SourceDestCheck = SourceDestCheck), SecurityGroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyNetworkInterfaceAttribute", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Reserved Instances
#' 
#' @param ReservedInstancesId List. The IDs of the Reserved Instances to modify.
#' @param ReservedInstancesConfigurationSetItemType List. The configuration settings for the Reserved Instances to modify.
#' @param ClientToken String. A unique, case-sensitive token you provide to ensure idempotency of your modification request.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ReservedInstancesId:
#' The IDs of the Reserved Instances to
#' modify.
#' @section ReservedInstancesConfigurationSetItemType:
#' The configuration settings for the Reserved
#' Instances to modify.
#' @section ClientToken:
#' A unique, case-sensitive token
#' you provide to ensure idempotency of your modification request. For more
#' information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @return A list object or a character vector
#' @export
ec2_modify_reserved_instances <- function(ReservedInstancesId, ReservedInstancesConfigurationSetItemType, 
    ClientToken = NULL, simplify = TRUE, others = list()) {
    ReservedInstancesId <- list_to_array("ReservedInstancesId", ReservedInstancesId)
    ReservedInstancesConfigurationSetItemType <- list_to_array("ReservedInstancesConfigurationSetItemType", 
        ReservedInstancesConfigurationSetItemType)
    parameters <- c(others, list(ClientToken = ClientToken), ReservedInstancesId, ReservedInstancesConfigurationSetItemType)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyReservedInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Snapshot Attribute
#' 
#' @param SnapshotId String. The ID of the snapshot.
#' @param Attribute String. The snapshot attribute to modify. Only volume creation permissions can be modified.\[optional\]
#' @param CreateVolumePermission Object. A JSON representation of the snapshot attribute modification.\[optional\]
#' @param UserGroup List. The group to modify for the snapshot.\[optional\]
#' @param OperationType String. The type of operation to perform to the attribute.\[optional\]
#' @param UserId List. The account ID to modify for the snapshot.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SnapshotId:
#' The
#' ID of the snapshot.
#' @section Attribute:
#' The snapshot attribute to
#' modify. Only volume creation permissions can be
#' modified.
#' @section CreateVolumePermission:
#' A JSON representation of the snapshot
#' attribute modification.
#' @section UserGroup:
#' The group to modify for the
#' snapshot.
#' @section OperationType:
#' The type of operation to perform to the
#' attribute.
#' @section UserId:
#' The account ID to modify for the
#' snapshot.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_snapshot_attribute <- function(SnapshotId, Attribute = NULL, CreateVolumePermission = NULL, 
    UserGroup = NULL, OperationType = NULL, UserId = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    UserGroup <- list_to_array("UserGroup", UserGroup)
    UserId <- list_to_array("UserId", UserId)
    parameters <- c(others, list(SnapshotId = SnapshotId, Attribute = Attribute, CreateVolumePermission = CreateVolumePermission, 
        OperationType = OperationType, DryRun = DryRun), UserGroup, UserId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifySnapshotAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Spot Fleet Request
#' 
#' @param SpotFleetRequestId String. The ID of the Spot Fleet request.
#' @param ExcessCapacityTerminationPolicy String. Indicates whether running Spot Instances should be terminated if the target capacity of the Spot...\[optional\]
#' @param LaunchTemplateConfig List. The launch template and overrides.\[optional\]
#' @param TargetCapacity Integer. The size of the fleet.\[optional\]
#' @param OnDemandTargetCapacity Integer. The number of On-Demand Instances in the fleet.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SpotFleetRequestId:
#' The ID of the Spot Fleet
#' request.
#' @section ExcessCapacityTerminationPolicy:
#' Indicates whether running Spot Instances
#' should be terminated if the target capacity of the Spot Fleet request is
#' decreased below the current size of the Spot
#' Fleet.
#' @section LaunchTemplateConfig:
#' The launch template and overrides. You can
#' only use this parameter if you specified a launch template
#' (`LaunchTemplateConfigs`) in your Spot Fleet request. If you specified
#' `LaunchSpecifications` in your Spot Fleet request, then omit this
#' parameter.
#' @section TargetCapacity:
#' The size of the
#' fleet.
#' @section OnDemandTargetCapacity:
#' The number of On-Demand Instances in the
#' fleet.
#' @return A list object or a character vector
#' @export
ec2_modify_spot_fleet_request <- function(SpotFleetRequestId, ExcessCapacityTerminationPolicy = NULL, 
    LaunchTemplateConfig = NULL, TargetCapacity = NULL, OnDemandTargetCapacity = NULL, simplify = TRUE, 
    others = list()) {
    LaunchTemplateConfig <- list_to_array("LaunchTemplateConfig", LaunchTemplateConfig)
    parameters <- c(others, list(SpotFleetRequestId = SpotFleetRequestId, ExcessCapacityTerminationPolicy = ExcessCapacityTerminationPolicy, 
        TargetCapacity = TargetCapacity, OnDemandTargetCapacity = OnDemandTargetCapacity), 
        LaunchTemplateConfig)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifySpotFleetRequest", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Subnet Attribute
#' 
#' Modifies a subnet attribute. You can only modify
#' one attribute at a time.
#' 
#' @param SubnetId String. The ID of the subnet.
#' @param AssignIpv6AddressOnCreation Object.   Specify `true` to indicate that network interfaces created in the specified subnet should be assigned...\[optional\]
#' @param MapPublicIpOnLaunch Object. Specify `true` to indicate that network interfaces attached to instances created in the specified...\[optional\]
#' @param MapCustomerOwnedIpOnLaunch Object.   Specify `true` to indicate that network interfaces attached to instances created in the specified...\[optional\]
#' @param CustomerOwnedIpv4Pool String.   The customer-owned IPv4 address pool associated with the subnet.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SubnetId:
#' The ID of the subnet.
#' @section AssignIpv6AddressOnCreation:
#' 
#' Specify `true` to indicate that network interfaces created in the
#' specified subnet should be assigned an IPv6 address. This includes a
#' network interface that\'s created when launching an instance into the
#' subnet (the instance therefore receives an IPv6 address).
#' 
#' If you enable the IPv6 addressing feature for your subnet, your network
#' interface or instance only receives an IPv6 address if it\'s created
#' using version `2016-11-15` or later of the Amazon EC2 API.
#' 
#' @section MapPublicIpOnLaunch:
#' Specify `true` to indicate that network interfaces
#' attached to instances created in the specified subnet should be assigned
#' a public IPv4 address.
#' @section MapCustomerOwnedIpOnLaunch:
#' 
#' Specify `true` to indicate that network interfaces attached to instances
#' created in the specified subnet should be assigned a customer-owned IPv4
#' address.
#' 
#' When this value is `true`, you must specify the customer-owned IP pool
#' using `CustomerOwnedIpv4Pool`.
#' 
#' @section CustomerOwnedIpv4Pool:
#' 
#' The customer-owned IPv4 address pool associated with the subnet.
#' 
#' You must set this value when you specify `true` for
#' `MapCustomerOwnedIpOnLaunch`.
#' 
#' @return A list object or a character vector
#' @export
ec2_modify_subnet_attribute <- function(SubnetId, AssignIpv6AddressOnCreation = NULL, MapPublicIpOnLaunch = NULL, 
    MapCustomerOwnedIpOnLaunch = NULL, CustomerOwnedIpv4Pool = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(SubnetId = SubnetId, AssignIpv6AddressOnCreation = AssignIpv6AddressOnCreation, 
        MapPublicIpOnLaunch = MapPublicIpOnLaunch, MapCustomerOwnedIpOnLaunch = MapCustomerOwnedIpOnLaunch, 
        CustomerOwnedIpv4Pool = CustomerOwnedIpv4Pool))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifySubnetAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Traffic Mirror Filter Network Services
#' 
#' @param TrafficMirrorFilterId String. The ID of the Traffic Mirror filter.
#' @param AddNetworkService List. The network service, for example Amazon DNS, that you want to mirror.\[optional\]
#' @param RemoveNetworkService List. The network service, for example Amazon DNS, that you no longer want to mirror.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorFilterId:
#' The ID of the Traffic Mirror
#' filter.
#' @section AddNetworkService:
#' The network service, for example Amazon DNS,
#' that you want to mirror.
#' @section RemoveNetworkService:
#' The network service, for
#' example Amazon DNS, that you no longer want to
#' mirror.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_traffic_mirror_filter_network_services <- function(TrafficMirrorFilterId, AddNetworkService = NULL, 
    RemoveNetworkService = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    AddNetworkService <- list_to_array("AddNetworkService", AddNetworkService)
    RemoveNetworkService <- list_to_array("RemoveNetworkService", RemoveNetworkService)
    parameters <- c(others, list(TrafficMirrorFilterId = TrafficMirrorFilterId, DryRun = DryRun), 
        AddNetworkService, RemoveNetworkService)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyTrafficMirrorFilterNetworkServices", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Traffic Mirror Filter Rule
#' 
#' @param TrafficMirrorFilterRuleId String. The ID of the Traffic Mirror rule.
#' @param TrafficDirection String. The type of traffic (`ingress` \| `egress`) to assign to the rule.\[optional\]
#' @param RuleNumber Integer. The number of the Traffic Mirror rule.\[optional\]
#' @param RuleAction String. The action to assign to the rule.\[optional\]
#' @param DestinationPortRange Object. The destination ports that are associated with the Traffic Mirror rule.\[optional\]
#' @param SourcePortRange Object. The port range to assign to the Traffic Mirror rule.\[optional\]
#' @param Protocol Integer. The protocol, for example TCP, to assign to the Traffic Mirror rule.\[optional\]
#' @param DestinationCidrBlock String. The destination CIDR block to assign to the Traffic Mirror rule.\[optional\]
#' @param SourceCidrBlock String. The source CIDR block to assign to the Traffic Mirror rule.\[optional\]
#' @param Description String. The description to assign to the Traffic Mirror rule.\[optional\]
#' @param RemoveField List.   The properties that you want to remove from the Traffic Mirror filter rule.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorFilterRuleId:
#' The ID of the Traffic
#' Mirror rule.
#' @section TrafficDirection:
#' The type of traffic (`ingress` \|
#' `egress`) to assign to the rule.
#' @section RuleNumber:
#' The number of the
#' Traffic Mirror rule. This number must be unique for each Traffic Mirror
#' rule in a given direction. The rules are processed in ascending order by
#' rule number.
#' @section RuleAction:
#' The action to assign to the
#' rule.
#' @section DestinationPortRange:
#' The destination ports that are associated with
#' the Traffic Mirror rule.
#' @section SourcePortRange:
#' The port range to assign to
#' the Traffic Mirror rule.
#' @section Protocol:
#' The protocol, for example
#' TCP, to assign to the Traffic Mirror rule.
#' @section DestinationCidrBlock:
#' The
#' destination CIDR block to assign to the Traffic Mirror
#' rule.
#' @section SourceCidrBlock:
#' The source CIDR block to assign to the Traffic
#' Mirror rule.
#' @section Description:
#' The description to assign to the
#' Traffic Mirror rule.
#' @section RemoveField:
#' 
#' The properties that you want to remove from the Traffic Mirror filter
#' rule.
#' 
#' When you remove a property from a Traffic Mirror filter rule, the
#' property is set to the default.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_traffic_mirror_filter_rule <- function(TrafficMirrorFilterRuleId, TrafficDirection = NULL, 
    RuleNumber = NULL, RuleAction = NULL, DestinationPortRange = NULL, SourcePortRange = NULL, 
    Protocol = NULL, DestinationCidrBlock = NULL, SourceCidrBlock = NULL, Description = NULL, 
    RemoveField = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    RemoveField <- list_to_array("RemoveField", RemoveField)
    parameters <- c(others, list(TrafficMirrorFilterRuleId = TrafficMirrorFilterRuleId, TrafficDirection = TrafficDirection, 
        RuleNumber = RuleNumber, RuleAction = RuleAction, DestinationPortRange = DestinationPortRange, 
        SourcePortRange = SourcePortRange, Protocol = Protocol, DestinationCidrBlock = DestinationCidrBlock, 
        SourceCidrBlock = SourceCidrBlock, Description = Description, DryRun = DryRun), RemoveField)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyTrafficMirrorFilterRule", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Traffic Mirror Session
#' 
#' Modifies a Traffic Mirror
#' session.
#' 
#' @param TrafficMirrorSessionId String. The ID of the Traffic Mirror session.
#' @param TrafficMirrorTargetId String. The Traffic Mirror target.\[optional\]
#' @param TrafficMirrorFilterId String. The ID of the Traffic Mirror filter.\[optional\]
#' @param PacketLength Integer. The number of bytes in each packet to mirror.\[optional\]
#' @param SessionNumber Integer.   The session number determines the order in which sessions are evaluated when an interface is used...\[optional\]
#' @param VirtualNetworkId Integer. The virtual network ID of the Traffic Mirror session.\[optional\]
#' @param Description String. The description to assign to the Traffic Mirror session.\[optional\]
#' @param RemoveField List.   The properties that you want to remove from the Traffic Mirror session.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TrafficMirrorSessionId:
#' The ID of the Traffic
#' Mirror session.
#' @section TrafficMirrorTargetId:
#' The Traffic Mirror target. The
#' target must be in the same VPC as the source, or have a VPC peering
#' connection with the source.
#' @section TrafficMirrorFilterId:
#' The ID of the Traffic
#' Mirror filter.
#' @section PacketLength:
#' The number of bytes in each packet to
#' mirror. These are bytes after the VXLAN header. To mirror a subset, set
#' this to the length (in bytes) to mirror. For example, if you set this
#' value to 100, then the first 100 bytes that meet the filter criteria are
#' copied to the target. Do not specify this parameter when you want to
#' mirror the entire packet.
#' @section SessionNumber:
#' 
#' The session number determines the order in which sessions are evaluated
#' when an interface is used by multiple sessions. The first session with a
#' matching filter is the one that mirrors the packets.
#' 
#' Valid values are 1-32766.
#' 
#' @section VirtualNetworkId:
#' The virtual network ID of the Traffic Mirror
#' session.
#' @section Description:
#' The description to assign to the Traffic
#' Mirror session.
#' @section RemoveField:
#' 
#' The properties that you want to remove from the Traffic Mirror session.
#' 
#' When you remove a property from a Traffic Mirror session, the property
#' is set to the default.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_traffic_mirror_session <- function(TrafficMirrorSessionId, TrafficMirrorTargetId = NULL, 
    TrafficMirrorFilterId = NULL, PacketLength = NULL, SessionNumber = NULL, VirtualNetworkId = NULL, 
    Description = NULL, RemoveField = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    RemoveField <- list_to_array("RemoveField", RemoveField)
    parameters <- c(others, list(TrafficMirrorSessionId = TrafficMirrorSessionId, TrafficMirrorTargetId = TrafficMirrorTargetId, 
        TrafficMirrorFilterId = TrafficMirrorFilterId, PacketLength = PacketLength, SessionNumber = SessionNumber, 
        VirtualNetworkId = VirtualNetworkId, Description = Description, DryRun = DryRun), 
        RemoveField)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyTrafficMirrorSession", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Transit Gateway
#' 
#' Modifies the specified transit gateway.
#' When you modify a transit gateway, the modified options are applied to
#' new transit gateway attachments only. Your existing transit gateway
#' attachments are not modified.
#' 
#' @param TransitGatewayId String. The ID of the transit gateway.
#' @param Description String. The description for the transit gateway.\[optional\]
#' @param Options Object. The options to modify.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayId:
#' The ID of the transit
#' gateway.
#' @section Description:
#' The description for the transit
#' gateway.
#' @section Options:
#' The options to
#' modify.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_transit_gateway <- function(TransitGatewayId, Description = NULL, Options = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayId = TransitGatewayId, Description = Description, 
        Options = Options, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyTransitGateway", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Transit Gateway Prefix List Reference
#' 
#' Modifies a reference
#' (route) to a prefix list in a specified transit gateway route
#' table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param PrefixListId String. The ID of the prefix list.
#' @param TransitGatewayAttachmentId String. The ID of the attachment to which traffic is routed.\[optional\]
#' @param Blackhole Logical. Indicates whether to drop traffic that matches this route.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section PrefixListId:
#' The ID of the prefix
#' list.
#' @section TransitGatewayAttachmentId:
#' The ID of the attachment to which traffic is
#' routed.
#' @section Blackhole:
#' Indicates whether to drop traffic that
#' matches this route.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_transit_gateway_prefix_list_reference <- function(TransitGatewayRouteTableId, 
    PrefixListId, TransitGatewayAttachmentId = NULL, Blackhole = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        PrefixListId = PrefixListId, TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        Blackhole = Blackhole, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyTransitGatewayPrefixListReference", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Transit Gateway Vpc Attachment
#' 
#' Modifies the specified VPC
#' attachment.
#' 
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param AddSubnetIds List. The IDs of one or more subnets to add. You can specify at most one subnet per Availability Zone.\[optional\]
#' @param RemoveSubnetIds List. The IDs of one or more subnets to remove.\[optional\]
#' @param Options Object.   The new VPC attachment options.  You cannot modify the IPv6 options.  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section AddSubnetIds:
#' The IDs of one or more subnets to add.
#' You can specify at most one subnet per Availability
#' Zone.
#' @section RemoveSubnetIds:
#' The IDs of one or more subnets to
#' remove.
#' @section Options:
#' 
#' The new VPC attachment options.
#' 
#' You cannot modify the IPv6 options.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_transit_gateway_vpc_attachment <- function(TransitGatewayAttachmentId, AddSubnetIds = NULL, 
    RemoveSubnetIds = NULL, Options = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    AddSubnetIds <- list_to_array("AddSubnetIds", AddSubnetIds)
    RemoveSubnetIds <- list_to_array("RemoveSubnetIds", RemoveSubnetIds)
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        Options = Options, DryRun = DryRun), AddSubnetIds, RemoveSubnetIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyTransitGatewayVpcAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Volume
#' 
#' @param VolumeId String. The ID of the volume.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Size Integer.   The target size of the volume, in GiB.\[optional\]
#' @param VolumeType String.   The target EBS volume type of the volume.\[optional\]
#' @param Iops Integer.   The target IOPS rate of the volume.\[optional\]
#' @param Throughput Integer.   The target throughput of the volume, in MiB/s.\[optional\]
#' @param MultiAttachEnabled Logical. Specifies whether to enable Amazon EBS Multi-Attach.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VolumeId:
#' The ID of the
#' volume.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Size:
#' 
#' The target size of the volume, in GiB. The target volume size must be
#' greater than or equal to the existing size of the volume.
#' 
#' The following are the supported volumes sizes for each volume type:
#' 
#' -   `gp2` and `gp3`: 1-16,384
#' 
#' -   `io1` and `io2`: 4-16,384
#' 
#' -   `st1` and `sc1`: 125-16,384
#' 
#' -   `standard`: 1-1,024
#' 
#' Default: If no size is specified, the existing size is retained.
#' 
#' @section VolumeType:
#' 
#' The target EBS volume type of the volume. For more information, see
#' [Amazon EBS volume
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' Default: If no type is specified, the existing type is retained.
#' 
#' @section Iops:
#' 
#' The target IOPS rate of the volume. This parameter is valid only for
#' `gp3`, `io1`, and `io2` volumes.
#' 
#' The following are the supported values for each volume type:
#' 
#' -   `gp3`: 3,000-16,000 IOPS
#' 
#' -   `io1`: 100-64,000 IOPS
#' 
#' -   `io2`: 100-64,000 IOPS
#' 
#' Default: If no IOPS value is specified, the existing value is retained.
#' 
#' @section Throughput:
#' 
#' The target throughput of the volume, in MiB/s. This parameter is valid
#' only for `gp3` volumes. The maximum value is 1,000.
#' 
#' Default: If no throughput value is specified, the existing value is
#' retained.
#' 
#' Valid Range: Minimum value of 125. Maximum value of 1000.
#' 
#' @section MultiAttachEnabled:
#' Specifies whether to enable Amazon EBS
#' Multi-Attach. If you enable Multi-Attach, you can attach the volume to
#' up to 16 [Nitro-based
#' instances](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances)
#' in the same Availability Zone. This parameter is supported with `io1`
#' and `io2` volumes only. For more information, see [Amazon EBS
#' Multi-Attach](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes-multi.html)
#' in the *Amazon Elastic Compute Cloud User
#' Guide*.
#' @return A list object or a character vector
#' @export
ec2_modify_volume <- function(VolumeId, DryRun = NULL, Size = NULL, VolumeType = NULL, Iops = NULL, 
    Throughput = NULL, MultiAttachEnabled = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VolumeId = VolumeId, DryRun = DryRun, Size = Size, VolumeType = VolumeType, 
        Iops = Iops, Throughput = Throughput, MultiAttachEnabled = MultiAttachEnabled))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVolume", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Volume Attribute
#' 
#' @param VolumeId String. The ID of the volume.
#' @param AutoEnableIO Object. Indicates whether the volume should be auto-enabled for I/O operations.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VolumeId:
#' The ID of the
#' volume.
#' @section AutoEnableIO:
#' Indicates whether the volume should be
#' auto-enabled for I/O operations.
#' @section DryRun:
#' Checks whether you
#' have the required permissions for the action, without actually making
#' the request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_volume_attribute <- function(VolumeId, AutoEnableIO = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(VolumeId = VolumeId, AutoEnableIO = AutoEnableIO, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVolumeAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Vpc Attribute
#' 
#' Modifies the specified attribute of the specified
#' VPC.
#' 
#' @param VpcId String. The ID of the VPC.
#' @param EnableDnsHostnames Object.   Indicates whether the instances launched in the VPC get DNS hostnames.\[optional\]
#' @param EnableDnsSupport Object.   Indicates whether the DNS resolution is supported for the VPC.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section EnableDnsHostnames:
#' 
#' Indicates whether the instances launched in the VPC get DNS hostnames.
#' If enabled, instances in the VPC get DNS hostnames; otherwise, they do
#' not.
#' 
#' You cannot modify the DNS resolution and DNS hostnames attributes in the
#' same request. Use separate requests for each attribute. You can only
#' enable DNS hostnames if you\'ve enabled DNS support.
#' 
#' @section EnableDnsSupport:
#' 
#' Indicates whether the DNS resolution is supported for the VPC. If
#' enabled, queries to the Amazon provided DNS server at the
#' 169.254.169.253 IP address, or the reserved IP address at the base of
#' the VPC network range \'plus two\' succeed. If disabled, the Amazon
#' provided DNS service in the VPC that resolves public DNS hostnames to IP
#' addresses is not enabled.
#' 
#' You cannot modify the DNS resolution and DNS hostnames attributes in the
#' same request. Use separate requests for each attribute.
#' 
#' @return A list object or a character vector
#' @export
ec2_modify_vpc_attribute <- function(VpcId, EnableDnsHostnames = NULL, EnableDnsSupport = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcId = VpcId, EnableDnsHostnames = EnableDnsHostnames, 
        EnableDnsSupport = EnableDnsSupport))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpcAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Vpc Endpoint
#' 
#' Modifies attributes of a specified VPC
#' endpoint. The attributes that you can modify depend on the type of VPC
#' endpoint (interface, gateway, or Gateway Load Balancer). For more
#' information, see [VPC
#' Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints.html)
#' in the *Amazon Virtual Private Cloud User
#' Guide*.
#' 
#' @param VpcEndpointId String. The ID of the endpoint.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ResetPolicy Logical. (Gateway endpoint) Specify `true` to reset the policy document to the default policy.\[optional\]
#' @param PolicyDocument String. (Interface and gateway endpoints) A policy to attach to the endpoint that controls access to the...\[optional\]
#' @param AddRouteTableId List. (Gateway endpoint) One or more route tables IDs to associate with the endpoint.\[optional\]
#' @param RemoveRouteTableId List. (Gateway endpoint) One or more route table IDs to disassociate from the endpoint.\[optional\]
#' @param AddSubnetId List. (Interface and Gateway Load Balancer endpoints) One or more subnet IDs in which to serve the endpoint....\[optional\]
#' @param RemoveSubnetId List. (Interface endpoint) One or more subnets IDs in which to remove the endpoint.\[optional\]
#' @param AddSecurityGroupId List. (Interface endpoint) One or more security group IDs to associate with the network interface.\[optional\]
#' @param RemoveSecurityGroupId List. (Interface endpoint) One or more security group IDs to disassociate from the network interface.\[optional\]
#' @param PrivateDnsEnabled Logical. (Interface endpoint) Indicates whether a private hosted zone is associated with the VPC.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcEndpointId:
#' The ID of the endpoint.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section ResetPolicy:
#' (Gateway
#' endpoint) Specify `true` to reset the policy document to the default
#' policy. The default policy allows full access to the
#' service.
#' @section PolicyDocument:
#' (Interface and gateway endpoints) A policy
#' to attach to the endpoint that controls access to the service. The
#' policy must be in valid JSON format.
#' @section AddRouteTableId:
#' (Gateway
#' endpoint) One or more route tables IDs to associate with the
#' endpoint.
#' @section RemoveRouteTableId:
#' (Gateway endpoint) One or more route table
#' IDs to disassociate from the endpoint.
#' @section AddSubnetId:
#' (Interface
#' and Gateway Load Balancer endpoints) One or more subnet IDs in which to
#' serve the endpoint. For a Gateway Load Balancer endpoint, you can
#' specify only one subnet.
#' @section RemoveSubnetId:
#' (Interface endpoint) One or
#' more subnets IDs in which to remove the
#' endpoint.
#' @section AddSecurityGroupId:
#' (Interface endpoint) One or more security
#' group IDs to associate with the network
#' interface.
#' @section RemoveSecurityGroupId:
#' (Interface endpoint) One or more security
#' group IDs to disassociate from the network
#' interface.
#' @section PrivateDnsEnabled:
#' (Interface endpoint) Indicates whether a
#' private hosted zone is associated with the VPC.
#' @return A list object or a character vector
#' @export
ec2_modify_vpc_endpoint <- function(VpcEndpointId, DryRun = NULL, ResetPolicy = NULL, PolicyDocument = NULL, 
    AddRouteTableId = NULL, RemoveRouteTableId = NULL, AddSubnetId = NULL, RemoveSubnetId = NULL, 
    AddSecurityGroupId = NULL, RemoveSecurityGroupId = NULL, PrivateDnsEnabled = NULL, simplify = TRUE, 
    others = list()) {
    AddRouteTableId <- list_to_array("AddRouteTableId", AddRouteTableId)
    RemoveRouteTableId <- list_to_array("RemoveRouteTableId", RemoveRouteTableId)
    AddSubnetId <- list_to_array("AddSubnetId", AddSubnetId)
    RemoveSubnetId <- list_to_array("RemoveSubnetId", RemoveSubnetId)
    AddSecurityGroupId <- list_to_array("AddSecurityGroupId", AddSecurityGroupId)
    RemoveSecurityGroupId <- list_to_array("RemoveSecurityGroupId", RemoveSecurityGroupId)
    parameters <- c(others, list(VpcEndpointId = VpcEndpointId, DryRun = DryRun, ResetPolicy = ResetPolicy, 
        PolicyDocument = PolicyDocument, PrivateDnsEnabled = PrivateDnsEnabled), AddRouteTableId, 
        RemoveRouteTableId, AddSubnetId, RemoveSubnetId, AddSecurityGroupId, RemoveSecurityGroupId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpcEndpoint", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Vpc Endpoint Connection Notification
#' 
#' Modifies a connection notification for VPC
#' endpoint or VPC endpoint service. You can change the SNS topic for the
#' notification, or the events for which to be notified.
#' 
#' @param ConnectionNotificationId String. The ID of the notification.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param ConnectionNotificationArn String. The ARN for the SNS topic for the notification.\[optional\]
#' @param ConnectionEvents List. One or more events for the endpoint. Valid values are `Accept`, `Connect`, `Delete`, and `Reject`.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ConnectionNotificationId:
#' The
#' ID of the notification.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section ConnectionNotificationArn:
#' The ARN for the SNS topic
#' for the notification.
#' @section ConnectionEvents:
#' One or more events for the
#' endpoint. Valid values are `Accept`, `Connect`, `Delete`, and
#' `Reject`.
#' @return A list object or a character vector
#' @export
ec2_modify_vpc_endpoint_connection_notification <- function(ConnectionNotificationId, DryRun = NULL, 
    ConnectionNotificationArn = NULL, ConnectionEvents = NULL, simplify = TRUE, others = list()) {
    ConnectionEvents <- list_to_array("ConnectionEvents", ConnectionEvents)
    parameters <- c(others, list(ConnectionNotificationId = ConnectionNotificationId, DryRun = DryRun, 
        ConnectionNotificationArn = ConnectionNotificationArn), ConnectionEvents)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpcEndpointConnectionNotification", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Vpc Endpoint Service Configuration
#' 
#' @param ServiceId String. The ID of the service.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param PrivateDnsName String. (Interface endpoint configuration) The private DNS name to assign to the endpoint service.\[optional\]
#' @param RemovePrivateDnsName Logical. (Interface endpoint configuration) Removes the private DNS name of the endpoint service.\[optional\]
#' @param AcceptanceRequired Logical. Indicates whether requests to create an endpoint to your service must be accepted.\[optional\]
#' @param AddNetworkLoadBalancerArn List. The Amazon Resource Names (ARNs) of Network Load Balancers to add to your service configuration.\[optional\]
#' @param RemoveNetworkLoadBalancerArn List. The Amazon Resource Names (ARNs) of Network Load Balancers to remove from your service configuration....\[optional\]
#' @param AddGatewayLoadBalancerArn List. The Amazon Resource Names (ARNs) of Gateway Load Balancers to add to your service configuration.\[optional\]
#' @param RemoveGatewayLoadBalancerArn List. The Amazon Resource Names (ARNs) of Gateway Load Balancers to remove from your service configuration....\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ServiceId:
#' The ID of the
#' service.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section PrivateDnsName:
#' (Interface endpoint
#' configuration) The private DNS name to assign to the endpoint
#' service.
#' @section RemovePrivateDnsName:
#' (Interface endpoint configuration) Removes
#' the private DNS name of the endpoint
#' service.
#' @section AcceptanceRequired:
#' Indicates whether requests to create an
#' endpoint to your service must be accepted.
#' @section AddNetworkLoadBalancerArn:
#' The
#' Amazon Resource Names (ARNs) of Network Load Balancers to add to your
#' service configuration.
#' @section RemoveNetworkLoadBalancerArn:
#' The Amazon Resource Names
#' (ARNs) of Network Load Balancers to remove from your service
#' configuration.
#' @section AddGatewayLoadBalancerArn:
#' The Amazon Resource Names (ARNs) of
#' Gateway Load Balancers to add to your service
#' configuration.
#' @section RemoveGatewayLoadBalancerArn:
#' The Amazon Resource Names (ARNs) of
#' Gateway Load Balancers to remove from your service
#' configuration.
#' @return A list object or a character vector
#' @export
ec2_modify_vpc_endpoint_service_configuration <- function(ServiceId, DryRun = NULL, PrivateDnsName = NULL, 
    RemovePrivateDnsName = NULL, AcceptanceRequired = NULL, AddNetworkLoadBalancerArn = NULL, 
    RemoveNetworkLoadBalancerArn = NULL, AddGatewayLoadBalancerArn = NULL, RemoveGatewayLoadBalancerArn = NULL, 
    simplify = TRUE, others = list()) {
    AddNetworkLoadBalancerArn <- list_to_array("AddNetworkLoadBalancerArn", AddNetworkLoadBalancerArn)
    RemoveNetworkLoadBalancerArn <- list_to_array("RemoveNetworkLoadBalancerArn", RemoveNetworkLoadBalancerArn)
    AddGatewayLoadBalancerArn <- list_to_array("AddGatewayLoadBalancerArn", AddGatewayLoadBalancerArn)
    RemoveGatewayLoadBalancerArn <- list_to_array("RemoveGatewayLoadBalancerArn", RemoveGatewayLoadBalancerArn)
    parameters <- c(others, list(ServiceId = ServiceId, DryRun = DryRun, PrivateDnsName = PrivateDnsName, 
        RemovePrivateDnsName = RemovePrivateDnsName, AcceptanceRequired = AcceptanceRequired), 
        AddNetworkLoadBalancerArn, RemoveNetworkLoadBalancerArn, AddGatewayLoadBalancerArn, 
        RemoveGatewayLoadBalancerArn)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpcEndpointServiceConfiguration", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Vpc Endpoint Service Permissions
#' 
#' @param ServiceId String. The ID of the service.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param AddAllowedPrincipals List. The Amazon Resource Names (ARN) of one or more principals.\[optional\]
#' @param RemoveAllowedPrincipals List. The Amazon Resource Names (ARN) of one or more principals.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ServiceId:
#' The ID of the
#' service.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section AddAllowedPrincipals:
#' The Amazon Resource Names
#' (ARN) of one or more principals. Permissions are granted to the
#' principals in this list. To grant permissions to all principals, specify
#' an asterisk (\*).
#' @section RemoveAllowedPrincipals:
#' The Amazon Resource Names (ARN) of
#' one or more principals. Permissions are revoked for principals in this
#' list.
#' @return A list object or a character vector
#' @export
ec2_modify_vpc_endpoint_service_permissions <- function(ServiceId, DryRun = NULL, AddAllowedPrincipals = NULL, 
    RemoveAllowedPrincipals = NULL, simplify = TRUE, others = list()) {
    AddAllowedPrincipals <- list_to_array("AddAllowedPrincipals", AddAllowedPrincipals)
    RemoveAllowedPrincipals <- list_to_array("RemoveAllowedPrincipals", RemoveAllowedPrincipals)
    parameters <- c(others, list(ServiceId = ServiceId, DryRun = DryRun), AddAllowedPrincipals, 
        RemoveAllowedPrincipals)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpcEndpointServicePermissions", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Vpc Peering Connection Options
#' 
#' @param VpcPeeringConnectionId String. The ID of the VPC peering connection.
#' @param AccepterPeeringConnectionOptions Object. The VPC peering connection options for the accepter VPC.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param RequesterPeeringConnectionOptions Object. The VPC peering connection options for the requester VPC.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcPeeringConnectionId:
#' The ID of the VPC peering
#' connection.
#' @section AccepterPeeringConnectionOptions:
#' The VPC peering connection options for
#' the accepter VPC.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section RequesterPeeringConnectionOptions:
#' The VPC peering connection
#' options for the requester VPC.
#' @return A list object or a character vector
#' @export
ec2_modify_vpc_peering_connection_options <- function(VpcPeeringConnectionId, AccepterPeeringConnectionOptions = NULL, 
    DryRun = NULL, RequesterPeeringConnectionOptions = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpcPeeringConnectionId = VpcPeeringConnectionId, AccepterPeeringConnectionOptions = AccepterPeeringConnectionOptions, 
        DryRun = DryRun, RequesterPeeringConnectionOptions = RequesterPeeringConnectionOptions))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpcPeeringConnectionOptions", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Modify Vpc Tenancy
#' 
#' @param VpcId String. The ID of the VPC.
#' @param InstanceTenancy String. The instance tenancy attribute for the VPC. 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcId:
#' The ID of the
#' VPC.
#' @section InstanceTenancy:
#' The instance tenancy attribute for the VPC.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_vpc_tenancy <- function(VpcId, InstanceTenancy, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(VpcId = VpcId, InstanceTenancy = InstanceTenancy, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpcTenancy", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Vpn Connection
#' 
#' @param VpnConnectionId String. The ID of the VPN connection.
#' @param TransitGatewayId String. The ID of the transit gateway.\[optional\]
#' @param CustomerGatewayId String. The ID of the customer gateway at your end of the VPN connection.\[optional\]
#' @param VpnGatewayId String. The ID of the virtual private gateway at the AWS side of the VPN connection.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpnConnectionId:
#' The ID of the VPN
#' connection.
#' @section TransitGatewayId:
#' The ID of the transit
#' gateway.
#' @section CustomerGatewayId:
#' The ID of the customer gateway at your end
#' of the VPN connection.
#' @section VpnGatewayId:
#' The ID of the virtual private
#' gateway at the AWS side of the VPN
#' connection.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_vpn_connection <- function(VpnConnectionId, TransitGatewayId = NULL, CustomerGatewayId = NULL, 
    VpnGatewayId = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpnConnectionId = VpnConnectionId, TransitGatewayId = TransitGatewayId, 
        CustomerGatewayId = CustomerGatewayId, VpnGatewayId = VpnGatewayId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpnConnection", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Vpn Connection Options
#' 
#' @param VpnConnectionId String. The ID of the Site-to-Site VPN connection. 
#' @param LocalIpv4NetworkCidr String.   The IPv4 CIDR on the customer gateway (on-premises) side of the VPN connection.\[optional\]
#' @param RemoteIpv4NetworkCidr String.   The IPv4 CIDR on the AWS side of the VPN connection.  Default: `0.0.0.0/0`  \[optional\]
#' @param LocalIpv6NetworkCidr String.   The IPv6 CIDR on the customer gateway (on-premises) side of the VPN connection.\[optional\]
#' @param RemoteIpv6NetworkCidr String.   The IPv6 CIDR on the AWS side of the VPN connection.  Default: `::/0`  \[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpnConnectionId:
#' The ID of the Site-to-Site
#' VPN connection. 
#' @section LocalIpv4NetworkCidr:
#' 
#' The IPv4 CIDR on the customer gateway (on-premises) side of the VPN
#' connection.
#' 
#' Default: `0.0.0.0/0`
#' 
#' @section RemoteIpv4NetworkCidr:
#' 
#' The IPv4 CIDR on the AWS side of the VPN connection.
#' 
#' Default: `0.0.0.0/0`
#' 
#' @section LocalIpv6NetworkCidr:
#' 
#' The IPv6 CIDR on the customer gateway (on-premises) side of the VPN
#' connection.
#' 
#' Default: `::/0`
#' 
#' @section RemoteIpv6NetworkCidr:
#' 
#' The IPv6 CIDR on the AWS side of the VPN connection.
#' 
#' Default: `::/0`
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_vpn_connection_options <- function(VpnConnectionId, LocalIpv4NetworkCidr = NULL, 
    RemoteIpv4NetworkCidr = NULL, LocalIpv6NetworkCidr = NULL, RemoteIpv6NetworkCidr = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpnConnectionId = VpnConnectionId, LocalIpv4NetworkCidr = LocalIpv4NetworkCidr, 
        RemoteIpv4NetworkCidr = RemoteIpv4NetworkCidr, LocalIpv6NetworkCidr = LocalIpv6NetworkCidr, 
        RemoteIpv6NetworkCidr = RemoteIpv6NetworkCidr, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpnConnectionOptions", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Vpn Tunnel Certificate
#' 
#' Modifies the VPN tunnel endpoint
#' certificate.
#' 
#' @param VpnConnectionId String. The ID of the AWS Site-to-Site VPN connection.
#' @param VpnTunnelOutsideIpAddress String. The external IP address of the VPN tunnel.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpnConnectionId:
#' The ID of the AWS
#' Site-to-Site VPN connection.
#' @section VpnTunnelOutsideIpAddress:
#' The external IP address
#' of the VPN tunnel.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_vpn_tunnel_certificate <- function(VpnConnectionId, VpnTunnelOutsideIpAddress, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpnConnectionId = VpnConnectionId, VpnTunnelOutsideIpAddress = VpnTunnelOutsideIpAddress, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpnTunnelCertificate", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Modify Vpn Tunnel Options
#' 
#' Modifies the options for a VPN tunnel
#' in an AWS Site-to-Site VPN connection. You can modify multiple options
#' for a tunnel in a single request, but you can only modify one tunnel at
#' a time. For more information, see [Site-to-Site VPN Tunnel Options for
#' Your Site-to-Site VPN
#' Connection](https://docs.aws.amazon.com/vpn/latest/s2svpn/VPNTunnels.html)
#' in the *AWS Site-to-Site VPN User Guide*.
#' 
#' @param VpnConnectionId String. The ID of the AWS Site-to-Site VPN connection.
#' @param VpnTunnelOutsideIpAddress String. The external IP address of the VPN tunnel.
#' @param TunnelOptions Object. The tunnel options to modify.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpnConnectionId:
#' The ID of the AWS
#' Site-to-Site VPN connection.
#' @section VpnTunnelOutsideIpAddress:
#' The external IP address
#' of the VPN tunnel.
#' @section TunnelOptions:
#' The tunnel options to
#' modify.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_modify_vpn_tunnel_options <- function(VpnConnectionId, VpnTunnelOutsideIpAddress, TunnelOptions, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(VpnConnectionId = VpnConnectionId, VpnTunnelOutsideIpAddress = VpnTunnelOutsideIpAddress, 
        TunnelOptions = TunnelOptions, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ModifyVpnTunnelOptions", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Monitor Instances
#' 
#' @param InstanceId List. The IDs of the instances.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The IDs of the
#' instances.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_monitor_instances <- function(InstanceId, DryRun = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    parameters <- c(others, list(DryRun = DryRun), InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "MonitorInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Move Address To Vpc
#' 
#' Moves an Elastic IP address from the EC2-Classic
#' platform to the EC2-VPC platform. The Elastic IP address must be
#' allocated to your account for more than 24 hours, and it must not be
#' associated with an instance. After the Elastic IP address is moved, it
#' is no longer available for use in the EC2-Classic platform, unless you
#' move it back using the RestoreAddressToClassic request. You cannot move
#' an Elastic IP address that was originally allocated for use in the
#' EC2-VPC platform to the EC2-Classic platform. 
#' 
#' @param PublicIp String. The Elastic IP address.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PublicIp:
#' The Elastic IP
#' address.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_move_address_to_vpc <- function(PublicIp, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(PublicIp = PublicIp, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "MoveAddressToVpc", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Provision Byoip Cidr
#' 
#' @param Cidr String. The public IPv4 or IPv6 address range, in CIDR notation.
#' @param CidrAuthorizationContext Object. A signed document that proves that you are authorized to bring the specified IP address range to...\[optional\]
#' @param PubliclyAdvertisable Logical.   (IPv6 only) Indicate whether the address range will be publicly advertised to the internet.\[optional\]
#' @param Description String. A description for the address range and the address pool.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param PoolTagSpecification List. The tags to apply to the address pool.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Cidr:
#' The public IPv4 or IPv6
#' address range, in CIDR notation. The most specific IPv4 prefix that you
#' can specify is /24. The most specific IPv6 prefix you can specify is
#' /56. The address range cannot overlap with another address range that
#' you\'ve brought to this or another Region.
#' @section CidrAuthorizationContext:
#' A signed
#' document that proves that you are authorized to bring the specified IP
#' address range to Amazon using BYOIP.
#' @section PubliclyAdvertisable:
#' 
#' (IPv6 only) Indicate whether the address range will be publicly
#' advertised to the internet.
#' 
#' Default: true
#' 
#' @section Description:
#' A description for the address range and the address
#' pool.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section PoolTagSpecification:
#' The tags to apply to the
#' address pool.
#' @return A list object or a character vector
#' @export
ec2_provision_byoip_cidr <- function(Cidr, CidrAuthorizationContext = NULL, PubliclyAdvertisable = NULL, 
    Description = NULL, DryRun = NULL, PoolTagSpecification = NULL, simplify = TRUE, others = list()) {
    PoolTagSpecification <- list_to_array("PoolTagSpecification", PoolTagSpecification)
    parameters <- c(others, list(Cidr = Cidr, CidrAuthorizationContext = CidrAuthorizationContext, 
        PubliclyAdvertisable = PubliclyAdvertisable, Description = Description, DryRun = DryRun), 
        PoolTagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ProvisionByoipCidr", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Purchase Host Reservation
#' 
#' Purchase a reservation with configurations that
#' match those of your Dedicated Host. You must have active Dedicated Hosts
#' in your account before you purchase a reservation. This action results
#' in the specified reservation being purchased and charged to your
#' account.
#' 
#' @param HostIdSet List. The IDs of the Dedicated Hosts with which the reservation will be associated.
#' @param OfferingId String. The ID of the offering.
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param CurrencyCode String. The currency in which the `totalUpfrontPrice`, `LimitPrice`, and `totalHourlyPrice` amounts are specified....\[optional\]
#' @param LimitPrice String. The specified limit is checked against the total upfront cost of the reservation (calculated as the offering\'s upfront cost multiplied by the host count)....\[optional\]
#' @param TagSpecification List. The tags to apply to the Dedicated Host Reservation during purchase.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section HostIdSet:
#' The IDs of the Dedicated Hosts with
#' which the reservation will be associated.
#' @section OfferingId:
#' The ID of
#' the offering.
#' @section ClientToken:
#' Unique, case-sensitive identifier that
#' you provide to ensure the idempotency of the request. For more
#' information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section CurrencyCode:
#' The
#' currency in which the `totalUpfrontPrice`, `LimitPrice`, and
#' `totalHourlyPrice` amounts are specified. At this time, the only
#' supported currency is `USD`.
#' @section LimitPrice:
#' The specified limit is
#' checked against the total upfront cost of the reservation (calculated as
#' the offering\'s upfront cost multiplied by the host count). If the total
#' upfront cost is greater than the specified price limit, the request
#' fails. This is used to ensure that the purchase does not exceed the
#' expected upfront cost of the purchase. At this time, the only supported
#' currency is `USD`. For example, to indicate a limit price of USD 100,
#' specify 100.00.
#' @section TagSpecification:
#' The tags to apply to the Dedicated
#' Host Reservation during purchase.
#' @return A list object or a character vector
#' @export
ec2_purchase_host_reservation <- function(HostIdSet, OfferingId, ClientToken = NULL, CurrencyCode = NULL, 
    LimitPrice = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    HostIdSet <- list_to_array("HostIdSet", HostIdSet)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(OfferingId = OfferingId, ClientToken = ClientToken, CurrencyCode = CurrencyCode, 
        LimitPrice = LimitPrice), HostIdSet, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "PurchaseHostReservation", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Purchase Reserved Instances Offering
#' 
#' @param InstanceCount Integer. The number of Reserved Instances to purchase.
#' @param ReservedInstancesOfferingId String. The ID of the Reserved Instance offering to purchase.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param LimitPrice Object. Specified for Reserved Instance Marketplace offerings to limit the total order and ensure that the...\[optional\]
#' @param PurchaseTime String. The time at which to purchase the Reserved Instance, in UTC format (for example, *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z)....\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceCount:
#' The number of
#' Reserved Instances to purchase.
#' @section ReservedInstancesOfferingId:
#' The ID of the
#' Reserved Instance offering to purchase.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section LimitPrice:
#' Specified
#' for Reserved Instance Marketplace offerings to limit the total order and
#' ensure that the Reserved Instances are not purchased at unexpected
#' prices.
#' @section PurchaseTime:
#' The time at which to purchase the Reserved
#' Instance, in UTC format (for example,
#' *YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z).
#' @return A list object or a character vector
#' @export
ec2_purchase_reserved_instances_offering <- function(InstanceCount, ReservedInstancesOfferingId, 
    DryRun = NULL, LimitPrice = NULL, PurchaseTime = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceCount = InstanceCount, ReservedInstancesOfferingId = ReservedInstancesOfferingId, 
        DryRun = DryRun, LimitPrice = LimitPrice, PurchaseTime = PurchaseTime))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "PurchaseReservedInstancesOffering", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Purchase Scheduled Instances
#' 
#' @param PurchaseRequest List. The purchase requests.
#' @param ClientToken String. Unique, case-sensitive identifier that ensures the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PurchaseRequest:
#' The purchase
#' requests.
#' @section ClientToken:
#' Unique, case-sensitive identifier that
#' ensures the idempotency of the request. For more information, see
#' [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_purchase_scheduled_instances <- function(PurchaseRequest, ClientToken = NULL, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    PurchaseRequest <- list_to_array("PurchaseRequest", PurchaseRequest)
    parameters <- c(others, list(ClientToken = ClientToken, DryRun = DryRun), PurchaseRequest)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "PurchaseScheduledInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Reboot Instances
#' 
#' @param InstanceId List. The instance IDs.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The
#' instance IDs.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reboot_instances <- function(InstanceId, DryRun = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    parameters <- c(others, list(DryRun = DryRun), InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RebootInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Register Image
#' 
#' @param Name String.   A name for your AMI.
#' @param ImageLocation String. The full path to your AMI manifest in Amazon S3 storage.\[optional\]
#' @param Architecture String.   The architecture of the AMI.\[optional\]
#' @param BlockDeviceMapping List.   The block device mapping entries.\[optional\]
#' @param Description String. A description for your AMI.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EnaSupport Logical.   Set to `true` to enable enhanced networking with ENA for the AMI and any instances that you launch...\[optional\]
#' @param KernelId String. The ID of the kernel.\[optional\]
#' @param BillingProduct List. The billing product codes.\[optional\]
#' @param RamdiskId String. The ID of the RAM disk.\[optional\]
#' @param RootDeviceName String. The device name of the root device volume (for example, `/dev/sda1`).\[optional\]
#' @param SriovNetSupport String.   Set to `simple` to enable enhanced networking with the Intel 82599 Virtual Function interface for...\[optional\]
#' @param VirtualizationType String.   The type of virtualization (`hvm` \| `paravirtual`).  Default: `paravirtual`  \[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Name:
#' 
#' A name for your AMI.
#' 
#' Constraints: 3-128 alphanumeric characters, parentheses (()), square
#' brackets (\[\]), spaces ( ), periods (.), slashes (/), dashes (-),
#' single quotes (\'), at-signs (@), or underscores(_)
#' 
#' @section ImageLocation:
#' The full path to your AMI manifest in Amazon S3
#' storage. The specified bucket must have the `aws-exec-read` canned
#' access control list (ACL) to ensure that it can be accessed by Amazon
#' EC2. For more information, see [Canned
#' ACLs](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl)
#' in the *Amazon S3 Service Developer Guide*.
#' @section Architecture:
#' 
#' The architecture of the AMI.
#' 
#' Default: For Amazon EBS-backed AMIs, `i386`. For instance store-backed
#' AMIs, the architecture specified in the manifest file.
#' 
#' @section BlockDeviceMapping:
#' 
#' The block device mapping entries.
#' 
#' If you specify an EBS volume using the ID of an EBS snapshot, you can\'t
#' specify the encryption state of the volume.
#' 
#' If you create an AMI on an Outpost, then all backing snapshots must be
#' on the same Outpost or in the Region of that Outpost. AMIs on an Outpost
#' that include local snapshots can be used to launch instances on the same
#' Outpost only. For more information, [Amazon EBS local snapshots on
#' Outposts](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshots-outposts.html#ami)
#' in the *Amazon Elastic Compute Cloud User Guide*.
#' 
#' @section Description:
#' A description for your
#' AMI.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section EnaSupport:
#' 
#' Set to `true` to enable enhanced networking with ENA for the AMI and any
#' instances that you launch from the AMI.
#' 
#' This option is supported only for HVM AMIs. Specifying this option with
#' a PV AMI can make instances launched from the AMI unreachable.
#' 
#' @section KernelId:
#' The ID of the kernel.
#' @section BillingProduct:
#' The
#' billing product codes. Your account must be authorized to specify
#' billing product codes. Otherwise, you can use the AWS Marketplace to
#' bill for the use of an AMI.
#' @section RamdiskId:
#' The ID of the RAM
#' disk.
#' @section RootDeviceName:
#' The device name of the root device volume (for
#' example, `/dev/sda1`).
#' @section SriovNetSupport:
#' 
#' Set to `simple` to enable enhanced networking with the Intel 82599
#' Virtual Function interface for the AMI and any instances that you launch
#' from the AMI.
#' 
#' There is no way to disable `sriovNetSupport` at this time.
#' 
#' This option is supported only for HVM AMIs. Specifying this option with
#' a PV AMI can make instances launched from the AMI unreachable.
#' 
#' @section VirtualizationType:
#' 
#' The type of virtualization (`hvm` \| `paravirtual`).
#' 
#' Default: `paravirtual`
#' 
#' @return A list object or a character vector
#' @export
ec2_register_image <- function(Name, ImageLocation = NULL, Architecture = NULL, BlockDeviceMapping = NULL, 
    Description = NULL, DryRun = NULL, EnaSupport = NULL, KernelId = NULL, BillingProduct = NULL, 
    RamdiskId = NULL, RootDeviceName = NULL, SriovNetSupport = NULL, VirtualizationType = NULL, 
    simplify = TRUE, others = list()) {
    BlockDeviceMapping <- list_to_array("BlockDeviceMapping", BlockDeviceMapping)
    BillingProduct <- list_to_array("BillingProduct", BillingProduct)
    parameters <- c(others, list(Name = Name, ImageLocation = ImageLocation, Architecture = Architecture, 
        Description = Description, DryRun = DryRun, EnaSupport = EnaSupport, KernelId = KernelId, 
        RamdiskId = RamdiskId, RootDeviceName = RootDeviceName, SriovNetSupport = SriovNetSupport, 
        VirtualizationType = VirtualizationType), BlockDeviceMapping, BillingProduct)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RegisterImage", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Register Instance Event Notification Attributes
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceTagAttribute Object. Information about the tag keys to register.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section InstanceTagAttribute:
#' Information about the tag
#' keys to register.
#' @return A list object or a character vector
#' @export
ec2_register_instance_event_notification_attributes <- function(DryRun = NULL, InstanceTagAttribute = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun, InstanceTagAttribute = InstanceTagAttribute))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RegisterInstanceEventNotificationAttributes", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Register Transit Gateway Multicast Group Members
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param GroupIpAddress String. The IP address assigned to the transit gateway multicast group.\[optional\]
#' @param NetworkInterfaceIds List. The group members\' network interface IDs to register with the transit gateway multicast group.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit gateway
#' multicast domain.
#' @section GroupIpAddress:
#' The IP address assigned to the
#' transit gateway multicast group.
#' @section NetworkInterfaceIds:
#' The group members\'
#' network interface IDs to register with the transit gateway multicast
#' group.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_register_transit_gateway_multicast_group_members <- function(TransitGatewayMulticastDomainId = NULL, 
    GroupIpAddress = NULL, NetworkInterfaceIds = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    NetworkInterfaceIds <- list_to_array("NetworkInterfaceIds", NetworkInterfaceIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        GroupIpAddress = GroupIpAddress, DryRun = DryRun), NetworkInterfaceIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RegisterTransitGatewayMulticastGroupMembers", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Register Transit Gateway Multicast Group Sources
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param GroupIpAddress String. The IP address assigned to the transit gateway multicast group.\[optional\]
#' @param NetworkInterfaceIds List. The group sources\' network interface IDs to register with the transit gateway multicast group.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit
#' gateway multicast domain.
#' @section GroupIpAddress:
#' The IP address assigned to
#' the transit gateway multicast group.
#' @section NetworkInterfaceIds:
#' The group
#' sources\' network interface IDs to register with the transit gateway
#' multicast group.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_register_transit_gateway_multicast_group_sources <- function(TransitGatewayMulticastDomainId = NULL, 
    GroupIpAddress = NULL, NetworkInterfaceIds = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    NetworkInterfaceIds <- list_to_array("NetworkInterfaceIds", NetworkInterfaceIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        GroupIpAddress = GroupIpAddress, DryRun = DryRun), NetworkInterfaceIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RegisterTransitGatewayMulticastGroupSources", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Reject Transit Gateway Multicast Domain Associations
#' 
#' Rejects a request to associate cross-account
#' subnets with a transit gateway multicast
#' domain.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param TransitGatewayAttachmentId String. The ID of the transit gateway attachment.\[optional\]
#' @param SubnetIds List. The IDs of the subnets to associate with the transit gateway multicast domain.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit
#' gateway multicast domain.
#' @section TransitGatewayAttachmentId:
#' The ID of the transit
#' gateway attachment.
#' @section SubnetIds:
#' The IDs of the subnets to
#' associate with the transit gateway multicast
#' domain.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reject_transit_gateway_multicast_domain_associations <- function(TransitGatewayMulticastDomainId = NULL, 
    TransitGatewayAttachmentId = NULL, SubnetIds = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    SubnetIds <- list_to_array("SubnetIds", SubnetIds)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, DryRun = DryRun), SubnetIds)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RejectTransitGatewayMulticastDomainAssociations", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Reject Transit Gateway Peering Attachment
#' 
#' Rejects a transit gateway peering attachment
#' request.
#' 
#' @param TransitGatewayAttachmentId String. The ID of the transit gateway peering attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the transit
#' gateway peering attachment.
#' @section DryRun:
#' Checks whether you have
#' the required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reject_transit_gateway_peering_attachment <- function(TransitGatewayAttachmentId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RejectTransitGatewayPeeringAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Reject Transit Gateway Vpc Attachment
#' 
#' @param TransitGatewayAttachmentId String. The ID of the attachment.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reject_transit_gateway_vpc_attachment <- function(TransitGatewayAttachmentId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(TransitGatewayAttachmentId = TransitGatewayAttachmentId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RejectTransitGatewayVpcAttachment", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Reject Vpc Endpoint Connections
#' 
#' Rejects one or more VPC endpoint connection
#' requests to your VPC endpoint service.
#' 
#' @param ServiceId String. The ID of the service.
#' @param VpcEndpointId List. The IDs of one or more VPC endpoints.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ServiceId:
#' The ID of the
#' service.
#' @section VpcEndpointId:
#' The IDs of one or more VPC
#' endpoints.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reject_vpc_endpoint_connections <- function(ServiceId, VpcEndpointId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    VpcEndpointId <- list_to_array("VpcEndpointId", VpcEndpointId)
    parameters <- c(others, list(ServiceId = ServiceId, DryRun = DryRun), VpcEndpointId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RejectVpcEndpointConnections", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Reject Vpc Peering Connection
#' 
#' Rejects a VPC
#' peering connection request. The VPC peering connection must be in the
#' `pending-acceptance` state. Use the DescribeVpcPeeringConnections
#' request to view your outstanding VPC peering connection requests. To
#' delete an active VPC peering connection, or to delete a VPC peering
#' connection request that you initiated, use
#' DeleteVpcPeeringConnection.
#' 
#' @param VpcPeeringConnectionId String. The ID of the VPC peering connection.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section VpcPeeringConnectionId:
#' The ID of the VPC peering
#' connection.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reject_vpc_peering_connection <- function(VpcPeeringConnectionId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(VpcPeeringConnectionId = VpcPeeringConnectionId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RejectVpcPeeringConnection", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Release Address
#' 
#' @param AllocationId String. \[EC2-VPC\] The allocation ID. Required for EC2-VPC.\[optional\]
#' @param PublicIp String. \[EC2-Classic\] The Elastic IP address. Required for EC2-Classic.\[optional\]
#' @param NetworkBorderGroup String.   The set of Availability Zones, Local Zones, or Wavelength Zones from which AWS advertises IP addresses....\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AllocationId:
#' \[EC2-VPC\] The allocation
#' ID. Required for EC2-VPC.
#' @section PublicIp:
#' \[EC2-Classic\] The
#' Elastic IP address. Required for EC2-Classic.
#' @section NetworkBorderGroup:
#' 
#' The set of Availability Zones, Local Zones, or Wavelength Zones from
#' which AWS advertises IP addresses.
#' 
#' If you provide an incorrect network border group, you will receive an
#' `InvalidAddress.NotFound` error. For more information, see [Error
#' Codes](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/errors-overview.html).
#' 
#' You cannot use a network border group with EC2 Classic. If you attempt
#' this operation on EC2 classic, you will receive an
#' `InvalidParameterCombination` error. For more information, see [Error
#' Codes](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/errors-overview.html).
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_release_address <- function(AllocationId = NULL, PublicIp = NULL, NetworkBorderGroup = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(AllocationId = AllocationId, PublicIp = PublicIp, NetworkBorderGroup = NetworkBorderGroup, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReleaseAddress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Release Hosts
#' 
#' @param HostId List. The IDs of the Dedicated Hosts to release.
#' @inheritParams CommonDoc
#' 
#' @section HostId:
#' The IDs of the Dedicated
#' Hosts to release.
#' @return A list object or a character vector
#' @export
ec2_release_hosts <- function(HostId, simplify = TRUE, others = list()) {
    HostId <- list_to_array("HostId", HostId)
    parameters <- c(others, list(), HostId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReleaseHosts", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Replace Iam Instance Profile Association
#' 
#' @param IamInstanceProfile Object. The IAM instance profile.
#' @param AssociationId String. The ID of the existing IAM instance profile association.
#' @inheritParams CommonDoc
#' 
#' @section IamInstanceProfile:
#' The IAM instance
#' profile.
#' @section AssociationId:
#' The ID of the existing IAM instance profile
#' association.
#' @return A list object or a character vector
#' @export
ec2_replace_iam_instance_profile_association <- function(IamInstanceProfile, AssociationId, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(IamInstanceProfile = IamInstanceProfile, AssociationId = AssociationId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReplaceIamInstanceProfileAssociation", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Replace Network Acl Association
#' 
#' @param AssociationId String. The ID of the current association between the original network ACL and the subnet.
#' @param NetworkAclId String. The ID of the new network ACL to associate with the subnet.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' The ID of the current association
#' between the original network ACL and the subnet.
#' @section NetworkAclId:
#' The
#' ID of the new network ACL to associate with the
#' subnet.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_replace_network_acl_association <- function(AssociationId, NetworkAclId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(AssociationId = AssociationId, NetworkAclId = NetworkAclId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReplaceNetworkAclAssociation", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Replace Network Acl Entry
#' 
#' Replaces an entry (rule) in a network ACL. For more
#' information, see [Network
#' ACLs](https://docs.aws.amazon.com/vpc/latest/userguide/VPC_ACLs.html) in
#' the *Amazon Virtual Private Cloud User Guide*.
#' 
#' @param Egress Logical.   Indicates whether to replace the egress rule.
#' @param NetworkAclId String. The ID of the ACL.
#' @param Protocol String. The protocol number.
#' @param RuleAction String. Indicates whether to allow or deny the traffic that matches the rule.
#' @param RuleNumber Integer. The rule number of the entry to replace.
#' @param CidrBlock String. The IPv4 network range to allow or deny, in CIDR notation (for example `172.16.0.0/24`).\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Icmp Object. ICMP protocol: The ICMP or ICMPv6 type and code.\[optional\]
#' @param Ipv6CidrBlock String. The IPv6 network range to allow or deny, in CIDR notation (for example `2001:bd8:1234:1a00::/64`).\[optional\]
#' @param PortRange Object. TCP or UDP protocols: The range of ports the rule applies to.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Egress:
#' 
#' Indicates whether to replace the egress rule.
#' 
#' Default: If no value is specified, we replace the ingress rule.
#' 
#' @section NetworkAclId:
#' The ID of the ACL.
#' @section Protocol:
#' The protocol
#' number. A value of \'-1\' means all protocols. If you specify \'-1\' or
#' a protocol number other than \'6\' (TCP), \'17\' (UDP), or \'1\' (ICMP),
#' traffic on all ports is allowed, regardless of any ports or ICMP types
#' or codes that you specify. If you specify protocol \'58\' (ICMPv6) and
#' specify an IPv4 CIDR block, traffic for all ICMP types and codes
#' allowed, regardless of any that you specify. If you specify protocol
#' \'58\' (ICMPv6) and specify an IPv6 CIDR block, you must specify an ICMP
#' type and code.
#' @section RuleAction:
#' Indicates whether to allow or deny
#' the traffic that matches the rule.
#' @section RuleNumber:
#' The rule number
#' of the entry to replace.
#' @section CidrBlock:
#' The IPv4 network range to
#' allow or deny, in CIDR notation (for example
#' `172.16.0.0/24`).
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Icmp:
#' ICMP protocol: The ICMP or
#' ICMPv6 type and code. Required if specifying protocol 1 (ICMP) or
#' protocol 58 (ICMPv6) with an IPv6 CIDR block.
#' @section Ipv6CidrBlock:
#' The
#' IPv6 network range to allow or deny, in CIDR notation (for example
#' `2001:bd8:1234:1a00::/64`).
#' @section PortRange:
#' TCP or UDP protocols:
#' The range of ports the rule applies to. Required if specifying protocol
#' 6 (TCP) or 17 (UDP).
#' @return A list object or a character vector
#' @export
ec2_replace_network_acl_entry <- function(Egress, NetworkAclId, Protocol, RuleAction, RuleNumber, 
    CidrBlock = NULL, DryRun = NULL, Icmp = NULL, Ipv6CidrBlock = NULL, PortRange = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(Egress = Egress, NetworkAclId = NetworkAclId, Protocol = Protocol, 
        RuleAction = RuleAction, RuleNumber = RuleNumber, CidrBlock = CidrBlock, DryRun = DryRun, 
        Icmp = Icmp, Ipv6CidrBlock = Ipv6CidrBlock, PortRange = PortRange))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReplaceNetworkAclEntry", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Replace Route
#' 
#' @param RouteTableId String. The ID of the route table.
#' @param DestinationCidrBlock String. The IPv4 CIDR address block used for the destination match.\[optional\]
#' @param DestinationIpv6CidrBlock String. The IPv6 CIDR address block used for the destination match.\[optional\]
#' @param DestinationPrefixListId String. The ID of the prefix list for the route.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param VpcEndpointId String. The ID of a VPC endpoint. Supported for Gateway Load Balancer endpoints only.\[optional\]
#' @param EgressOnlyInternetGatewayId String. \[IPv6 traffic only\] The ID of an egress-only internet gateway.\[optional\]
#' @param GatewayId String. The ID of an internet gateway or virtual private gateway.\[optional\]
#' @param InstanceId String. The ID of a NAT instance in your VPC.\[optional\]
#' @param LocalTarget Logical. Specifies whether to reset the local route to its default target (`local`).\[optional\]
#' @param NatGatewayId String. \[IPv4 traffic only\] The ID of a NAT gateway.\[optional\]
#' @param TransitGatewayId String. The ID of a transit gateway.\[optional\]
#' @param LocalGatewayId String. The ID of the local gateway.\[optional\]
#' @param CarrierGatewayId String. \[IPv4 traffic only\] The ID of a carrier gateway.\[optional\]
#' @param NetworkInterfaceId String. The ID of a network interface.\[optional\]
#' @param VpcPeeringConnectionId String. The ID of a VPC peering connection.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section RouteTableId:
#' The ID of the route
#' table.
#' @section DestinationCidrBlock:
#' The IPv4 CIDR address block used for the
#' destination match. The value that you provide must match the CIDR of an
#' existing route in the table.
#' @section DestinationIpv6CidrBlock:
#' The IPv6 CIDR address
#' block used for the destination match. The value that you provide must
#' match the CIDR of an existing route in the
#' table.
#' @section DestinationPrefixListId:
#' The ID of the prefix list for the
#' route.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section VpcEndpointId:
#' The ID of a VPC endpoint.
#' Supported for Gateway Load Balancer endpoints
#' only.
#' @section EgressOnlyInternetGatewayId:
#' \[IPv6 traffic only\] The ID of an egress-only
#' internet gateway.
#' @section GatewayId:
#' The ID of an internet gateway or
#' virtual private gateway.
#' @section InstanceId:
#' The ID of a NAT instance in
#' your VPC.
#' @section LocalTarget:
#' Specifies whether to reset the local route
#' to its default target (`local`).
#' @section NatGatewayId:
#' \[IPv4 traffic
#' only\] The ID of a NAT gateway.
#' @section TransitGatewayId:
#' The ID of a transit
#' gateway.
#' @section LocalGatewayId:
#' The ID of the local
#' gateway.
#' @section CarrierGatewayId:
#' \[IPv4 traffic only\] The ID of a carrier
#' gateway.
#' @section NetworkInterfaceId:
#' The ID of a network
#' interface.
#' @section VpcPeeringConnectionId:
#' The ID of a VPC peering
#' connection.
#' @return A list object or a character vector
#' @export
ec2_replace_route <- function(RouteTableId, DestinationCidrBlock = NULL, DestinationIpv6CidrBlock = NULL, 
    DestinationPrefixListId = NULL, DryRun = NULL, VpcEndpointId = NULL, EgressOnlyInternetGatewayId = NULL, 
    GatewayId = NULL, InstanceId = NULL, LocalTarget = NULL, NatGatewayId = NULL, TransitGatewayId = NULL, 
    LocalGatewayId = NULL, CarrierGatewayId = NULL, NetworkInterfaceId = NULL, VpcPeeringConnectionId = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(RouteTableId = RouteTableId, DestinationCidrBlock = DestinationCidrBlock, 
        DestinationIpv6CidrBlock = DestinationIpv6CidrBlock, DestinationPrefixListId = DestinationPrefixListId, 
        DryRun = DryRun, VpcEndpointId = VpcEndpointId, EgressOnlyInternetGatewayId = EgressOnlyInternetGatewayId, 
        GatewayId = GatewayId, InstanceId = InstanceId, LocalTarget = LocalTarget, NatGatewayId = NatGatewayId, 
        TransitGatewayId = TransitGatewayId, LocalGatewayId = LocalGatewayId, CarrierGatewayId = CarrierGatewayId, 
        NetworkInterfaceId = NetworkInterfaceId, VpcPeeringConnectionId = VpcPeeringConnectionId))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReplaceRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Replace Route Table Association
#' 
#' @param AssociationId String. The association ID.
#' @param RouteTableId String. The ID of the new route table to associate with the subnet.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AssociationId:
#' The association
#' ID.
#' @section RouteTableId:
#' The ID of the new route table to associate with
#' the subnet.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_replace_route_table_association <- function(AssociationId, RouteTableId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(AssociationId = AssociationId, RouteTableId = RouteTableId, 
        DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReplaceRouteTableAssociation", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Replace Transit Gateway Route
#' 
#' Replaces the specified route in the specified
#' transit gateway route table.
#' 
#' @param DestinationCidrBlock String. The CIDR range used for the destination match.
#' @param TransitGatewayRouteTableId String. The ID of the route table.
#' @param TransitGatewayAttachmentId String. The ID of the attachment.\[optional\]
#' @param Blackhole Logical. Indicates whether traffic matching this route is to be dropped.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DestinationCidrBlock:
#' The CIDR range used for the
#' destination match. Routing decisions are based on the most specific
#' match.
#' @section TransitGatewayRouteTableId:
#' The ID of the route
#' table.
#' @section TransitGatewayAttachmentId:
#' The ID of the
#' attachment.
#' @section Blackhole:
#' Indicates whether traffic matching this
#' route is to be dropped.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_replace_transit_gateway_route <- function(DestinationCidrBlock, TransitGatewayRouteTableId, 
    TransitGatewayAttachmentId = NULL, Blackhole = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(DestinationCidrBlock = DestinationCidrBlock, TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        TransitGatewayAttachmentId = TransitGatewayAttachmentId, Blackhole = Blackhole, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReplaceTransitGatewayRoute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Report Instance Status
#' 
#' @param InstanceId List. The instances.
#' @param ReasonCode List.   The reason codes that describe the health state of your instance.
#' @param Status String. The status of all instances listed.
#' @param Description String. Descriptive text about the health state of your instance.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EndTime String. The time at which the reported instance health state ended.\[optional\]
#' @param StartTime String. The time at which the reported instance health state began.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The
#' instances.
#' @section ReasonCode:
#' 
#' The reason codes that describe the health state of your instance.
#' 
#' -   `instance-stuck-in-state`: My instance is stuck in a state.
#' 
#' -   `unresponsive`: My instance is unresponsive.
#' 
#' -   `not-accepting-credentials`: My instance is not accepting my
#'     credentials.
#' 
#' -   `password-not-available`: A password is not available for my
#'     instance.
#' 
#' -   `performance-network`: My instance is experiencing performance
#'     problems that I believe are network related.
#' 
#' -   `performance-instance-store`: My instance is experiencing
#'     performance problems that I believe are related to the instance
#'     stores.
#' 
#' -   `performance-ebs-volume`: My instance is experiencing performance
#'     problems that I believe are related to an EBS volume.
#' 
#' -   `performance-other`: My instance is experiencing performance
#'     problems.
#' 
#' -   `other`: \[explain using the description parameter\]
#' 
#' @section Status:
#' The status of all instances
#' listed.
#' @section Description:
#' Descriptive text about the health state of
#' your instance.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section EndTime:
#' The time at which the
#' reported instance health state ended.
#' @section StartTime:
#' The time at
#' which the reported instance health state began.
#' @return A list object or a character vector
#' @export
ec2_report_instance_status <- function(InstanceId, ReasonCode, Status, Description = NULL, 
    DryRun = NULL, EndTime = NULL, StartTime = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    ReasonCode <- list_to_array("ReasonCode", ReasonCode)
    parameters <- c(others, list(Status = Status, Description = Description, DryRun = DryRun, 
        EndTime = EndTime, StartTime = StartTime), InstanceId, ReasonCode)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ReportInstanceStatus", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Request Spot Fleet
#' 
#' @param SpotFleetRequestConfig Object. The configuration for the Spot Fleet request.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section SpotFleetRequestConfig:
#' The
#' configuration for the Spot Fleet request.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_request_spot_fleet <- function(SpotFleetRequestConfig, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(SpotFleetRequestConfig = SpotFleetRequestConfig, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RequestSpotFleet", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Request Spot Instances
#' 
#' @param AvailabilityZoneGroup String.   The user-specified name for a logical grouping of requests.\[optional\]
#' @param BlockDurationMinutes Integer.   The required duration for the Spot Instances (also known as Spot blocks), in minutes.\[optional\]
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceCount Integer.   The maximum number of Spot Instances to launch.  Default: 1  \[optional\]
#' @param LaunchGroup String.   The instance launch group.\[optional\]
#' @param LaunchSpecification Object. The launch specification.\[optional\]
#' @param SpotPrice String. The maximum price per hour that you are willing to pay for a Spot Instance.\[optional\]
#' @param Type String.   The Spot Instance request type.  Default: `one-time`  \[optional\]
#' @param ValidFrom String.   The start date of the request.\[optional\]
#' @param ValidUntil String.   The end date of the request, in UTC format (*YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z).\[optional\]
#' @param TagSpecification List. The key-value pair for tagging the Spot Instance request on creation.\[optional\]
#' @param InstanceInterruptionBehavior String. The behavior when a Spot Instance is interrupted. The default is `terminate`.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AvailabilityZoneGroup:
#' 
#' The user-specified name for a logical grouping of requests.
#' 
#' When you specify an Availability Zone group in a Spot Instance request,
#' all Spot Instances in the request are launched in the same Availability
#' Zone. Instance proximity is maintained with this parameter, but the
#' choice of Availability Zone is not. The group applies only to requests
#' for Spot Instances of the same instance type. Any additional Spot
#' Instance requests that are specified with the same Availability Zone
#' group name are launched in that same Availability Zone, as long as at
#' least one instance from the group is still active.
#' 
#' If there is no active instance running in the Availability Zone group
#' that you specify for a new Spot Instance request (all instances are
#' terminated, the request is expired, or the maximum price you specified
#' falls below current Spot price), then Amazon EC2 launches the instance
#' in any Availability Zone where the constraint can be met. Consequently,
#' the subsequent set of Spot Instances could be placed in a different zone
#' from the original request, even if you specified the same Availability
#' Zone group.
#' 
#' Default: Instances are launched in any available Availability Zone.
#' 
#' @section BlockDurationMinutes:
#' 
#' The required duration for the Spot Instances (also known as Spot
#' blocks), in minutes. This value must be a multiple of 60 (60, 120, 180,
#' 240, 300, or 360).
#' 
#' The duration period starts as soon as your Spot Instance receives its
#' instance ID. At the end of the duration period, Amazon EC2 marks the
#' Spot Instance for termination and provides a Spot Instance termination
#' notice, which gives the instance a two-minute warning before it
#' terminates.
#' 
#' You can\'t specify an Availability Zone group or a launch group if you
#' specify a duration.
#' 
#' New accounts or accounts with no previous billing history with AWS are
#' not eligible for Spot Instances with a defined duration (also known as
#' Spot blocks).
#' 
#' @section ClientToken:
#' Unique, case-sensitive identifier that you provide
#' to ensure the idempotency of the request. For more information, see [How
#' to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html)
#' in the *Amazon EC2 User Guide for Linux
#' Instances*.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section InstanceCount:
#' 
#' The maximum number of Spot Instances to launch.
#' 
#' Default: 1
#' 
#' @section LaunchGroup:
#' 
#' The instance launch group. Launch groups are Spot Instances that launch
#' together and terminate together.
#' 
#' Default: Instances are launched and terminated individually
#' 
#' @section LaunchSpecification:
#' The launch specification.
#' @section SpotPrice:
#' The
#' maximum price per hour that you are willing to pay for a Spot Instance.
#' The default is the On-Demand price.
#' @section Type:
#' 
#' The Spot Instance request type.
#' 
#' Default: `one-time`
#' 
#' @section ValidFrom:
#' 
#' The start date of the request. If this is a one-time request, the
#' request becomes active at this date and time and remains active until
#' all instances launch, the request expires, or the request is canceled.
#' If the request is persistent, the request becomes active at this date
#' and time and remains active until it expires or is canceled.
#' 
#' The specified start date and time cannot be equal to the current date
#' and time. You must specify a start date and time that occurs after the
#' current date and time.
#' 
#' @section ValidUntil:
#' 
#' The end date of the request, in UTC format
#' (*YYYY*-*MM*-*DD*T*HH*:*MM*:*SS*Z).
#' 
#' -   For a persistent request, the request remains active until the
#'     `ValidUntil` date and time is reached. Otherwise, the request
#'     remains active until you cancel it.
#' 
#' -   For a one-time request, the request remains active until all
#'     instances launch, the request is canceled, or the `ValidUntil` date
#'     and time is reached. By default, the request is valid for 7 days
#'     from the date the request was created.
#' 
#' @section TagSpecification:
#' The key-value pair for tagging the Spot Instance
#' request on creation. The value for `ResourceType` must be
#' `spot-instances-request`, otherwise the Spot Instance request fails. To
#' tag the Spot Instance request after it has been created, see
#' [CreateTags](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html).
#' 
#' @section InstanceInterruptionBehavior:
#' The behavior when a Spot Instance is interrupted.
#' The default is `terminate`.
#' @return A list object or a character vector
#' @export
ec2_request_spot_instances <- function(AvailabilityZoneGroup = NULL, BlockDurationMinutes = NULL, 
    ClientToken = NULL, DryRun = NULL, InstanceCount = NULL, LaunchGroup = NULL, LaunchSpecification = NULL, 
    SpotPrice = NULL, Type = NULL, ValidFrom = NULL, ValidUntil = NULL, TagSpecification = NULL, 
    InstanceInterruptionBehavior = NULL, simplify = TRUE, others = list()) {
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(AvailabilityZoneGroup = AvailabilityZoneGroup, BlockDurationMinutes = BlockDurationMinutes, 
        ClientToken = ClientToken, DryRun = DryRun, InstanceCount = InstanceCount, LaunchGroup = LaunchGroup, 
        LaunchSpecification = LaunchSpecification, SpotPrice = SpotPrice, Type = Type, ValidFrom = ValidFrom, 
        ValidUntil = ValidUntil, InstanceInterruptionBehavior = InstanceInterruptionBehavior), 
        TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RequestSpotInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Reset Address Attribute
#' 
#' Resets the attribute of the specified IP address.
#' For requirements, see [Using reverse DNS for email
#' applications](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html#Using_Elastic_Addressing_Reverse_DNS).
#' 
#' @param AllocationId String. \[EC2-VPC\] The allocation ID.
#' @param Attribute String. The attribute of the IP address.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section AllocationId:
#' \[EC2-VPC\] The
#' allocation ID.
#' @section Attribute:
#' The attribute of the IP
#' address.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reset_address_attribute <- function(AllocationId, Attribute, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(AllocationId = AllocationId, Attribute = Attribute, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ResetAddressAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Reset Ebs Default Kms Key Id
#' 
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reset_ebs_default_kms_key_id <- function(DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ResetEbsDefaultKmsKeyId", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Reset Fpga Image Attribute
#' 
#' Resets the specified attribute of the specified
#' Amazon FPGA Image (AFI) to its default value. You can only reset the
#' load permission attribute.
#' 
#' @param FpgaImageId String. The ID of the AFI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Attribute String. The attribute.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section FpgaImageId:
#' The ID of the
#' AFI.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Attribute:
#' The
#' attribute.
#' @return A list object or a character vector
#' @export
ec2_reset_fpga_image_attribute <- function(FpgaImageId, DryRun = NULL, Attribute = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(FpgaImageId = FpgaImageId, DryRun = DryRun, Attribute = Attribute))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ResetFpgaImageAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Reset Image Attribute
#' 
#' @param Attribute String. The attribute to reset (currently you can only reset the launch permission attribute).
#' @param ImageId String. The ID of the AMI.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' The attribute to reset (currently you can
#' only reset the launch permission attribute).
#' @section ImageId:
#' The ID
#' of the AMI.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reset_image_attribute <- function(Attribute, ImageId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, ImageId = ImageId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ResetImageAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Reset Instance Attribute
#' 
#' @param Attribute String.   The attribute to reset.
#' @param InstanceId String. The ID of the instance.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' 
#' The attribute to reset.
#' 
#' You can only reset the following attributes: `kernel` \| `ramdisk` \|
#' `sourceDestCheck`. To change an instance attribute, use
#' ModifyInstanceAttribute.
#' 
#' @section InstanceId:
#' The ID of the instance.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reset_instance_attribute <- function(Attribute, InstanceId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, InstanceId = InstanceId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ResetInstanceAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Reset Network Interface Attribute
#' 
#' Resets a network interface attribute. You can
#' specify only one attribute at a time.
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param SourceDestCheck String. The source/destination checking attribute. Resets the value to `true`.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of
#' the network interface.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section SourceDestCheck:
#' The source/destination
#' checking attribute. Resets the value to `true`.
#' @return A list object or a character vector
#' @export
ec2_reset_network_interface_attribute <- function(NetworkInterfaceId, DryRun = NULL, SourceDestCheck = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId, DryRun = DryRun, 
        SourceDestCheck = SourceDestCheck))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ResetNetworkInterfaceAttribute", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Reset Snapshot Attribute
#' 
#' @param Attribute String. The attribute to reset.
#' @param SnapshotId String. The ID of the snapshot.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section Attribute:
#' The
#' attribute to reset. Currently, only the attribute for permission to
#' create volumes can be reset.
#' @section SnapshotId:
#' The ID of the
#' snapshot.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_reset_snapshot_attribute <- function(Attribute, SnapshotId, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    parameters <- c(others, list(Attribute = Attribute, SnapshotId = SnapshotId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "ResetSnapshotAttribute", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Restore Address To Classic
#' 
#' Restores an Elastic IP address that was previously
#' moved to the EC2-VPC platform back to the EC2-Classic platform. You
#' cannot move an Elastic IP address that was originally allocated for use
#' in EC2-VPC. The Elastic IP address must not be associated with an
#' instance or network interface.
#' 
#' @param PublicIp String. The Elastic IP address.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PublicIp:
#' The Elastic IP
#' address.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_restore_address_to_classic <- function(PublicIp, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(PublicIp = PublicIp, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RestoreAddressToClassic", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Restore Managed Prefix List Version
#' 
#' Restores the entries
#' from a previous version of a managed prefix list to a new version of the
#' prefix list.
#' 
#' @param PrefixListId String. The ID of the prefix list.
#' @param PreviousVersion Integer. The version to restore.
#' @param CurrentVersion Integer. The current version number for the prefix list.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section PrefixListId:
#' The ID of the prefix
#' list.
#' @section PreviousVersion:
#' The version to
#' restore.
#' @section CurrentVersion:
#' The current version number for the prefix
#' list.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_restore_managed_prefix_list_version <- function(PrefixListId, PreviousVersion, CurrentVersion, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(PrefixListId = PrefixListId, PreviousVersion = PreviousVersion, 
        CurrentVersion = CurrentVersion, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RestoreManagedPrefixListVersion", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Revoke Client Vpn Ingress
#' 
#' Removes an ingress authorization rule
#' from a Client VPN endpoint. 
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint with which the authorization rule is associated.
#' @param TargetNetworkCidr String. The IPv4 address range, in CIDR notation, of the network for which access is being removed.
#' @param AccessGroupId String. The ID of the Active Directory group for which to revoke access. \[optional\]
#' @param RevokeAllGroups Logical. Indicates whether access should be revoked for all clients.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN
#' endpoint with which the authorization rule is
#' associated.
#' @section TargetNetworkCidr:
#' The IPv4 address range, in CIDR
#' notation, of the network for which access is being
#' removed.
#' @section AccessGroupId:
#' The ID of the Active Directory group for
#' which to revoke access. 
#' @section RevokeAllGroups:
#' Indicates whether access
#' should be revoked for all clients.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_revoke_client_vpn_ingress <- function(ClientVpnEndpointId, TargetNetworkCidr, AccessGroupId = NULL, 
    RevokeAllGroups = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, TargetNetworkCidr = TargetNetworkCidr, 
        AccessGroupId = AccessGroupId, RevokeAllGroups = RevokeAllGroups, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RevokeClientVpnIngress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Revoke Security Group Egress
#' 
#' @param GroupId String. The ID of the security group.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param IpPermissions List. The sets of IP permissions.\[optional\]
#' @param CidrIp String. Not supported. Use a set of IP permissions to specify the CIDR.\[optional\]
#' @param FromPort Integer. Not supported. Use a set of IP permissions to specify the port.\[optional\]
#' @param IpProtocol String. Not supported. Use a set of IP permissions to specify the protocol name or number.\[optional\]
#' @param ToPort Integer. Not supported. Use a set of IP permissions to specify the port.\[optional\]
#' @param SourceSecurityGroupName String. Not supported. Use a set of IP permissions to specify a destination security group.\[optional\]
#' @param SourceSecurityGroupOwnerId String. Not supported. Use a set of IP permissions to specify a destination security group.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section GroupId:
#' The ID of
#' the security group.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section IpPermissions:
#' The sets of IP permissions.
#' You can\'t specify a destination security group and a CIDR IP address
#' range in the same set of permissions.
#' @section CidrIp:
#' Not supported.
#' Use a set of IP permissions to specify the CIDR.
#' @section FromPort:
#' Not
#' supported. Use a set of IP permissions to specify the
#' port.
#' @section IpProtocol:
#' Not supported. Use a set of IP permissions to
#' specify the protocol name or number.
#' @section ToPort:
#' Not supported.
#' Use a set of IP permissions to specify the port.
#' @section SourceSecurityGroupName:
#' Not
#' supported. Use a set of IP permissions to specify a destination security
#' group.
#' @section SourceSecurityGroupOwnerId:
#' Not supported. Use a set of IP permissions to
#' specify a destination security group.
#' @return A list object or a character vector
#' @export
ec2_revoke_security_group_egress <- function(GroupId, DryRun = NULL, IpPermissions = NULL, 
    CidrIp = NULL, FromPort = NULL, IpProtocol = NULL, ToPort = NULL, SourceSecurityGroupName = NULL, 
    SourceSecurityGroupOwnerId = NULL, simplify = TRUE, others = list()) {
    IpPermissions <- list_to_array("IpPermissions", IpPermissions)
    parameters <- c(others, list(GroupId = GroupId, DryRun = DryRun, CidrIp = CidrIp, FromPort = FromPort, 
        IpProtocol = IpProtocol, ToPort = ToPort, SourceSecurityGroupName = SourceSecurityGroupName, 
        SourceSecurityGroupOwnerId = SourceSecurityGroupOwnerId), IpPermissions)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RevokeSecurityGroupEgress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Revoke Security Group Ingress
#' 
#' @param CidrIp String. The CIDR IP address range.\[optional\]
#' @param FromPort Integer. The start of port range for the TCP and UDP protocols, or an ICMP type number.\[optional\]
#' @param GroupId String. The ID of the security group.\[optional\]
#' @param GroupName String. \[EC2-Classic, default VPC\] The name of the security group.\[optional\]
#' @param IpPermissions List. The sets of IP permissions.\[optional\]
#' @param IpProtocol String. The IP protocol name (`tcp`, `udp`, `icmp`) or number (see [Protocol Numbers](http://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml))....\[optional\]
#' @param SourceSecurityGroupName String. \[EC2-Classic, default VPC\] The name of the source security group.\[optional\]
#' @param SourceSecurityGroupOwnerId String. \[EC2-Classic\] The AWS account ID of the source security group, if the source security group is...\[optional\]
#' @param ToPort Integer. The end of port range for the TCP and UDP protocols, or an ICMP code number.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section CidrIp:
#' The CIDR IP
#' address range. You can\'t specify this parameter when specifying a
#' source security group.
#' @section FromPort:
#' The start of port range for
#' the TCP and UDP protocols, or an ICMP type number. For the ICMP type
#' number, use `-1` to specify all ICMP types.
#' @section GroupId:
#' The ID
#' of the security group. You must specify either the security group ID or
#' the security group name in the request. For security groups in a
#' nondefault VPC, you must specify the security group
#' ID.
#' @section GroupName:
#' \[EC2-Classic, default VPC\] The name of the
#' security group. You must specify either the security group ID or the
#' security group name in the request.
#' @section IpPermissions:
#' The sets of IP
#' permissions. You can\'t specify a source security group and a CIDR IP
#' address range in the same set of permissions.
#' @section IpProtocol:
#' The IP
#' protocol name (`tcp`, `udp`, `icmp`) or number (see [Protocol
#' Numbers](http://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml)).
#' Use `-1` to specify all.
#' @section SourceSecurityGroupName:
#' \[EC2-Classic, default
#' VPC\] The name of the source security group. You can\'t specify this
#' parameter in combination with the following parameters: the CIDR IP
#' address range, the start of the port range, the IP protocol, and the end
#' of the port range. For EC2-VPC, the source security group must be in the
#' same VPC. To revoke a specific rule for an IP protocol and port range,
#' use a set of IP permissions instead.
#' @section SourceSecurityGroupOwnerId:
#' \[EC2-Classic\]
#' The AWS account ID of the source security group, if the source security
#' group is in a different account. You can\'t specify this parameter in
#' combination with the following parameters: the CIDR IP address range,
#' the IP protocol, the start of the port range, and the end of the port
#' range. To revoke a specific rule for an IP protocol and port range, use
#' a set of IP permissions instead.
#' @section ToPort:
#' The end of port
#' range for the TCP and UDP protocols, or an ICMP code number. For the
#' ICMP code number, use `-1` to specify all ICMP codes for the ICMP
#' type.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_revoke_security_group_ingress <- function(CidrIp = NULL, FromPort = NULL, GroupId = NULL, 
    GroupName = NULL, IpPermissions = NULL, IpProtocol = NULL, SourceSecurityGroupName = NULL, 
    SourceSecurityGroupOwnerId = NULL, ToPort = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    IpPermissions <- list_to_array("IpPermissions", IpPermissions)
    parameters <- c(others, list(CidrIp = CidrIp, FromPort = FromPort, GroupId = GroupId, 
        GroupName = GroupName, IpProtocol = IpProtocol, SourceSecurityGroupName = SourceSecurityGroupName, 
        SourceSecurityGroupOwnerId = SourceSecurityGroupOwnerId, ToPort = ToPort, DryRun = DryRun), 
        IpPermissions)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RevokeSecurityGroupIngress", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Run Instances
#' 
#' @param MaxCount Integer.   The maximum number of instances to launch.
#' @param MinCount Integer.   The minimum number of instances to launch.
#' @param BlockDeviceMapping List. The block device mapping entries.\[optional\]
#' @param ImageId String. The ID of the AMI.\[optional\]
#' @param InstanceType String.   The instance type.\[optional\]
#' @param Ipv6AddressCount Integer.   \[EC2-VPC\] The number of IPv6 addresses to associate with the primary network interface.\[optional\]
#' @param Ipv6Address List.   \[EC2-VPC\] The IPv6 addresses from the range of the subnet to associate with the primary network...\[optional\]
#' @param KernelId String.   The ID of the kernel.\[optional\]
#' @param KeyName String.   The name of the key pair.\[optional\]
#' @param Monitoring Object. Specifies whether detailed monitoring is enabled for the instance.\[optional\]
#' @param Placement Object. The placement for the instance.\[optional\]
#' @param RamdiskId String.   The ID of the RAM disk to select.\[optional\]
#' @param SecurityGroupId List.   The IDs of the security groups.\[optional\]
#' @param SecurityGroup List.   \[EC2-Classic, default VPC\] The names of the security groups.\[optional\]
#' @param SubnetId String.   \[EC2-VPC\] The ID of the subnet to launch the instance into.\[optional\]
#' @param UserData String. The user data to make available to the instance.\[optional\]
#' @param AdditionalInfo String. Reserved.\[optional\]
#' @param ClientToken String.   Unique, case-sensitive identifier you provide to ensure the idempotency of the request.\[optional\]
#' @param DisableApiTermination Logical.   If you set this parameter to `true`, you can\'t terminate the instance using the Amazon EC2 console,...\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param EbsOptimized Logical.   Indicates whether the instance is optimized for Amazon EBS I/O.\[optional\]
#' @param IamInstanceProfile Object. The name or Amazon Resource Name (ARN) of an IAM instance profile.\[optional\]
#' @param InstanceInitiatedShutdownBehavior String.   Indicates whether an instance stops or terminates when you initiate shutdown from the instance...\[optional\]
#' @param NetworkInterface List. The network interfaces to associate with the instance.\[optional\]
#' @param PrivateIpAddress String.   \[EC2-VPC\] The primary IPv4 address.\[optional\]
#' @param ElasticGpuSpecification List. An elastic GPU to associate with the instance.\[optional\]
#' @param ElasticInferenceAccelerator List.   An elastic inference accelerator to associate with the instance.\[optional\]
#' @param TagSpecification List. The tags to apply to the resources during launch.\[optional\]
#' @param LaunchTemplate Object. The launch template to use to launch the instances.\[optional\]
#' @param InstanceMarketOptions Object.   The market (purchasing) option for the instances.\[optional\]
#' @param CreditSpecification Object.   The credit option for CPU usage of the burstable performance instance.\[optional\]
#' @param CpuOptions Object. The CPU options for the instance.\[optional\]
#' @param CapacityReservationSpecification Object. Information about the Capacity Reservation targeting option.\[optional\]
#' @param HibernationOptions Object.   Indicates whether an instance is enabled for hibernation.\[optional\]
#' @param LicenseSpecification List. The license configurations.\[optional\]
#' @param MetadataOptions Object. The metadata options for the instance.\[optional\]
#' @param EnclaveOptions Object.   Indicates whether the instance is enabled for AWS Nitro Enclaves.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section MaxCount:
#' 
#' The maximum number of instances to launch. If you specify more instances
#' than Amazon EC2 can launch in the target Availability Zone, Amazon EC2
#' launches the largest possible number of instances above `MinCount`.
#' 
#' Constraints: Between 1 and the maximum number you\'re allowed for the
#' specified instance type. For more information about the default limits,
#' and how to request an increase, see [How many instances can I run in
#' Amazon
#' EC2](https://aws.amazon.com/ec2/faqs/#How_many_instances_can_I_run_in_Amazon_EC2/)
#' in the Amazon EC2 FAQ.
#' 
#' @section MinCount:
#' 
#' The minimum number of instances to launch. If you specify a minimum that
#' is more instances than Amazon EC2 can launch in the target Availability
#' Zone, Amazon EC2 launches no instances.
#' 
#' Constraints: Between 1 and the maximum number you\'re allowed for the
#' specified instance type. For more information about the default limits,
#' and how to request an increase, see [How many instances can I run in
#' Amazon
#' EC2](https://aws.amazon.com/ec2/faqs/#How_many_instances_can_I_run_in_Amazon_EC2/)
#' in the Amazon EC2 General FAQ.
#' 
#' @section BlockDeviceMapping:
#' The block device mapping
#' entries.
#' @section ImageId:
#' The ID of the AMI. An AMI ID is required to
#' launch an instance and must be specified here or in a launch
#' template.
#' @section InstanceType:
#' 
#' The instance type. For more information, see [Instance
#' types](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' Default: `m1.small`
#' 
#' @section Ipv6AddressCount:
#' 
#' \[EC2-VPC\] The number of IPv6 addresses to associate with the primary
#' network interface. Amazon EC2 chooses the IPv6 addresses from the range
#' of your subnet. You cannot specify this option and the option to assign
#' specific IPv6 addresses in the same request. You can specify this option
#' if you\'ve specified a minimum number of instances to launch.
#' 
#' You cannot specify this option and the network interfaces option in the
#' same request.
#' 
#' @section Ipv6Address:
#' 
#' \[EC2-VPC\] The IPv6 addresses from the range of the subnet to associate
#' with the primary network interface. You cannot specify this option and
#' the option to assign a number of IPv6 addresses in the same request. You
#' cannot specify this option if you\'ve specified a minimum number of
#' instances to launch.
#' 
#' You cannot specify this option and the network interfaces option in the
#' same request.
#' 
#' @section KernelId:
#' 
#' The ID of the kernel.
#' 
#' We recommend that you use PV-GRUB instead of kernels and RAM disks. For
#' more information, see
#' [PV-GRUB](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' @section KeyName:
#' 
#' The name of the key pair. You can create a key pair using
#' [CreateKeyPair](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html)
#' or
#' [ImportKeyPair](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html).
#' 
#' If you do not specify a key pair, you can\'t connect to the instance
#' unless you choose an AMI that is configured to allow users another way
#' to log in.
#' 
#' @section Monitoring:
#' Specifies whether detailed monitoring is enabled
#' for the instance.
#' @section Placement:
#' The placement for the
#' instance.
#' @section RamdiskId:
#' 
#' The ID of the RAM disk to select. Some kernels require additional
#' drivers at launch. Check the kernel requirements for information about
#' whether you need to specify a RAM disk. To find kernel requirements, go
#' to the AWS Resource Center and search for the kernel ID.
#' 
#' We recommend that you use PV-GRUB instead of kernels and RAM disks. For
#' more information, see
#' [PV-GRUB](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' @section SecurityGroupId:
#' 
#' The IDs of the security groups. You can create a security group using
#' [CreateSecurityGroup](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html).
#' 
#' If you specify a network interface, you must specify any security groups
#' as part of the network interface.
#' 
#' @section SecurityGroup:
#' 
#' \[EC2-Classic, default VPC\] The names of the security groups. For a
#' nondefault VPC, you must use security group IDs instead.
#' 
#' If you specify a network interface, you must specify any security groups
#' as part of the network interface.
#' 
#' Default: Amazon EC2 uses the default security group.
#' 
#' @section SubnetId:
#' 
#' \[EC2-VPC\] The ID of the subnet to launch the instance into.
#' 
#' If you specify a network interface, you must specify any subnets as part
#' of the network interface.
#' 
#' @section UserData:
#' The user data to make available to the instance.
#' For more information, see [Running commands on your Linux instance at
#' launch](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/user-data.html)
#' (Linux) and [Adding User
#' Data](https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-metadata.html#instancedata-add-user-data)
#' (Windows). If you are using a command line tool, base64-encoding is
#' performed for you, and you can load the text from a file. Otherwise, you
#' must provide base64-encoded text. User data is limited to 16
#' KB.
#' @section AdditionalInfo:
#' Reserved.
#' @section ClientToken:
#' 
#' Unique, case-sensitive identifier you provide to ensure the idempotency
#' of the request. If you do not specify a client token, a randomly
#' generated token is used for the request to ensure idempotency.
#' 
#' For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' 
#' Constraints: Maximum 64 ASCII characters
#' 
#' @section DisableApiTermination:
#' 
#' If you set this parameter to `true`, you can\'t terminate the instance
#' using the Amazon EC2 console, CLI, or API; otherwise, you can. To change
#' this attribute after launch, use
#' [ModifyInstanceAttribute](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html).
#' Alternatively, if you set `InstanceInitiatedShutdownBehavior` to
#' `terminate`, you can terminate the instance by running the shutdown
#' command from the instance.
#' 
#' Default: `false`
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section EbsOptimized:
#' 
#' Indicates whether the instance is optimized for Amazon EBS I/O. This
#' optimization provides dedicated throughput to Amazon EBS and an
#' optimized configuration stack to provide optimal Amazon EBS I/O
#' performance. This optimization isn\'t available with all instance types.
#' Additional usage charges apply when using an EBS-optimized instance.
#' 
#' Default: `false`
#' 
#' @section IamInstanceProfile:
#' The name or Amazon Resource Name (ARN) of an IAM
#' instance profile.
#' @section InstanceInitiatedShutdownBehavior:
#' 
#' Indicates whether an instance stops or terminates when you initiate
#' shutdown from the instance (using the operating system command for
#' system shutdown).
#' 
#' Default: `stop`
#' 
#' @section NetworkInterface:
#' The network interfaces to associate with the
#' instance. If you specify a network interface, you must specify any
#' security groups and subnets as part of the network
#' interface.
#' @section PrivateIpAddress:
#' 
#' \[EC2-VPC\] The primary IPv4 address. You must specify a value from the
#' IPv4 address range of the subnet.
#' 
#' Only one private IP address can be designated as primary. You can\'t
#' specify this option if you\'ve specified the option to designate a
#' private IP address as the primary IP address in a network interface
#' specification. You cannot specify this option if you\'re launching more
#' than one instance in the request.
#' 
#' You cannot specify this option and the network interfaces option in the
#' same request.
#' 
#' @section ElasticGpuSpecification:
#' An elastic GPU to associate with the instance. An
#' Elastic GPU is a GPU resource that you can attach to your Windows
#' instance to accelerate the graphics performance of your applications.
#' For more information, see [Amazon EC2 Elastic
#' GPUs](https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html)
#' in the *Amazon EC2 User Guide*.
#' @section ElasticInferenceAccelerator:
#' 
#' An elastic inference accelerator to associate with the instance. Elastic
#' inference accelerators are a resource you can attach to your Amazon EC2
#' instances to accelerate your Deep Learning (DL) inference workloads.
#' 
#' You cannot specify accelerators from different generations in the same
#' request.
#' 
#' @section TagSpecification:
#' The tags to apply to the resources during launch.
#' You can only tag instances and volumes on launch. The specified tags are
#' applied to all instances or volumes that are created during launch. To
#' tag a resource after it has been created, see
#' [CreateTags](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html).
#' @section LaunchTemplate:
#' The
#' launch template to use to launch the instances. Any parameters that you
#' specify in RunInstances override the same parameters in the launch
#' template. You can specify either the name or ID of a launch template,
#' but not both.
#' @section InstanceMarketOptions:
#' 
#' The market (purchasing) option for the instances.
#' 
#' For RunInstances, persistent Spot Instance requests are only supported
#' when **InstanceInterruptionBehavior** is set to either `hibernate` or
#' `stop`.
#' 
#' @section CreditSpecification:
#' 
#' The credit option for CPU usage of the burstable performance instance.
#' Valid values are `standard` and `unlimited`. To change this attribute
#' after launch, use
#' [ModifyInstanceCreditSpecification](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html).
#' For more information, see [Burstable performance
#' instances](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' Default: `standard` (T2 instances) or `unlimited` (T3/T3a instances)
#' 
#' @section CpuOptions:
#' The CPU options for the instance. For more
#' information, see [Optimizing CPU
#' options](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-optimize-cpu.html)
#' in the *Amazon EC2 User Guide*.
#' @section CapacityReservationSpecification:
#' Information about
#' the Capacity Reservation targeting option. If you do not specify this
#' parameter, the instance\'s Capacity Reservation preference defaults to
#' `open`, which enables it to run in any open Capacity Reservation that
#' has matching attributes (instance type, platform, Availability
#' Zone).
#' @section HibernationOptions:
#' 
#' Indicates whether an instance is enabled for hibernation. For more
#' information, see [Hibernate your
#' instance](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' You can\'t enable hibernation and AWS Nitro Enclaves on the same
#' instance.
#' 
#' @section LicenseSpecification:
#' The license configurations.
#' @section MetadataOptions:
#' The
#' metadata options for the instance. For more information, see [Instance
#' metadata and user
#' data](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html).
#' @section EnclaveOptions:
#' 
#' Indicates whether the instance is enabled for AWS Nitro Enclaves. For
#' more information, see [What is AWS Nitro
#' Enclaves?](https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave.html)
#' in the *AWS Nitro Enclaves User Guide*.
#' 
#' You can\'t enable AWS Nitro Enclaves and hibernation on the same
#' instance.
#' 
#' @return A list object or a character vector
#' @export
ec2_run_instances <- function(MaxCount, MinCount, BlockDeviceMapping = NULL, ImageId = NULL, 
    InstanceType = NULL, Ipv6AddressCount = NULL, Ipv6Address = NULL, KernelId = NULL, KeyName = NULL, 
    Monitoring = NULL, Placement = NULL, RamdiskId = NULL, SecurityGroupId = NULL, SecurityGroup = NULL, 
    SubnetId = NULL, UserData = NULL, AdditionalInfo = NULL, ClientToken = NULL, DisableApiTermination = NULL, 
    DryRun = NULL, EbsOptimized = NULL, IamInstanceProfile = NULL, InstanceInitiatedShutdownBehavior = NULL, 
    NetworkInterface = NULL, PrivateIpAddress = NULL, ElasticGpuSpecification = NULL, ElasticInferenceAccelerator = NULL, 
    TagSpecification = NULL, LaunchTemplate = NULL, InstanceMarketOptions = NULL, CreditSpecification = NULL, 
    CpuOptions = NULL, CapacityReservationSpecification = NULL, HibernationOptions = NULL, 
    LicenseSpecification = NULL, MetadataOptions = NULL, EnclaveOptions = NULL, simplify = TRUE, 
    others = list()) {
    BlockDeviceMapping <- list_to_array("BlockDeviceMapping", BlockDeviceMapping)
    Ipv6Address <- list_to_array("Ipv6Address", Ipv6Address)
    SecurityGroupId <- list_to_array("SecurityGroupId", SecurityGroupId)
    SecurityGroup <- list_to_array("SecurityGroup", SecurityGroup)
    NetworkInterface <- list_to_array("NetworkInterface", NetworkInterface)
    ElasticGpuSpecification <- list_to_array("ElasticGpuSpecification", ElasticGpuSpecification)
    ElasticInferenceAccelerator <- list_to_array("ElasticInferenceAccelerator", ElasticInferenceAccelerator)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    LicenseSpecification <- list_to_array("LicenseSpecification", LicenseSpecification)
    parameters <- c(others, list(MaxCount = MaxCount, MinCount = MinCount, ImageId = ImageId, 
        InstanceType = InstanceType, Ipv6AddressCount = Ipv6AddressCount, KernelId = KernelId, 
        KeyName = KeyName, Monitoring = Monitoring, Placement = Placement, RamdiskId = RamdiskId, 
        SubnetId = SubnetId, UserData = UserData, AdditionalInfo = AdditionalInfo, ClientToken = ClientToken, 
        DisableApiTermination = DisableApiTermination, DryRun = DryRun, EbsOptimized = EbsOptimized, 
        IamInstanceProfile = IamInstanceProfile, InstanceInitiatedShutdownBehavior = InstanceInitiatedShutdownBehavior, 
        PrivateIpAddress = PrivateIpAddress, LaunchTemplate = LaunchTemplate, InstanceMarketOptions = InstanceMarketOptions, 
        CreditSpecification = CreditSpecification, CpuOptions = CpuOptions, CapacityReservationSpecification = CapacityReservationSpecification, 
        HibernationOptions = HibernationOptions, MetadataOptions = MetadataOptions, EnclaveOptions = EnclaveOptions), 
        BlockDeviceMapping, Ipv6Address, SecurityGroupId, SecurityGroup, NetworkInterface, 
        ElasticGpuSpecification, ElasticInferenceAccelerator, TagSpecification, LicenseSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RunInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Run Scheduled Instances
#' 
#' @param LaunchSpecification Object. The launch specification.
#' @param ScheduledInstanceId String. The Scheduled Instance ID.
#' @param ClientToken String. Unique, case-sensitive identifier that ensures the idempotency of the request.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param InstanceCount Integer.   The number of instances.  Default: 1  \[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LaunchSpecification:
#' The launch specification. You must match the
#' instance type, Availability Zone, network, and platform of the schedule
#' that you purchased.
#' @section ScheduledInstanceId:
#' The Scheduled Instance
#' ID.
#' @section ClientToken:
#' Unique, case-sensitive identifier that ensures
#' the idempotency of the request. For more information, see [Ensuring
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @section InstanceCount:
#' 
#' The number of instances.
#' 
#' Default: 1
#' 
#' @return A list object or a character vector
#' @export
ec2_run_scheduled_instances <- function(LaunchSpecification, ScheduledInstanceId, ClientToken = NULL, 
    DryRun = NULL, InstanceCount = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(LaunchSpecification = LaunchSpecification, ScheduledInstanceId = ScheduledInstanceId, 
        ClientToken = ClientToken, DryRun = DryRun, InstanceCount = InstanceCount))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "RunScheduledInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Search Local Gateway Routes
#' 
#' Searches for routes in the specified local gateway
#' route table.
#' 
#' @param LocalGatewayRouteTableId String. The ID of the local gateway route table.
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section LocalGatewayRouteTableId:
#' The ID of the local gateway route
#' table.
#' @section Filter:
#' One or more filters.
#' @section MaxResults:
#' The
#' maximum number of results to return with a single call. To retrieve the
#' remaining results, make another call with the returned `nextToken`
#' value.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_search_local_gateway_routes <- function(LocalGatewayRouteTableId, Filter, MaxResults = NULL, 
    NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(LocalGatewayRouteTableId = LocalGatewayRouteTableId, MaxResults = MaxResults, 
        NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "SearchLocalGatewayRoutes", parameters = parameters, 
        simplify = simplify, token_name = "NextToken")
}


#' Search Transit Gateway Multicast Groups
#' 
#' Searches one or more transit gateway
#' multicast groups and returns the group membership
#' information.
#' 
#' @param TransitGatewayMulticastDomainId String. The ID of the transit gateway multicast domain.\[optional\]
#' @param MaxResults Integer. The maximum number of results to return with a single call.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayMulticastDomainId:
#' The ID of the transit
#' gateway multicast domain.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `group-ip-address` - The IP address of the transit gateway multicast
#'     group.
#' 
#' -   `is-group-member` - The resource is a group member. Valid values are
#'     `true` \| `false`.
#' 
#' -   `is-group-source` - The resource is a group source. Valid values are
#'     `true` \| `false`.
#' 
#' -   `member-type` - The member type. Valid values are `igmp` \|
#'     `static`.
#' 
#' -   `resource-id` - The ID of the resource.
#' 
#' -   `resource-type` - The type of resource. Valid values are `vpc` \|
#'     `vpn` \| `direct-connect-gateway` \| `tgw-peering`.
#' 
#' -   `source-type` - The source type. Valid values are `igmp` \|
#'     `static`.
#' 
#' -   `state` - The state of the subnet association. Valid values are
#'     `associated` \| `associated` \| `disassociated` \| `disassociating`.
#' 
#' -   `subnet-id` - The ID of the subnet.
#' 
#' -   `transit-gateway-attachment-id` - The id of the transit gateway
#'     attachment.
#' 
#' @section MaxResults:
#' The maximum number of results to return with a
#' single call. To retrieve the remaining results, make another call with
#' the returned `nextToken` value.
#' @section DryRun:
#' Checks whether you have the
#' required permissions for the action, without actually making the
#' request, and provides an error response. If you have the required
#' permissions, the error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_search_transit_gateway_multicast_groups <- function(TransitGatewayMulticastDomainId = NULL, 
    Filter = NULL, MaxResults = NULL, NextToken = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayMulticastDomainId = TransitGatewayMulticastDomainId, 
        MaxResults = MaxResults, NextToken = NextToken, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "SearchTransitGatewayMulticastGroups", 
        parameters = parameters, simplify = simplify, token_name = "NextToken")
}


#' Search Transit Gateway Routes
#' 
#' Searches for routes in the specified
#' transit gateway route table.
#' 
#' @param TransitGatewayRouteTableId String. The ID of the transit gateway route table.
#' @param MaxResults Integer. The maximum number of routes to return.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section TransitGatewayRouteTableId:
#' The ID of the transit
#' gateway route table.
#' @section Filter:
#' 
#' One or more filters. The possible values are:
#' 
#' -   `attachment.transit-gateway-attachment-id`- The id of the transit
#'     gateway attachment.
#' 
#' -   `attachment.resource-id` - The resource id of the transit gateway
#'     attachment.
#' 
#' -   `attachment.resource-type` - The attachment resource type. Valid
#'     values are `vpc` \| `vpn` \| `direct-connect-gateway` \| `peering`
#'     \| `connect`.
#' 
#' -   `prefix-list-id` - The ID of the prefix list.
#' 
#' -   `route-search.exact-match` - The exact match of the specified
#'     filter.
#' 
#' -   `route-search.longest-prefix-match` - The longest prefix that
#'     matches the route.
#' 
#' -   `route-search.subnet-of-match` - The routes with a subnet that match
#'     the specified CIDR filter.
#' 
#' -   `route-search.supernet-of-match` - The routes with a CIDR that
#'     encompass the CIDR filter. For example, if you have 10.0.1.0/29 and
#'     10.0.1.0/31 routes in your route table and you specify
#'     supernet-of-match as 10.0.1.0/30, then the result returns
#'     10.0.1.0/29.
#' 
#' -   `state` - The state of the route (`active` \| `blackhole`).
#' 
#' -   `type` - The type of route (`propagated` \| `static`).
#' 
#' @section MaxResults:
#' The maximum number of routes to
#' return.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_search_transit_gateway_routes <- function(TransitGatewayRouteTableId, Filter, MaxResults = NULL, 
    DryRun = NULL, simplify = TRUE, others = list()) {
    Filter <- list_to_filter(Filter)
    parameters <- c(others, list(TransitGatewayRouteTableId = TransitGatewayRouteTableId, 
        MaxResults = MaxResults, DryRun = DryRun), Filter)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "SearchTransitGatewayRoutes", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Send Diagnostic Interrupt
#' 
#' @param InstanceId String. The ID of the instance.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The ID of the
#' instance.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_send_diagnostic_interrupt <- function(InstanceId, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(InstanceId = InstanceId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "SendDiagnosticInterrupt", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Start Instances
#' 
#' @param InstanceId List. The IDs of the instances.
#' @param AdditionalInfo String. Reserved.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The IDs of the
#' instances.
#' @section AdditionalInfo:
#' Reserved.
#' @section DryRun:
#' Checks
#' whether you have the required permissions for the action, without
#' actually making the request, and provides an error response. If you have
#' the required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_start_instances <- function(InstanceId, AdditionalInfo = NULL, DryRun = NULL, simplify = TRUE, 
    others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    parameters <- c(others, list(AdditionalInfo = AdditionalInfo, DryRun = DryRun), InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "StartInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Start Network Insights Analysis
#' 
#' Starts analyzing the specified path. If the path is
#' reachable, the operation returns the shortest feasible
#' path.
#' 
#' @param NetworkInsightsPathId String. The ID of the path.
#' @param ClientToken String. Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
#' @param FilterInArn List. The Amazon Resource Names (ARN) of the resources that the path must traverse.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param TagSpecification List. The tags to apply.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section NetworkInsightsPathId:
#' The ID of
#' the path.
#' @section ClientToken:
#' Unique, case-sensitive identifier that you
#' provide to ensure the idempotency of the request. For more information,
#' see [How to Ensure
#' Idempotency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html).
#' @section FilterInArn:
#' The
#' Amazon Resource Names (ARN) of the resources that the path must
#' traverse.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section TagSpecification:
#' The tags to
#' apply.
#' @return A list object or a character vector
#' @export
ec2_start_network_insights_analysis <- function(NetworkInsightsPathId, ClientToken, FilterInArn = NULL, 
    DryRun = NULL, TagSpecification = NULL, simplify = TRUE, others = list()) {
    FilterInArn <- list_to_array("FilterInArn", FilterInArn)
    TagSpecification <- list_to_array("TagSpecification", TagSpecification)
    parameters <- c(others, list(NetworkInsightsPathId = NetworkInsightsPathId, ClientToken = ClientToken, 
        DryRun = DryRun), FilterInArn, TagSpecification)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "StartNetworkInsightsAnalysis", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Start Vpc Endpoint Service Private Dns Verification
#' 
#' @param ServiceId String. The ID of the endpoint service.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ServiceId:
#' The ID of the endpoint
#' service.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_start_vpc_endpoint_service_private_dns_verification <- function(ServiceId, DryRun = NULL, 
    simplify = TRUE, others = list()) {
    parameters <- c(others, list(ServiceId = ServiceId, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "StartVpcEndpointServicePrivateDnsVerification", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Stop Instances
#' 
#' @param InstanceId List. The IDs of the instances.
#' @param Hibernate Logical.   Hibernates the instance if the instance was enabled for hibernation at launch.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Force Logical.   Forces the instances to stop.\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The IDs of the
#' instances.
#' @section Hibernate:
#' 
#' Hibernates the instance if the instance was enabled for hibernation at
#' launch. If the instance cannot hibernate successfully, a normal shutdown
#' occurs. For more information, see [Hibernate your
#' instance](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' Default: `false`
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @section Force:
#' 
#' Forces the instances to stop. The instances do not have an opportunity
#' to flush file system caches or file system metadata. If you use this
#' option, you must perform file system check and repair procedures. This
#' option is not recommended for Windows instances.
#' 
#' Default: `false`
#' 
#' @return A list object or a character vector
#' @export
ec2_stop_instances <- function(InstanceId, Hibernate = NULL, DryRun = NULL, Force = NULL, 
    simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    parameters <- c(others, list(Hibernate = Hibernate, DryRun = DryRun, Force = Force), 
        InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "StopInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Terminate Client Vpn Connections
#' 
#' Terminates active Client VPN endpoint connections.
#' This action can be used to terminate a specific client connection, or up
#' to five connections established by a specific user.
#' 
#' @param ClientVpnEndpointId String. The ID of the Client VPN endpoint to which the client is connected.
#' @param ConnectionId String. The ID of the client connection to be terminated.\[optional\]
#' @param Username String. The name of the user who initiated the connection.\[optional\]
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section ClientVpnEndpointId:
#' The ID of the Client VPN endpoint to which the
#' client is connected.
#' @section ConnectionId:
#' The ID of the client connection
#' to be terminated.
#' @section Username:
#' The name of the user who initiated
#' the connection. Use this option to terminate all active connections for
#' the specified user. This option can only be used if the user has
#' established up to five connections.
#' @section DryRun:
#' Checks whether
#' you have the required permissions for the action, without actually
#' making the request, and provides an error response. If you have the
#' required permissions, the error response is `DryRunOperation`.
#' Otherwise, it is `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_terminate_client_vpn_connections <- function(ClientVpnEndpointId, ConnectionId = NULL, 
    Username = NULL, DryRun = NULL, simplify = TRUE, others = list()) {
    parameters <- c(others, list(ClientVpnEndpointId = ClientVpnEndpointId, ConnectionId = ConnectionId, 
        Username = Username, DryRun = DryRun))
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "TerminateClientVpnConnections", 
        parameters = parameters, simplify = simplify, token_name = NULL)
}


#' Terminate Instances
#' 
#' @param InstanceId List.   The IDs of the instances.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' 
#' The IDs of the instances.
#' 
#' Constraints: Up to 1000 instance IDs. We recommend breaking up this
#' request into smaller batches.
#' 
#' @section DryRun:
#' Checks whether you have the required permissions
#' for the action, without actually making the request, and provides an
#' error response. If you have the required permissions, the error response
#' is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_terminate_instances <- function(InstanceId, DryRun = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    parameters <- c(others, list(DryRun = DryRun), InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "TerminateInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Unassign Ipv6 Addresses
#' 
#' Unassigns one or more IPv6 addresses from a network
#' interface.
#' 
#' @param Ipv6Addresses List. The IPv6 addresses to unassign from the network interface.
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @inheritParams CommonDoc
#' 
#' @section Ipv6Addresses:
#' The IPv6 addresses to
#' unassign from the network interface.
#' @section NetworkInterfaceId:
#' The ID of the
#' network interface.
#' @return A list object or a character vector
#' @export
ec2_unassign_ipv6_addresses <- function(Ipv6Addresses, NetworkInterfaceId, simplify = TRUE, 
    others = list()) {
    Ipv6Addresses <- list_to_array("Ipv6Addresses", Ipv6Addresses)
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId), Ipv6Addresses)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "UnassignIpv6Addresses", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Unassign Private Ip Addresses
#' 
#' Unassigns one or more secondary private
#' IP addresses from a network interface.
#' 
#' @param NetworkInterfaceId String. The ID of the network interface.
#' @param PrivateIpAddress List. The secondary private IP addresses to unassign from the network interface.
#' @inheritParams CommonDoc
#' 
#' @section NetworkInterfaceId:
#' The ID of the network
#' interface.
#' @section PrivateIpAddress:
#' The secondary private IP addresses to
#' unassign from the network interface. You can specify this option
#' multiple times to unassign more than one IP
#' address.
#' @return A list object or a character vector
#' @export
ec2_unassign_private_ip_addresses <- function(NetworkInterfaceId, PrivateIpAddress, simplify = TRUE, 
    others = list()) {
    PrivateIpAddress <- list_to_array("PrivateIpAddress", PrivateIpAddress)
    parameters <- c(others, list(NetworkInterfaceId = NetworkInterfaceId), PrivateIpAddress)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "UnassignPrivateIpAddresses", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Unmonitor Instances
#' 
#' Disables
#' detailed monitoring for a running instance. For more information, see
#' [Monitoring your instances and
#' volumes](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-cloudwatch.html)
#' in the *Amazon EC2 User Guide*.
#' 
#' @param InstanceId List. The IDs of the instances.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @inheritParams CommonDoc
#' 
#' @section InstanceId:
#' The IDs of the
#' instances.
#' @section DryRun:
#' Checks whether you have the required
#' permissions for the action, without actually making the request, and
#' provides an error response. If you have the required permissions, the
#' error response is `DryRunOperation`. Otherwise, it is
#' `UnauthorizedOperation`.
#' @return A list object or a character vector
#' @export
ec2_unmonitor_instances <- function(InstanceId, DryRun = NULL, simplify = TRUE, others = list()) {
    InstanceId <- list_to_array("InstanceId", InstanceId)
    parameters <- c(others, list(DryRun = DryRun), InstanceId)
    parameters <- parameters[!vapply(parameters, is.empty, logical(1))]
    make_request(service_request = ec2_request, action = "UnmonitorInstances", parameters = parameters, 
        simplify = simplify, token_name = NULL)
}


#' Update Security Group Rule Descriptions Egress
#' 
#' @param IpPermissions List. The IP permissions for the security group rule.
#' @param DryRun Logical. Checks whether you have the required permissions for the action, without actually making the request,...\[optional\]
#' @param Gro