  #' Print the results of the sequential p-value estimation.
#'
#' @param x an object of class avseqmc_progress containing progress on the
#'   sequential estimation.
#' @param ... Ignored.
#'
#' @return No return value, called for side effects.
#' @exportS3Method
print.avseqmc_progress <- function(x, ...){
  cat(sprintf("p-value estimate:            %.4f (rounded to 4 decimals) \n", x$ptilde[length(x$ptilde)]))
  cat(sprintf("Resampling risk:             %.4f (rounded to 4 decimals) \n", x$epsilon))
  cat(sprintf("Total number of samples:     %d\n",   x$n))
}


#' Plot the results of the sequential p-value estimation.
#'
#' @param x an object of class avseqmc_progress containing progress on the
#'   sequential estimation.
#' @param ... Ignored.
#'  
#' @return No return value, called for side effects.
#' @exportS3Method
#' @importFrom graphics axis axTicks lines legend mtext
plot.avseqmc_progress <- function(x, ...){
  plot(cumsum(x$Bn), x$ptilde, type='l', ylim=c(0,1), xlim=c(1,sum(x$Bn)),
       xlab="Sample index", ylab="Estimate",
       main="Sequential p-value estimate", xaxt="n", lwd=2, col="steelblue")
  axis(1, at = ifelse(axTicks(1) == 0, 1, axTicks(1))) #remove tick at 0 and place a tick at 1.
  lines(cumsum(x$Bn), cumsum(x$B)/cumsum(x$Bn), lty=3)
  if(!is.null(x$Ltilde)){
    lines(cumsum(x$Bn), x$Ltilde, lty=2)
    legend("topright",c("Sequential p-value estimate","Non-sequential p-value estimate","Lower confidence limit"),
           lty=c(1,3,2), col=c("steelblue","black","black"), lwd=c(2,1,1))
  } else {
    legend("topright",c("Sequential p-value estimate","Non-sequential p-value estimate"),
           lty=c(1,3), col=c("steelblue","black","black"), lwd=c(2,1))
  }
  rounded.pvalue <- round(x$ptilde[length(x$ptilde)],2)
  mtext(bquote(~tilde(p)[tau](bold(x)) == .(rounded.pvalue)), side=4, line=0.5, at=rounded.pvalue, col="steelblue")
}
