#' Calculates the consistency ratio of each decision-maker
#'
#' @author Frankie Cho
#'
#' @description The `ahp.cr` function calculates the consistency ratio of each decision-maker, defined by the following equation:
#'
#' \deqn{CR = (\lambda-n)/((n-1)(RI))}
#'
#' Where \eqn{\lambda} is the maximum eigenvalue of the pairwise comparison matrix, \eqn{n} is the number of attributes, and RI is the random index. Following \insertCite{Saaty2007fuzzy;textual}{ahpsurvey}, the RI is a function of \eqn{n} and is the consistency ratio of randomly generated pairwise comparison matrices.
#'
#' @param ahpmat A list of pairwise comparison matrices of each decision maker generated by `ahp.mat`.
#' @param atts a list of attributes in the correct order. The RI is asymptotic as it approaches n=15, thus it is set to be equal to 1.6 if the number of attributes exceeds 16.
#'
#' @return A `list` of consistency ratios of each decision-maker.
#'
#' @include ahp_mat.R
#'
#' @examples
#'
#' data(city200)
#' atts <- c('cult', 'fam', 'house', 'jobs', 'trans')
#'
#' cityahp <- ahp.mat(df = city200, atts = atts, negconvert = TRUE) 
#' ahp.cr(cityahp, atts)
#'
#'@references
#'\insertAllCited{}
#'
#'@export

ahp.cr <- function(ahpmat, atts) {
    respmat <- ahpmat
    maxeigen <- vector()
    CI <- vector()
    CR <- vector()
    RI <- c(0, 0, 0.52, 0.89, 1.11, 1.25, 1.35, 1.4, 1.45, 1.49, 1.52, 1.54, 1.56, 1.58, 
        1.59, rep(1.6, 100))
    ## RI from Saaty 2008
    
    ## Computing the maximum eigenvalue of the matrix and using it for calculation for the
    ## CR
    for (i in 1:length(respmat)) {
        eigenvalues <- eigen(respmat[[i]])$values
        maxeigen[i] <- max(Re(eigenvalues[abs(Im(eigenvalues)) < 0.000001]))
        CI[i] <- (maxeigen[i] - length(atts))/(length(atts) - 1)
        CR[i] <- CI[i]/RI[length(atts)]
        ## As a general rule, CR should be lower than 0.1.
    }
    CR
}
