\name{method_energy}
\alias{method_energy}

\title{Energy Balancing}

\description{
This page explains the details of estimating weights using energy balancing by setting \code{method = "energy"} in the call to \fun{weightit} or \fun{weightitMSM}. This method can be used with binary, multinomial, and continuous treatments.

In general, this method relies on estimating weights by minimizing an energy statistic related to covariate balance. For binary and multinomial treatments, this is the energy distance, which is a multivariate distance between distributions, between treatment groups. For continuous treatments, this is the sum of the distance covariance between the treatment variable and the covariates and the energy distances between the treatment and covariates in the weighted sample and their distributions in the original sample. This method relies on code written for \pkg{WeightIt} using \pkgfun{osqp}{osqp} from the \CRANpkg{osqp} package to perform the optimization. This method may be slow or memory-intensive for large datasets.

\subsection{Binary Treatments}{
For binary treatments, this method estimates the weights using \code{osqp()} using formulas described by Huling and Mak (2022). The following estimands are allowed: ATE, ATT, and ATC.
}
\subsection{Multinomial Treatments}{
For multinomial treatments, this method estimates the weights using \code{osqp()} using formulas described by Huling and Mak (2022). The following estimands are allowed: ATE and ATT.
}
\subsection{Continuous Treatments}{
For continuous treatments, this method estimates the weights using \code{osqp()} using formulas described by Huling, Greifer, and Chen (2021).
}
\subsection{Longitudinal Treatments}{
For longitudinal treatments, the weights are the product of the weights estimated at each time point. This method is not guaranteed to yield optimal balance at each time point. NOTE: the use of energy balancing with longitudinal treatments has not been validated!
}
\subsection{Sampling Weights}{
Sampling weights are supported through \code{s.weights} in all scenarios. In some cases, sampling weights will cause the optimization to fail due to lack of convexity or infeasible constraints.
}
\subsection{Missing Data}{
In the presence of missing data, the following value(s) for \code{missing} are allowed:
\describe{
\item{\code{"ind"} (default)}{
First, for each variable with missingness, a new missingness indicator variable is created which takes the value 1 if the original covariate is \code{NA} and 0 otherwise. The missingness indicators are added to the model formula as main effects. The missing values in the covariates are then replaced with the covariate medians (this value is arbitrary and does not affect estimation). The weight estimation then proceeds with this new formula and set of covariates. The covariates output in the resulting \code{weightit} object will be the original covariates with the \code{NA}s.
}
}
}
}
\section{Additional Arguments}{
The following following additional arguments can be specified:
\describe{
\item{\code{dist.mat}}{the name of the method used to compute the distance matrix of the covariates or the numeric distance matrix itself. Allowable options include \code{"scaled_euclidean"} for the Euclidean (L2) distance on the scaled covariates (the default), \code{"mahalanobis"} for the Mahalanobis distance, and \code{"euclidean"} for the raw Euclidean distance. Abbreviations allowed. Note that some user-supplied distance matrices can cause the R session to abort due to a bug within \pkg{osqp}, so this argument should be used with caution. A distance matrix must be a square, symmetric, numeric matrix with zeros along the diagonal and a row and column for each unit. Can also be supplied as the output of a call to \fun{dist}.
}
\item{\code{lambda}}{a positive numeric scalar used to penalize the square of the weights. This value divided by the square of the total sample size is added to the diagonal of the quadratic part of the loss function. Higher values favor weights with less variability. Note this is distinct from the lambda value described in Huling and Mak (2022), which penalizes the complexity of individual treatment rules rather than the weights, but does correspond to lambda from Huling et al. (2021). Default is .0001, which is essentially 0.
}
}
For binary and multinomial treatments, the following additional argument can be specified:
\describe{
\item{\code{improved}}{\code{logical}; whether to use the improved energy balancing weights as described by Huling and Mak (2022) when \code{estimand = "ATE"}. This involves optimizing balance not only between each treatment group and the overall sample, but also between each pair of treatment groups. Huling and Mak (2022) found that the improved energy balancing weights generally outperformed standard energy balancing. Default is \code{TRUE}; set to \code{FALSE} to use the standard energy balancing weights instead (not recommended).
}
}

For continuous treatments, the following additional arguments can be specified:
\describe{
\item{\code{d.moments}}{
The number of moments of the treatment and covariate distributions that are constrained to be the same in the weighted sample as in the original sample. For example, setting \code{d.moments = 3} ensures that the mean, variance, and skew of the treatment and covariates are the same in the weighted sample as in the unweighted sample. \code{d.moments} should be greater than or equal to \code{moments} and will be automatically set accordingly if not (or if not specified).
}
\item{\code{dimension.adj}}{
\code{logical}; whether to include the dimensionality adjustment described by Huling et al. (2021). If \code{TRUE}, the default, the energy distance for the covariates is weighted \eqn{\sqrt{p}} times as much as the energy distance for the treatment, where \eqn{p} is the number of covariates. If \code{FALSE}, the two energy distances are given equal weights. Default is \code{TRUE}.
}
}

The \code{moments} argument functions differently for \code{method = "energy"} from how it does with other methods. When unspecified or set to zero, energy balancing weights are estimated as described by Huling and Mak (2022) for binary and multi-category treatments or by Huling et al. (2021) for continuous treatments. When \code{moments} is set to an integer larger than 0, additional balance constraints on the requested moments of the covariates are also included, guaranteeing exact moment balance on these covariates while minimizing the energy distance of the weighted sample. For binary and multinomial treatments, this involves exact balance on the means of the entered covariates; for continuous treatments, this involves exact balance on the treatment-covariate correlations of the entered covariates.
}
\section{Additional Outputs}{
\describe{
\item{\code{obj}}{
When \code{include.obj = TRUE}, the output of the call to \pkgfun{osqp}{solve_osqp}, which contains the dual variables and convergence information.
}
}
}

\details{
Energy balancing is a method of estimating weights using optimization without a propensity score. The weights are the solution to a constrain quadratic optimization problem where the objective function concerns covariate balance as measured by the energy distance and (for continuous treatments) the distance covariance.

Energy balancing for binary and multi-category treatments involves minimizing the energy distance between the treatment groups and between each treatment group and a target group (e.g., the full sample for the ATE). The energy distance is a scalar measure of the difference between two multivariate distributions and is equal to 0 when the two distributions are identical.

Energy balancing for continuous treatments involves minimizing the distance covariance between the treatment and the covariates; the distance covariance is a scalar measure of the association between two (possibly multivariate) distributions that is equal to 0 when the two distributions are independent. In addition, the energy distances between the treatment and covariate distributions in the weighted sample and the treatment and covariate distributions in the original sample are minimized.

The primary benefit of energy balancing is that all features of the covariate distribution are balanced, not just means, as with other optimization-based methods like entropy balancing. Still, it is possible to add additional balance constraints to require balance on individual terms using the \code{moments} argument, just like with entropy balancing. Energy balancing can sometimes yield weights with high variability; the \code{lambda} argument can be supplied to penalize highly variable weights to increase the effective sample size at the expense of balance.
}


\note{
Sometimes the optimization can fail to converge because the problem is not convex. A warning will be displayed if so. In these cases, try simply re-fitting the weights without changing anything. If the method repeatedly fails, you should try another method or change the supplied parameters (though this is uncommon). Increasing \code{max_iter} might help.

If it seems like the weights are balancing the covariates but you still get a failure to converge, this usually indicates that more iterations are needs to find the optimal solutions. This can occur when \code{moments} or \code{int} are specified. \code{max_iter} should be increased, and setting \code{verbose = TRUE} allows you to monitor the process and examine if the optimization is approaching convergence.
}
\author{
Noah Greifer, using code from Jared Huling's \CRANpkg{independenceWeights} package for continuous treatments.
}
\references{

\strong{Binary and Multinomial treatments}

Huling, J. D., & Mak, S. (2022). Energy Balancing of Covariate Distributions (arXiv:2004.13962). arXiv. \doi{10.48550/arXiv.2004.13962}

\strong{Continuous treatments}

Huling, J. D., Greifer, N., & Chen, G. (2021). Independence weights for causal inference with continuous exposures (arXiv:2107.07086). arXiv. \doi{10.48550/arXiv.2107.07086}

}
\seealso{
\fun{weightit}, \fun{weightitMSM}
}

\examples{\dontshow{if (requireNamespace("osqp", quietly = TRUE)) force(\{ # examplesIf}
library("cobalt")
data("lalonde", package = "cobalt")

#Examples may not converge, but may after several runs
\dontrun{
#Balancing covariates between treatment groups (binary)
(W1 <- weightit(treat ~ age + educ + married +
                  nodegree + re74, data = lalonde,
                method = "energy", estimand = "ATE"))
summary(W1)
bal.tab(W1)

#Balancing covariates with respect to race (multinomial)
(W2 <- weightit(race ~ age + educ + married +
                  nodegree + re74, data = lalonde,
                method = "energy", estimand = "ATT",
                focal = "black"))
summary(W2)
bal.tab(W2)


#Balancing covariates with respect to re75 (continuous)
(W3 <- weightit(re75 ~ age + educ + married +
                  nodegree + re74, data = lalonde,
                method = "energy", moments = 1))
summary(W3)
bal.tab(W3, poly = 2)
}
\dontshow{\}) # examplesIf}}
