
#' Read STICS observation file (*.obs) in the context of txt files
#'
#' @description Read STICS observation file(s) for sole and/or mixed crops.
#'
#' @param dirpath  Directory path
#' @param filename A vector of observation file name(s). Optional, see details.
#' @param mixed    (optional) Is the simulation on mixed species (boolean)
#'
#' @details If \code{filename} is not specified (or equal to \code{NULL}),
#'          the function tries first to match the \code{mod_s*} names for
#'          the same \*.obs names, and then to use the \code{.obs} file if there
#'           is one only (used for sole crops or for both mixed crops).
#'           If there are no .obs files, or two but not specified for reading,
#'           the function returns \code{NULL}
#'          If \code{mixed} is not specified (or equal to \code{NULL}),
#'          the function tries to read the number of species from the
#'          input files.
#'
#' @return A data.frame of the STICS-formatted observations.
#'         Return is \code{NULL} if no files were found,
#'         or more files than usable. If mixed crops (two `filename` provided),
#'         the function binds them and add a new column called "Plant"
#'         which corresponds to the name of each observation file.
#'
# @seealso \code{\link{read_output}}
#'
#'
#' @examples
#' \dontrun{
#'
#' obs_table <- get_obs_txt()
#' }
#'
#' @keywords internal
#'
#' @noRd
#'
get_obs_txt <- function(dirpath,
                        filename = NULL,
                        mixed = NULL) {
  . <- NULL # to avoid CRAN note for pipe
  if (is.null(mixed)) {
    if (file.exists(file.path(dirpath, "new_travail.usm"))) {
      nbplants <-
        get_usm_txt(filepath = file.path(dirpath,
                                         "new_travail.usm"))$nbplantes %>%
        as.numeric()
      if (nbplants > 1) {
        mixed <- TRUE
      } else {
        mixed <- FALSE
      }
    } else {
      if (length(list.files(dirpath) %>% .[grep("\\.obs$", .)]) == 1) {
        # If there is only one .obs file, the value of mixed doesn't matter
        mixed <- FALSE
      } else {
        stop(
          "mixed= NULL, there are several .obs files, and new_travail.usm",
          " cannot be found, please set the mixed parameter"
        )
      }
    }
  }

  # If no filename is given, trying to:
  # (1) use the mod_s* names or
  # (2) use the *.obs file if there is only one
  if (is.null(filename)) {
    if (mixed) {
      plant_name <-
        list.files(dirpath) %>%
        .[grep("mod_sp", .)] %>%
        gsub("mod_sp", "", .) %>%
        strsplit(., "\\.") %>%
        {
          if (length(.) > 0) {
            .[[1]] %>% .[1]
          }
        }
      plant_name <-
        list.files(dirpath) %>%
        .[grep("mod_sa", .)] %>%
        gsub("mod_sa", "", .) %>%
        strsplit(., "\\.") %>%
        {
          if (length(.) > 0) {
            .[[1]] %>% .[1]
          }
        } %>%
        c(plant_name, .)
    } else {
      plant_name <-
        list.files(dirpath) %>%
        .[grep("mod_s", .)] %>%
        gsub("mod_s", "", .) %>%
        strsplit(., "\\.") %>%
        {
          if (length(.) > 0) {
            .[[1]] %>% .[1]
          }
        }
    }

    # If the *.obs names are the same used for mod_s* files,
    # read them accordingly...
    if (all(file.exists(file.path(dirpath, paste0(plant_name, ".obs"))))) {
      obs_table <- get_file_int(dirpath, paste0(plant_name, ".obs"))
      warning(
        "Observation file names read from matching mod_s* ",
        "file names.\nmod_s* names:",
        plant_name,
        "\n*.obs:",
        paste0(plant_name, ".obs")
      )
    } else {
      # ...else try to read a single *.obs file (multiple .obs
      # file are not allowed)
      obs_files <- list.files(dirpath) %>% .[grep("\\.obs$", .)]
      if (length(obs_files) == 1) {
        obs_table <- get_file_int(dirpath, obs_files)
        warning(
          "Observation file guessed from the only '.obs' file in dirpath",
          plant_name, "\n*.obs:", paste0(plant_name, ".obs")
        )
      } else {
        warning(
          "\nObservation file names do not match mod_s* ",
          "file names and several *.obs ",
          "file names are present. ",
          "Please provide the *.obs file names using the ",
          "filename parameter"
        )
        obs_table <- NULL
      }
    }
  } else {
    obs_table <- get_file_int(dirpath, filename)
  }

  return(obs_table)
}
