\name{kulldorff}
\alias{kulldorff}
\title{
	Kulldorff Cluster Detection Method
}
\description{
Kulldorff spatial cluster detection method for a study region with \code{n} areas.  The method constructs \emph{zones} by consecutively aggregating nearest-neighboring areas until a proportion of the total study population is included.  Given the observed number of cases, the likelihood of each zone is computed using either binomial or poisson likelihoods. The procedure reports the zone that is the \emph{most likely cluster} and generates significance measures via Monte Carlo sampling.  Further, \emph{secondary clusters}, whose Monte Carlo p-values are below the \eqn{\alpha}-threshold, are reported as well.  
}
\usage{
kulldorff(geo, cases, population, expected.cases=NULL, pop.upper.bound, n.simulations, alpha.level, plot=TRUE)
}
\arguments{
  \item{geo}{an \code{n x 2} table of the (x,y)-coordinates of the area centroids}
  \item{cases}{aggregated case counts for all \code{n} areas}
  \item{population}{aggregated population counts for all \code{n} areas}
  \item{expected.cases}{expected numbers of disease for all \code{n} areas}
  \item{pop.upper.bound}{the upper bound on the proportion of the total population each zone can include}
  \item{n.simulations}{number of Monte Carlo samples used for significance measures}
  \item{alpha.level}{\eqn{\alpha}-level threshold used to declare significance}
  \item{plot}{flag for whether to plot histogram of Monte Carlo samples of the log-likelihood of the most likely cluster}
}
\details{
If \code{expected.cases} is specified to be \code{NULL}, then the binomial likelihood is used.  Otherwise, a Poisson model is assumed.  Typical values of \code{n.simulations} are \code{99}, \code{999}, \code{9999}...  
}
\value{
List containing:
  \item{most.likely.cluster}{information on the most likely cluster}
  \item{secondary.clusters}{information on secondary clusters, if none \code{NULL} is returned}
  \item{type}{type of likelihood}
  \item{log.lkhd}{log-likelihood of each zone considered}
  \item{simulated.log.lkhd}{\code{n.simulations} Monte Carlo samples of the log-likelihood of the most likely cluster}
}
\note{
The \code{most.likely.cluster} and \code{secondary.clusters} list elements are themselves lists reporting:\cr\cr
\tabular{ll}{
  \code{location.IDs.included} \tab ID's of areas in cluster, in order of distance\cr
  \code{population} \tab population of cluster\cr
  \code{number.of.cases} \tab number of cases in cluster\cr
  \code{expected.cases} \tab expected number of cases in cluster\cr
  \code{SMR} \tab estimated SMR of cluster\cr
  \code{log.likelihood.ratio} \tab log-likelihood of cluster\cr
  \code{monte.carlo.rank} \tab rank of lkhd of cluster within Monte Carlo simulated values\cr
  \code{p.value} \tab Monte Carlo \eqn{p}-value\cr
}
}
\references{ 
 SatScan:  Software for the spatial, temporal, and space-time scan statistics \url{http://www.satscan.org/}

  Kulldorff, M. (1997)
 A spatial scan statistic.
  \emph{Communications in Statistics: Theory and Methods}, \bold{26}, 1481--1496. %\url{http://www.satscan.org/papers/k-cstm1997.pdf}

  Kulldorff M. and Nagarwalla N. (1995) Spatial disease clusters: Detection and Inference.
\emph{Statistics in Medicine}, \bold{14}, 799--810.  %\url{http://www.satscan.org/papers/k-sm1995.pdf}
}
\author{Albert Y. Kim}
\seealso{\code{\link{pennLC}}, \code{\link{expected}}}
\examples{
## Load Pennsylvania Lung Cancer Data
data(pennLC)
data <- pennLC$data

## Process geographical information and convert to grid
geo <- pennLC$geo[,2:3]
geo <- latlong2grid(geo)

## Get aggregated counts of population and cases for each county
population <- tapply(data$population,data$county,sum)
cases <- tapply(data$cases,data$county,sum)

## Based on the 16 strata levels, computed expected numbers of disease
n.strata <- 16
expected.cases <- expected(data$population, data$cases, n.strata)

## Set Parameters
pop.upper.bound <- 0.5
n.simulations <- 999
alpha.level <- 0.05
plot <- TRUE

## Kulldorff using Binomial likelihoods
binomial <- kulldorff(geo, cases, population, NULL, pop.upper.bound, n.simulations, alpha.level, plot)
cluster <- binomial$most.likely.cluster$location.IDs.included

## plot
plot(pennLC$spatial.polygon,axes=TRUE)
plot(pennLC$spatial.polygon[cluster],add=TRUE,col="red")
title("Most Likely Cluster")

## Kulldorff using Poisson likelihoods
poisson <- kulldorff(geo, cases, population, expected.cases, pop.upper.bound, n.simulations, alpha.level, plot)
cluster <- poisson$most.likely.cluster$location.IDs.included

## plot
plot(pennLC$spatial.polygon,axes=TRUE)
plot(pennLC$spatial.polygon[cluster],add=TRUE,col="red")
title("Most Likely Cluster Controlling for Strata")




}
\keyword{file}

