% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distanceFromEachPoint.R
\name{directionFromEachPoint}
\alias{directionFromEachPoint}
\alias{.pointDirection}
\title{Calculate distances and directions between many points and many grid cells}
\usage{
directionFromEachPoint(from, to = NULL, landscape)

.pointDirection(from, to)
}
\arguments{
\item{from}{matrix with 2 or 3 columns, x and y, representing x and y coordinates
of \code{from} cell, and optional \code{id}, which will be returned,
and if \code{id} column is in \code{to}, it will be matched with that.}

\item{to}{matrix with 2  or 3 columns (or optionally more, all of which will be returned),
x and y, representing x and y coordinates of \code{to} cells, and
optional \code{id} which will be matched with \code{id} from \code{from}.
It makes no sense to have \code{id} column here with no \code{id} column
in \code{from}.}

\item{landscape}{(optional) \code{RasterLayer} or \code{SpatRaster}.
This is only used if \code{to = NULL}, in which case all cells are considered \code{to}.}
}
\value{
A sorted matrix on \code{id} with same number of rows as \code{to},
but with one extra column, \code{angles} indicating the angle in radians between from and to.
For speed, this angle will be between \code{-pi/2} and \code{3*pi/2}.
If the user wants this between say, \code{0} and \code{2*pi},
then \verb{angles \\\%\\\% (2*pi)} will do the trick. See example.
}
\description{
This is a modification of \code{\link[terra:distance]{terra::distance()}} for the case
of many points.
This version can often be faster for a single point because it does not return
a \code{RasterLayer}.
This is different than \code{\link[terra:distance]{terra::distance()}} because it does
not take the minimum distance from the set of points to all cells.
Rather this returns the every pair-wise point distance.
As a result, this can be used for doing inverse distance weightings, seed rain,
cumulative effects of distance-based processes etc.
If memory limitation is an issue, \code{maxDistance} will keep memory use down,
but with the consequences that there will be a maximum distance returned.
This function has the potential to use a lot of memory if there are a lot of
\code{from} and \code{to} points.

Internal function.
}
\details{
\code{directionFromEachPoint} calls \code{.pointDirection}, which is
not intended to be called directly by the user.

If knowing the which from cell matches with which to cell is important,
put a column "id" (e.g., starting cell) in the \code{from} matrix.
}
\examples{
library(terra)

origDTThreads <- data.table::setDTthreads(2L)
origNcpus <- options(Ncpus = 2L)

N <- 2
dirRas <- terra::rast(terra::ext(0,40,0,40), res = 1)
coords <- cbind(x = round(runif(N, xmin(dirRas), xmax(dirRas))) + 0.5,
                y = round(runif(N, xmin(dirRas), xmax(dirRas))) + 0.5,
                id = 1:N)

dirs1 <- directionFromEachPoint(from = coords, landscape = dirRas)

# helper function for converting radians to degrees
deg2 <- function(radian) (radian * 180)/pi
dirs1[, "angles"] <- deg2(dirs1[,"angles"] \%\% (2*pi))
indices <- cellFromXY(dirRas,dirs1[, c("x", "y")])
minDir <- tapply(dirs1[, "angles"], indices, function(x) min(x)) # minimum angle
dirRas[] <- as.vector(minDir)

if (interactive()) {
  terra::plot(dirRas)
  start <- terra::vect(coords[, c("x", "y"), drop = FALSE])
  terra::plot(start, add = TRUE)
}

# clean up
data.table::setDTthreads(origDTThreads)
options(Ncpus = origNcpus)

}
\seealso{
\code{\link[=distanceFromEachPoint]{distanceFromEachPoint()}}, which will also return directions if \code{angles = TRUE}.
}
\author{
Eliot McIntire
}
\keyword{internal}
