% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/depthfilter.R
\name{depthfilter}
\alias{depthfilter}
\title{Filter locations by water depth}
\usage{
depthfilter(
  sdata,
  bathymetry,
  extract = "bilinear",
  qi = 4,
  tide,
  tidal.plane,
  type = "HT",
  height = 0,
  filter = TRUE
)
}
\arguments{
\item{sdata}{A data frame containing columns with the following headers: "id", "DateTime", "lat", "lon", "qi". 
See the data \code{\link{turtle}} for an example.
The function filters the input data by a unique "id" (e.g. transmitter number, identifier for each animal). 
"DateTime" is the GMT date & time of each location in class \code{\link[base:DateTimeClasses]{POSIXct}} or \code{\link[base]{character}} with the following format "2012-06-03 01:33:46".
"lat" and "lon" are the latitude and longitude of each location in decimal degrees. 
"qi" is the quality index associated with each location fix. 
The input values can be either the number of GPS satellites or Argos Location Classes. 
Argos Location Classes will be converted to numerical values, where "A", "B", "Z" will be replaced with "-1", "-2", "-3" respectively.
The greater number indicates a higher accuracy.}

\item{bathymetry}{A RasterLayer object containing bathymetric data in metres. 
Negative and positive values indicate below and above the water respectively. 
Geographic coordinate system is WGS84.}

\item{extract}{Method to extract cell values from the raster layer as inherited from the \code{\link[raster]{extract}} function of the raster package. 
Default is bilinear.}

\item{qi}{An integer specifying a threshold of quality index. 
\emph{depthfilter} does not filter a location that is associated with a quality index higher than this threshold. 
Default is 4.}

\item{tide}{A data frame containing columns with the following headers: "tideDT", "reading", "standard.port". 
"tideDT" is date & time in class \code{\link[base:DateTimeClasses]{POSIXct}} at each observation. "reading" is the observed tidal height in metres. 
"standard.port" is the identifier of each tidal station.}

\item{tidal.plane}{A data frame containing columns with the following headers: 
"standard.port", "secondary.port", "lat", "lon", "timeDiff", "datumDiff". 
"standard.port" is the identifier for a tidal observation station. 
"secondary.port" is the identifier for a station at which tide is only predicted using tidal records observed at the related standard port. 
"lat" and "lon" are the latitude and longitude of each secondary port in decimal degrees. 
"timeDiff" is the time difference between standard port and its associated secondary port. 
"datumDiff" is the baseline difference in metres if bathymetry and tidal observations/predictions 
uses different datum (e.g. LAT and MSL).}

\item{type}{The type of water depth considered in the filtering process.
"exp" is for the water depth experienced by the animal at the time.
This option may be applicable to species that remain in water at all times (e.g. dugongs, dolphins, etc).
"HT" is for the water depth at the nearest high tide (default). 
This option is useful for animals that use inter-tidal zones at high tide and may remain there even after the tide drops (e.g. some sea turtles).}

\item{height}{A numerical value to adjust the water depth an animal is likely to use. Default is 0 m. 
This parameter is useful if the minimum water depth used by the animal is known. 
For example, a dugong is unlikely to use water shallower than its body height (e.g. ~0.5 m)
so it may be sensible to consider the fix is an error if the estimated water depth is shallower than its body height.
A negative value indicates below the water surface. 
For the dugong example, to remove locations for which the water depth was <0.5 m, 
it should be specified as; height = -0.5.
By supplying the body height to this argument, all the locations recorded shallower than its body will be removed.}

\item{filter}{Default is TRUE. 
If FALSE, the function does not filter locations but it still returns estimates of the water depth experienced by the animal at each location.}
}
\value{
When \emph{filter} option is enabled, this function filters the input data and returns with two additional columns; "depth.exp", "depth.HT". 
"depth.exp" is the estimated water depth at each location at the time of location fixing. 
"depth.HT" is the estimated water depth at the nearest high tide at each location.
}
\description{
Function to filter locations according to bathymetry and tide.
}
\details{
The function examines each location according to the water depth experienced by the animal or the water depth at the nearest high tide. 
The function looks for the closest match between each fix and tidal observations or predictions in temporal and spatial scales. 
When \emph{filter} is disabled, the function does not filter locations but returns the estimated water depth of each location with 
the tide effect considered (bathymetry + tide).
}
\note{
The input data must not contain temporal or spatial duplicates.
}
\examples{

#### Load data sets
## Fastloc GPS data obtained from a green turtle
data(turtle)

## Bathymetry model developed by Beaman (2010)
data(bathymodel)

## A tidal plane for the example site
data(tidalplane)

## Tidal observations and predictions for the example site
data(tidedata)

## Maps for the example site
data(SandyStrait)


#### Remove temporal and/or spatial duplicates
turtle.dup <- dupfilter(turtle)


#### Remove biologically unrealistic fixes 
turtle.dd <- ddfilter(turtle.dup, vmax=9.9, qi=4, ia=90, vmaxlp=2.0)


#### Apply depthfilter
turtle.dep <- depthfilter(sdata=turtle.dd, 
                          bathymetry=bathymodel, 
                          tide=tidedata, 
                          tidal.plane=tidalplane)


#### Plot data removed or retained by depthfilter
map_track(turtle.dd, bgmap=SandyStrait, point.bg = "red", point.size = 2, line.size = 0.5, 
        axes.lab.size = 0, title.size=0, sb.distance=10, multiplot = FALSE)[[1]] + 
geom_point(aes(x=lon, y=lat), data=turtle.dep, size=2, fill="yellow", shape=21)+
geom_point(aes(x=x, y=y), data=data.frame(x=c(152.68, 152.68), y=c(-25.3, -25.34)), 
           size=3, fill=c("yellow", "red"), shape=21) + 
annotate("text", x=c(152.7, 152.7), y=c(-25.3, -25.34), label=c("Retained", "Removed"), 
         colour="black", size=4, hjust = 0)
}
\references{
Shimada T, Limpus C, Jones R, Hazel J, Groom R, Hamann M (2016) 
Sea turtles return home after intentional displacement from coastal foraging areas. 
\emph{Marine Biology} 163:1-14 \doi{10.1007/s00227-015-2771-0}

Beaman, R.J. (2010) Project 3DGBR: A high-resolution depth model for the Great Barrier Reef and Coral Sea. 
\emph{Marine and Tropical Sciences Research Facility (MTSRF) Project 2.5i.1a Final Report}, MTSRF, Cairns, Australia, pp. 13 plus Appendix 1.
}
\seealso{
\code{\link{dupfilter}}, \code{\link{ddfilter}}
}
\author{
Takahiro Shimada
}
