#' Marginal Fisher Analysis
#'
#' Marginal Fisher Analysis (MFA) is a supervised linear dimension reduction method.
#' The intrinsic graph characterizes the intraclass compactness and connects each data point with
#' its neighboring pionts of the same class, while the penalty graph connects the marginal points
#' and characterizes the interclass separability.
#'
#' @param X an \eqn{(n\times p)} matrix or data frame whose rows are observations.
#' @param label a length-\eqn{n} vector of data class labels.
#' @param ndim an integer-valued target dimension.
#' @param preprocess  an additional option for preprocessing the data.
#' Default is "center". See also \code{\link{aux.preprocess}} for more details.
#' @param k1 the number of same-class neighboring points (homogeneous neighbors).
#' @param k2 the number of different-class neighboring points (heterogeneous neighbors).
#'
#' @return a named list containing
#' \describe{
#' \item{Y}{an \eqn{(n\times ndim)} matrix whose rows are embedded observations.}
#' \item{trfinfo}{a list containing information for out-of-sample prediction.}
#' \item{projection}{a \eqn{(p\times ndim)} whose columns are basis for projection.}
#' }
#'
#' @examples
#' ## generate data of 3 types with clear difference
#' dt1  = aux.gensamples(n=20)-100
#' dt2  = aux.gensamples(n=20)
#' dt3  = aux.gensamples(n=20)+100
#'
#' ## merge the data and create a label correspondingly
#' X      = rbind(dt1,dt2,dt3)
#' label  = rep(1:3, each=20)
#'
#' ## try different numbers for neighborhood size
#' out1 = do.mfa(X, label, k1=5, k2=5)
#' out2 = do.mfa(X, label, k1=10,k2=10)
#' out3 = do.mfa(X, label, k1=25,k2=25)
#'
#' ## visualize
#' opar <- par(no.readonly=TRUE)
#' par(mfrow=c(1,3))
#' plot(out1$Y, main="MFA::nbd size=5")
#' plot(out2$Y, main="MFA::nbd size=10")
#' plot(out3$Y, main="MFA::nbd size=25")
#' par(opar)
#'
#' @references
#' \insertRef{yan_graph_2007}{Rdimtools}
#'
#' @author Kisung You
#' @rdname linear_MFA
#' @concept linear_methods
#' @export
do.mfa <- function(X, label, ndim=2, preprocess=c("center","scale","cscale","decorrelate","whiten"),
                    k1=max(ceiling(nrow(X)/10),2), k2=max(ceiling(nrow(X)/10),2)){
  #------------------------------------------------------------------------
  ## PREPROCESSING
  #   1. data matrix
  aux.typecheck(X)
  n = nrow(X)
  p = ncol(X)
  #   2. label : check and return a de-factored vector
  #   For this example, there should be no degenerate class of size 1.
  label  = check_label(label, n)
  ulabel = unique(label)
  if (any(is.na(label))||(any(is.infinite(label)))){stop("* Supervised Learning : any element of 'label' as NA or Inf will simply be considered as a class, not missing entries.")  }
  #   3. ndim
  ndim = as.integer(ndim)
  if (!check_ndim(ndim,p)){stop("* do.mfa : 'ndim' is a positive integer in [1,#(covariates)).")}
  #   4. preprocess
  if (missing(preprocess)){    algpreprocess = "center"  }
  else {    algpreprocess = match.arg(preprocess)  }
  #   5. k1 and k2
  k1 = as.integer(k1)
  k2 = as.integer(k2)
  if (!check_NumMM(k1,1,n/2,compact=FALSE)){stop("* do.mfa : 'k1' should be an integer in [2,nrow(X)/2).")}
  if (!check_NumMM(k2,1,n/2,compact=FALSE)){stop("* do.mfa : 'k2' should be an integer in [2,nrow(X)/2).")}


  #------------------------------------------------------------------------
  ## COMPUTATION : PRELIMINARY
  #   1. preprocessing
  tmplist = aux.preprocess.hidden(X,type=algpreprocess,algtype="linear")
  trfinfo = tmplist$info
  pX      = tmplist$pX

  #   2. PCA preprocessing
  eigtest = eigen(cov(pX), only.values=TRUE)
  pcadim  = sum(eigtest$values > 0)
  if (pcadim <= ndim){
    warning("* do.mfa : target 'ndim' is larger than intrinsic data dimension achieved from PCA.")
    projection_first = diag(p)
    pcapX = pX
  } else{
    projection_first = aux.adjprojection(eigen(cov(pX))$vectors[,1:pcadim])
    pcapX = pX%*%projection_first
  }

  #------------------------------------------------------------------------
  ## COMPUTATION : MAIN PART FOR MFA
  #   1. compute homogeneous (intraclass) and heterogeneous (interclass) neighborhood
  #   1-1. compute neighborhood
  logicalmat = aux.nbdlogical(pcapX, label, k1, k2)
  mat_hom    = logicalmat$hom
  mat_het    = logicalmat$het
  #   1-2. OR class under symmetrization
  W  = array(as.logical(mat_hom + t(mat_hom)),c(n,n))*1.0; diag(W)=0
  Wp = array(as.logical(mat_het + t(mat_het)),c(n,n))*1.0; diag(Wp)=0
  #   1-3. D and Dp as well
  Dp = diag(rowSums(Wp))
  D  = diag(rowSums(W))

  #   2. formulation as generalized eigenvalue problem
  LHS = t(pcapX)%*%(D-W)%*%pcapX
  RHS = t(pcapX)%*%(Dp-Wp)%*%pcapX

  #   3. use lowest eigenvectors as of MINIMIZATION Problem
  projection_second = aux.geigen(LHS,RHS,ndim,maximal=FALSE)

  #------------------------------------------------------------------------
  ## RETURN
  #   1. merge two projection matrix
  projection = (projection_first%*%projection_second)

  #   2. return output
  result = list()
  result$Y = pX%*%projection
  result$trfinfo = trfinfo
  result$projection = projection
  return(result)
}
