% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adjoint.R
\name{ADjoint}
\alias{ADjoint}
\title{AD adjoint code from R}
\usage{
ADjoint(f, df, name = NULL)
}
\arguments{
\item{f}{R function representing the function value.}

\item{df}{R function representing the reverse mode derivative.}

\item{name}{Internal name of this atomic.}
}
\value{
A function that allows for numeric and taped evaluation.
}
\description{
Writing custom AD adjoint derivatives from R
}
\details{
Reverse mode derivatives (adjoint code) can be implemented from R using the function \code{ADjoint}. It takes as input a function of a single argument \code{f(x)} representing the function value, and another function of \emph{three} arguments \code{df(x, y, dy)} representing the adjoint derivative wrt \code{x} defined as \verb{d/dx sum( f(x) * dy )}. Both \code{y} and \code{dy} have the same length as \code{f(x)}. The argument \code{y} can be assumed equal to \code{f(x)} to avoid recalculation during the reverse pass. It should be assumed that all arguments \code{x}, \code{y}, \code{dy} are vectors without any attributes. In case of matrix functions, the argument dimensions therefore have to be recovered from the lengths (see \code{logdet} example).
Higher order derivatives automatically work provided that \code{df} is composed by functions that \code{RTMB} already knows how to differentiate.
}
\note{
\code{ADjoint} may be useful when you need a special atomic function which is not yet available in \code{RTMB}, or just to experiment with reverse mode derivatives.
However, the approach may cause a \emph{significant overhead} compared to native \code{RTMB} derivatives. In addition, the approach is \emph{not thread safe}, i.e. calling R functions cannot be done in parallel using OpenMP.
}
\examples{
############################################################################
## Lambert W-function defined by W(y*exp(y))=y
W <- function(x) {
  logx <- log(x)
  y <- pmax(logx, 0)
  while (any(abs(logx - log(y) - y) > 1e-9, na.rm = TRUE)) {
      y <- y - (y - exp(logx - y)) / (1 + y)
  }
  y
}
## Derivatives
dW <- function(x, y, dy) {
   dy / (exp(y) * (1. + y))
}
## Define new derivative symbol
LamW <- ADjoint(W, dW)
## Test derivatives
(F <- MakeTape(function(x)sum(LamW(x)), numeric(3)))
F(1:3)
F$print()                ## Note the 'name'
F$jacobian(1:3)          ## gradient
F$jacfun()$jacobian(1:3) ## hessian
############################################################################
## Log determinant
logdet <- ADjoint(
   function(x) {
       dim(x) <- rep(sqrt(length(x)), 2)
       determinant(x, log=TRUE)$modulus
   },
   function(x, y, dy) {
       dim(x) <- rep(sqrt(length(x)), 2)
       t(solve(x)) * dy
   },
   name = "logdet")
MakeTape(logdet, diag(2))
}
