#' Nonlinearity tests
#' @details
#' This function runs a set of nonlinearity tests on the RR time series 
#' implemented in other R packages including:
#' \itemize{
#'    \item Teraesvirta's neural metwork test for nonlinearity (\code{\link[tseries]{terasvirta.test}}).
#'    \item White neural metwork test for nonlinearity (\code{\link[tseries]{white.test}}).
#'    \item Keenan's one-degree test for nonlinearity (\code{\link[TSA]{Keenan.test}}).
#'    \item Perform the McLeod-Li test for conditional heteroscedascity (ARCH). (\code{\link[TSA]{McLeod.Li.test}}).
#'    \item Perform the Tsay's test for quadratic nonlinearity in a time series. (\code{\link[TSA]{Tsay.test}}).
#'    \item Perform the Likelihood ratio test for threshold nonlinearity. (\code{\link[TSA]{tlrt}}).
#' }
#' @param HRVData Structure containing the RR time series.
#' @param indexNonLinearAnalysis Reference to the data structure that will contain the nonlinear analysis
#' @return A \emph{HRVData} structure containing a \emph{NonlinearityTests} field storing the results of each of the tests. 
#' The \emph{NonlinearityTests} list is stored under the \emph{NonLinearAnalysis} structure.
NonlinearityTests <-function(HRVData, indexNonLinearAnalysis =length(HRVData$NonLinearAnalysis)){
  
  CheckAnalysisIndex(indexNonLinearAnalysis, length(HRVData$NonLinearAnalysis),"nonlinear")
  CheckNIHR(HRVData)
  VerboseMessage(HRVData$Verbose, "Performing nonlinearity tests")  
  
  tseries = HRVData$Beat$RR
  # apply all tests
  HRVData$NonLinearAnalysis[[indexNonLinearAnalysis]]$NonlinearityTests = 
    nonlinearityTest(time.series = tseries, verbose = HRVData$Verbose)
  
  
  return(HRVData)
  
}

#' Surrogate data testing 
#' @details
#' This function tests the null hypothesis (H0) stating that the series 
#' is a gaussian linear process. The test is performed by generating several 
#' surrogate data according to H0 and comparing the values of a discriminating
#' statistic between both original data and the surrogate data. If the value of 
#' the statistic is significantly different for the original series than for the
#' surrogate set, the null hypothesis is rejected and nonlinearity assumed.  
#' 
#' To test with a significance level of \eqn{\alpha}{alpha} if the statistic
#' from the original data is smaller than the expected  value under the null
#' hypothesis (a one-side test),  \eqn{K/\alpha - 1}{K/alpha - 1} surrogates
#' are generated. The null hypothesis is then rejected if the statistic from
#'  the data has one of the K smallest values. For a two-sided test, 
#' \eqn{2K/\alpha - 1}{2K/alpha - 1} surrogates are generated. The null 
#' hypothesis is rejected if the statistic from the data gives one of the K smallest
#' or largest values.
#' 
#' The  surrogate data is generated by using a phase randomization procedure.
#' @param HRVData Structure containing the RR time series.
#' @param indexNonLinearAnalysis Reference to the data structure that will contain the nonlinear analysis
#' @param significance Significance of the test.
#' @param oneSided Logical value. If \emph{TRUE}, the routine runs a one-side
#' test. If \emph{FALSE}, a two-side test is applied (default).
#' @param alternative Specifies the concrete type of one-side test that should be 
#' performed: If the the user wants to test if the statistic from the original
#' data is smaller (\emph{alternative="smaller"}) or larger 
#' (\emph{alternative="larger"}) than the expected value under the
#'  null hypothesis.
#' @param K Integer controlling the number of surrogates to be generated (see
#' details). 
#' @param doPlot Logical value. If TRUE, a graphical representation of the statistic value for
#' both surrogates and original data is shown.
#' @param useFunction The function that computes the discriminating statistic that shall be used for testing.
#' @param main an overall title for the plot.
#' @param xlab a title for the x axis.
#' @param ylab a title for the y axis. 
#' @param ... Additional arguments for the \emph{useFunction} function.
#' @return A \emph{HRVData} structure containing a \emph{SurrogateTest} field storing the 
#' statistics computed for the set (\emph{surrogates.statistics} field) and the RR time series 
#' (\emph{data.statistic field}). The \emph{SurrogateTest} list is stored under the \emph{NonLinearAnalysis} structure.
#' @references SCHREIBER, Thomas; SCHMITZ, Andreas. Surrogate time series. Physica D: 
#' Nonlinear Phenomena, 2000, vol. 142, no 3, p. 346-382.
#' @examples
#' \dontrun{
#' data(HRVProcessedData)
#' # rename for convenience
#' HRVData = HRVProcessedData
#' # Select a small window that looks stationary
#' HRVData = Window(HRVData,start = 0, end=800)
#' HRVData = CreateNonLinearAnalysis(HRVData)
#' HRVData = SetVerbose(HRVData,TRUE)
#' HRVData = SurrogateTest(HRVData, indexNonLinearAnalysis = 1,
#'                         significance = 0.05, oneSided = FALSE,
#'                         K = 5, useFunction = timeAsymmetry2)
#' 
#' }
SurrogateTest <- function(HRVData, 
                          indexNonLinearAnalysis = length(HRVData$NonLinearAnalysis), 
                          significance = 0.05, oneSided=FALSE,
                          alternative = c("smaller","larger"), K=1,
                          useFunction,
                          xlab="Values of the statistic", ylab="",
                          main="Surrogate data testing on the RR intervals",
                          doPlot = TRUE, ...){
  
  CheckAnalysisIndex(indexNonLinearAnalysis, length(HRVData$NonLinearAnalysis),"nonlinear")
  CheckNIHR(HRVData)
  tseries = HRVData$Beat$RR
  # apply all tests
  HRVData$NonLinearAnalysis[[indexNonLinearAnalysis]]$SurrogateTest = 
    surrogateTest(time.series = tseries, significance = significance,
                  one.sided = oneSided,alternative = alternative,
                  K = K, xlab = xlab, ylab = ylab, main = main,
                  verbose = HRVData$Verbose, 
                  do.plot = doPlot, FUN = useFunction, ...)  
  
  return(HRVData)
}
