###########################################################################/**
# @RdocFunction devIsOpen
#
# @title "Checks if a device is open or not"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{which}{An index (@numeric) or a label (@character).}
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns @TRUE if the device is open, otherwise @FALSE.
# }
#
# @examples "../incl/deviceUtils.Rex"
#
# @author
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devIsOpen <- function(which=dev.cur(), ...) {
  devList <- .devList();
  dev <- devList[which];
  label <- names(dev);
  (!is.na(label) && dev[[1]] != "");
} # devIsOpen()





###########################################################################/**
# @RdocFunction devList
#
# @title "Lists the indices of the open devices named by their labels"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns a named @integer @vector.
# }
#
# @author
#
# \seealso{
#   \code{\link[grDevices:dev]{dev.list}()}.
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devList <- function(...) {
  devList <- .devList();

  # Return only opened devices
  isOpen <- sapply(devList, FUN=function(dev) (dev != ""));
  names(isOpen) <- names(devList);
  idxs <- which(isOpen);

  # Exclude the "null" device
  idxs <- idxs[-1];

  if (length(idxs) == 0)
    idxs <- NULL;

  idxs;
}



###########################################################################/**
# @RdocFunction devGetLabel
#
# @title "Gets the label of a device"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{which}{An index (@numeric) or a label (@character).}
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns a @character string.
# }
#
# @author
#
# \seealso{
#   @see "devSetLabel" and @see "devList".
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devGetLabel <- function(which=dev.cur(), ...) {
  devList <- .devList();
  dev <- devList[which];
  label <- names(dev);
  if (is.na(label) || dev[[1]] == "")
    stop("No such device: ", which);
  label;
} # devGetLabel()



###########################################################################/**
# @RdocFunction devSetLabel
#
# @title "Sets the label of a device"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{which}{An index (@numeric) or a label (@character).}
#   \item{label}{A @character string.}
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns nothing.
# }
#
# @author
#
# \seealso{
#   @see "devGetLabel" and @see "devList".
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devSetLabel <- function(which=dev.cur(), label, ...) {
  if (is.character(which))
    which <- .devIndexOf(which);
  devList <- .devList();
  if (devList[[which]] == "")
    stop("No such device: ", which);

  # Update the label
  if (is.null(label))
    label <- "";
  names(devList)[which] <- label;

  assign(".Devices", devList, envir=baseenv());
}





###########################################################################/**
# @RdocFunction devSet
#
# @title "Activates a device"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{which}{An index (@numeric) or a label (@character).
#     If neither, then a label corresponding to the checksum of
#     \code{which} as generated by @see "digest::digest" is used.}
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns what \code{\link[grDevices:dev]{dev.set}()} returns.
# }
#
# @author
#
# \seealso{
#   @see "devOff" and @see "devDone".
#   Internally, \code{\link[grDevices:dev]{dev.set}()} is used.
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devSet <- function(which=dev.next(), ...) {
  args <- list(...);

  # Argument 'which':
  if (!is.numeric(which) || length(which) != 1) {
    if (length(which) != 1 || !is.character(which)) {
      require("digest") || throw("Package not loaded: digest");
      which <- digest(which);
    }

    if (is.character(which)) {
      args$label <- which;
      which <- .devIndexOf(which, error=FALSE);
      # If not existing, open the next available one
      if (is.na(which))
        which <- .devNextAvailable();
    }
  }

  if (which < 2) {
    stop("Cannot set device: ", which);
  }


  if (devIsOpen(which)) {
    # Active already existing device
    return(dev.set(which));
  }

  # Identify set devices that needs to be opened inorder for
  # the next device to get the requested index
  if (which == 2) {
    toBeOpened <- c();
  } else {
    toBeOpened <- setdiff(2:(which-1), dev.list());
  }

  toBeClosed <- list();
  len <- length(toBeOpened);
  if (len > 0) {
    for (idx in toBeOpened) {
      # Create a dummy postscript device (which is non-visible)
      pathname <- tempfile();
      toBeClosed[[idx]] <- pathname;
      postscript(file=pathname);
    }
  }

  # Open the device
  res <- do.call("devNew", args=args);

  # Close temporarily opened devices
  for (kk in seq(along=toBeClosed)) {
    pathname <- toBeClosed[[kk]];
    if (!is.null(pathname)) {
      dev.set(kk);
      dev.off();
      file.remove(pathname);
    }
  }

  invisible(res);
} # devSet()




###########################################################################/**
# @RdocFunction devOff
#
# @title "Closes a device"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{which}{An index (@numeric) or a label (@character).}
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns what \code{\link[grDevices:dev]{dev.off}()} returns.
# }
#
# @author
#
# \seealso{
#   @see "devDone".
#   Internally, \code{\link[grDevices:dev]{dev.off}()} is used.
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devOff <- function(which=dev.cur(), ...) {
  # Identify device
  which <- devSet(which);

  # Reset the label
  devSetLabel(which, label=NULL);

  # Close device
  dev.off(which);
} # devOff()




###########################################################################/**
# @RdocFunction devDone
#
# @title "Closes an open device unless it is a on-screen (interactive) device"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{which}{An index (@numeric) or a label (@character).}
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns nothing.
# }
#
# @author
#
# \seealso{
#   @see "devOff".
#   @see "grDevices::dev.interactive".
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devDone <- function(which=dev.cur(), ...) {
  # Do nothing?
  if (is.numeric(which) && length(which) == 1 && which <= 1) {
    return(invisible());
  }

  which <- devSet(which);
  if (which != 1) {
    type <- tolower(names(which));
    type <- gsub(":.*", "", type);
    
    isOnScreen <- (type %in% deviceIsInteractive());
    if (!isOnScreen)
      devOff(which);
  }
} # devDone()




###########################################################################/**
# @RdocFunction devNew
#
# @title "Opens a new device"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{type}{A @character string specifying the type of device to be 
#     opened. This string should match the name of an existing device 
#     @function.}
#   \item{...}{Additional arguments passed to the device @function, e.g.
#     \code{width} and \code{height}.}
#   \item{aspectRatio}{A @numeric ratio specifying the aspect ratio
#     of the image.  See below.}
#   \item{par}{An optional named @list of graphical settings applied,
#     that is, passed to @see "graphics::par", immediately after
#     opening the device.}
#   \item{label}{An optional @character string specifying the label of the
#     opened device.}
# }
#
# \value{
#   Returns what the device @function returns.
# }
#
# \section{Aspect ratio}{
#   The aspect ratio of an image is the height relative to the width.
#   If argument \code{height} is not given (or @NULL), it is 
#   calculated as \code{aspectRatio*width} as long as they are given.
#   Likewise, if argument \code{width} is not given (or @NULL), it is
#   calculated as \code{width/aspectRatio} as long as they are given.
#   If neither \code{width} nor \code{height} is given, or if both
#   are given, then \code{aspectRatio} is ignored.
# }
#
# @author
#
# \seealso{
#   @see "devDone" and @see "devOff".
#   For simplified generation of image files, see @see "devEval".
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devNew <- function(type=getOption("device"), ..., aspectRatio=1, par=NULL, label=NULL) {
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # Argument 'type':
  if (is.function(type)) {
  } else {
    type <- as.character(type);
  }

  # Argument 'aspectRatio':
  if (!is.null(aspectRatio)) {
    aspectRatio <- Arguments$getDouble(aspectRatio, range=c(0,Inf));
  }

  # Argument 'par':
  if (!is.null(par)) {
    if (!is.list(par) || is.null(names(par))) {
      throw("Argument 'par' has to be a named list: ", mode(par));
    }
  }

  # Argument 'label':
  if (!is.null(label)) {
    if (any(label == names(devList())))
      stop("Cannot open device. Label is already used: ", label);
  }


  # Arguments to be passed to the device function
  args <- list(...);

  # Drop 'width' and 'height' if NULL
  args$width <- args$width;
  args$height <- args$height;

  # Update argument 'height' by aspect ratio?
  if (!is.null(aspectRatio)) {
    width <- args$width;
    height <- args$height;

    if (is.null(width) && is.null(height)) {
      warning("Argument 'aspectRatio' was ignored because none of 'width' and 'height' were given: ", aspectRatio);
    } else if (!is.null(width) && !is.null(height)) {
      warning("Argument 'aspectRatio' was ignored because both 'width' and 'height' were given: ", aspectRatio);
    } else if (!is.null(width)) {
      args$height <- aspectRatio * width;
    } else if (!is.null(height)) {
      args$width <- height / aspectRatio;
    }
  }


  # Exclude 'file' and 'filename' arguments?
  if (is.character(type)) {
    knownInteractive <- grDevices:::.known_interactive.devices;
    if (is.element(tolower(type), tolower(knownInteractive))) {
      keep <- !is.element(names(args), c("file", "filename"));
      args <- args[keep];
    }
  }

  # Open device by calling device function
  res <- do.call(type, args=args);

  devSetLabel(label=label);

  # Default and user-specific parameters
  parT <- getOption("devNew/args/par", list());
  parT <- c(parT, par);
  if (length(parT) > 0) {
    par(parT);
  }

  invisible(res);
} # devNew()



###########################################################################/**
# @RdocFunction devEval
#
# @title "Opens a new device, evaluate (graphing) code, and closes device"
#
# \description{
#  @get "title".
# }
# 
# @synopsis
#
# \arguments{
#   \item{type}{Specifies the type of device to be used by
#     @see "R.utils::devNew".}
#   \item{expr}{The @expression of graphing commands to be evaluated.}
#   \item{envir}{The @environment where \code{expr} should be evaluated.}
#   \item{name, tags}{The fullname name of the image is specified
#     as the name with optional comma-separated tags appended.}
#   \item{ext}{The filename extension of the image file generated, if any.
#    By default, it is inferred from argument \code{type}.}
#   \item{...}{Additional arguments passed to @see "R.utils::devNew".}
#   \item{filename}{The filename of the image saved, if any.
#     See also below.}
#   \item{path}{The directory where then image should be saved, if any.}
#   \item{force}{If @TRUE, and the image file already exists, then it is
#     overwritten, otherwise not.}
# }
#
# \value{
#   Returns a named @list with items specifying for instance
#   the pathname, the fullname etc of the generated image.
#   \emph{Note that the return value may be changed in future releases.}
# }
#
# \section{Generated image file}{
#   If created, the generated image file is saved in the directory
#   specfied by argument \code{path} with a filename consisting of
#   the \code{name} followed by optional comma-separated \code{tags}
#   and a filename extension given by argument \code{ext}.
# }
#
# @author
#
# \seealso{
#   @see "devNew".
# }
#
# @keyword device
# @keyword utilities
#*/########################################################################### 
devEval <- function(type=getOption("device"), expr, envir=parent.frame(), name="Rplot", tags=NULL, ..., ext=substitute(type), filename=sprintf("%s.%s", paste(c(name, tags), collapse=","), ext), path=getOption("devEval/args/path", "figures/"), force=getOption("devEval/args/force", TRUE)) {
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # Argument 'filename' & 'path':
  pathname <- Arguments$getWritablePathname(filename, path=path);

  # Argument 'name' and 'tags':
  fullname <- paste(c(name, tags), collapse=",");
  fullname <- unlist(strsplit(fullname, split=",", fixed=TRUE));
  fullname <- trim(fullname);
  fullname <- fullname[nchar(fullname) > 0];
  fullname <- paste(fullname, collapse=",");

  # Argument 'force':
  force <- Arguments$getLogical(force);

  # Result object
  res <- list(
    type = type,
    name = name,
    tags = tags,
    fullname = fullname,
    filename = filename,
    path = path,
    pathname = pathname
  );

  if (force || !isFile(pathname)) {
    devNew(type, pathname, ...);
    on.exit({
      devDone();

      # Archive file?
      if (isPackageLoaded("R.archive")) {
        # To please R CMD check
        getArchiveOption <- archiveFile <- NULL;
        if (getArchiveOption("devEval", FALSE)) archiveFile(pathname);
      }
    }, add=TRUE);
  
    eval(expr, envir=envir);
  }

  res;
} # devEval()


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
# BEGIN: Local functions
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
.devList <- function() {
  if (exists(".Devices")) {
    devList <- get(".Devices");
  } else {
    devList <- list("null device");
  }

  labels <- names(devList);
  if (is.null(labels)) {
    labels <- paste("Device", seq(along=devList), sep=" ");
    names(devList) <- labels;
    assign(".Devices", devList, envir=baseenv());
  } else {
    # Update the names
    labels <- names(devList);
    idxs <- which(nchar(labels) == 0);
    if (length(idxs) > 0) {
      labels[idxs] <- paste("Device", idxs, sep=" ");
    }
    names(devList) <- labels;
  }

  devList;
} # .devList()

.devIndexOf <- function(label, error=TRUE) {
  devList <- .devList();
  idx <- match(label, names(devList));
  if (is.na(idx) || devList[[idx]] == "") {
    if (error)
      stop("No such device: ", label);
  }
  idx;
} # .devIndexOf()


.devNextAvailable <- function() {
  # All open devices
  devList <- dev.list();

  if (length(devList) == 0)
    return(as.integer(2));

  devPossible <- seq(from=2, to=max(devList)+1);
  devFree <- setdiff(devPossible, devList);

  devFree[1];
} # .devNextAvailable()

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
# END: Local functions
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

############################################################################
# HISTORY: 
# 2011-03-29
# o Now argument 'force' of devEval() defaults to 
#   getOption("devEval/args/force", TRUE).
# 2011-03-18
# o Now devEval() does a better job of "cleaning up" 'name' and 'tags'.
# o Now argument 'path' of devEval() defaults to 
#   getOption("devEval/args/path", "figures/").
# o devNew() gained option 'devNew/args/par', which can be used to specify 
#   the default graphical parameters for devNew().  Any additional 
#   parameters passed via argument 'par' will override such default ones,
#   if both specifies the same parameter.
# 2011-03-16
# o Now R.archive:ing is only done if the R.archive package is loaded.
# o DOCUMENTATION: The title of devDone() was incorrect.
# 2011-03-10
# o Now argument 'aspectRatio' of devNew() defaults to 1 (instead of @NULL).
# 2011-03-09
# o Added support for automatic file archiving in devEval().
# 2011-02-20
# o Changed argument 'force' of devEval() to default to TRUE.
# o Added argument 'par' to devNew() allowing for applying graphical
#   settings at same time as the device is opened, which is especially
#   useful when using devEval().
# 2011-02-14
# o Now devEval() returns a named list.
# o GENERALIZED: Argument 'aspectRatio' to devNew() can now updated
#   either 'height' or 'width', depending on which is given.
# 2011-02-13
# o Added devEval().
# o Added argument 'aspectRatio' to devNew(), which updates/set the 
#   'height', if argument 'width' is given, otherwise ignored.
# 2008-10-26
# o Now argument 'which' to devSet() can be any object.  If not a single
#   numeric or a single character string, then a checksum character string
#   is generated using digest::digest(which).
# 2008-10-16
# o Now devDone(which=1) does nothing.  Before it gave an error.
# o BUG FIX: Argument 'type' of devNew() did not take function:s.
# 2008-09-08
# o Now devNew() filters out arguments 'file' and 'filename' if the device
#   is interactive.
# 2008-08-01
# o Added devList() and removed devLabels().
# o Added internal .devNextAvailable().
# o Added argument 'error=TRUE' to internal .devIndexOf().
# 2008-07-31
# o Now devSet(idx) opens a new device with index 'idx' if not already
#   opened.
# 2008-07-29
# o Using term 'label' instead of 'name' everywhere, e.g. devLabels().
#   This was changed because the help pages on 'dev.list' etc. already
#   use the term 'name' for a different purpose, e.g. 'windows'.
# o Renamed devOpen() to devNew() to be consistent with dev.new().
# o Added Rdoc comments.
# 2008-07-18
# o Created.
############################################################################
