% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EM.MIM2.R
\name{EM.MIM2}
\alias{EM.MIM2}
\title{EM Algorithm for QTL MIM with Selective Genotyping}
\usage{
EM.MIM2(
  QTL,
  marker,
  geno,
  D.matrix,
  cp.matrix = NULL,
  y,
  yu = NULL,
  sele.g = "n",
  tL = NULL,
  tR = NULL,
  type = "RI",
  ng = 2,
  cM = TRUE,
  E.vector0 = NULL,
  X = NULL,
  beta0 = NULL,
  variance0 = NULL,
  crit = 10^-5,
  stop = 1000,
  conv = TRUE,
  console = TRUE
)
}
\arguments{
\item{QTL}{matrix. A q*2 matrix contains the QTL information, where
the row dimension q is the number of QTLs in the chromosomes. The
first column labels the chromosomes where the QTLs are located, and
the second column labels the positions of QTLs (in morgan (M) or
centimorgan (cM)). Note that chromosome and position must be divided
in order.}

\item{marker}{matrix. A k*2 matrix contains the marker information,
where the row dimension k is the number of markers in the chromosomes.
The first column labels the chromosomes where the markers are located,
and the second column labels the positions of QTLs (in morgan (M) or
centimorgan (cM)). Note that chromosomes and positions must be divided
in order.}

\item{geno}{matrix. A n*k matrix contains the k markers of the n
individuals. The marker genotypes of P1 homozygote (MM),
heterozygote (Mm), and P2 homozygote (mm) are coded as 2, 1, and 0,
respectively, and NA for missing value.}

\item{D.matrix}{matrix. The design matrix of QTL effects which is a
g*p matrix, where g is the number of possible QTL genotypes, and p
is the number of effects considered in the MIM model. The design
matrix can be easily generated by the function D.make().}

\item{cp.matrix}{matrix. The conditional probability matrix is an
n*g matrix, where n is the number of individuals who are genotyped,
and g is the number of possible genotypes of QTLs. If cp.matrix=NULL,
the function will calculate the conditional probability matrix for
selective genotyping.}

\item{y}{vector. A vector that contains the phenotype values of
individuals with genotypes.}

\item{yu}{vector. A vector that contains the phenotype value
of the individuals without genotypes.}

\item{sele.g}{character. If sele.g="n", it will consider that the
data is not a selective genotyping data but the complete genotyping
data. If sele.g="p", it will consider that the data is a selective
genotyping data, and use the proposed model (Lee 2014) to analyze.
If sele.g="t", it will consider that the data is a selective
genotyping data, and use the truncated model (Lee 2014) to analyze.
If sele.g="f, it will consider that the data is a selective
genotyping data, and use the frequency-based model (Lee 2014) to
analyze. Note that the yu must be input when sele.g="p" of "f".}

\item{tL}{numeric. The lower truncation point of phenotype value
when sele.g="t". Note that when sele.g="t" and tL=NULL, the yu
must be input, and the function will consider the minimum of yu
as the lower truncation point.}

\item{tR}{numeric. The upper truncation point of phenotype value
when sele.g="t". Note that when sele.g="t" and tR=NULL, the yu
must be input, and the function will consider the maximum of yu
as the upper truncation point.}

\item{type}{character. The population type of the dataset. Include
backcross (type="BC"), advanced intercross population (type="AI"), and
recombinant inbred population (type="RI").}

\item{ng}{integer. The generation number of the population type. For
example, the BC1 population is type="BC" with ng=1; the AI F3
population is type="AI" with ng=3.}

\item{cM}{logical. Specify the unit of marker position. cM=TRUE for
centimorgan. Or cM=FALSE for morgan.}

\item{E.vector0}{vector. The initial value for QTL effects. The
number of elements corresponds to the column dimension of the
design matrix. If E.vector0=NULL, the initial value will be 0
for all effects.}

\item{X}{matrix. The design matrix of the fixed factors except for
QTL effects. It is an n*k matrix, where n is the number of
individuals, and k is the number of fixed factors. If X=NULL,
the matrix will be an n*1 matrix that all elements are 1.}

\item{beta0}{vector. The initial value for effects of the fixed
factors. The number of elements corresponds to the column dimension
of the fixed factor design matrix.  If beta0=NULL, the initial
value will be the average of y.}

\item{variance0}{numeric. The initial value for variance. If
variance0=NULL, the initial value will be the variance of
phenotype values.}

\item{crit}{numeric. The convergence criterion of EM algorithm.
The E and M steps will be iterated until a convergence criterion
is satisfied. It must be between 0 and 1.}

\item{stop}{numeric. The stopping criterion of EM algorithm.
The E and M steps will be stopped when the iteration number reaches the
stop criterion, and it will treat the algorithm as failing to converge.}

\item{conv}{logical. If being False, it will ignore the inability to
converge, and output the last result in the process of EM algorithm
before the stopping criterion.}

\item{console}{logical. To decide whether the process of algorithm
will be shown in the R console or not.}
}
\value{
\item{QTL}{The QTL imformation of this analysis.}
\item{E.vector}{The QTL effects are calculated by the EM algorithm.}
\item{beta}{The effects of the fixed factors are calculated by the EM
algorithm.}
\item{variance}{The variance is calculated by the EM algorithm.}
\item{PI.matrix}{The posterior probabilities matrix after the
process of the EM algorithm.}
\item{log.likelihood}{The log-likelihood value of this model.}
\item{LRT}{The LRT statistic of this model.}
\item{R2}{The coefficient of determination of this model. This
can be used as an estimate of heritability.}
\item{y.hat}{The fitted values of trait values with genotyping are
calculated by the estimated values from the EM algorithm.}
\item{yu.hat}{The fitted values of trait values without genotyping
are calculated by the estimated values from the EM algorithm.}
\item{iteration.number}{The iteration number of the EM algorithm.}
\item{model}{The model of this analysis, contains complete a
genotyping model, a proposed model, a truncated model, and a
frequency-based model.}
}
\description{
Expectation-maximization algorithm for QTL multiple interval mapping.
This function can handle the genotype data, witch is selective genotyping.
}
\examples{
# load the example data
load(system.file("extdata", "exampledata.RDATA", package = "QTLEMM"))

# make the seletive genotyping data
ys <- y[y > quantile(y)[4] | y < quantile(y)[2]]
yu <- y[y >= quantile(y)[2] & y <= quantile(y)[4]]
geno.s <- geno[y > quantile(y)[4] | y < quantile(y)[2],]

# run and result
D.matrix <- D.make(3, type = "RI", aa = c(1, 3, 2, 3), dd = c(1, 2, 1, 3), ad = c(1, 2, 2, 3))
result <- EM.MIM2(QTL, marker, geno.s, D.matrix, y = ys, yu = yu, sele.g = "p")
result$E.vector
}
\references{
KAO, C.-H. and Z.-B. ZENG 1997 General formulas for obtaining the maximum
likelihood estimates and the asymptotic variance-covariance matrix in QTL
mapping when using the EM algorithm. Biometrics 53, 653-665.

KAO, C.-H., Z.-B. ZENG and R. D. TEASDALE 1999 Multiple interval mapping
for Quantitative Trait Loci. Genetics 152: 1203-1216.

H.-I LEE, H.-A. HO and C.-H. KAO 2014 A new simple method for improving
QTL mapping under selective genotyping. Genetics 198: 1685-1698.
}
\seealso{
\code{\link[QTLEMM]{D.make}}
\code{\link[QTLEMM]{Q.make}}
\code{\link[QTLEMM]{EM.MIM}}
}
