% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obwoe.R
\name{obwoe_gains}
\alias{obwoe_gains}
\title{Gains Table Statistics for Credit Risk Scorecard Evaluation}
\usage{
obwoe_gains(
  obj,
  target = NULL,
  feature = NULL,
  use_column = c("auto", "bin", "woe", "direct"),
  sort_by = c("id", "woe", "event_rate", "bin"),
  n_groups = NULL
)
}
\arguments{
\item{obj}{Input object: an \code{"obwoe"} object, a \code{data.frame} from
\code{\link{obwoe_apply}}, or any \code{data.frame} containing a grouping
variable and target values.}

\item{target}{Integer vector of binary target values (0/1) or the name of
the target column in \code{obj}. Required for \code{data.frame} inputs.
For \code{"obwoe"} objects, the target is extracted automatically.}

\item{feature}{Character string specifying the feature/variable to analyze.
For \code{"obwoe"} objects: defaults to the feature with highest IV.
For \code{data.frame} objects: can be any column name representing groups
(e.g., \code{"age_bin"}, \code{"age_woe"}, \code{"score_decile"}).}

\item{use_column}{Character string specifying which column type to use when
\code{obj} is a \code{data.frame} from \code{\link{obwoe_apply}}:
\describe{
  \item{\code{"bin"}}{Use the \code{<feature>_bin} column (default)}
  \item{\code{"woe"}}{Use the \code{<feature>_woe} column (groups by WoE values)}
  \item{\code{"auto"}}{Automatically detect: use \code{_bin} if available}
  \item{\code{"direct"}}{Use the \code{feature} column name directly (for any variable)}
}}

\item{sort_by}{Character string specifying sort order for bins:
\describe{
  \item{\code{"woe"}}{Descending WoE (highest risk first) - default}
  \item{\code{"event_rate"}}{Descending event rate}
  \item{\code{"bin"}}{Alphabetical/natural order}
}}

\item{n_groups}{Integer. For continuous variables (e.g., scores), the number
of groups (deciles) to create. Default is \code{NULL} (use existing groups).
Set to 10 for standard decile analysis.}
}
\value{
An S3 object of class \code{"obwoe_gains"} containing:
\describe{
  \item{\code{table}}{Data frame with 18 statistics per bin (see Details)}
  \item{\code{metrics}}{Named list of global performance metrics:
    \describe{
      \item{\code{ks}}{Kolmogorov-Smirnov statistic (\%)}
      \item{\code{gini}}{Gini coefficient (\%)}
      \item{\code{auc}}{Area Under ROC Curve}
      \item{\code{total_iv}}{Total Information Value}
      \item{\code{ks_bin}}{Bin where maximum KS occurs}
    }
  }
  \item{\code{feature}}{Feature/variable name analyzed}
  \item{\code{n_bins}}{Number of bins/groups}
  \item{\code{n_obs}}{Total observations}
  \item{\code{event_rate}}{Overall event rate}
}
}
\description{
Computes a comprehensive gains table (also known as a lift table or
decile analysis) for evaluating the discriminatory power of credit scoring
models and optimal binning transformations. The gains table is a fundamental
tool in credit risk management for model validation, cutoff selection,
and regulatory reporting (Basel II/III, IFRS 9).

This function accepts three types of input:
\enumerate{
  \item An \code{"obwoe"} object from \code{\link{obwoe}} (uses stored binning)
  \item A \code{data.frame} from \code{\link{obwoe_apply}} (uses bin/WoE columns)
  \item Any \code{data.frame} with a grouping variable (e.g., score deciles)
}
}
\details{
\subsection{Gains Table Construction}{

The gains table is constructed by:
\enumerate{
  \item Sorting observations by risk score or WoE (highest risk first)
  \item Grouping into bins (pre-defined or created via quantiles)
  \item Computing bin-level and cumulative statistics
}

The table enables assessment of model rank-ordering ability: a well-calibrated
model should show monotonically increasing event rates as risk score increases.
}

\subsection{Bin-Level Statistics (18 metrics)}{

\tabular{lll}{
  \strong{Column} \tab \strong{Formula} \tab \strong{Description} \cr
  \code{bin} \tab - \tab Bin label or interval \cr
  \code{count} \tab \eqn{n_i} \tab Total observations in bin \cr
  \code{count_pct} \tab \eqn{n_i / N} \tab Proportion of total population \cr
  \code{pos_count} \tab \eqn{n_{i,1}} \tab Event count (Bad, target=1) \cr
  \code{neg_count} \tab \eqn{n_{i,0}} \tab Non-event count (Good, target=0) \cr
  \code{pos_rate} \tab \eqn{n_{i,1} / n_i} \tab Event rate (Bad rate) in bin \cr
  \code{neg_rate} \tab \eqn{n_{i,0} / n_i} \tab Non-event rate (Good rate) \cr
  \code{pos_pct} \tab \eqn{n_{i,1} / N_1} \tab Distribution of events \cr
  \code{neg_pct} \tab \eqn{n_{i,0} / N_0} \tab Distribution of non-events \cr
  \code{odds} \tab \eqn{n_{i,1} / n_{i,0}} \tab Odds of event \cr
  \code{log_odds} \tab \eqn{\ln(\text{odds})} \tab Log-odds (logit) \cr
  \code{woe} \tab \eqn{\ln(p_i / q_i)} \tab Weight of Evidence \cr
  \code{iv} \tab \eqn{(p_i - q_i) \cdot WoE_i} \tab Information Value contribution \cr
  \code{cum_pos_pct} \tab \eqn{\sum_{j \le i} p_j} \tab Cumulative events captured \cr
  \code{cum_neg_pct} \tab \eqn{\sum_{j \le i} q_j} \tab Cumulative non-events \cr
  \code{ks} \tab \eqn{|F_1(i) - F_0(i)|} \tab KS statistic at bin \cr
  \code{lift} \tab \eqn{\text{pos\_rate} / \bar{p}} \tab Lift over random \cr
  \code{capture_rate} \tab \eqn{cum\_pos\_pct} \tab Cumulative capture rate
}
}

\subsection{Global Performance Metrics}{

\strong{Kolmogorov-Smirnov (KS) Statistic}:
Maximum absolute difference between cumulative distributions of events
and non-events. Measures the model's ability to separate populations.

\deqn{KS = \max_i |F_1(i) - F_0(i)|}

\tabular{ll}{
  \strong{KS Range} \tab \strong{Interpretation} \cr
  < 20\% \tab Poor discrimination \cr
  20-40\% \tab Acceptable \cr
  40-60\% \tab Good \cr
  60-75\% \tab Very good \cr
  > 75\% \tab Excellent (verify for data leakage)
}

\strong{Gini Coefficient}:
Measure of inequality between event and non-event distributions.
Equivalent to 2*AUC - 1, representing the area between the Lorenz
curve and the line of equality.

\deqn{Gini = 2 \times AUC - 1}

\strong{Area Under ROC Curve (AUC)}:
Probability that a randomly chosen event is ranked higher than a
randomly chosen non-event. Computed via the trapezoidal rule.

\strong{Total Information Value (IV)}:
Sum of IV contributions across all bins. See \code{\link{obwoe}} for
interpretation guidelines.
}

\subsection{Use Cases}{

\strong{Model Validation}:
Verify rank-ordering (monotonic event rates) and acceptable KS/Gini.

\strong{Cutoff Selection}:
Identify the bin where the model provides optimal separation for
business rules (e.g., auto-approve above score X).

\strong{Population Stability}:
Compare gains tables over time to detect model drift.

\strong{Regulatory Reporting}:
Generate metrics required by Basel II/III and IFRS 9 frameworks.
}
}
\examples{
\donttest{
# =============================================================================
# Example 1: From obwoe Object (Standard Usage)
# =============================================================================
set.seed(42)
n <- 1000
df <- data.frame(
  age = rnorm(n, 40, 15),
  income = exp(rnorm(n, 10, 0.8)),
  score = rnorm(n, 600, 100),
  target = rbinom(n, 1, 0.15)
)

model <- obwoe(df, target = "target")
gains <- obwoe_gains(model, feature = "age")
print(gains)

# Access metrics
cat("KS:", gains$metrics$ks, "\%\n")
cat("Gini:", gains$metrics$gini, "\%\n")

# =============================================================================
# Example 2: From obwoe_apply Output - Using Bin Column
# =============================================================================
scored <- obwoe_apply(df, model)

# Default: uses age_bin column
gains_bin <- obwoe_gains(scored,
  target = df$target, feature = "age",
  use_column = "bin"
)

# =============================================================================
# Example 3: From obwoe_apply Output - Using WoE Column
# =============================================================================
# Group by WoE values (continuous analysis)
gains_woe <- obwoe_gains(scored,
  target = df$target, feature = "age",
  use_column = "woe", n_groups = 5
)

# =============================================================================
# Example 4: Any Variable - Score Decile Analysis
# =============================================================================
# Create score deciles manually
df$score_decile <- cut(df$score,
  breaks = quantile(df$score, probs = seq(0, 1, 0.1)),
  include.lowest = TRUE, labels = 1:10
)

# Analyze score deciles directly
gains_score <- obwoe_gains(df,
  target = "target", feature = "score_decile",
  use_column = "direct"
)
print(gains_score)

# =============================================================================
# Example 5: Automatic Decile Creation
# =============================================================================
# Use n_groups to automatically create quantile groups
gains_auto <- obwoe_gains(df,
  target = "target", feature = "score",
  use_column = "direct", n_groups = 10
)
}
}
\references{
Siddiqi, N. (2006). Credit Risk Scorecards: Developing and Implementing
Intelligent Credit Scoring. \emph{John Wiley & Sons}.
\doi{10.1002/9781119201731}

Thomas, L. C., Edelman, D. B., & Crook, J. N. (2002). Credit Scoring
and Its Applications. \emph{SIAM Monographs on Mathematical Modeling
and Computation}. \doi{10.1137/1.9780898718317}

Anderson, R. (2007). The Credit Scoring Toolkit: Theory and Practice
for Retail Credit Risk Management. \emph{Oxford University Press}.

Hand, D. J., & Henley, W. E. (1997). Statistical Classification Methods
in Consumer Credit Scoring: A Review. \emph{Journal of the Royal
Statistical Society: Series A}, 160(3), 523-541.
\doi{10.1111/j.1467-985X.1997.00078.x}
}
\seealso{
\code{\link{obwoe}} for optimal binning,
\code{\link{obwoe_apply}} for scoring new data,
\code{\link{plot.obwoe_gains}} for visualization (cumulative gains, KS, lift).
}
