#' Generalised Pairs Plots for MoEClust Mixture Models
#'
#' Produces a matrix of plots showing pairwise relationships between continuous response variables and continuous/categorical associated covariates, as well as the clustering achieved, according to fitted MoEClust mixture models.
#' @param res An object of class \code{"MoEClust"} generated by \code{\link{MoE_clust}}, or an object of class \code{"MoECompare"} generated by \code{\link{MoE_compare}}.
#' @param subset A list giving named arguments for producing only a subset of panels:
#' \describe{
#' \item{\code{show.map}}{Logical indicating whether to show panels involving the MAP classification (defaults to \code{TRUE}, unless there is only one component, in which case the MAP classification is never plotted.).}
#' \item{\code{data.ind}}{For subsetting response variables: a vector of column indices corresponding to the variables in the columns of \code{res$data} which should be shown. Defaults to all.}
#' \item{\code{cov.ind}}{For subsetting covariates: a vector of column indices corresponding to the covariates in the columns \code{res$net.covs} which should be shown. Defaults to all.}
#' }
#' @param response.type The type of plot desired for the scatter plots comparing continuous response variables. Defaults to \code{"points"}, but points can also be sized according to their associated clustering \code{"uncertainty"}, or the bivariate \code{"density"} contours can be displayed (see \code{density.pars}). Caution is advised, until further updates, when interpreting \code{"density"} plots in the presence of covariates.
#' @param residuals Logical indicating whether to treat the data as the raw data (when \code{FALSE}, the default) or the augmented data comprising the residuals from the expert network (when \code{TRUE}). In the latter case, the mean and (co)variance parameters are taken to be the mean and (co)variance of the residuals. Only relevant if expert network covariates were supplied under \code{res}, otherwise coerced to \code{FALSE}.
#' @param scatter.type A vector of length 2 (or 1) giving the plot type for the upper and lower triangular portions of the plot, respectively, pertaining to the associated covariates. Defaults to \code{"lm"} for covariate vs. response panels and \code{"points"} otherwise. Only relevant for models with continuous covariates in the gating &/or expert network. \code{"ci"} and \code{"lm"} type plots are only produced for plots pairing covariates with response, and never response vs. response or covariate vs. covariate. Note that lines &/or confidence intervals will only be drawn for continuous covariates included in the expert network; to include covariates included only in the gating network also, use the options \code{"lm2"} or \code{"ci2"}.
#' @param conditional A vector of length 2 (or 1) giving the plot type for the upper and lower triangular portions of the plot, respectively, for plots involving a mix of categorical and continuous variables. Defaults to \code{"stripplot"} in the upper triangle and \code{"boxplot"} in the lower triangle (see \code{\link[lattice]{panel.stripplot}} and \code{\link[lattice]{panel.bwplot}}). \code{"barcode"} and \code{"violin"} plots can also be produced. Only relevant for models with categorical covariates in the gating &/or expert network. Comparisons of two categorical variables (which can only ever be covariates) are always displayed via mosaic plots (see \code{\link[vcd]{strucplot}}).
#' @param addEllipses Controls whether to add MVN ellipses with axes corresponding to the within-cluster covariances for the response data (\code{"yes"} or \code{"no"}). The options \code{"inner"} and \code{"outer"} (the default) will colour the axes or the perimeter of those ellipses, respectively, according to the cluster they represent (according to \code{scatter.pars$lci.col}). The option \code{"both"} will obviously colour both the axes and the perimeter. Ellipses are only ever drawn for multivariate data, and only when \code{response.type} is \code{"points"} or \code{"uncertainty"}.
#'
#' Ellipses are centered on the posterior mean of the fitted values when there are expert network covariates, otherwise on the posterior mean of the response variables. In the presence of expert network covariates, the component-specific covariance matrices are also modified for plotting purposes via the function \code{\link{expert_covar}}, in order to account for the extra variability of the means, usually resulting in bigger shapes & sizes for the MVN ellipses.
#' @param border.col A vector of length 5 (or 1) containing \emph{border} colours for plots against the MAP classification, respponse vs. response, covariate vs. response, response vs. covariate, and covariate vs. covariate panels, respectively.
#'
#' Defaults to \code{c("purple", "black", "brown", "brown", "navy")}.
#' @param bg.col A vector of length 5 (or 1) containing \emph{background} colours for plots against the MAP classification, response vs. response, covariate vs. response, response vs. covariate, and covariate vs. covariate panels, respectively.
#'
#' Defaults to \code{c("cornsilk", "white", "palegoldenrod", "palegoldenrod", "cornsilk")}.
#' @param diagonal By default, the diagonal from the top left to the bottom right is used for displaying the marginal distributions of variables. Specifying \code{"off"} will place the diagonal running from the top right down to the bottom left.
#' @param outer.margins A list of length 4 with units as components named bottom, left, top, and right, giving the outer margins; the defaults uses two lines of text. A vector of length 4 with units (ordered properly) will work, as will a vector of length 4 with numeric variables (interpreted as lines).
#' @param outer.labels The default is \code{NULL}, for alternating labels around the perimeter. If \code{"all"}, all labels are printed, and if \code{"none"}, no labels are printed.
#' @param outer.rot A 2-vector (\code{x}, \code{y}) rotating the top/bottom outer labels \code{x} degrees and the left/right outer labels \code{y} degrees. Only works for categorical labels of boxplot and mosaic panels. Defaults to \code{c(0, 90)}.
#' @param gap The gap between the tiles; defaulting to 0.05 of the width of a tile.
#' @param buffer The fraction by which to expand the range of quantitative variables to provide plots that will not truncate plotting symbols. Defaults to 2 percent of range currently.
#' @param scatter.pars A list supplying select parameters for the continuous vs. continuous scatter plots.
#'
#' \code{NULL} is equivalent to \code{list(scat.pch=res$classification, scat.size=unit(0.25, "char"), scat.col=res$classification, lci.col=res$classification)}, where \code{lci.col} gives the colour of the fitted lines &/or confidence intervals when \code{scatter.type} is one of \code{"ci"} or \code{"lm"} and the colour of the ellipses when \code{addEllipses} is one of \code{"outer"}, \code{"inner"}, or \code{"both"}. Note that \code{scatter.pars$scat.size} will be modified on an observation by observation level when \code{response.type} is \code{"uncertainty"}.
#' @param density.pars A list supplying select parameters for visualising the bivariate density contours, only when \code{response.type} is \code{"density"}.
#'
#' \code{NULL} is equivalent to \code{list(grid.size=c(100, 100), dcol="grey30", nlevels=11, show.labels=TRUE)}, where \code{density.pars$grid.size} is a vector of length two giving the number of points in the x & y direction of the grid over which the density is evaluated, respectively.
#' @param stripplot.pars A list supplying select parameters for continuous vs. categorical panels when one of the entries of \code{conditional} is \code{"stripplot"}.
#'
#' \code{NULL} is equivalent to \code{list(strip.pch=res$classification, strip.size=unit(0.5, "char"), strip.col=res$classification, jitter=TRUE)}.
#' @param barcode.pars A list supplying select parameters for continuous vs. categorical panels  when one of the entries of \code{conditional} is \code{"boxplot"}.
#'
#' \code{NULL} is equivalent to \code{list(nint=0, ptsize=unit(0.25, "char"), ptpch=1, bcspace=NULL, use.points=FALSE)}. See the help file for \code{barcode::barcode}.
#' @param mosaic.pars A list supplying select parameters for categorical vs. categorical panels. \code{NULL}. Currently \code{shade, gp_labels, gp} and \code{gp_args} are passed through to \code{\link[vcd]{strucplot}} for producing mosaic tiles.
#' @param axis.pars A list supplying select parameters for controlling axes.
#'
#' \code{NULL} is equivalent to \code{list(n.ticks=5, axis.fontsize=9)}. The argument \code{n.ticks} will be overwritten for categorical variables with fewer than 5 levels.
#' @param diag.pars A list supplying select parameters for panels along the diagonal.
#'
#' \code{NULL} is equivalent to \code{list(diag.fontsize=9, show.hist=TRUE, hist.color=hist.color, show.counts=TRUE)}, where \code{hist.color} is a vector of length 4, giving the colours for the response variables, gating covariates, expert covariates, and covariates entering both networks, respectively. By default, response variables are \code{"black"} and covariates of any kind are \code{"grey"}. The MAP classification is always coloured by cluster membership. \code{show.counts} is only relevant for categorical variables.
#' @param ... Catches unused arguments. Alternatively, named arguments can be passed directly here to any/all of \code{scatter.pars, barcode.pars, mosaic.pars, axis.pars} and \code{diag.pars}.
#'
#' @importFrom lattice "current.panel.limits" "panel.abline" "panel.bwplot" "panel.histogram" "panel.lines" "panel.points" "panel.rect" "panel.stripplot" "panel.text" "panel.violin" "trellis.grobname" "trellis.par.get" "trellis.par.set"
#' @importFrom matrixStats "colMeans2" "rowLogSumExps"
#' @importFrom mclust "sigma2decomp"
#' @importFrom vcd "strucplot"
#'
#' @return A generalised pairs plot showing all pairwise relationships between clustered response variables and associated gating &/or expert network continuous &/or categorical variables, coloured according to the MAP classification, with the marginal distributions of each variable along the diagonal.
#' @note For \code{MoEClust} models with more than one associated covariate (entering either network), fitted lines produced in continuous covariate vs. continuous response scatter plots via \code{scatter.type="lm"} or \code{scatter.type="ci"} will \strong{NOT} correspond to the coefficients in the expert network (\code{res$expert}).
#'
#' \code{\link{plot.MoEClust}} is a wrapper to \code{\link{MoE_gpairs}} which accepts the default arguments, and also produces other types of plots. Caution is advised producing generalised pairs plots when the dimension of the data is large.
#' @export
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @references K. Murphy and T. B. Murphy (2017). Parsimonious Model-Based Clustering with Covariates. \emph{To appear}. <\href{https://arxiv.org/abs/1711.05632}{arXiv:1711.05632}>.
#'
#' Emerson, J.W., Green, W.A., Schloerke, B., Crowley, J., Cook, D., Hofmann, H. and Wickham, H. (2013). The Generalized Pairs Plot. \emph{Journal of Computational and Graphical Statistics}, 22(1):79-91.
#' @seealso \code{\link{MoE_clust}}, \code{\link{plot.MoEClust}}, \code{\link[lattice]{panel.stripplot}}, \code{\link[lattice]{panel.bwplot}}, \code{\link[lattice]{panel.violin}}, \code{\link[vcd]{strucplot}}
#' @keywords plotting
#' @examples
#' \dontrun{
#' data(ais)
#' res   <- MoE_clust(ais[,3:7], G=2, gating= ~ BMI, expert= ~ sex,
#'                    network.data=ais, modelNames="EVE")
#' MoE_gpairs(res)
#'
#' # Produce the same plot, but with a violin plot in the lower triangle.
#' # Add confidence intervals to the scatter plots. Remove the "Fe" variable.
#' # Size points in the response vs. response panels by their clustering uncertainty.
#' MoE_gpairs(res, conditional=c("stripplot", "violin"), data.ind=1:4,
#'            scatter.type=c("ci", "points"), response.type="uncertainty")
#'
#' # Instead show the bivariate density contours of the reponse variables.
#' # Use different colours for histograms of covariates in the gating/expert/both networks.
#' # Also use different colours for response vs. covariate & covariate vs. response panels.
#' MoE_gpairs(res, response.type="density", show.labels=FALSE,
#'            hist.color=c("black", "cyan", "hotpink", "chartreuse"),
#'            bg.col=c("whitesmoke", "white", "mintcream", "mintcream", "floralwhite"))}
MoE_gpairs          <- function(res, response.type = c("points", "uncertainty", "density"), subset = list(...), residuals = FALSE, scatter.type = c("lm", "points"), conditional = c("stripplot", "boxplot"),
                                addEllipses = c("outer", "yes", "no", "inner", "both"), border.col = c("purple", "black", "brown", "brown", "navy"), bg.col = c("cornsilk", "white", "palegoldenrod", "palegoldenrod", "cornsilk"),
                                diagonal = c("on", "off"), outer.margins = list(bottom=grid::unit(2, "lines"), left=grid::unit(2, "lines"), top=grid::unit(2, "lines"), right=grid::unit(2, "lines")), outer.labels = NULL, outer.rot = c(0, 90),
                                gap = 0.05, buffer = 0.02, scatter.pars = list(...), density.pars = list(...), stripplot.pars = list(...), barcode.pars = list(...), mosaic.pars = list(...), axis.pars = list(...), diag.pars = list(...), ...) {
  UseMethod("MoE_gpairs")
}

#' @method MoE_gpairs MoEClust
#' @export
MoE_gpairs.MoEClust <- function(res, response.type = c("points", "uncertainty", "density"), subset = list(...), residuals = FALSE, scatter.type = c("lm", "points"), conditional = c("stripplot", "boxplot"),
                                addEllipses = c("outer", "yes", "no", "inner", "both"), border.col = c("purple", "black", "brown", "brown", "navy"), bg.col = c("cornsilk", "white", "palegoldenrod", "palegoldenrod", "cornsilk"),
                                diagonal = c("on", "off"), outer.margins = list(bottom=grid::unit(2, "lines"), left=grid::unit(2, "lines"), top=grid::unit(2, "lines"), right=grid::unit(2, "lines")), outer.labels = NULL, outer.rot = c(0, 90),
                                gap = 0.05, buffer = 0.02, scatter.pars = list(...), density.pars = list(...), stripplot.pars = list(...), barcode.pars = list(...), mosaic.pars = list(...), axis.pars = list(...), diag.pars = list(...), ...) {

  res   <- if(inherits(res, "MoECompare")) res$optimal else res
  if(attr(res, "Expert") && (length(residuals) > 1 ||
                             !is.logical(residuals))) stop("'residuals' must be a single logical indicator", call.=FALSE)
  resid <- residuals     && attr(res, "Expert")
  G     <- res$G
  Gseq  <- seq_len(G)
  dat   <- if(resid) res$resid.data else res$data
  rownames(dat)     <- if(resid) seq_len(res$n  * G)  else rownames(dat)
  net   <- res$net.covs
  both  <- attr(net, "Both")
  gate  <- setdiff(attr(net, "Gating"), both)
  expx  <- setdiff(attr(net, "Expert"), both)
  if(is.null(subset$show.map)) {
    subset$show.map <- (G + !is.na(res$hypvol)) > 1
  } else if(length(subset$show.map) > 1  ||
            !is.logical(subset$show.map))             stop("'subset$show.map' should be a single logical indicator", call.=FALSE)
  if(is.null(subset$data.ind)) {
    subset$data.ind <- seq_len(ncol(dat))
  } else if(length(subset$data.ind) < 1  ||
            !all(is.numeric(subset$data.ind)) ||
    !all(subset$data.ind    %in% seq_len(ncol(dat)))) stop("Invalid 'subset$data.ind'", call.=FALSE)
  if(is.null(subset$cov.ind))  {
    subset$cov.ind <- seq_len(ncol(net))
  } else if(length(subset$cov.ind)  < 1  ||
            !all(is.numeric(subset$cov.ind))  ||
    !all(subset$cov.ind     %in% seq_len(ncol(net)))) stop("Invalid 'subset$cov.ind'", call.=FALSE)
  subset$data.ind  <- sort(unique(subset$data.ind))
  subset$cov.ind   <- sort(unique(subset$cov.ind))
  if((length(c(subset$data.ind,
     subset$cov.ind)) + subset$show.map) <= 1)        stop("Not enough columns to plot based on arguments supplied to 'subset'!", call.=FALSE)
  dat   <- dat[,subset$data.ind, drop=FALSE]
  net   <- net[,subset$cov.ind,  drop=FALSE]
  dcol  <- ncol(dat)  + subset$show.map
  z     <- if(resid) do.call(rbind, replicate(G, list(res$z))) else res$z
  class <- if(resid) stats::setNames(rep(res$classification, G), seq_len(nrow(dat))) else res$classification
  uni.c <- unique(class[class > 0])
  class <- factor(class)
  x     <- if(ncol(net) == 0) as.data.frame(dat)      else cbind(dat, if(resid) do.call(rbind, replicate(G, net, simplify=FALSE)) else net)
  x     <- if(subset$show.map) cbind(MAP  = class, x) else x
  clust <- as.character(class)
  zc    <- function(x) length(unique(x)) <= 1
  saxzc <- vapply(x, zc, logical(1L))
  nrm   <- sum(saxzc, na.rm=TRUE)
  if(any(saxzc, na.rm=TRUE)) {                        warning(paste(nrm, "column", ifelse(nrm > 1, "s", ""), " with less than two distinct values eliminated"), call.=FALSE)
   dcol <- sum(which(saxzc) < dcol)
   x    <- x[,!saxzc]
  }
  N     <- ncol(x)
  Nseq  <- seq_len(N)
  both  <- which(names(net) %in% gsub("[[:space:]]", ".", gsub("[[:punct:]]", ".", both))) + dcol
  gate  <- which(names(net) %in% gsub("[[:space:]]", ".", gsub("[[:punct:]]", ".", gate))) + dcol
  expx  <- which(names(net) %in% gsub("[[:space:]]", ".", gsub("[[:punct:]]", ".", expx))) + dcol
  both[is.na(both)] <- 0L
  gate[is.na(gate)] <- 0L
  expx[is.na(expx)] <- 0L
  both  <- if(length(both) == 0) 0L else both
  gate  <- if(length(gate) == 0) 0L else gate
  expx  <- if(length(expx) == 0) 0L else expx
  U     <- sort(unique(clust))
  L     <- length(U)
  noise <- any(clust == 0) || !is.na(res$hypvol)
  if(L  <= length(mclust.options("classPlotSymbols"))) {
    symbols   <- mclust.options("classPlotSymbols")
    if(noise) {
      symbols[symbols == 16] <- symbols[G + 1]
      symbols[G + 1]         <- 16
    }
  } else if(L <= 9)  {
    symbols   <- as.character(1:9)
  } else if(L <= 26) {
    symbols   <- LETTERS
  } else if(length(symbols)  == 1) {
    symbols   <- rep(symbols, L)
  }
  if(L <= length(mclust.options("classPlotColors"))) {
    colors    <- mclust.options("classPlotColors")[seq_len(L)]
    if(noise)  {
      colors[colors == "black"] <- colors[G + 1]
      colors[G + 1]             <- "grey65"
    }
  } else if(length(colors) == 1) colors <- rep(colors, L)
  if(length(symbols) < L) {                           warning("More symbols needed to show classification", call.=FALSE)
    symbols <- rep(16, L)
  }
  if(length(colors)  < L) {                           warning("More colors needed to show classification", call.=FALSE)
    colors <- rep("black", L)
  }
  scatter.type <- if(length(scatter.type) == 1) rep(scatter.type, 2) else scatter.type
  conditional  <- if(length(conditional)  == 1) rep(conditional,  2) else conditional
  bg.col       <- if(length(bg.col)       == 1) rep(bg.col,       5) else bg.col
  border.col   <- if(length(border.col)   == 1) rep(border.col,   5) else border.col
  if(length(bg.col)     != 5)                         stop("'bg.col' must be a vector of length 1 or 5 containing valid colours", call.=FALSE)
  if(length(border.col) != 5)                         stop("'border.col' must be a vector of length 1 or 5 containing valid colours", call.=FALSE)
  if(!missing(response.type)    && (length(response.type) > 1 ||
     !is.character(response.type)))                   stop("'response.type' must be a single character string", call.=FALSE)
  response.type         <- match.arg(response.type)
  if(length(scatter.type)  != 2 ||
            !all(is.character(scatter.type)))         stop("'scatter.type' must be a character vector of length 2", call.=FALSE)
  if(length(conditional)   != 2 ||
            !all(is.character(conditional)))          stop("'conditional' must be a character vector of length 2", call.=FALSE)
  if(!all(scatter.type %in% c("ci", "lm", "points",
                              "ci2", "lm2")))         stop("The entries of 'scatter.type' must be one of 'points', 'ci', 'lm', 'ci2', or 'lm2'", call.=FALSE)
  if(!all(conditional  %in% c("stripplot", "violin",
                             "boxplot", "barcode")))  stop("The entries of 'conditional' must be one of 'stripplot', 'boxplot', 'violin' or 'barcode'", call.=FALSE)
  if(!missing(addEllipses) && (length(addEllipses) > 1 ||
     !is.character(addEllipses)))                     stop("'addEllipses' must be a single character string", call.=FALSE)
  addEllipses  <- match.arg(addEllipses)
  addEllipses  <- ifelse(G == 0,   "no",    addEllipses)
  drawEllipses <- addEllipses   != "no"
  colEllipses  <- addEllipses   != "yes" && drawEllipses
  if(res$d  > 1) {
    res$parameters$varianceX    <- if(isTRUE(drawEllipses)) suppressMessages(expert_covar(res)) else res$parameters$variance
  }
  upr.gate <- grepl("2", scatter.type[1])
  low.gate <- grepl("2", scatter.type[2])
  upr.exp  <- ifelse(upr.gate, substr(scatter.type[1], 1, nchar(scatter.type[1]) - 1), scatter.type[1])
  low.exp  <- ifelse(low.gate, substr(scatter.type[2], 1, nchar(scatter.type[2]) - 1), scatter.type[2])
  upr.cond <- conditional[1]
  low.cond <- conditional[2]
  if(!missing(diagonal) && (length(diagonal) > 1 ||
      !is.character(diagonal)))                       stop("'diagonal' must be a single character string", call.=FALSE)
  diagonal <- match.arg(diagonal)
  if(!is.list(outer.margins)) {
    if(length(outer.margins) == 4) {
      outer.margins     <- if(inherits(outer.margins[1], "units")) list(bottom=outer.margins[1], left=outer.margins[2], top=outer.margins[3], right=outer.margins[4]) else list(bottom=grid::unit(outer.margins[1], "lines"), left=grid::unit(outer.margins[2], "lines"), top=grid::unit(outer.margins[3], "lines"), right=grid::unit(outer.margins[4], "lines"))
    } else                                            stop("'outer.margins' are not valid", call.=FALSE)
  }
  if(is.null(outer.labels)) {
    lab1   <- switch(names(x)[1], MAP=1, 2)
    lab2   <- switch(names(x)[1], MAP=2, 1)
    outer.labels$top    <- vector("logical", N)
    outer.labels$top[seq(lab1,  N, by=2)] <- TRUE
    outer.labels$left   <- vector("logical", N)
    outer.labels$left[seq(lab2, N, by=2)] <- TRUE
    outer.labels$right  <- !outer.labels$left
    outer.labels$bottom <- !outer.labels$top
  } else {
    if(pmatch(as.character(outer.labels),         "all", nomatch=FALSE)) {
      all.labeling      <- TRUE
    } else if(pmatch(as.character(outer.labels), "none", nomatch=FALSE)) {
      all.labeling      <- FALSE
    } else                                            stop("Invalid 'outer.labels'", call.=FALSE)
    outer.labels        <- NULL
    outer.labels$top    <-
    outer.labels$left   <-
    outer.labels$bottom <-
    outer.labels$right  <- rep(all.labeling, N)
  }
  if(length(outer.rot)  != 2    ||
    !all(is.numeric(outer.rot)) ||
     any(outer.rot       < 0))                        stop("Invalid 'outer.rot': must be a strictly non-negative numeric vector of length 2", call.=FALSE)
  class                 <- as.integer(levels(class))[class]
  class[which(class     == 0)]  <- G  + 1
  x[,1]                 <- if(names(x)[1] == "MAP") factor(class) else x[,1]
  if(length(gap)        != 1    || (!is.numeric(gap)    ||
     gap    < 0))                                     stop("'gap' must be single strictly non-negative number", call.=FALSE)
  if(length(buffer)     != 1    || (!is.numeric(buffer) ||
     buffer < 0))                                     stop("'buffer' must be single strictly non-negative number", call.=FALSE)
  if(is.null(scatter.pars$scat.pch))  {
    scatter.pars$pch    <- symbols[class]
  } else scatter.pars$pch         <- scatter.pars$scat.pch
  if(is.null(scatter.pars$scat.size)) {
    scatter.pars$size   <- grid::unit(0.25, "char")
  } else scatter.pars$size        <- scatter.pars$scat.size
  if(length(scatter.pars$size)   > 1 ||
       !inherits(scatter.pars$size,         "unit"))  stop("'scatter.pars$scat.size' must be a single item of class 'unit'", call.=FALSE)
  uncertainty           <- res$uncertainty
  uncertainty           <- grid::unit((uncertainty - min(uncertainty))/diff(range(uncertainty)) * as.numeric(scatter.pars$size), attr(scatter.pars$size, "unit"))
  if(is.null(scatter.pars$scat.col)) {
    scatter.pars$col    <- colors[class]
  } else scatter.pars$col    <- scatter.pars$scat.col
  if(is.null(scatter.pars$lci.col))  {
    scatter.pars$lci.col     <- colors[Gseq]
  }
  if(is.null(density.pars$grid.size)) {
    density.pars$grid.size   <- c(100, 100)
  } else if(length(density.pars$grid.size)  != 2 || !all(is.numeric(density.pars$grid.size)) ||
            any(density.pars$grid.size       < 10))   stop("Invalid 'density.pars$grid.size'", call.=FALSE)
  if(is.null(density.pars$dcol))     {
    density.pars$dcol    <- "grey50"
  } else if(length(density.pars$dcol) > 1   ||
            !is.character(density.pars$dcol))         stop("Invalid 'density.pars$dcol", call.=FALSE)
  if(is.null(density.pars$nlevels))  {
    density.pars$nlevels     <- 11
  } else if(length(density.pars$nlevels) > 1 || !is.numeric(density.pars$nlevels) ||
            density.pars$nlevels            <= 1)     stop("Invalid 'density.pars$nlevels'", call.=FALSE)
  if(is.null(density.pars$show.labels))  {
    density.pars$show.labels <- TRUE
  } else if(length(density.pars$show.labels) > 1 ||
            !is.logical(density.pars$show.labels))    stop("Invalid 'density.pars$show.labels", call.=FALSE)
  if(is.null(axis.pars$n.ticks))     {
    axis.pars$n.ticks   <- 5
  }
  if(is.null(axis.pars$axis.fontsize))   {
    axis.pars$fontsize  <- 9
  } else axis.pars$fontsize       <- axis.pars$axis.fontsize
  if(axis.pars$n.ticks   < 3)        {                warning("Fewer than 3 axis ticks might cause problems", call.=FALSE)
    axis.pars$n.ticks   <- 3
  }
  if(is.null(diag.pars$diag.fontsize))   {
    diag.pars$fontsize  <- 9
  } else diag.pars$fontsize       <- diag.pars$diag.fontsize
  if(is.null(diag.pars$show.hist))   {
    diag.pars$show.hist <- TRUE
  }
  if(is.null(diag.pars$hist.color))  {
   diag.pars$hist.color <- c("black", "grey", "grey", "grey")
  } else {
   diag.pars$hist.color <- if(length(diag.pars$hist.color) == 1) rep(diag.pars$hist.color, 4) else diag.pars$hist.color
   if(length(diag.pars$hist.color) != 4)              stop("'diag.pars$hist.color' must be a vector of length 1 or 4", call.=FALSE)
  }
  hist.col              <- diag.pars$hist.color
  diag.pars$hist.color  <- replace(Nseq,                 Nseq  <=  dcol, hist.col[1])
  diag.pars$hist.color  <- replace(diag.pars$hist.color, Nseq %in% gate, hist.col[2])
  diag.pars$hist.color  <- replace(diag.pars$hist.color, Nseq %in% expx, hist.col[3])
  diag.pars$hist.color  <- replace(diag.pars$hist.color, Nseq %in% both, hist.col[4])
  if(is.null(diag.pars$show.counts)) {
    diag.pars$show.counts  <- TRUE
  } else if(length(diag.pars$show.counts) > 1 ||
            !is.logical(diag.pars$show.counts))       stop("'diag.pars$show.counts' must be a single logical indicator", call.=FALSE)
  if(is.null(stripplot.pars$strip.pch)) {
    stripplot.pars$pch  <- symbols[class]
  } else stripplot.pars$pch       <- stripplot.pars$strip.pch
  if(is.null(stripplot.pars$strip.size) ||
     any(names(list(...)) == "size")) {
    stripplot.pars$size <- grid::unit(0.5, "char")
  } else stripplot.pars$size      <- stripplot.pars$strip.size
  if(length(stripplot.pars$size) > 1    ||
            !inherits(stripplot.pars$size, "unit"))   stop("'stripplot.pars$strip.size' must be a single item of class 'unit'", call.=FALSE)
  if(is.null(stripplot.pars$strip.col))  {
    stripplot.pars$col  <- colors[class]
  } else stripplot.pars$col       <- stripplot.pars$strip.col
  if(is.null(stripplot.pars$jitter)) {
  stripplot.pars$jitter <- TRUE
  }
  if(is.null(barcode.pars$nint))     {
    barcode.pars$nint   <- 0
  }
  if(is.null(barcode.pars$ptsize))   {
    barcode.pars$ptsize <- grid::unit(0.25, "char")
  }
  if(is.null(barcode.pars$ptpch))    {
    barcode.pars$ptpch  <- 1
  }
  if(is.null(barcode.pars$use.points)) {
    barcode.pars$use.points <- FALSE
  }
  if(is.null(mosaic.pars$gp_labels)) {
    mosaic.pars$gp_labels   <- grid::gpar(fontsize=9)
  }
  if(is.null(mosaic.pars$gp_args))   {
    mosaic.pars$gp_args <- list()
  }
  if(is.null(mosaic.pars$shade))     {
    mosaic.pars$shade   <- NULL
  }
  noise.cols <- scatter.pars$col
  scatter.pars$ecol     <- unique(noise.cols[class != G + 1])[match(Gseq, uni.c)]
  noise.cols <- unique(noise.cols)[match(if(noise) c(Gseq, G + 1) else Gseq, unique(class))]
  grid::grid.newpage()
  vp.main    <- grid::viewport(x=outer.margins$bottom, y=outer.margins$left,
                               width=grid::unit(1,  "npc") - outer.margins$right - outer.margins$left,
                               height=grid::unit(1, "npc") - outer.margins$top   - outer.margins$bottom,
                               just=c("left", "bottom"), name="main", clip="off")
  grid::pushViewport(vp.main)
  for(i   in Nseq) {
    for(j in Nseq) {
      bg     <- if((i == 1 || j == 1) && names(x)[1] == "MAP") bg.col[1]     else if(i <= dcol && j <= dcol) bg.col[2]     else if(i > dcol && j > dcol) bg.col[5]     else if(j > dcol && i <= dcol) bg.col[3]     else bg.col[4]
      border <- if((i == 1 || j == 1) && names(x)[1] == "MAP") border.col[1] else if(i <= dcol && j <= dcol) border.col[2] else if(i > dcol && j > dcol) border.col[5] else if(j > dcol && i <= dcol) border.col[3] else border.col[4]
      labelj <- switch(diagonal, on=j, off=N - j + 1)
      x[is.infinite(x[,i]), i] <- NA
      x[is.infinite(x[,j]), j] <- NA
      vp     <- grid::viewport(x=(labelj - 1)/N, y=1 - i/N, width=1/N, height=1/N, just=c("left", "bottom"), name=as.character(i * N + j))
      grid::pushViewport(vp)
      vp.in  <- grid::viewport(x=0.5, y=0.5, width=1 - gap, height=1 - gap, just=c("center", "center"), name=paste("IN", as.character(i * N + j)))
      grid::pushViewport(vp.in)
      xpos   <- NULL
      if(i == 1 && outer.labels$top[j])    {
        xpos <- FALSE
      }
      if(i == N && outer.labels$bottom[j]) {
        xpos <- TRUE
      }
      ypos   <- NULL
      if(j == N && outer.labels$right[i])  {
        ypos <- FALSE
      }
      if(j == 1 && outer.labels$left[i])   {
        ypos <- TRUE
      }
      if(!is.null(ypos) && diagonal == "off") {
        ypos <- !ypos
      }
      if(i == j) {
        .diag_panel(x=x[,i], varname=names(x)[i], diag.pars=diag.pars, hist.col=if(i == 1 && names(x)[i] == "MAP") list(noise.cols) else diag.pars$hist.color, axis.pars=axis.pars, xpos=xpos, ypos=ypos, buffer=buffer, index=i, outer.rot=outer.rot)
      } else {
        if(is.factor(x[,i]) + is.factor(x[,j]) == 1) {
          if(i < j & upr.cond != "barcode") .boxplot_panel(x=x[,j], y=x[,i], type=upr.cond, axis.pars=axis.pars, xpos=xpos, ypos=ypos, buffer=buffer, stripplot.pars=stripplot.pars, outer.rot=outer.rot, bg=bg, box.fill=if((i == 1 || j == 1) && names(x)[1] == "MAP") noise.cols else if(i > dcol && j > dcol) "grey" else "white", border=border, col.ind=i <= dcol)
          if(i > j & low.cond != "barcode") .boxplot_panel(x=x[,j], y=x[,i], type=low.cond, axis.pars=axis.pars, xpos=xpos, ypos=ypos, buffer=buffer, stripplot.pars=stripplot.pars, outer.rot=outer.rot, bg=bg, box.fill=if((i == 1 || j == 1) && names(x)[1] == "MAP") noise.cols else if(i > dcol && j > dcol) "grey" else "white", border=border, col.ind=FALSE)
          if(i < j & upr.cond == "barcode") {
            grid::pushViewport(grid::viewport(gp=grid::gpar(fill=bg)))
            if(is.factor(x[,i])) {
              .bar_code(x=split(x[,j], x[,i])[length(levels(x[,i])):1], horizontal=TRUE, xlim=NULL,
                        labelloc=ypos, axisloc=xpos, labelouter=TRUE, newpage=FALSE, fontsize=axis.pars$fontsize,
                        buffer=buffer, nint=barcode.pars$nint, ptsize=barcode.pars$ptsize, ptpch=barcode.pars$ptpch,
                        bcspace=barcode.pars$bcspace, use.points=barcode.pars$use.points, outerbox=border)
            } else {
              if(!is.null(ypos)) ypos <- !ypos
              .bar_code(x=split(x[,i], x[,j])[length(levels(x[,j])):1], horizontal=FALSE, xlim=NULL,
                        labelloc=xpos, axisloc=ypos, labelouter=TRUE, newpage=FALSE, fontsize=axis.pars$fontsize,
                        buffer=buffer, nint=barcode.pars$nint, ptsize=barcode.pars$ptsize, ptpch=barcode.pars$ptpch,
                        bcspace=barcode.pars$bcspace, use.points=barcode.pars$use.points, outerbox=border)
            }
            grid::popViewport()
          }
          if(i > j & low.cond == "barcode") {
            grid::pushViewport(grid::viewport(gp=grid::gpar(fill=bg)))
            if(is.factor(x[,i])) {
              .bar_code(x=split(x[,j], x[,i])[length(levels(x[,i])):1], horizontal=TRUE, xlim=NULL,
                        labelloc=ypos, axisloc=xpos, labelouter=TRUE, newpage=FALSE, fontsize=axis.pars$fontsize,
                        buffer=buffer, nint=barcode.pars$nint, ptsize=barcode.pars$ptsize, ptpch=barcode.pars$ptpch,
                        bcspace=barcode.pars$bcspace, use.points=barcode.pars$use.points, outerbox=border)
            } else {
              if(!is.null(ypos)) ypos <- !ypos
              .bar_code(x=split(x[,i], x[,j])[length(levels(x[,j])):1], horizontal=FALSE, xlim=NULL,
                        labelloc=xpos, axisloc=ypos, labelouter=TRUE, newpage=FALSE, fontsize=axis.pars$fontsize,
                        buffer=buffer, nint=barcode.pars$nint, ptsize=barcode.pars$ptsize, ptpch=barcode.pars$ptpch,
                        bcspace=barcode.pars$bcspace, use.points=barcode.pars$use.points, outerbox=border)
            }
            grid::popViewport()
          }
        }
        if(!any(is.factor(x[,i]), is.factor(x[,j])))  {
          if(response.type == "density" && all(j <= dcol, i <= dcol)) {
            .density_panel(cbind(x[,j], x[,i]), dimens=c(j - subset$show.map, i - subset$show.map), res, density.pars, axis.pars, xpos, ypos, buffer, outer.rot, bg, border, resid)
          } else {
            .scatter_panel(x=x[,j], y=x[,i], type=ifelse(j > dcol && i <= dcol, ifelse(upr.gate || (j %in% c(expx, both)), upr.exp, "points"), ifelse(j <= dcol && i <= dcol, ifelse(drawEllipses, "ellipses", "points"), ifelse(j <= dcol && (low.gate || (i %in% c(expx, both))), low.exp, "points"))),
                           scatter.pars=scatter.pars, axis.pars=axis.pars, xpos=xpos, ypos=ypos, buffer=buffer, z=z, G=G, res=res, dimens=c(j - subset$show.map, i - subset$show.map), outer.rot=outer.rot, bg=bg, mvn.type=addEllipses, border=border,
                           uncertainty=if(response.type == "uncertainty" && (i <= dcol && j <= dcol)) uncertainty else NA, mvn.col=if(colEllipses) scatter.pars$ecol else NULL, residuals=resid)
          }
        }
        if(all(is.factor(x[,i]),  is.factor(x[,j]))) {
          .mosaic_panel(x=x[,j], y=x[,i], mosaic.pars=mosaic.pars, mosaic.outer=if(j == 1) noise.cols else if(i == 1) rep(noise.cols, each=nlevels(x[,j])) else mosaic.pars$gp$col, axis.pars=axis.pars, xpos=xpos, ypos=ypos, outer.rot=outer.rot, bg=bg)
        }
      }
      grid::popViewport(1)
      grid::upViewport()
    }
  }
  grid::popViewport()
  invisible()
}

#' Plot MoEClust Gating Network
#'
#' Plots the gating network for fitted MoEClust models, i.e. the observation index against the mixing proportions for that observation, coloured by cluster.
#' @param res An object of class \code{"MoEClust"} generated by \code{\link{MoE_clust}}, or an object of class \code{"MoECompare"} generated by \code{\link{MoE_compare}}.
#' @param type,xlab,ylab,ylim,col These graphical parameters retain their definitions from \code{\link[graphics]{matplot}}.
#' @param ... Catches unused arguments, or additional arguments to be passed to \code{\link[graphics]{matplot}}.
#'
#' @return A plot of the gating network of the fitted model.
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @note \code{\link{plot.MoEClust}} is a wrapper to \code{\link{MoE_plotGate}} which accepts the default arguments, and also produces other types of plots.
#'
#' By default, the noise component (if any) will be coloured \code{"grey65"}.
#' @seealso \code{\link{MoE_clust}}, \code{\link{plot.MoEClust}}, \code{\link[graphics]{matplot}}
#' @keywords plotting
#' @export
#'
#' @examples
#' data(ais)
#' res <- MoE_clust(ais[,3:7], gating= ~ sex, G=3, modelNames="EEE", network.data=ais,
#'                  noise.init=sample(c(TRUE, FALSE), replace=TRUE, size=nrow(ais)))
#' MoE_plotGate(res)
MoE_plotGate  <- function(res, type = "l", xlab = "Observation", ylab = expression(tau[g]), ylim = c(0, 1), col = NULL, ...) {
  UseMethod("MoE_plotGate")
}

#' @method MoE_plotGate MoEClust
#' @export
MoE_plotGate.MoEClust   <- function(res, type = "l", xlab = "Observation", ylab = expression(tau[g]), ylim = c(0, 1), col = NULL, ...) {
  res         <- if(inherits(res, "MoECompare")) res$optimal        else res
  Tau         <- .mat_byrow(res$parameters$pro, nrow=res$n, ncol=ncol(res$z))
  G           <- res$G
  cX          <- missing(col)
  nX          <- is.na(res$hypvol)
  col         <- if(cX)                                  seq_len(G) else col
  col         <- if(cX  && !nX) c(rep(col, length.out=G), "grey65") else col
    graphics::matplot(Tau, type=type, xlab=xlab, ylab=ylab, ylim=ylim, col=col, ...)
    invisible()
}

#' Model Selection Criteria Plot for MoEClust Mixture Models
#'
#' Plots the BIC, ICL, or AIC values of a fitted \code{MoEClust} object.
#' @param res An object of class \code{"MoEClust"} generated by \code{\link{MoE_clust}}, or an object of class \code{"MoECompare"} generated by \code{\link{MoE_compare}}.
#' @param criterion The criterion to be plotted. Defaults to \code{"bic"}.
#' @param ... Catches other arguments, or additional arguments to be passed to \code{\link[mclust]{plot.mclustBIC}} (or equivalent functions for the other \code{criterion} arguments).
#'
#' @importFrom mclust "plot.mclustBIC" "plot.mclustICL"
#' @return A plot of the values of the chosen \code{criterion}.
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @keywords plotting
#' @export
#' @note \code{\link{plot.MoEClust}} is a wrapper to \code{\link{MoE_plotCrit}} which accepts the default arguments, and also produces other types of plots.
#' @seealso \code{\link{MoE_clust}}, \code{\link{plot.MoEClust}}, \code{\link[mclust]{plot.mclustBIC}}
#'
#' @examples
#' \dontrun{
#' data(ais)
#' res <- MoE_clust(ais[,3:7], expert= ~ sex, network.data=ais)
#' MoE_plotCrit(res)}
MoE_plotCrit <- function(res, criterion = c("bic", "icl", "aic"), ...) {
  UseMethod("MoE_plotCrit")
}

#' @method MoE_plotCrit MoEClust
#' @export
MoE_plotCrit.MoEClust   <- function(res, criterion = c("bic", "icl", "aic"), ...) {
  if(!missing(criterion)      &&
     (length(criterion)  > 1  ||
      !is.character(criterion)))                      stop("'criterion' must be a single character string", call.=FALSE)
  res        <- if(inherits(res, "MoECompare")) res$optimal else res
  criterion  <- match.arg(criterion)
  crit       <- switch(criterion, bic=res$BIC, icl=res$ICL, res$AIC)
  crit       <- replace(crit, !is.finite(crit), NA)
    switch(criterion, bic=plot.mclustBIC(crit, ...), icl=plot.mclustICL(crit, ...), plot.mclustAIC(crit, ...))
    invisible()
}

#' Plot the Log-Likelihood of a MoEClust Mixture Model
#'
#' Plots the log-likelihood at every iteration of the EM algorithm used to fit a MoEClust mixture model.
#' @param res An object of class \code{"MoEClust"} generated by \code{\link{MoE_clust}}, or an object of class \code{"MoECompare"} generated by \code{\link{MoE_compare}}.
#' @param type,xlab,ylab,xaxt These graphical parameters retain their usual definitions from \code{\link[graphics]{plot}}.
#' @param ... Catches unused arguments, or additional arguments to be passed to \code{\link[graphics]{plot}}.
#'
#' @return A plot of the log-likelihood versus the EM iterations.
#' @note \code{\link{plot.MoEClust}} is a wrapper to \code{\link{MoE_plotLogLik}} which accepts the default arguments, and also produces other types of plots.
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @keywords plotting
#' @export
#' @seealso \code{\link{MoE_clust}}, \code{\link{plot.MoEClust}},
#'
#' @examples
#' data(ais)
#' res <- MoE_clust(ais[,3:7], gating= ~ BMI, expert= ~ sex,
#'                  G=2, modelNames="EVE", network.data=ais)
#' MoE_plotLogLik(res)
MoE_plotLogLik          <- function(res, type = "l", xlab = "Iteration", ylab = "Log-Likelihood", xaxt = "n", ...) {
  UseMethod("MoE_plotLogLik")
}

#' @method MoE_plotLogLik MoEClust
#' @export
MoE_plotLogLik.MoEClust <- function(res, type = "l", xlab = "Iteration", ylab = "Log-Likelihood", xaxt = "n", ...) {
  res        <- if(inherits(res, "MoECompare")) res$optimal else res
  xll        <- res$loglik
  if(all(xll != cummax(xll)))                         warning("Log-likelihoods are not strictly increasing", call.=FALSE)
  graphics::plot(xll, type = ifelse(length(xll) == 1, "p", type), xlab = xlab, ylab = ylab, xaxt = xaxt)
  if(length(xaxt) == 1  && is.character(xaxt)) {
    seqll    <- seq_along(xll)
    llseq    <- pretty(seqll)
    llseq    <- if(any(llseq  != floor(llseq))) seqll else  llseq
    if(xaxt  == "n") graphics::axis(1, at = llseq, labels = llseq)
  } else                                              stop("'xaxt' must be a single character string", call.=FALSE)
  invisible()
}

#' Plot MoEClust Results
#'
#' Plot results for fitted MoE_clust mixture models with gating &/or expert network covariates: generalised pairs plots, model selection criteria, the log-likelihood vs. the EM iterations, and the gating network are all currently visualisable.
#' @param x An object of class \code{"MoEClust"} generated by \code{\link{MoE_clust}}, or an object of class \code{"MoECompare"} generated by \code{\link{MoE_compare}}.
#' @param what The type of graph requested:
#' \describe{
#' \item{\code{gpairs}}{A generalised pairs plot. To further customise this plot, arguments to \code{\link{MoE_gpairs}} can be supplied.}
#' \item{\code{gating}}{The gating network. To further customise this plot, arguments to \code{\link{MoE_plotGate}} and \code{\link[graphics]{matplot}} can be supplied.}
#' \item{\code{criterion}}{The model selection criteria. To further customise this plot, arguments to \code{\link{MoE_plotCrit}} and \code{\link[mclust]{plot.mclustBIC}} can be supplied.}
#' \item{\code{loglik}}{The log-likelihood vs. the iterations of the EM algorithm. To further customise this plot, arguments to \code{\link{MoE_plotLogLik}} and \code{\link[graphics]{plot}} can be supplied.}
#' }
#' By default, all of the above graphs are produced.
#' @param ... Optional arguments to be passed to \code{\link{MoE_gpairs}}, \code{\link{MoE_plotGate}}, \code{\link{MoE_plotCrit}}, \code{\link{MoE_plotGate}}, \code{\link[graphics]{matplot}}, \code{\link[mclust]{plot.mclustBIC}} and \code{\link[graphics]{plot}}.
#'
#' @details For more flexibility in plotting, use \code{\link{MoE_gpairs}}, \code{\link{MoE_plotGate}}, \code{\link{MoE_plotCrit}} and \code{\link{MoE_plotLogLik}}.
#' @importFrom lattice "current.panel.limits" "panel.abline" "panel.bwplot" "panel.histogram" "panel.lines" "panel.points" "panel.rect" "panel.stripplot" "panel.text" "panel.violin" "trellis.grobname" "trellis.par.get" "trellis.par.set"
#' @importFrom matrixStats "colMeans2" "rowLogSumExps"
#' @importFrom mclust "plot.mclustBIC" "plot.mclustICL"
#' @importFrom vcd "strucplot"
#' @note Caution is advised producing generalised pairs plots when the dimension of the data is large.
#'
#' Other types of plots are available by first calling \code{\link{as.Mclust}} on the fitted object, and then calling \code{\link[mclust]{plot.Mclust}} on the results.
#' @return The visualisation according to \code{"what"} of the results of a fitted \code{MoEClust} model.
#' @seealso \code{\link{MoE_clust}}, \code{\link{MoE_gpairs}}, \code{\link{MoE_plotGate}}, \code{\link{MoE_plotCrit}}, \code{\link{as.Mclust}}, \code{\link[mclust]{plot.Mclust}}
#' @references K. Murphy and T. B. Murphy (2017). Parsimonious Model-Based Clustering with Covariates. \emph{To appear}. <\href{https://arxiv.org/abs/1711.05632}{arXiv:1711.05632}>.
#' @author Keefe Murphy - \href{keefe.murphy@ucd.ie}{<keefe.murphy@ucd.ie>}
#' @export
#' @keywords plotting main
#'
#' @examples
#' \dontrun{
#' data(ais)
#' res <- MoE_clust(ais[,3:7], gating= ~ BMI, expert= ~ sex,
#'                  G=2, modelNames="EVE", network.data=ais)
#'
#' # Plot the gating network
#' plot(res, what="gating")
#'
#' # Plot the log-likelihood
#' plot(res, what="loglik")
#'
#' # Produce a generalised pairs plot
#' plot(res, what="gpairs")
#'
#' # Modify the gpairs plot by passing arguments to MoE_gpairs()
#' plot(res, what="gpairs", response.type="density",
#'      scatter.type="ci", jitter=FALSE, show.counts=FALSE)}
plot.MoEClust <- function(x, what=c("gpairs", "gating", "criterion", "loglik"), ...) {
  if(!missing(what) && !all(is.character(what)))      stop("'what' must be a character string", call.=FALSE)
  what        <- match.arg(what, several.ok=TRUE)
  if(interactive()  && length(what) > 1) {
    title     <- c("MoEClust Plots\n\n<Press 0 to exit>")
    what.tmp  <- c(gpairs="Generalised Pairs Plot", gating="Gating Network",
                   criterion="Model Selection Criteria", loglik="Log-Likelihood")
    choice    <- utils::menu(what.tmp, graphics=FALSE, title=title)
    while(choice    != 0) {
      switch(what[choice], gpairs=MoE_gpairs(x, ...), gating=MoE_plotGate(x, ...),
             criterion=MoE_plotCrit(x, ...), loglik=MoE_plotLogLik(x, ...))
      choice  <- utils::menu(what.tmp, graphics=FALSE, title=title)
    }
  } else if(length(what)  > 1) {                      stop("'what' must be a single character string for non-interactive sessions", call.=FALSE)
  } else {
    switch(what, gpairs=MoE_gpairs(x,  ...), gating=MoE_plotGate(x,   ...),
           criterion=MoE_plotCrit(x,   ...), loglik=MoE_plotLogLik(x, ...))
  }
    invisible()
}

.grid_1   <- function(n, range = c(0, 1), edge = TRUE) {
  if(any(n < 0 | round(n) != n))                      stop("'n' must be nonpositive and integer", call.=FALSE)
  G       <- vector("numeric", n)
  if(edge) {
    G     <- seq(from=min(range), to=max(range), by=abs(diff(range))/(n - 1))
  } else   {
    lj    <- abs(diff(range))
    incr  <- lj/(2 * n)
    G     <- seq(from=min(range) + incr, to=max(range) - incr, by=2 * incr)
  }
    G
}

.grid_2   <- function(x, y) {
  lx      <- length(x)
  ly      <- length(y)
  xy      <- matrix(0, nrow=lx * ly, ncol=2)
  l       <- 0
  for(j   in seq_len(ly))   {
    for(i in seq_len(lx))   {
      l   <- l + 1
      xy[l,]  <- c(x[i], y[j])
    }
  }
    xy
}

#' @importFrom lattice "panel.lines" "panel.points"
.mvn2D_panel <- function(mu, sigma, k = 15, alone = FALSE, col = rep("grey30", 3), pch = 8, lty = c(1, 2), lwd = c(1, 1)) {
  ev         <- eigen(sigma, symmetric = TRUE)
  s          <- sqrt(rev(sort(ev$values)))
  V          <- ev$vectors[,rev(order(ev$values))]
  theta      <- (0:k)  * (pi/(2 * k))
  x          <- s[1]   * cos(theta)
  y          <- s[2]   * sin(theta)
  xy         <- sweep(tcrossprod(cbind(c(x, -x, -x, x), c(y, y, -y, -y)), V), MARGIN=2, STATS=mu, FUN="+", check.margin=FALSE)
  l          <- length(x)
  i          <- seq_len(l)
  for(k in seq_len(4)) {
    panel.lines(xy[i,], col=col[1], lty=lty[1], lwd=lwd[1])
    i        <- i + l
  }
  x          <- s[1]
  y          <- s[2]
  xy         <- sweep(tcrossprod(cbind(c(x, -x, 0, 0), c(0, 0, y, -y)),   V), MARGIN=2, STATS=mu, FUN="+", check.margin=FALSE)
  panel.lines(xy[1:2,], col=col[2], lty=lty[2], lwd=lwd[2])
  panel.lines(xy[3:4,], col=col[2], lty=lty[2], lwd=lwd[2])
  panel.points(mu[1], mu[2], col=col[3], pch=pch)
    invisible()
}

.draw_axis <- function(x, y, axis.pars, xpos, ypos, cat.labels = NULL, horiz = NULL, xlim = NULL, ylim = NULL, outer.rot) {
  x.fac    <- is.factor(x)
  y.fac    <- is.factor(y)
  x        <- if(x.fac) x         else as.numeric(x)
  y        <- if(y.fac) y         else as.numeric(y)
  if(is.null(xlim)) {
    px     <- if(x.fac) levels(x) else pretty(x,    axis.pars$n.ticks)
    px     <- if(x.fac) px        else px[px > min(x, na.rm=TRUE)    & px < max(x, na.rm=TRUE)]
  } else {
    px     <- if(x.fac) levels(x) else pretty(xlim, axis.pars$n.ticks)
    px     <- if(x.fac) px        else px[px > min(xlim, na.rm=TRUE) & px < max(xlim, na.rm=TRUE)]
  }
  if(is.null(ylim)) {
    py     <- if(y.fac) levels(y) else pretty(y,    axis.pars$n.ticks)
    py     <- if(y.fac) py        else py[py > min(y, na.rm=TRUE)    & py < max(y, na.rm=TRUE)]
  } else {
    py     <- if(y.fac) levels(y) else pretty(ylim, axis.pars$n.ticks)
    py     <- if(y.fac) py        else py[py > min(ylim, na.rm=TRUE) & py < max(ylim, na.rm=TRUE)]
  }
  k        <- length(cat.labels)
  if(!is.null(xpos)) {
    if(!is.null(cat.labels) && !horiz) {
      grid::grid.text(cat.labels, x=grid::unit(seq_len(k), "native"), y=grid::unit(rep(1 * (1 - xpos), k), "npc") + grid::unit(rep(-1 * xpos + 1 * (1 - xpos), k), "lines"), rot=outer.rot[1], gp=grid::gpar(fontsize=axis.pars$fontsize))
    } else  grid::grid.xaxis(at=if(x.fac) seq_along(px) else px, gp=grid::gpar(fontsize=axis.pars$fontsize), main=xpos, label=px)
  }
  if(!is.null(ypos)) {
    if(!is.null(cat.labels) && horiz)  {
      grid::grid.text(cat.labels, y=grid::unit(seq_len(k), "native"), x=grid::unit(rep(1 * (1 - ypos), k), "npc") + grid::unit(rep(-1 * ypos + 1 * (1 - ypos), k), "lines"), rot=outer.rot[2], gp=grid::gpar(fontsize=axis.pars$fontsize))
    } else  grid::grid.yaxis(at=if(y.fac) seq_along(py) else py, gp=grid::gpar(fontsize=axis.pars$fontsize), main=ypos, label=py)
  }
}

#' @importFrom lattice "panel.bwplot" "panel.stripplot" "trellis.par.get" "trellis.par.set"
.boxplot_panel <- function(x, y, type, axis.pars, xpos, ypos, buffer, stripplot.pars, outer.rot, bg, box.fill, border, col.ind) {
  xlim         <- NULL
  ylim         <- NULL
  old.color    <- trellis.par.get("box.rectangle")$col
  trellis.par.set(name="box.rectangle", value=list(col="black"))
  trellis.par.set(name="box.umbrella",  value=list(col="black"))
  trellis.par.set(name="box.dot",       value=list(col="black"))
  trellis.par.set(name="plot.symbol",   value=list(col="black"))
  if(is.factor(x)) {
    cat.labels <- levels(x)
    k          <- length(levels(x))
    cat.var    <- as.numeric(x)
    cont.var   <- y
    horiz      <- FALSE
  } else {
    cat.labels <- levels(y)
    k          <- length(levels(y))
    cat.labels <- cat.labels[k:1]
    cat.var    <- k + 1 - as.numeric(y)
    cont.var   <- x
    horiz      <- TRUE
    stripplot.pars$col <- if(col.ind) unique(stripplot.pars$col)[match(levels(y), unique(y))][as.numeric(y)] else stripplot.pars$col
  }
  grid::grid.rect(gp=grid::gpar(fill=bg, col=border))
  if(horiz) {
    xlim       <- range(cont.var, na.rm=TRUE) + c(-buffer * (max(cont.var, na.rm=TRUE) - min(cont.var, na.rm=TRUE)), buffer * (max(cont.var, na.rm=TRUE) - min(cont.var, na.rm=TRUE)))
    grid::pushViewport(grid::viewport(xscale=xlim, yscale=c(0.5, max(cat.var, na.rm=TRUE) + 0.5)))
    if(is.null(ypos)) cat.labels <- NULL
    .draw_axis(x=cont.var, y=cat.var, axis.pars=axis.pars, xpos=xpos, ypos=ypos, cat.labels=cat.labels, horiz=horiz, xlim=xlim, ylim=ylim, outer.rot=outer.rot)
    grid::popViewport(1)
    grid::pushViewport(grid::viewport(xscale=xlim, yscale=c(0.5, max(cat.var, na.rm=TRUE) + 0.5), clip=TRUE))
    if(type == "boxplot")   panel.bwplot(cont.var, cat.var, horizontal=horiz, col="black", pch="|", gp=grid::gpar(box.umbrella=list(col="black")), fill=box.fill)
    if(type == "stripplot") panel.stripplot(cont.var, cat.var, horizontal=horiz, jitter.data=stripplot.pars$jitter, col=stripplot.pars$col, cex=stripplot.pars$size, pch=stripplot.pars$pch)
    if(type == "violin")    .violin_panel(cont.var, cat.var, horizontal=horiz, col=box.fill)
  } else {
    ylim       <- range(cont.var, na.rm=TRUE) + c(-buffer * (max(cont.var, na.rm=TRUE) - min(cont.var, na.rm=TRUE)), buffer * (max(cont.var, na.rm=TRUE) - min(cont.var, na.rm=TRUE)))
    grid::pushViewport(grid::viewport(yscale=ylim, xscale=c(0.5, max(cat.var, na.rm=TRUE) + 0.5)))
    if(is.null(xpos)) cat.labels <- NULL
    .draw_axis(x=cat.var, y=cont.var, axis.pars=axis.pars, xpos=xpos, ypos=ypos, cat.labels=cat.labels, horiz=horiz, xlim=xlim, ylim=ylim, outer.rot=outer.rot)
    grid::popViewport(1)
    grid::pushViewport(grid::viewport(yscale=ylim, xscale=c(0.5, max(cat.var, na.rm=TRUE) + 0.5), clip=TRUE))
    if(type == "boxplot")   panel.bwplot(cat.var, cont.var, horizontal=horiz, col="black", pch="|", gp=grid::gpar(box.umbrella=list(col="black")), fill=box.fill)
    if(type == "stripplot") panel.stripplot(cat.var, cont.var, horizontal=horiz, jitter.data=stripplot.pars$jitter, col=stripplot.pars$col, cex=stripplot.pars$size, pch=stripplot.pars$pch)
    if(type == "violin")    .violin_panel(cat.var, cont.var, horizontal=horiz, col=box.fill)
  }
  grid::popViewport(1)
  trellis.par.set(name="box.rectangle", value=list(col=old.color))
  trellis.par.set(name="box.umbrella",  value=list(col=old.color))
  trellis.par.set(name="box.dot",       value=list(col=old.color))
  trellis.par.set(name="plot.symbol",   value=list(col=old.color))
}

#' @importFrom lattice "current.panel.limits" "llines" "ltext" "trellis.par.get"
.contour_panel <- function(x, y, zz, density.pars) {
  levs         <- pretty(zz, density.pars$nlevels)
  cLines       <- grDevices::contourLines(x, y, zz, levels=levs)
  text         <- trellis.par.get("add.text")
  tmp          <- list(col=density.pars$dcol, alpha=text$alpha, cex=text$cex/1.5, fontfamily=text$fontfamily, fontface=text$fontface, font=text$font)
  labels       <- c(tmp, list(labels=format(levs, trim=TRUE)))
  ux           <- sort(unique(x[!is.na(x)]))
  uy           <- sort(unique(y[!is.na(y)]))
  cpl          <- current.panel.limits(unit = "cm")
  asp          <- diff(cpl$ylim)/diff(cpl$xlim)
  ccount       <- 0
  for(val in cLines) {
    ccount     <- ccount  + 1
    llines(val, col=density.pars$dcol, identifier=paste("levelplot", "line", ccount, sep="."))
    if(length(val$x) > 5 && isTRUE(density.pars$show.labels)) {
      rx       <- range(ux)
      ry       <- range(uy)
      slopes   <- diff(val$y)/diff(val$x)
      depth    <- pmin(pmin(val$x - rx[1], rx[2] - val$x)/diff(rx), pmin(val$y - ry[1], ry[2] -val$y)/diff(ry))
      txtloc   <- min(which.max(depth), length(slopes))
      rotangle <- atan(asp * slopes[txtloc] * diff(rx)/diff(ry)) * 180/base::pi
      ltext(labels$labels[match(val$level, levs)], srt=rotangle, adj=c(0.5, 0), col=labels$col, alpha=labels$alpha, cex=labels$cex, font=labels$font, fontfamily=labels$fontfamily,
            fontface=labels$fontface, x=0.5 * (val$x[txtloc]  + val$x[txtloc + 1]), y=0.5 * (val$y[txtloc] + val$y[txtloc + 1]), identifier=paste("levelplot", "label", ccount, sep="."))
    }
  }
}

#' @importFrom matrixStats "colMeans2" "rowLogSumExps"
.density_panel <- function(dat, dimens, res, density.pars, axis.pars, xpos, ypos, buffer, outer.rot, bg, border, residuals) {
  par          <- res$parameters
  G            <- res$G
  mu           <- if(residuals) array(0, c(2, G)) else par$mean[dimens,, drop=FALSE]
  sigma        <- array(dim=c(2, 2, G))
  for(k in seq_len(G))  sigma[,,k] <- par$variance$sigma[dimens,dimens,k]
  xlim         <- range(dat[,1], na.rm=TRUE) + c(-buffer * (max(dat[,1], na.rm=TRUE) - min(dat[,1], na.rm=TRUE)), buffer * (max(dat[,1], na.rm=TRUE) - min(dat[,1], na.rm=TRUE)))
  ylim         <- range(dat[,2], na.rm=TRUE) + c(-buffer * (max(dat[,2], na.rm=TRUE) - min(dat[,2], na.rm=TRUE)), buffer * (max(dat[,2], na.rm=TRUE) - min(dat[,2], na.rm=TRUE)))
  lx           <- density.pars$grid.size[1]
  ly           <- density.pars$grid.size[2]
  x            <- .grid_1(n=lx, range=xlim, edge=TRUE)
  y            <- .grid_1(n=ly, range=ylim, edge=TRUE)
  xy           <- .grid_2(x, y)
  Vnull        <- is.null(res$parameters$Vinv)
  ltau         <- .mat_byrow(log(if(attr(res, "Gating")) colMeans2(res$z) else par$pro), nrow=nrow(xy), ncol=G + !Vnull)
  ltau         <- if(Vnull) ltau else ltau[,-(G + 1)]
  zz           <- matrix(exp(rowLogSumExps(MoE_dens("VVV", data=xy, mus=mu, sigs=sigma, log.tau=ltau, Vinv=NULL))), lx, ly)
  grid::pushViewport(grid::viewport(xscale=xlim, yscale=ylim))
  .draw_axis(x=x, y=y, axis.pars=axis.pars, xpos=xpos, ypos=ypos, cat.labels=NULL, horiz=NULL, xlim=xlim, ylim=ylim, outer.rot=outer.rot)
  grid::popViewport(1)
  grid::pushViewport(grid::viewport(xscale=xlim, yscale=ylim, clip=TRUE))
  grid::grid.rect(gp=grid::gpar(fill=bg, col=border))
  .contour_panel(x, y, zz, density.pars)
  grid::popViewport(1)
}

#' @importFrom lattice "panel.histogram"
.diag_panel <- function(x, varname, diag.pars, hist.col, axis.pars, xpos, ypos, buffer, index, outer.rot) {
  x         <- x[!is.na(x)]
  xlim      <- range(as.numeric(x), na.rm=TRUE) + c(-buffer * (max(as.numeric(x), na.rm=TRUE) - min(as.numeric(x), na.rm=TRUE)), buffer * (max(as.numeric(x), na.rm=TRUE) - min(as.numeric(x), na.rm=TRUE)))
  ylim      <- xlim
  grid::pushViewport(grid::viewport(xscale=xlim, yscale=ylim))
  what      <- if(is.factor(x)) x else as.numeric(x)
  .draw_axis(x=what, y=what, axis.pars=axis.pars, xpos=xpos, ypos=ypos, cat.labels=NULL, horiz=NULL, xlim=xlim, ylim=ylim, outer.rot=outer.rot)
  grid::popViewport(1)
  grid::pushViewport(grid::viewport(xscale=xlim, yscale=ylim, clip=TRUE))
  if(!diag.pars$show.hist) {
    grid::grid.rect()
    grid::grid.text(varname, 0.5, 0.5, gp=grid::gpar(fontsize=diag.pars$fontsize, fontface=2))
  }
  grid::popViewport(1)
  if(diag.pars$show.hist)  {
    if(!is.factor(x)) {
      grid::pushViewport(grid::viewport(xscale=xlim, yscale=c(0, 100), clip=TRUE))
      panel.histogram(as.numeric(x), breaks=NULL, type="percent", col=if(index == 1) hist.col[[index]] else hist.col[index])
    } else {
      grid::pushViewport(grid::viewport(xscale=c(min(as.numeric(x), na.rm=TRUE) - 1, max(as.numeric(x), na.rm=TRUE) + 1), yscale=c(0, 100), clip=TRUE))
      tabx <- table(x)
      show.counts    <- if(isTRUE(diag.pars$show.counts)) as.numeric(tabx) else FALSE
      .barchart_panel(seq_along(tabx), 100 * tabx/sum(tabx), horizontal=FALSE, col=if(index == 1) hist.col[[index]] else hist.col[index], show.counts=show.counts, fontsize=diag.pars$fontsize)
    }
    grid::grid.text(varname, 0.5, 0.9, gp=grid::gpar(fontsize=diag.pars$fontsize))
    grid::popViewport(1)
  }
}

#' @importFrom vcd "strucplot"
.mosaic_panel <- function(x, y, mosaic.pars, mosaic.outer, axis.pars, xpos, ypos, outer.rot, bg) {
  mosaic.pars$gp$fill  <- bg
  mosaic.pars$gp$col   <- mosaic.outer
  if(!is.null(xpos)       && !is.null(ypos)) {
    strucplot(table(y, x), margins=c(0, 0, 0, 0), newpage=FALSE, pop=FALSE, keep_aspect_ratio=FALSE, shade=mosaic.pars$shade, legend=FALSE, gp=mosaic.pars$gp,
              gp_args=mosaic.pars$gp_args, labeling_args=list(tl_labels=c(xpos, !ypos), gp_labels=mosaic.pars$gp_labels, varnames=c(FALSE, FALSE), rot_labels=rep(outer.rot, 2)))
  } else if(is.null(xpos) && is.null(ypos))  {
      strucplot(table(y, x), margins=c(0, 0, 0, 0), shade=mosaic.pars$shade, legend=FALSE, gp=mosaic.pars$gp, gp_args=mosaic.pars$gp_args, newpage=FALSE, pop=FALSE, keep_aspect_ratio=FALSE, labeling=NULL)
  } else if(is.null(xpos)) {
      strucplot(table(y, x), margins=c(0, 0, 0, 0), newpage=FALSE, pop=FALSE, keep_aspect_ratio=FALSE, shade=mosaic.pars$shade, legend=FALSE, gp=mosaic.pars$gp, gp_args=mosaic.pars$gp_args,
                labeling_args=list(labels=c(TRUE, FALSE), tl_labels=c(ypos, FALSE), gp_labels=mosaic.pars$gp_labels, varnames=c(FALSE, FALSE), rot_labels=rep(outer.rot, 2)))
  } else strucplot(table(y, x), margins=c(0, 0, 0, 0), newpage=FALSE, pop=FALSE, keep_aspect_ratio=FALSE, shade=mosaic.pars$shade, legend=FALSE, gp=mosaic.pars$gp, gp_args=mosaic.pars$gp_args,
                     labeling_args=list(labels=c(FALSE, TRUE), tl_labels=c(FALSE, !xpos), gp_labels=mosaic.pars$gp_labels, varnames=c(FALSE, FALSE), rot_labels=rep(outer.rot, 2)))
}

#' @importFrom lattice "panel.abline"
.scatter_panel <- function(x, y, type, scatter.pars, axis.pars, xpos, ypos, buffer, z, G, res, dimens, outer.rot, bg, uncertainty, mvn.col, mvn.type, border, residuals) {
  xlim    <- range(x, na.rm=TRUE) + c(-buffer * (max(x, na.rm=TRUE) - min(x, na.rm=TRUE)), buffer * (max(x, na.rm=TRUE) - min(x, na.rm=TRUE)))
  ylim    <- range(y, na.rm=TRUE) + c(-buffer * (max(y, na.rm=TRUE) - min(y, na.rm=TRUE)), buffer * (max(y, na.rm=TRUE) - min(y, na.rm=TRUE)))
  grid::pushViewport(grid::viewport(xscale=xlim, yscale=ylim))
  .draw_axis(x=x, y=y, axis.pars=axis.pars, xpos=xpos, ypos=ypos, cat.labels=NULL, horiz=NULL, xlim=xlim, ylim=ylim, outer.rot=outer.rot)
  grid::popViewport(1)
  grid::pushViewport(grid::viewport(xscale=xlim, yscale=ylim, clip=TRUE))
  grid::grid.rect(gp=grid::gpar(fill=bg, col=border))
  grid::grid.points(x, y, pch=scatter.pars$pch, size=if(all(is.na(uncertainty))) scatter.pars$size else uncertainty, gp=grid::gpar(col=scatter.pars$col))
  switch(type, ellipses= {
    mu    <- array(if(isTRUE(residuals)) 0 else res$parameters$mean[dimens,], c(2, G))
    sigma <- array(res$parameters$varianceX$sigma[dimens, dimens,], c(2, 2, G))
    for(g in seq_len(G)) .mvn2D_panel(mu=mu[,g], sigma=sigma[,,g], k=15, col=if(mvn.type == "inner") c("grey30", mvn.col[g], mvn.col[g]) else if(mvn.type == "outer") c(mvn.col[g], "grey30", "grey30") else if(mvn.type == "both") rep(mvn.col[g], 3))
  }, lm=   {
    for(g in seq_len(G)) {
      xy.lm <- stats::lm(y ~ x, weights=z[,g])
      panel.abline(xy.lm$coef[1], xy.lm$coef[2], col=scatter.pars$lci.col[g], lwd=1)
    }
  }, ci=   {
    for(g in seq_len(G)) {
      xy.lm <- stats::lm(y ~ x, weights=z[,g])
      xy    <- data.frame(x = seq(min(x, na.rm=TRUE), max(x, na.rm=TRUE), length.out=25))
      yhat  <- stats::predict(xy.lm, newdata=xy, interval="confidence")
      ci    <- data.frame(lower=yhat[, "lwr"], upper=yhat[, "upr"])
      panel.abline(xy.lm$coef[1], xy.lm$coef[2], col=scatter.pars$lci.col[g], lwd=1)
      grid::grid.lines(x=xy$x, y=ci$lower, default.units="native", gp=grid::gpar(col=scatter.pars$lci.col[g], lwd=1, lty=2))
      grid::grid.lines(x=xy$x, y=ci$upper, default.units="native", gp=grid::gpar(col=scatter.pars$lci.col[g], lwd=1, lty=2))
    }
  })
  grid::popViewport(1)
}

#' @importFrom lattice "current.panel.limits" "trellis.grobname"
.violin_panel  <- function(x, y, box.ratio = 1, box.width = box.ratio/(1 + box.ratio), horizontal = TRUE, alpha = plot.polygon$alpha,
                           border = plot.polygon$border, lty = plot.polygon$lty, lwd = plot.polygon$lwd, col = plot.polygon$col,
                           varwidth = FALSE, bw = NULL, adjust = NULL, kernel = NULL, window = NULL, width = NULL, n = 50,
                           from = NULL, to = NULL, cut = NULL, na.rm = TRUE, ..., identifier = "violin") {
  if(all(is.na(x) | is.na(y))) return()
  x           <- as.numeric(x)
  y           <- as.numeric(y)
  plot.polygon           <- trellis.par.get("plot.polygon")
  darg        <- list()
  darg$bw     <- bw
  darg$adjust <- adjust
  darg$kernel <- kernel
  darg$window <- window
  darg$width  <- width
  darg$n      <- n
  darg$from   <- from
  darg$to     <- to
  darg$cut    <- cut
  darg$na.rm  <- na.rm
  my.density  <- function(x) {
    answer    <- try(do.call(stats::density, c(list(x=x), darg)), silent=TRUE)
    if(!inherits(answer, "try-error"))     answer              else list(x=rep(x[1], 3), y=c(0, 1, 0))
  }
  numeric.list           <- if(horizontal) split(x, factor(y)) else split(y, factor(x))
  levels.fos  <- as.numeric(names(numeric.list))
  d.list      <- lapply(numeric.list, my.density)
  dx.list     <- lapply(d.list, "[[", "x")
  dy.list     <- lapply(d.list, "[[", "y")
  max.d       <- vapply(dy.list, max, numeric(1L))
  if(varwidth)   max.d[] <- max(max.d)
  cur.limits  <- current.panel.limits()
  xscale      <- cur.limits$xlim
  yscale      <- cur.limits$ylim
  height      <- box.width
  col         <- if(length(col) == 1)      rep(col, n)         else col
  if(horizontal) {
    for(i in seq_along(levels.fos)) {
      if(is.finite(max.d[i])) {
        grid::pushViewport(grid::viewport(y=grid::unit(levels.fos[i], "native"), height=grid::unit(height, "native"), yscale=c(max.d[i] * c(-1, 1)), xscale=xscale))
        grid::grid.polygon(x=c(dx.list[[i]], rev(dx.list[[i]])), y=c(dy.list[[i]], -rev(dy.list[[i]])), default.units="native",
                           name=trellis.grobname(identifier, type="panel", group=0), gp=grid::gpar(fill=col[i], col=border, lty=lty, lwd=lwd, alpha=alpha))
        grid::popViewport()
      }
    }
  }
  else {
    for(i in seq_along(levels.fos)) {
      if(is.finite(max.d[i])) {
        grid::pushViewport(grid::viewport(x=grid::unit(levels.fos[i], "native"), width=grid::unit(height, "native"), xscale=c(max.d[i] * c(-1, 1)), yscale=yscale))
        grid::grid.polygon(y=c(dx.list[[i]], rev(dx.list[[i]])),
                           x=c(dy.list[[i]], -rev(dy.list[[i]])), default.units="native",
                           name=trellis.grobname(identifier, type="panel", group=0), gp=grid::gpar(fill=col[i], col=border, lty=lty, lwd=lwd, alpha=alpha))
        grid::popViewport()
      }
    }
  }
  invisible()
}

.bar_code       <- function(x, outer.margins = list(bottom = grid::unit(2, "lines"), left = grid::unit(2, "lines"), top = grid::unit(2, "lines"), right = grid::unit(2, "lines")),
                            horizontal = TRUE, xlim = NULL, nint = 0, main = "", xlab = "", labelloc = TRUE, axisloc = TRUE, labelouter = FALSE,  newpage = TRUE,
                            fontsize = 9, ptsize = grid::unit(0.25, "char"), ptpch = 1, bcspace = NULL, use.points = FALSE, buffer = 0.02, log = FALSE, outerbox = "black") {
  if(!is.null(labelloc)) {
    if(labelloc == "right" || labelloc == "top")
      labelloc  <- FALSE
    if(labelloc == "left"  || labelloc == "bottom")
      labelloc  <- TRUE
  }
  if(!is.null(axisloc))  {
    if(axisloc  == "right" || axisloc  == "top")
      axisloc   <- FALSE
    if(axisloc  == "left"  || axisloc  == "bottom")
      axisloc   <- TRUE
  }
  x             <- if(is.vector(x) && !is.list(x))          list(x) else x
  names(x)      <- if(is.null(names(x))) as.character(seq_along(x)) else names(x)
  x             <- if(is.matrix(x))                as.data.frame(x) else x
  xlaboffset    <- grid::unit(2.5, "lines")
  if(newpage)      grid::grid.newpage()
  grid::grid.text(main, 0.5, grid::unit(1, "npc") - grid::unit(1, "lines"), gp=grid::gpar(fontface="bold"))
  if(!is.null(axisloc) && !axisloc && main != ""  && !labelouter) outer.margins$top <- outer.margins$top + grid::unit(2, "lines")
  if(!is.null(axisloc) && xlab != ""  && !labelouter)   {
    if(axisloc)  {
      if(horizontal) { outer.margins$bottom       <- outer.margins$bottom + grid::unit(1.5, "lines")
      } else outer.margins$top     <- outer.margins$top + grid::unit(1.5, "lines")
    } else       {
      if(horizontal) { outer.margins$top          <- outer.margins$top    + grid::unit(1.5, "lines")
      } else     {
        outer.margins$left  <- outer.margins$left  + grid::unit(1.5, "lines")
        outer.margins$right <- outer.margins$right - grid::unit(1.5, "lines")
      }
    }
  }
  if(horizontal) {
    thisangle   <- 0
    thisjust    <- c("left", "bottom")
  } else         {
    thisangle   <- 90
    thisjust    <- c("left", "top")
    grid::pushViewport(grid::viewport(x=0, y=0, width=grid::convertHeight(grid::unit(1, "npc"), "inches"), height=grid::convertWidth(grid::unit(1, "npc"), "inches"), just=c("left", "bottom")))
  }
  outer.margins <- if(labelouter) list(bottom=grid::unit(0, "lines"), left=grid::unit(0, "lines"), top=grid::unit(0, "lines"), right=grid::unit(0, "lines")) else outer.margins
  vp.main       <- grid::viewport(x=outer.margins$left, y=outer.margins$bottom,
                                  width=grid::unit(1,  "npc") - outer.margins$right - outer.margins$left,
                                  height=grid::unit(1, "npc") - outer.margins$top   - outer.margins$bottom,
                                  just=thisjust, angle=thisangle, name="main", clip="off")
  grid::pushViewport(vp.main)
  grid::grid.rect(gp=grid::gpar(col=outerbox))
  .barcode_panel(x, horizontal=horizontal, nint=nint, xlim=xlim, labelloc=labelloc, labelouter=labelouter,
                 fontsize=fontsize, ptsize=ptsize, bcspace=bcspace, use.points=use.points, xlab=xlab,
                 xlaboffset=xlaboffset, axisloc=axisloc, buffer=0.02, log=log)
  grid::popViewport(1)
  if(!horizontal) grid::popViewport(1)
}

#' @importFrom lattice "current.panel.limits" "panel.abline" "panel.rect" "panel.text" "trellis.par.get"
.barchart_panel <- function(x, y, box.ratio = 1, box.width = box.ratio/(1 + box.ratio),
                            horizontal = TRUE, origin = NULL, reference = TRUE, stack = FALSE,
                            groups = NULL, col = if(is.null(groups)) plot.polygon$col else superpose.polygon$col,
                            border = if(is.null(groups)) plot.polygon$border else superpose.polygon$border,
                            lty = if(is.null(groups)) plot.polygon$lty else superpose.polygon$lty,
                            lwd = if(is.null(groups)) plot.polygon$lwd else superpose.polygon$lwd,
                            show.counts = FALSE, ..., fontsize = 9, identifier = "barchart") {
  plot.polygon      <- trellis.par.get("plot.polygon")
  superpose.polygon <- trellis.par.get("superpose.polygon")
  reference.line    <- trellis.par.get("reference.line")
  keep          <- (function(x, y, groups, subscripts, ...) {
    !is.na(x) & !is.na(y) & if(is.null(groups)) TRUE else !is.na(groups[subscripts]) })(x=x, y=y, groups=groups, ...)
  if(!any(keep))   return()
  x             <- as.numeric(x[keep])
  y             <- as.numeric(y[keep])
  if(!is.null(groups))    {
    groupSub    <- function(groups, subscripts, ...) groups[subscripts[keep]]
    if(!is.factor(groups)) groups <- factor(groups)
    nvals       <- nlevels(groups)
    groups      <- as.numeric(groupSub(groups, ...))
  }
  if(horizontal) {
    if(is.null(groups))   {
      if(is.null(origin)) {
        origin  <- current.panel.limits()$xlim[1]
        reference   <- FALSE
      }
      height    <- box.width
      if(reference)  panel.abline(v=origin, col=reference.line$col, lty=reference.line$lty, lwd=reference.line$lwd, identifier=paste(identifier, "abline", sep="."))
      panel.rect(x=rep(origin, length(y)), y=y, height=rep(height, length(y)), width=x - origin, border=border, col=col, lty=lty, lwd=lwd, just=c("left", "centre"), identifier=identifier)
    } else if(stack)      {
      if(!is.null(origin) && origin != 0)             warning("'origin' forced to 0 for stacked bars", call.=FALSE)
      col       <- rep(col,    length.out=nvals)
      border    <- rep(border, length.out=nvals)
      lty       <- rep(lty,    length.out=nvals)
      lwd       <- rep(lwd,    length.out=nvals)
      height    <- box.width
      if(reference) panel.abline(v=origin, col=reference.line$col, lty=reference.line$lty, lwd=reference.line$lwd, identifier=paste(identifier, "abline", sep="."))
      for(i in unique(y)) {
        ok      <- y == i
        ord     <- sort.list(groups[ok])
        pos     <- x[ok][ord] > 0
        nok     <- sum(pos, na.rm=TRUE)
        if(nok   > 0) panel.rect(x=cumsum(c(0, x[ok][ord][pos][-nok])), y=rep(i, nok), col=col[groups[ok][ord][pos]], border=border[groups[ok][ord][pos]], lty=lty[groups[ok][ord][pos]],
                                 lwd=lwd[groups[ok][ord][pos]], height=rep(height, nok), width=x[ok][ord][pos], just=c("left", "centre"), identifier=paste(identifier, "pos", i, sep="."))
        neg     <- x[ok][ord] < 0
        nok     <- sum(neg, na.rm=TRUE)
        if(nok   > 0) panel.rect(x=cumsum(c(0, x[ok][ord][neg][-nok])), y=rep(i, nok), col=col[groups[ok][ord][neg]], border=border[groups[ok][ord][neg]], lty=lty[groups[ok][ord][neg]],
                                 lwd=lwd[groups[ok][ord][neg]], height=rep(height, nok), width=x[ok][ord][neg], just=c("left", "centre"), identifier=paste(identifier, "neg", i, sep="."))
      }
    } else {
      if(is.null(origin)) {
        origin  <- current.panel.limits()$xlim[1]
        reference   <- FALSE
      }
      col       <- rep(col,    length.out=nvals)
      border    <- rep(border, length.out=nvals)
      lty       <- rep(lty,    length.out=nvals)
      lwd       <- rep(lwd,    length.out=nvals)
      height    <- box.width/nvals
      if(reference) panel.abline(v=origin, col=reference.line$col, lty=reference.line$lty, lwd=reference.line$lwd, identifier=paste(identifier, "abline", sep="."))
      for(i in unique(y)) {
        ok      <- y == i
        nok     <- sum(ok, na.rm=TRUE)
        panel.rect(x=rep(origin, nok), y=(i + height * (groups[ok] - (nvals + 1)/2)), col=col[groups[ok]], border=border[groups[ok]], lty=lty[groups[ok]],
                   lwd=lwd[groups[ok]], height=rep(height, nok), width=x[ok] - origin, just=c("left", "centre"), identifier=paste(identifier, "y", i, sep="."))
      }
    }
  } else {
    if(is.null(groups))   {
      if(is.null(origin)) {
        origin  <- current.panel.limits()$ylim[1]
        reference   <- FALSE
      }
      width     <- box.width
      y.fix     <- y - origin
      fix.y     <- ifelse(!identical(show.counts, FALSE), 80, 90)
      y.fix     <- if(max(y.fix) > fix.y) (y.fix * fix.y)/max(y.fix) else y.fix
      if(reference) panel.abline(h=origin, col=reference.line$col, lty=reference.line$lty, lwd=reference.line$lwd, identifier=paste(identifier, "abline", sep="."))
      panel.rect(x=x, y=rep(origin, length(x)), col=col, border=border, lty=lty, lwd=lwd, width=rep(width, length(x)), height=y.fix, just=c("centre", "bottom"), identifier=identifier)
      if(!identical(show.counts, FALSE)) {
        panel.text(x=x, y=y.fix, label=show.counts, adj=c(0.5, -0.5), identifier=identifier, gp=grid::gpar(fontsize=fontsize), cex=0.8)
      }
    } else if(stack) {
      if(!is.null(origin) && origin != 0)             warning("'origin' forced to 0 for stacked bars", call.=FALSE)
      col       <- rep(col,    length.out=nvals)
      border    <- rep(border, length.out=nvals)
      lty       <- rep(lty,    length.out=nvals)
      lwd       <- rep(lwd,    length.out=nvals)
      width     <- box.width
      if(reference) panel.abline(h=origin, col=reference.line$col, lty=reference.line$lty, lwd=reference.line$lwd, identifier=paste(identifier, "abline", sep="."))
      for(i in unique(x)) {
        ok      <- x == i
        ord     <- sort.list(groups[ok])
        pos     <- y[ok][ord] > 0
        nok     <- sum(pos, na.rm=TRUE)
        if(nok   > 0) panel.rect(x=rep(i, nok), y=cumsum(c(0, y[ok][ord][pos][-nok])), col=col[groups[ok][ord][pos]], border=border[groups[ok][ord][pos]], lty=lty[groups[ok][ord][pos]],
                                 lwd=lwd[groups[ok][ord][pos]], width=rep(width, nok), height=y[ok][ord][pos], just=c("centre", "bottom"), identifier=paste(identifier, "pos", i, sep="."))
        neg     <- y[ok][ord] < 0
        nok     <- sum(neg, na.rm=TRUE)
        if(nok   > 0) panel.rect(x=rep(i, nok), y=cumsum(c(0, y[ok][ord][neg][-nok])), col=col[groups[ok][ord][neg]], border=border[groups[ok][ord][neg]], lty=lty[groups[ok][ord][neg]],
                                 lwd=lwd[groups[ok][ord][neg]], width=rep(width, nok), height=y[ok][ord][neg], just=c("centre", "bottom"), identifier=paste(identifier, "neg", i, sep="."))
      }
    } else {
      if(is.null(origin)) {
        origin  <- current.panel.limits()$ylim[1]
        reference   <- FALSE
      }
      col       <- rep(col,    length.out=nvals)
      border    <- rep(border, length.out=nvals)
      lty       <- rep(lty,    length.out=nvals)
      lwd       <- rep(lwd,    length.out=nvals)
      width     <- box.width/nvals
      if(reference) panel.abline(h=origin, col=reference.line$col, lty=reference.line$lty, lwd=reference.line$lwd, identifier=paste(identifier, "abline", sep="."))
      for(i in unique(x)) {
        ok      <- x == i
        nok     <- sum(ok, na.rm=TRUE)
        panel.rect(x=(i + width * (groups[ok] - (nvals + 1)/2)), y=rep(origin, nok), col=col[groups[ok]], border=border[groups[ok]], lty=lty[groups[ok]],
                   lwd=lwd[groups[ok]], width=rep(width, nok), height=y[ok] - origin, just=c("centre", "bottom"), identifier=paste(identifier, "x", i, sep="."))
      }
    }
  }
}

.barcode_panel  <- function(x, horizontal = TRUE, xlim = NULL, labelloc = TRUE, axisloc = TRUE, labelouter = FALSE,
                            nint = 0, fontsize = 9, ptsize = grid::unit(0.25, "char"), ptpch = 1, bcspace = NULL, xlab = "",
                            xlaboffset = grid::unit(2.5, "lines"), use.points = FALSE, buffer = 0.02, log = FALSE) {
  if(!is.list(x))                                     stop("x must be a list", call.=FALSE)
  K             <- length(x)
  for(i in 1:K)    x[[i]]   <- x[[i]][!is.na(x[[i]])]
  maxct         <- 0
  ux     <- unlist(x)
  if(is.null(xlim)) {
    minx <- min(ux) - buffer * (max(ux) - min(ux))
    maxx <- max(ux) + buffer * (max(ux) - min(ux))
  } else  {
    minx <- xlim[1]
    maxx <- xlim[2]
  }
  xleft  <- grid::unit(1, "strwidth", names(x)[1])
  for(i in 1:K)     {
    y    <- x[[i]]
    if(length(y) > 0) {
      z  <- if(nint > 0) graphics::hist(y, breaks=pretty(ux, n=nint), plot=FALSE)$counts else table(y)
      maxct     <- max(maxct, max(z))
      xleft     <- max(xleft, grid::unit(1, "strwidth", names(x)[i]))
    }
  }
  maxct  <- ifelse(log, log(maxct + 3), maxct + 3)
  xleft  <- 1.05 * xleft
  if(is.null(labelloc) || !labelloc)    {
    xright      <- xleft
    xleft       <- grid::unit(0, "npc")
    xtextloc    <- grid::unit(1, "npc") - xright
    xtextalign  <- "left"
  } else  {
    xright      <- grid::unit(0, "npc")
    xtextloc    <- xleft
    xtextalign  <- "right"
  }
  if(labelouter) {
    xleft       <- grid::unit(0, "npc")
    xright      <- grid::unit(0, "npc")
    xtextloc    <- grid::unit(ifelse(is.null(labelloc) || !labelloc, 1.02, -1.02), "npc")
  }
  if(is.null(bcspace)) bcspace <- max(0.2, 1.5/(maxct + 1))
  grid::pushViewport(grid::viewport(x=xleft, y=grid::unit(0, "npc"), width=grid::unit(1, "npc") - xleft - xright,
                     height=grid::unit(1, "npc"), xscale=c(minx, maxx), just=c("left", "bottom")))
  if(!is.null(axisloc)) {
    grid::grid.xaxis(main=axisloc, gp=grid::gpar(fontsize = fontsize))
    if(axisloc)  { grid::grid.text(xlab, x=grid::unit(0.5, "npc"), y = grid::unit(0, "npc") - xlaboffset)
    } else         grid::grid.text(xlab, x=grid::unit(0.5, "npc"), y = grid::unit(1, "npc") + xlaboffset)
  }
  grid::popViewport(1)
  for(i in 1:K)  {
    y    <- x[[i]]
    if(!is.null(labelloc)) grid::grid.text(names(x)[i], x=xtextloc, y=grid::unit((i - 1)/K, "npc") + 0.5 * grid::unit(1/K, "npc"), just=xtextalign, gp=grid::gpar(fontsize=fontsize))
    if(nint > 0) {
     zhist <- graphics::hist(y, breaks=pretty(unlist(x), n=nint), plot=FALSE)
     z     <- zhist$counts
     mids  <- zhist$mids
    } else  {
     z     <- table(y)
     mids  <- as.numeric(names(z))
    }
    if(length(mids) > 0) {
      vp.barcode <- grid::viewport(x=xleft, y=grid::unit((i - 1)/K, "npc") + grid::unit(0.05/K, "npc"), width=grid::unit(1, "npc") - xleft - xright, height=grid::unit(1/K, "npc") * bcspace - grid::unit(0.05/K, "npc"), xscale=c(minx, maxx), yscale=c(0, 1), just=c("left", "bottom"), name="barcode", clip="off")
      grid::pushViewport(vp.barcode)
      grid::grid.segments(grid::unit(mids[z > 0], "native"), 0, grid::unit(mids[z > 0], "native"), 1)
      grid::popViewport(1)
      vp.hist    <- grid::viewport(x=xleft, y=grid::unit((i - 1)/K, "npc") + grid::unit(1/K, "npc") * bcspace, width=grid::unit(1, "npc") - xright - xleft, height=grid::unit(1/K, "npc") - grid::unit(1/K, "npc") * bcspace, xscale=c(minx, maxx), yscale=c(0, 1), just=c("left", "bottom"), name="hist", clip="off")
      grid::pushViewport(vp.hist)
      vp.buffer  <- grid::viewport(x=0, y=0.05, width=1, height=0.9, just=c("left", "bottom"), xscale=c(minx, maxx), yscale=c(0, 1))
      grid::pushViewport(vp.buffer)
      for(j in 1:length(z)) {
        if(z[j] > 1) {
          xx     <- rep(mids[j], z[j] - 1)
          yy     <- if(log) (log(2 + 1:(z[j] - 1)))/maxct else (1:(z[j] - 1))/maxct
          if(use.points) {
            grid::grid.points(grid::unit(xx, "native"), yy, pch=ptpch, size=ptsize)
          } else  {
            yy   <- if(log) c(yy, log(2 + z[j])/maxct)    else c(yy, (z[j])/maxct)
            grid::grid.segments(grid::unit(mids[j], "native"), grid::unit(1/maxct, "npc"), grid::unit(mids[j], "native"), grid::unit(max(yy), "npc"))
          }
        }
      }
      grid::popViewport(2)
    }
  }
}

#' @method plot mclustAIC
#' @importFrom mclust "plot.mclustBIC"
#' @export
plot.mclustAIC  <- function (x, ylab = "AIC", ...) {
  plot.mclustBIC(x, ylab = ylab, ...)
}

#' @method plot mclustDF
#' @importFrom mclust "plot.mclustBIC"
#' @export
plot.mclustDF   <- function (x, ylab = "DF", ...)  {
  plot.mclustBIC(x, ylab = ylab, ...)
}
