# Run regenerative Markov chain on slaves and return results

runMarkovChainRegenIP <- function(genNextState, genNextStateArgList=NULL, 
                       initialPoint, 
                       regenProbs, regenProbsArgList=NULL,
                       transDens, transDensArgList=NULL,
                       smallMeasure, smallMeasureArgList=NULL,
                       smallFn, smallFnArgList=NULL, numericParams,
                       nsim, nTours) {

  # Check that slaves are present.
  commSize <- mpi.comm.size()
  if(commSize < 2) 
    stop("No slaves running. Please spawn slaves before calling 
          runMarkovChainRegenIP().\n") else
    cat("Found", commSize - 1, "slaves running.\n", sep=" ")

  # Check that exactly one of nsim or nTours is supplied
  nsimMissing <- missing(nsim)
  nToursMissing <- missing(nTours)
  if(nsimMissing & nToursMissing)
    stop("Either nsim or nTours has to be supplied.\n") else
  if(!nsimMissing & !nToursMissing)
    stop("Either nsim or nTours has to be supplied, but not both.\n")
  if(nsimMissing)
    mpi.bcast.Robj2slave(nTours) else
    mpi.bcast.Robj2slave(nsim)

  # Check that either regenProbs, or all of smallFn, smallMeasure and 
  # transDens are supplied.
  regenProbsMissing <- missing(regenProbs)
  if(regenProbsMissing){
    transDensMissing <- missing(transDens)
    smallFnMissing <- missing(smallFn)
    smallMeasureMissing <- missing(smallMeasure)
    if(transDensMissing || smallFnMissing || smallMeasureMissing)
      stop("Either regenProbs(), or ALL of transDens(), smallFn() and 
            smallMeasure() have to be supplied.\n")
  }

  # check that the number of numeric parameters is known
  if(missing(numericParams))
    numericParams <- 1:length(initialPoint)
  mpi.bcast.Robj2slave(numericParams)

# CHECKS ON genNextState AND ITS ARGUMENTS. ######################
  # check that the genNextState argument is a function.
  if(class(genNextState)!="function")
    stop("genNextState() has to be of 'function' class.\n")

  # check that genNextStateArgList is not missing and is a list.
  if(!is.null(genNextStateArgList)){
    if(class(genNextStateArgList)!="list") 
      stop("genNextStateArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(genNextState) <- c(genNextStateArgList, x=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if one-step generation function works .. ")
  firstPointGenerated <- genNextState(x=initialPoint)
  if(sum(is.na(firstPointGenerated)) > 0) 
    stop("NA's generated in genNextState().\n")

  # check length of output argument
  if(length(firstPointGenerated)!=length(initialPoint))
    stop("Input/output states for genNextState() do not match in length.\n")
# END OF CHECKS ON genNextState AND ITS ARGUMENTS. ###############
  cat("genNextState() OK.\n")
  mpi.bcast.Robj2slave(genNextState)

  if(regenProbsMissing) {
# CHECKS ON transDens AND ITS ARGUMENTS. ######################
  # check that the genNextState argument is a function.
  if(class(transDens)!="function")
    stop("transDens() has to be of 'function' class.\n")

  # check that transDensArgList is not missing and is a list.
  if(!is.null(transDensArgList)){
    if(class(transDensArgList)!="list")
      stop("transDensArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(transDens) <- c(transDensArgList, x=1, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if transition density function works .. ")
  pxy <- transDens(x=initialPoint, y=firstPointGenerated)
  if(is.na(pxy))
    stop("NA's generated by transition density function.\n")

  # check length of output argument
  if(length(pxy)!=1)
    stop("Transition density function should return a real scalar.\n")

# END OF CHECKS ON transDens AND ITS ARGUMENTS. ###############
  cat("transDens() OK.\n")
  mpi.bcast.Robj2slave(transDens)

# CHECKS ON smallMeasure AND ITS ARGUMENTS. ######################
  # check that the smallMeasure argument is a function.
  if(class(smallMeasure)!="function")
    stop("smallMeasure() has to be of 'function' class.\n")

  # check that smallMeasureArgList is not missing and is a list.
  if(!is.null(smallMeasureArgList)){
    if(class(smallMeasureArgList)!="list")
      stop("smallMeasureArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(smallMeasure) <- c(smallMeasureArgList, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if small measure function works .. ")
  nuY <- smallMeasure(y=firstPointGenerated)
  if(is.na(nuY))
    stop("NA's generated by small measure function.\n")

  # check length of output argument
  if(length(nuY)!=1)
    stop("Small measure function should return a real scalar.\n")
# END OF CHECKS ON smallMeasure AND ITS ARGUMENTS. ###############
  cat("smallMeasure() OK.\n")
  mpi.bcast.Robj2slave(smallMeasure)

# CHECKS ON smallFn AND ITS ARGUMENTS. ######################
  # check that the smallFn argument is a function.
  if(class(smallFn)!="function")
    stop("smallFn() has to be of 'function' class.\n")

  # check that smallFnArgList is not missing and is a list.
  if(!is.null(smallFnArgList)){
    if(class(smallFnArgList)!="list")
      stop("smallFnArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(smallFn) <- c(smallFnArgList, x=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if small function works .. ")
  sX <- smallFn(x=firstPointGenerated)
  if(is.na(sX))
    stop("NA's generated by small function.\n")

  # check length of output argument
  if(length(sX)!=1)
    stop("Small function should return a real scalar.\n")
# END OF CHECKS ON smallMeasure AND ITS ARGUMENTS. ###############
  cat("smallFn () OK.\n")
  mpi.bcast.Robj2slave(smallFn)
  }
  else {
# CHECKS ON regenProbs AND ITS ARGUMENTS. ######################
  # check that the regenProbs argument is a function.
  if(class(regenProbs)!="function")
    stop("regenProbs() has to be of 'function' class.\n")

  # When regenProbsArgList is not NULL, check that it is a list.
  if(!is.null(regenProbsArgList)){
    if(class(regenProbsArgList)!="list")
      stop("regenProbsArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(regenProbs) <- c(regenProbsArgList, x=1, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if regenProbs function works .. ")
  firstRegenProb <- regenProbs(x=initialPoint, y=firstPointGenerated)
  if(is.na(firstRegenProb))
    stop("NA's generated by regenProbs function with x=", initialPoint, 
        "and y=", firstPointGenerated, ".\n", sep=" ")

  # check length of output argument
  if(length(firstRegenProb)!=1)
    stop("regenProbs function should return a real scalar.\n")
# END OF CHECKS ON regenProbs AND ITS ARGUMENTS. ###############
  cat("regenProbs() OK.\n")
  mpi.bcast.Robj2slave(regenProbs)
  }
  cat("All checks done.\n\n")

  if(regenProbsMissing) {
    if(nsimMissing) {
    A <- mpi.remote.exec(runMarkovChainRegenSslave, genNextState=genNextState, 
                         initialPoint = initialPoint, 
                         transDens = transDens, 
                         smallMeasure = smallMeasure,
                         smallFn = smallFn, 
                         numericParams=numericParams, nTours=nTours)
    } 
    else {
    A <- mpi.remote.exec(runMarkovChainRegenSslave, genNextState=genNextState, 
                         initialPoint = initialPoint, 
                         transDens = transDens, 
                         smallMeasure = smallMeasure,
                         smallFn = smallFn, 
                         numericParams=numericParams, nsim=nsim)

    }
  }
  else {
    if(nsimMissing) {
    A <- mpi.remote.exec(runMarkovChainRegenSslave, genNextState=genNextState, 
                         initialPoint = initialPoint, 
                         regenProbs = regenProbs,
                         numericParams=numericParams, nTours=nTours)
    } 
    else {
    A <- mpi.remote.exec(runMarkovChainRegenSslave, genNextState=genNextState, 
                         initialPoint = initialPoint, 
                         regenProbs = regenProbs,
                         numericParams=numericParams, nsim=nsim)
    }
  }

  if(nsimMissing) {
    tourMatrix <- NULL
    for(ii in 1:(commSize-1)) {
      tmpTours <- nrow(A[[ii]])
      cat(tmpTours, "tours from slave", ii, ".\n", sep=" ")
      tourMatrix <- rbind(tourMatrix, A[[ii]])
    }
    return(tourMatrix)
  }
  else {
    tourMatrix <- NULL
    for(ii in 1:(commSize - 1)) {
      if(!("N" %in% colnames(A[[ii]]))) {
        cat("No tours from slave", ii, ".\n", sep=" ")
        next
      }
      tmpTours <- nrow(A[[ii]])
      cat(tmpTours, "tours from slave", ii, ".\n", sep=" ")
      tourMatrix <- rbind(tourMatrix, A[[ii]])
    }
    if(is.null(tourMatrix)) {
      cat("Returning raw output from slave 1.\n")
      return(A[[1]])
    }
    else 
      return(tourMatrix)
  }

##  mpi.bcast.Robj2slave(genNextStateArgList)
##  mpi.bcast.Robj2slave(initialPoint)
##  mpi.bcast.Robj2slave(nsim)
##
##  A <-  mpi.remote.exec(runMarkovChainNoRegenS, genNextState, 
##                        genNextStateArgList, initialPoint, nsim)
##  A
}
