% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HCRs.R
\name{HCR_ramp}
\alias{HCR_ramp}
\alias{HCR40_10}
\alias{HCR60_20}
\title{Linearly ramped harvest control rules}
\usage{
HCR_ramp(Assessment, reps = 1, LRP, TRP, rel_min = 0, rel_max = 1,
  RP_type = c("SSB_SSB0", "SSB_SSBMSY"), ...)

HCR40_10(Assessment, reps = 1, ...)

HCR60_20(Assessment, reps = 1, ...)
}
\arguments{
\item{Assessment}{An object of class \linkS4class{Assessment} with estimates of
FMSY or UMSY, vulnerable biomass, and spawning biomass depletion in terminal year.}

\item{reps}{The number of stochastic samples of the TAC recommendation.}

\item{LRP}{Numeric, the limit reference point.}

\item{TRP}{Numeric, the target reference point.}

\item{rel_min}{The relative maximum value (e.g. a multiple of FMSY) if \code{Brel < LRP}.}

\item{rel_max}{The relative maximum value (e.g. a multiple of FMSY) if \code{Brel > TRP}.}

\item{RP_type}{The reference point metric for TRP and LRP (\code{"SSB_SSB0"} for spawning depletion by default,
or \code{"SSB_SSBMSY"} for spawning biomass relative to MSY).}

\item{...}{Miscellaneous arguments.}
}
\value{
An object of class \linkS4class{Rec} with the TAC recommendation.
}
\description{
An output control rule with a ramp that reduces the TAC recommendation linearly
with respect to fishing mortality (F) or harvest rate (U) when the relative biomass (i.e.,
spawning depletion or spawning biomass relative to that at MSY) is less than
the target reference point (TRP). The TAC reduction is linearly reduced with respect to F
to a minimum value when the relative biomass is less than the limit reference point (LRP). For example,
the TRP and LRP for spawning depletion is 0.4 and 0.1, respectively, in the 40-10 control rule.
Class HCR objects are typically used with function \link{make_MP}.
}
\details{
\code{HCR_ramp} is the generic ramped-HCR function where user specifies LRP, TRP, and
relative biomass metric, as well as minimum and maximum values for adjusting the fishing mortality.

\code{HCR40_10} is a common U.S. west coast control rule (LRP and TRP of 0.1 and 0.4 spawning depletion,
respectively), while \code{HCR60_20} is more conservative than 40-10, with LRP and TRP of 0.2 and 0.6
spawning depletion, respectively).
}
\examples{
# 40-10 linear ramp
Brel <- seq(0, 1, length.out = 200)
plot(Brel, HCRlin(Brel, 0.1, 0.4), xlab = "Estimated SSB/SSB0",
ylab = "Prescribed F relative to FMSY", main = "40-10 harvest control rule",
type = "l", col = "blue")
abline(v = c(0.1, 0.4), col = "red", lty = 2)

# create a 40-10 MP to run in closed-loop MSE
DD_40_10 <- make_MP(DD_TMB, HCR40_10)

# Alternatively,
DD_40_10 <- make_MP(DD_TMB, HCR_ramp, LRP = 0.1, TRP = 0.4)

# An SCA with LRP and TRP at 0.4 and 0.8, respectively, of SSB/SSBMSY
SCA_80_40 <- make_MP(SCA, HCR_ramp, LRP = 0.4, TRP = 0.8, RP_type = "SSB_SSBMSY")

# A conservative HCR that fishes at 75\% of FMSY at B > 80\% BMSY but only reduces F
# to 10\% of FMSY if B < 40\% BMSY.
SCA_conservative <- make_MP(SCA, HCR_ramp, LRP = 0.4, TRP = 0.8, rel_max = 0.75,
rel_min = 0.1, RP_type = "SSB_SSBMSY")

# Figure of this conservative HCR
Brel <- seq(0, 1, length.out = 200)
Frel <- HCRlin(Brel, 0.4, 0.8, rel_max = 0.75, rel_min = 0.1)
plot(Brel, Frel, xlab = "Estimated SSB/SSB_MSY", ylab = "Prescribed F relative to FMSY",
type = "l", col = "blue")
abline(v = c(0.4, 0.8), col = "red", lty = 2)

\dontrun{
myOM <- DLMtool::runMSE(DLMtool::testOM, MPs = c("FMSYref", "DD_40_10"))
}
}
\references{
Deroba, J.J. and Bence, J.R. 2008. A review of harvest policies: Understanding relative
performance of control rules. Fisheries Research 94:210-223.

Edwards, C.T.T. and Dankel, D.J. (eds.). 2016. Management Science in Fisheries: an introduction
to simulation methods. Routledge, New York, NY. 460 pp.

Punt, A. E, Dorn, M. W., and Haltuch, M. A. 2008. Evaluation of threshold management strategies
for groundfish off the U.S. West Coast. Fisheries Research 94:251-266.

Restrepo, V.R. and Power, J.E. 1999. Precautionary control rules in US fisheries
management: specification and performance. ICES Journal of Marine Science 56:846-852.
}
\seealso{
\link{HCR_MSY} \link{HCRlin} \link{make_MP}
}
\author{
Q. Huynh & T. Carruthers
}
