% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getTile.R
\name{getTile}
\alias{getTile}
\title{Get MODIS Tile ID(s)}
\usage{
getTile(x = NULL, tileH = NULL, tileV = NULL)
}
\arguments{
\item{x}{Extent information, see Details. Ignored if \code{tileH} and
\code{tileV} are specified.}

\item{tileH, tileV}{\code{numeric} or \code{character}. Horizontal and 
vertical tile number(s) of the 
\href{https://nsidc.org/data/docs/daac/mod10_modis_snow/landgrid.html}{MODIS Sinusoidal grid}
(e.g., \code{tileH = 1:5}). If specified, no cropping is performed and the 
full tile(s) (if more than one then also mosaicked) is (are) processed!}
}
\value{
A \code{MODISextent} object.
}
\description{
Get MODIS tile ID(s) for a specific geographic area.
}
\details{
If \code{x} is of class (see Examples for use cases)
\tabular{ll}{
  \code{missing}:\cr
  \tab If 'tileH,tileV' are specified, 'x' will be ignored. If no such tile 
  indices are provided and 'x' is missing, a viewer window pops up that 
  allows interactive tile selection from the global MODIS Sinusoidal grid.\cr
  \cr 
  \code{character}:\cr
  \tab Either the country name of a \code{map} object (see \code{\link{map}})
  or a valid file path of a raster image or ESRI shapefile (shp). The former 
  approach also supports pattern matching via regular expressions.\cr
  \cr
  \code{Raster*}:\cr
  \tab Spatial extent, resolution, and projection of the specified 
  \code{Raster*} are determined automatically. This information is used by 
  \code{\link{runGdal}} to create perfectly matching files. If the 
  \code{Raster*} comes with no valid CRS, 
  \href{http://spatialreference.org/ref/epsg/wgs-84/}{EPSG:4326} is assumed.\cr
  \cr
  \code{Extent}:\cr
  \tab Boundary coordinates from \code{Extent} objects are assumed to be in 
  \href{http://spatialreference.org/ref/epsg/wgs-84/}{EPSG:4326} as well as 
  such objects have no projection information attached.\cr
  \cr
  Other:\cr
  \tab \code{Spatial}, \code{sf}, or \code{map} object.
}
}
\note{
\strong{MODIS} does no longer support the tile identification and automated 
download of MERIS and SRTM data. At least as far as the latter is concerned, 
easy data access is granted through \code{\link{getData}}.
}
\examples{
\dontrun{
# ex 1 ############
# interactive tile selection
getTile()

# ex 2: Spatial (taken from ?rgdal::readOGR) ############
dsn <- system.file("vectors/Up.tab", package = "rgdal")[1]
Up <- rgdal::readOGR(dsn, "Up")
getTile(Up)

# ex 3: sf ############
library(mapview)
getTile(franconia)

# ex 4: tileH,tileV ############
getTile(tileH = 18:19, tileV = 4)

# ex 5: Raster* with valid CRS ############
rst1 <- raster(xmn = 9.2, xmx = 17.47, ymn = 46.12, ymx = 49.3)
getTile(rst1)

# this also works for projected data
rst3 <- projectExtent(rst1, crs = "+init=epsg:32633")
getTile(rst3)

# ex 6: Raster* without CRS or, alternatively, Extent -> treated as EPSG:4326 ############
mat2 <- matrix(seq(180 * 360), byrow = TRUE, ncol = 360)
rst2 <- raster(mat2)
getTile(rst2)
getTile(extent(rst1))

# ex 7: map names as returned by search4map() ############
getTile("Austria")
getTile(c("Austria", "Germany"))

# or search for specific map name patterns (use with caution):
m1 <- search4map("Per")
getTile(m1)

# or use 'map' objects directly (remember to use map(..., fill = TRUE)): 
m2 <- map("state", region = "new jersey", fill = TRUE)
getTile(m2)
}

}
\seealso{
\code{\link{extent}}, \code{\link{map}}, \code{\link{search4map}}.
}
\author{
Matteo Mattiuzzi, Florian Detsch
}
