% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{wKModes}
\alias{wKModes}
\title{Weighted K-Modes Clustering}
\usage{
wKModes(data,
        modes,
        weights = NULL,
        iter.max = .Machine$integer.max,
        freq.weighted = FALSE,
        fast = TRUE,
        ...)
}
\arguments{
\item{data}{A matrix or data frame of categorical data. Objects have to be in rows, variables in columns.}

\item{modes}{Either the number of modes or a set of initial (distinct) cluster modes (where each mode is a row and \code{modes} has the same number of colums as \code{data}). If a number, a random set of (distinct) rows in \code{data} is chosen as the initial modes.}

\item{weights}{Optional numeric vector containing non-negative observation-specific case weights.}

\item{iter.max}{The maximum number of iterations allowed. Defaults to \code{.Machine$integer.max}. The algorithm terminates when \code{iter.max} is reached or when the partition ceases to change between iterations.}

\item{freq.weighted}{A logical indicating whether the usual simple-matching (Hamming) distance between objects is used, or a frequency weighted version of this distance. Default to \code{FALSE}; when \code{TRUE}, the frequency weights are computed within the algorithm and are \emph{not} user-specified. Distinct from the observation-level \code{weights} above, the frequency weights are assigned on a per-feature basis and derived from the categories represented in each column of \code{data}.}

\item{fast}{A logical indicating whether a fast version of the algorithm should be applied. Defaults to \code{TRUE}.}

\item{...}{Catches unused arguments.}
}
\value{
An object of class \code{"wKModes"} which is a list with the following components:
\describe{
\item{\code{cluster}}{A vector of integers indicating the cluster to which each object is allocated.}
\item{\code{size}}{The number of objects in each cluster.}
\item{\code{modes}}{A matrix of cluster modes.}
\item{\code{withindiff}}{The within-cluster (weighted) simple-matching distance for each cluster.}
\item{\code{iterations}}{The number of iterations the algorithm reached.}
\item{\code{weighted}}{A logical indicating whether observation-level \code{weights} were used or not throughout the algorithm.}
\item{\code{freq.weighted}}{A logical indicating whether feature-level \code{freq.weights} were used or not in the computation of the distances.}}
}
\description{
Perform k-modes clustering on categorical data with observation-specific sampling weights.
}
\details{
The k-modes algorithm (Huang, 1997) is an extension of the k-means algorithm by MacQueen (1967).

The data given by \code{data} is clustered by the k-modes method (Huang, 1997) which aims to partition the objects into k groups such that the distance from objects to the assigned cluster modes is minimised. 

By default, the simple-matching (Hamming) distance is used to determine the dissimilarity of two objects. It is computed by counting the number of mismatches in all variables. Alternatively, this distance can be weighted by the frequencies of the categories in data, using the \code{freq.weighted}n argument (see Huang, 1997, for details).

If an initial matrix of modes is supplied, it is possible that no object will be closest to one or more modes. In this case, fewer clusters than the number of supplied modes will be returned and a warning will be printed.

If called using \code{fast = TRUE}, the reassignment of the data to clusters is done for the entire data set before recomputation of the modes is done. For computational reasons, this option should be chosen for all but the most moderate of data sizes.
}
\note{
This code is adapted from the \code{kmodes} function in the \pkg{klaR} package. Specifically, modifications were made to allow for the incorporation of observation-specific sampling \code{weights}, with a view to using this function as a means to initialise the allocations for MEDseq models (see the \code{\link{MEDseq_control}} argument \code{init.z} and the related options \code{"kmodes"} and \code{"kmodes2"}). Notably, the \code{wKModes} function, when invokved inside \code{\link{MEDseq_fit}}, is used regardless of whether the weights are true sampling weights, or the weights are merely aggregation weights, or there are no weights at all.
}
\examples{
\dontshow{suppressMessages(require(TraMineR)); suppressMessages(require(WeightedCluster))}
# Load the MVAD data & aggregate the state sequences
data(mvad)
aggMVAD  <- wcAggregateCases(mvad[,17:86], weights=mvad$weight)

# Create a state sequence object without the first two (summer) time points
states   <- c("EM", "FE", "HE", "JL", "SC", "TR")
labels   <- c("Employment", "Further Education", "Higher Education", 
              "Joblessness", "School", "Training")
mvad.seq <- seqdef(mvad[aggMVAD$aggIndex, 17:86], 
                   states=states, labels=labels, 
                   weights=aggMVAD$aggWeights)

# Run k-modes without the weights
resX     <- wKModes(mvad.seq, 2)

# Run k-modes with the weights
resW     <- wKModes(mvad.seq, 2, weights=aggMVAD$aggWeights)

# Examine the modal sequences of both solutions
seqformat(seqdef(resX$modes), from="STS", to="SPS", compress=TRUE)
seqformat(seqdef(resW$modes), from="STS", to="SPS", compress=TRUE)
}
\references{
Huang, Z. (1997). A fast clustering algorithm to cluster very large categorical data sets in data mining. In H. Lu, H. Motoda, and H. Luu (Eds.), \emph{KDD: Techniques and Applications}, pp. 21-34. Singapore: World Scientific.

MacQueen, J. (1967). Some methods for classification and analysis of multivariate observations. In L. M. L. Cam and J. Neyman (Eds.), \emph{Proceedings of the Fifth Berkeley Symposium on  Mathematical Statistics and Probability}, Volume 1, pp. 281-297. Berkeley, CA, USA: University of California Press.
}
\seealso{
\code{\link{MEDseq_control}}, \code{\link{MEDseq_fit}}
}
\author{
Keefe Murphy - <\email{keefe.murphy@mu.ie}>
}
\keyword{utility}
