% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_secondary.R
\name{estimate_secondary}
\alias{estimate_secondary}
\title{Estimate a Secondary Observation from a Primary Observation}
\usage{
estimate_secondary(
  reports,
  secondary = secondary_opts(),
  delays = delay_opts(list(mean = 2.5, mean_sd = 0.5, sd = 0.47, sd_sd = 0.25, max = 30)),
  truncation = trunc_opts(),
  obs = obs_opts(),
  burn_in = 14,
  CrIs = c(0.2, 0.5, 0.9),
  priors = NULL,
  model = NULL,
  verbose = interactive(),
  ...
)
}
\arguments{
\item{reports}{A data frame containing the \code{date} of report and both
\code{primary} and \code{secondary} reports.}

\item{secondary}{A call to \code{secondary_opts()} or a list containing the
following  binary variables: cumulative, historic, primary_hist_additive,
current, primary_current_additive. These parameters control the structure of
the secondary model, see \code{secondary_opts()} for details.}

\item{delays}{A call to \code{\link[=delay_opts]{delay_opts()}} defining delay distributions between
primary and secondary observations See the documentation of \code{delay_opts()}
for details. By default a diffuse prior  is assumed with a mean of 14 days
and standard deviation of 7 days (with a standard deviation of 0.5 and 0.25
respectively on the log scale).}

\item{truncation}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}} A list of options as
generated by \code{trunc_opts()} defining the truncation of observed data.
Defaults to \code{trunc_opts()}. See \code{estimate_truncation()} for an approach to
estimating truncation from data.}

\item{obs}{A list of options as generated by \code{obs_opts()} defining the
observation model. Defaults to \code{obs_opts()}.}

\item{burn_in}{Integer, defaults to 14 days. The number of data points to
use for estimation but not to fit to at the beginning of the time series.
This must be less than the number of observations.}

\item{CrIs}{Numeric vector of credible intervals to calculate.}

\item{priors}{A \code{data.frame} of named priors to be used in model fitting
rather than the defaults supplied from other arguments. This is typically
useful if wanting to inform an estimate from the posterior of another model
fit.}

\item{model}{A compiled stan model to override the default model. May be
useful for package developers or those developing extensions.}

\item{verbose}{Logical, should model fitting progress be returned. Defaults
to \code{interactive()}.}

\item{...}{Additional parameters to pass to \code{rstan::sampling}.}
}
\value{
A list containing: \code{predictions} (a data frame ordered by date with
the primary, and secondary observations, and a summary of the model
estimated secondary observations), \code{posterior} which contains a summary of
the entire model posterior, \code{data} (a list of data used to fit the
model), and \code{fit} (the \code{stanfit} object).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}
Estimates the relationship between a primary and secondary observation, for
example hospital admissions and deaths or hospital admissions and bed
occupancy. See \code{secondary_opts()} for model structure options. See parameter
documentation for model defaults and options. See the examples for case
studies using synthetic data and
\href{https://gist.github.com/seabbs/4f09d7609df298db7a86c31612ff9d17}{here}
for an example of forecasting Covid-19 deaths from Covid-19 cases. See
\href{https://gist.github.com/seabbs/4dad3958ca8d83daca8f02b143d152e6}{here} for
a prototype function that may be used to estimate and forecast a secondary
observation from a primary across multiple regions and
\href{https://github.com/epiforecasts/covid.german.forecasts/blob/master/rt-forecast/death-from-cases.R}{here}
for an application forecasting Covid-19 deaths in Germany and Poland.
}
\examples{
\donttest{
# set number of cores to use
old_opts <- options()
options(mc.cores = ifelse(interactive(), 4, 1))

# load data.table for manipulation
library(data.table)

#### Incidence data example ####

# make some example secondary incidence data
cases <- example_confirmed
cases <- as.data.table(cases)[, primary := confirm]
# Assume that only 40 percent of cases are reported
cases[, scaling := 0.4]
# Parameters of the assumed log normal delay distribution
cases[, meanlog := 1.8][, sdlog := 0.5]

# Simulate secondary cases
cases <- simulate_secondary(cases, type = "incidence")
#
# fit model to example data specifying a weak prior for fraction reported
# with a secondary case
inc <- estimate_secondary(cases[1:60],
  obs = obs_opts(scale = list(mean = 0.2, sd = 0.2), week_effect = FALSE)
)
plot(inc, primary = TRUE)

# forecast future secondary cases from primary
inc_preds <- forecast_secondary(inc, cases[61:.N][, value := primary])
plot(inc_preds, new_obs = cases, from = "2020-05-01")

#### Prevalence data example ####

# make some example prevalence data
cases <- example_confirmed
cases <- as.data.table(cases)[, primary := confirm]
# Assume that only 30 percent of cases are reported
cases[, scaling := 0.3]
# Parameters of the assumed log normal delay distribution
cases[, meanlog := 1.6][, sdlog := 0.8]

# Simulate secondary cases
cases <- simulate_secondary(cases, type = "prevalence")

# fit model to example prevalence data
prev <- estimate_secondary(cases[1:100],
  secondary = secondary_opts(type = "prevalence"),
  obs = obs_opts(
    week_effect = FALSE,
    scale = list(mean = 0.4, sd = 0.1)
  )
)
plot(prev, primary = TRUE)

# forecast future secondary cases from primary
prev_preds <- forecast_secondary(prev, cases[101:.N][, value := primary])
plot(prev_preds, new_obs = cases, from = "2020-06-01")

options(old_opts)
}
}
\author{
Sam Abbott
}
