% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TREGO.nsteps.R
\name{TREGO.nsteps}
\alias{TREGO.nsteps}
\title{Trust-region based EGO algorithm.}
\usage{
TREGO.nsteps(
  model,
  fun,
  nsteps,
  lower,
  upper,
  control = NULL,
  kmcontrol = NULL,
  trcontrol = NULL,
  trace = 0,
  n.cores = 1,
  ...
)
}
\arguments{
\item{model}{an object of class \code{\link[DiceKriging]{km}},}

\item{fun}{the objective function to be minimized,}

\item{nsteps}{an integer representing the desired number of iterations,}

\item{lower, upper}{vector of lower and upper bounds for the variables to be optimized over,}

\item{control}{an optional list of control parameters for optimization.
For now only the number of \code{restarts} can be set.}

\item{kmcontrol}{an optional list representing the control variables for
the re-estimation of the kriging model.}

\item{trcontrol}{an optional list of control parameters for the trust-region scheme: 
\code{sigma} the initial size of the trust region,
\code{x0} its initial center, 
\code{beta} the contraction factor,
\code{alpha} its dilatation factor, 
\code{kappa} the forcing factor, 
\code{crit} the criterion used inside the TR (either "EI" or "gpmean"),
\code{GLratio} number of consecutive global and local steps,
\code{algo} either "TREGO" or "TRIKE",
\code{minsigma} minimal sigma value,
\code{maxsigma} maximal sigma value,
\code{minEI} stopping criterion for TRIKE,
\code{local.model} Boolean; if TRUE, a local model is used within the trust region,
\code{local.trend, local.covtype} trend and covariance for the local model,
\code{n.local.min} minimal number of points used to build the local model,}

\item{trace}{between -1 (no trace) and 3 (full messages)}

\item{n.cores}{number of cores used for EI maximisation}

\item{...}{additional parameters to be given to \code{fun}}
}
\value{
A list with components:

\item{par}{a data frame representing the additional points visited during
the algorithm,}

\item{value}{a data frame representing the response values at the points
given in \code{par},}

\item{npoints}{an integer representing the number of parallel computations
(=1 here),}

\item{nsteps}{an integer representing the desired number of iterations
(given in argument),}

\item{lastmodel}{an object of class \code{\link[DiceKriging]{km}}
corresponding to the last kriging model fitted. If warping is true, 
\code{y} values are normalized (warped) and will not match \code{value}.}

\item{all.success}{ a vector of Boolean indicating the successful steps according
to the sufficient decrease condtion}

\item{all.steps}{ a vector of Boolean indicating which steps were global}

\item{all.sigma}{ history of trust region size}

\item{all.x0}{ history of trust region centers}

\item{local.model}{ if trcontrol$local.model=TRUE, the latest local model}
}
\description{
Executes \emph{nsteps} iterations of the TREGO method to an object of class
\code{\link[DiceKriging]{km}}.  At each step, a kriging model is
re-estimated (including covariance parameters re-estimation) based on the
initial design points plus the points visited during all previous
iterations; then a new point is obtained by maximizing the Expected
Improvement criterion (\code{\link{EI}}) over either the entire search space
or restricted to a trust region. The trust region is updated at each iteration
based on a sufficient decrease condition.
}
\examples{


set.seed(123)
###############################################################
### 	10 ITERATIONS OF TREGO ON THE BRANIN FUNCTION, 	 ####
###	 STARTING FROM A 9-POINTS FACTORIAL DESIGN         ####
###############################################################

# a 9-points factorial design, and the corresponding response
d <- 2
n <- 9
design.fact <- expand.grid(seq(0,1,length=3), seq(0,1,length=3)) 
names(design.fact)<-c("x1", "x2")
design.fact <- data.frame(design.fact) 
names(design.fact)<-c("x1", "x2")
response.branin <- apply(design.fact, 1, branin)
response.branin <- data.frame(response.branin) 
names(response.branin) <- "y" 

# model identification
fitted.model1 <- km(~1, design=design.fact, response=response.branin, 
covtype="gauss", control=list(pop.size=50,trace=FALSE), parinit=c(0.5, 0.5))

# TREGO n steps
nsteps <- 5
lower <- rep(0, d) 
upper <- rep(1, d)     
oEGO <- TREGO.nsteps(model=fitted.model1, fun=branin, nsteps=nsteps, 
lower=lower, upper=upper)
print(oEGO$par)
print(oEGO$value)

# graphics
n.grid <- 15 # Was 20, reduced to 15 for speeding up compilation
x.grid <- y.grid <- seq(0,1,length=n.grid)
design.grid <- expand.grid(x.grid, y.grid)
response.grid <- apply(design.grid, 1, branin)
z.grid <- matrix(response.grid, n.grid, n.grid)
contour(x.grid, y.grid, z.grid, 40)
title("Branin function")
points(design.fact[,1], design.fact[,2], pch=17, col="blue")
points(oEGO$par, pch=19, col="red")
text(oEGO$par[,1], oEGO$par[,2], labels=1:nsteps, pos=3)
}
\references{
Diouane, Picheny, Le Riche, Scotto Di Perrotolo (2021),
\emph{TREGO: a Trust-Region Framework for Efficient Global Optimization}, ArXiv
}
\seealso{
\code{\link{EI}}, \code{\link{max_crit}}, \code{\link{EI.grad}}
}
\author{
Victor Picheny
}
\keyword{optimize}
