% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutate_node_attrs.R
\name{mutate_node_attrs}
\alias{mutate_node_attrs}
\title{Mutate a set of node attribute values}
\usage{
mutate_node_attrs(graph, ...)
}
\arguments{
\item{graph}{a graph object of class
\code{dgr_graph}.}

\item{...}{expressions used for the mutation
of node attributes. LHS of each expression is
either an existing or new node attribute name.
The RHS can consist of any valid R code that
uses node attributes as variables. Expressions
are evaluated in the order provided, so, node
attributes created or modified are ready to
use in subsequent expressions.}
}
\value{
a graph object of class
\code{dgr_graph}.
}
\description{
Within a graph's internal node
data frame (ndf), mutate numeric node
attribute values using one or more expressions.
}
\examples{
# Create a graph with 3 nodes
graph <-
  create_graph() \%>\%
  add_path(n = 3) \%>\%
  set_node_attrs(
    node_attr = width,
    values = c(1.4, 0.3, 1.1))

# Get the graph's internal ndf
# to show which node attributes
# are available
get_node_df(graph)
#>   id type label width
#> 1  1 <NA>     1   1.4
#> 2  2 <NA>     2   0.3
#> 3  3 <NA>     3   1.1

# Mutate the `width` node
# attribute, dividing each
# value by 2
graph <-
  graph \%>\%
  mutate_node_attrs(
    width = width / 2)

# Get the graph's internal
# ndf to show that the node
# attribute `width` had its
# values changed
get_node_df(graph)
#>   id type label width
#> 1  1 <NA>     1  0.70
#> 2  2 <NA>     2  0.15
#> 3  3 <NA>     3  0.55

# Create a new node attribute,
# called `length`, that is the
# log of values in `width` plus
# 2 (and, also, round all values
# to 2 decimal places)
graph <-
  graph \%>\%
  mutate_node_attrs(
    length = (log(width) + 2) \%>\%
               round(2))

# Get the graph's internal ndf
# to show that the node attribute
# values had been mutated
#>   id type label width length
#> 1  1 <NA>     1  0.70   1.64
#> 2  2 <NA>     2  0.15   0.10
#> 3  3 <NA>     3  0.55   1.40

# Create a new node attribute
# called `area`, which is the
# product of the `width` and
# `length` attributes
graph <-
  graph \%>\%
  mutate_node_attrs(
    area = width * length)

# Get the graph's internal ndf
# to show that the node attribute
# values had been multiplied
# together (with new attr `area`)
get_node_df(graph)
#>   id type label width length  area
#> 1  1 <NA>     1  0.70   1.64 1.148
#> 2  2 <NA>     2  0.15   0.10 0.015
#> 3  3 <NA>     3  0.55   1.40 0.770
}
