/*
 * AbstractModelSpecifics.cpp
 *
 *  Created on: Jul 13, 2012
 *      Author: msuchard
 */

#include <stdexcept>
#include <set>

#include "AbstractModelSpecifics.h"
#include "ModelData.h"
#include "engine/ModelSpecifics.h"

#ifdef HAVE_OPENCL
#include "engine/GpuModelSpecifics.hpp"
#endif // HAVE_OPENCL

namespace bsccs {

template <class Model, typename RealType>
AbstractModelSpecifics* deviceFactory(
        const ModelData<RealType>& modelData,
        const DeviceType deviceType,
        const std::string& deviceName) {
    AbstractModelSpecifics* model = nullptr;

    switch (deviceType) {
    case DeviceType::CPU :
        model = new ModelSpecifics<Model,RealType>(modelData);
        break;
#ifdef HAVE_OPENCL
    case DeviceType::GPU :
        model = new GpuModelSpecifics<Model,RealType>(modelData, deviceName);
        break;
#endif // HAVE_OPENCL
    default:
        break; // nullptr
    }
    return model;
}

template <typename RealType>
AbstractModelSpecifics* precisionFactory(
        const ModelType modelType,
        const ModelData<RealType>& modelData,
        const DeviceType deviceType,
        const std::string& deviceName);

template <>
AbstractModelSpecifics* precisionFactory<float>(
        const ModelType modelType,
        const ModelData<float>& modelData,
        const DeviceType deviceType,
        const std::string& deviceName) {

    AbstractModelSpecifics* model = nullptr;

    switch (modelType) {
    case ModelType::LOGISTIC :
        model = deviceFactory<LogisticRegression<float>,float>(modelData, deviceType, deviceName);
        break;
    case ModelType::POISSON :
        model = deviceFactory<PoissonRegression<float>,float>(modelData, deviceType, deviceName);
        break;
    case ModelType::CONDITIONAL_POISSON :
        model = deviceFactory<ConditionalPoissonRegression<float>,float>(modelData, deviceType, deviceName);
        break;
    case ModelType::COX :
        model = deviceFactory<BreslowTiedCoxProportionalHazards<float>,float>(modelData, deviceType, deviceName);
        break;
    default:
        break;
    }

    return model;
}

template <>
AbstractModelSpecifics* precisionFactory<double>(
        const ModelType modelType,
        const ModelData<double>& modelData,
        const DeviceType deviceType,
        const std::string& deviceName) {

    AbstractModelSpecifics* model = nullptr;

    switch (modelType) {
    case ModelType::SELF_CONTROLLED_MODEL :
        model =  deviceFactory<SelfControlledCaseSeries<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::CONDITIONAL_LOGISTIC :
        model =  deviceFactory<ConditionalLogisticRegression<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::TIED_CONDITIONAL_LOGISTIC :
        model =  deviceFactory<TiedConditionalLogisticRegression<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::LOGISTIC :
        model = deviceFactory<LogisticRegression<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::NORMAL :
        model = deviceFactory<LeastSquares<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::POISSON :
        model = deviceFactory<PoissonRegression<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::CONDITIONAL_POISSON :
        model = deviceFactory<ConditionalPoissonRegression<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::COX_RAW :
        model = deviceFactory<CoxProportionalHazards<double>,double>(modelData, deviceType, deviceName);
        break;
    case ModelType::COX :
        model = deviceFactory<BreslowTiedCoxProportionalHazards<double>,double>(modelData, deviceType, deviceName);
        break;
    default:
        break;
    }
    return model;
}

AbstractModelSpecifics* AbstractModelSpecifics::factory(const ModelType modelType,
                                                        const AbstractModelData& abstractModelData,
                                                        const DeviceType deviceType,
                                                        const std::string& deviceName) {
    AbstractModelSpecifics* model = nullptr;

    if (modelType != ModelType::LOGISTIC && deviceType == DeviceType::GPU) {
        return model; // Implementing lr first on GPU.
    }

    switch(abstractModelData.getPrecisionType()) {
    case PrecisionType::FP64 :
        model = precisionFactory<double>(modelType,
                                         static_cast<const ModelData<double>&>(abstractModelData),
                                         deviceType, deviceName);
        break;
    case PrecisionType::FP32 :
        model = precisionFactory<float>(modelType,
                                        static_cast<const ModelData<float>&>(abstractModelData),
                                        deviceType, deviceName);
        break;
    default :
        break;
    }

    return model;
}

AbstractModelSpecifics::AbstractModelSpecifics(const AbstractModelData& input)
	: //oY(input.getYVectorRef()), oZ(input.getZVectorRef()),
	  //oPid(input.getPidVectorRef()),
	  // modelData(input),
//	  hXI(static_cast<CompressedDataMatrix*>(const_cast<ModelData*>(&modelData))),
// 	  hY(const_cast<real*>(input.getYVectorRef().data())), //hZ(const_cast<real*>(input.getZVectorRef().data())),
	  // hY(input.getYVectorRef()),
// 	  hOffs(const_cast<real*>(input.getTimeVectorRef().data())),
	  // hOffs(input.getTimeVectorRef()),
// 	  hPid(const_cast<int*>(input.getPidVectorRef().data()))
// 	  hPid(input.getPidVectorRef())
      hPidOriginal(input.getPidVectorRef()), hPid(const_cast<int*>(hPidOriginal.data())),
      boundType(MmBoundType::METHOD_2) {

	// Do nothing
}

AbstractModelSpecifics::~AbstractModelSpecifics() { }

void AbstractModelSpecifics::makeDirty(void) {
	hessianCrossTerms.erase(hessianCrossTerms.begin(), hessianCrossTerms.end());

//	for (HessianSparseMap::iterator it = hessianSparseCrossTerms.begin();
//			it != hessianSparseCrossTerms.end(); ++it) {
//		delete it->second;
//	}
}

int AbstractModelSpecifics::getAlignedLength(int N) {
	return (N / 16) * 16 + (N % 16 == 0 ? 0 : 16);
}


// template <typename RealType>
// void AbstractModelSpecifics::setPidForAccumulation(const RealType* weights) {
//
// 	hPidInternal =  hPidOriginal; // Make copy
// 	hPid = hPidInternal.data(); // Point to copy
// 	accReset.clear();
//
// 	const int ignore = -1;
//
// 	// Find first non-zero weight
// 	size_t index = 0;
// 	while(weights != nullptr && weights[index] == 0.0 && index < K) {
// 		hPid[index] = ignore;
// 		index++;
// 	}
//
// 	int lastPid = hPid[index];
// 	real lastTime = hOffs[index];
// 	real lastEvent = hY[index];
//
// 	int pid = hPid[index] = 0;
//
// 	for (size_t k = index + 1; k < K; ++k) {
// 		if (weights == nullptr || weights[k] != 0.0) {
// 			int nextPid = hPid[k];
//
// 			if (nextPid != lastPid) { // start new strata
// 				pid++;
// 				accReset.push_back(pid);
// 				lastPid = nextPid;
// 			} else {
//
// 				if (lastEvent == 1.0 && lastTime == hOffs[k] && lastEvent == hY[k]) {
// 					// In a tie, do not increment denominator
// 				} else {
// 					pid++;
// 				}
// 			}
// 			lastTime = hOffs[k];
// 			lastEvent = hY[k];
//
// 			hPid[k] = pid;
// 		} else {
// 			hPid[k] = ignore;
// 		}
// 	}
// 	pid++;
// 	accReset.push_back(pid);
//
// 	// Save number of denominators
// 	N = pid;
//
// 	if (weights != nullptr) {
// 		for (size_t i = 0; i < K; ++i) {
// 			if (hPid[i] == ignore) hPid[i] = N; // do NOT accumulate, since loops use: i < N
// 		}
// 	}
// 	setupSparseIndices(N); // ignore pid == N (pointing to removed data strata)
// }

// void AbstractModelSpecifics::setupSparseIndices(const int max) {
// 	sparseIndices.clear(); // empty if full!
//
// 	for (size_t j = 0; j < J; ++j) {
// 		if (modelData.getFormatType(j) == DENSE || modelData.getFormatType(j) == INTERCEPT) {
// 			sparseIndices.push_back(NULL);
// 		} else {
// 			std::set<int> unique;
// 			const size_t n = modelData.getNumberOfEntries(j);
// 			const int* indicators = modelData.getCompressedColumnVector(j);
// 			for (size_t j = 0; j < n; j++) { // Loop through non-zero entries only
// 				const int k = indicators[j];
// 				const int i = hPid[k];  // TODO container-overflow #Generate some simulated data: #Fit the model
// 				if (i < max) {
// 					unique.insert(i);
// 				}
// 			}
// 			auto indices = bsccs::make_shared<IndexVector>(unique.begin(), unique.end());
//             sparseIndices.push_back(indices);
// 		}
// 	}
// }

// void AbstractModelSpecifics::deviceInitialization() {
//     // Do nothing
// }
//
// void AbstractModelSpecifics::initialize(
// 		int iN,
// 		int iK,
// 		int iJ,
// 		const void*,
// 		real* iNumerPid,
// 		real* iNumerPid2,
// 		real* iDenomPid,
// //		int* iNEvents,
// 		real* iXjY,
// 		std::vector<std::vector<int>* >* iSparseIndices,
// 		const int* iPid_unused,
// 		real* iOffsExpXBeta,
// 		real* iXBeta,
// 		real* iOffs,
// 		real* iBeta,
// 		const real* iY_unused//,
// //		real* iWeights
// 		) {
// 	N = iN;
// 	K = iK;
// 	J = iJ;
// 	offsExpXBeta.resize(K);
// 	hXBeta.resize(K);
//
// 	if (allocateXjY()) {
// 		hXjY.resize(J);
// 	}
//
// 	if (allocateXjX()) {
// 		hXjX.resize(J);
// 	}
//
// 	if (initializeAccumulationVectors()) {
// 		setPidForAccumulation(static_cast<double*>(nullptr)); // calls setupSparseIndices() before returning
//  	} else {
// 		// TODO Suspect below is not necessary for non-grouped data.
// 		// If true, then fill with pointers to CompressedDataColumn and do not delete in destructor
// 		setupSparseIndices(N); // Need to be recomputed when hPid change!
// 	}
//
//
//
// 	size_t alignedLength = getAlignedLength(N + 1);
// // 	numerDenomPidCache.resize(3 * alignedLength, 0);
// // 	numerPid = numerDenomPidCache.data();
// // 	denomPid = numerPid + alignedLength; // Nested in denomPid allocation
// // 	numerPid2 = numerPid + 2 * alignedLength;
// 	denomPid.resize(alignedLength);
// 	numerPid.resize(alignedLength);
// 	numerPid2.resize(alignedLength);
//
// 	deviceInitialization();
//
// }

} // namespace
