% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gcalibrate.R
\name{gcalibrate}
\alias{gcalibrate}
\title{Calibration for Gaussian Outcomes}
\usage{
gcalibrate(
  y,
  tr,
  t1,
  t2,
  calitype = c("worstcase", "multicali", "null"),
  mu_y_dt = NULL,
  sigma_y_t = NULL,
  mu_u_dt = NULL,
  cov_u_t = NULL,
  nU = NULL,
  R2 = 1,
  gamma = NULL,
  R2_constr = 1,
  nc_index = NULL,
  ...
)
}
\arguments{
\item{y}{\code{data.frame}, \code{matrix} or \code{vector}. Gaussian outcome variable.}

\item{tr}{\code{data.frame}. Treatment variables with rows corresponding to observations and columns
to variables.}

\item{t1}{\code{data.frame}. First treatment arms of interest.
May contain a single or multiple treatments in rows.}

\item{t2}{\code{data.frame}. Second treatment arms of interest,
which has same number of row as \code{t1}.}

\item{calitype}{character. The calibration method to be applied. Can be one of: \cr
"worstcase" - apply worst-case calibration when considering a single contrast. \cr
"multicali" - apply mutlivariate calibration when considering multiple contrasts.\cr
"null" - apply calibration with user-specified sensitivity parameter, \eqn{\gamma}. \cr}

\item{mu_y_dt}{an optional scalar or vector that contains naive estimates of treatment effects
ignoring confounding.}

\item{sigma_y_t}{an optional scalar of the standard deviation of outcome conditional on treatments.}

\item{mu_u_dt}{an optional matrix of difference in conditional confounder means, \eqn{E(U \mid t1) - E(U \mid t2)},
with latent variables in columns.}

\item{cov_u_t}{an optional covariance matrix of confounders conditional on treatments.}

\item{nU}{Number of latent confounders to consider.}

\item{R2}{an optional scalar or vector specifying the proportion of residual variance in outcome given the
treatment that can be explained by confounders.}

\item{gamma}{sensitivity parameter vector. Must be given when \code{calitype = "null"}.}

\item{R2_constr}{an optional scalar or vector specifying the upper limit constraint on \eqn{R^2} .
By default, \code{R2_constr = 1}.}

\item{nc_index}{an optional vector containing indexes of negative control treatments. If not \code{NULL},
worstcase calibration will be executed with constraints imposed by negative control treatments.}

\item{...}{further arguments passed to \code{\link{kEstimate}}, \code{\link{pca}} or
\code{\link{get_opt_gamma}}.}
}
\value{
\code{gcalibrate} returns a list containing the following components:
\describe{
  \item{\code{est_df}}{a \code{data.frame} with naive and calibrated estimates of average treatment effects.}
  \item{\code{R2}}{a vector of \eqn{R^2} with elements corresponding to columns of \code{est_df}.}
  \item{\code{gamma}}{a matrix returned when \code{calitype = "multicali"} or \code{"worstcase"}.
  If \code{calitype = "multicali"}, optimized gamma are in columns,
  respectively resulting in estimates in columns of \code{est_df}.
  If \code{calitype = "worstcase"}, gamma are in rows,
  which respectively lead to the worstcase ignorance region with \eqn{R^2=1} for each contrast of interest.}
  \item{\code{rv}}{a \code{numeric vector} returned when \code{calitype = "worstcase"},
  with elements being the robustness value or \code{NA} if the ignorance region doesn't
  contains 0 for each contrast of interest.}
}
}
\description{
Calibrates the naive estimates to account for unobserved confounding when outcome
variables are Gaussian. The calibration can be done with user-specific sensitivity parameters or
with our pre-provided calibration methods, the worst-case calibration for a single contrast
or multivariate calibration for multiple contrasts.
}
\examples{
# load the example data #
y <- GaussianT_GaussianY$y
tr <- subset(GaussianT_GaussianY, select = -c(y))

# worst-case calibration #
t1 <- data.frame(diag(ncol(tr)))
t2 <- data.frame(matrix(0, nrow = ncol(tr), ncol = ncol(tr)))
colnames(t1) = colnames(t2) <- colnames(tr)
est_g1 <- gcalibrate(y = y, tr = tr, t1 = t1, t2 = t2, nU = 3,
                     calitype = "worstcase", R2 = c(0.3, 1))
plot_estimates(est_g1)
# with negative conotrls #
est_g1_nc <- gcalibrate(y = y, tr = tr, t1 = t1, t2 = t2, nU = 3,
                        calitype = "worstcase", R2 = c(0.3, 1), nc_index = c(3, 6))
plot_estimates(est_g1_nc)

\donttest{
# multivariate calibration #
est_g2 <- gcalibrate(y = y, tr = tr, t1 = tr[1:10,], t2 = tr[11:20,], nU = 3,
                     calitype = "multicali", R2_constr = c(1, 0.15))
plot_estimates(est_g2)

# user-specified calibration #
est_g3 <- gcalibrate(y = y, tr = tr, t1 = tr[1:2,], t2 = tr[3:4,],
                     nU = 3, calitype = "null",
                     gamma = c(0.96, -0.29, 0), R2 = c(0.2, 0.6, 1))
plot_estimates(est_g3)
# apply gamma that maximizes the bias for the first contrast considered in est_g1 #
est_g4 <- gcalibrate(y = y, tr = tr, t1 = tr[1:2,], t2 = tr[3:4,],
                     nU = 3, calitype = "null",
                     gamma = est_g1$gamma[1,], R2 = c(0.2, 0.6, 1))
plot_estimates(est_g4)
}
}
