% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/doubly_robust.R
\name{doubly_robust}
\alias{doubly_robust}
\title{Doubly Robust Model}
\usage{
doubly_robust(
  data,
  out.mod = NULL,
  p.mod = NULL,
  f = NA,
  family = gaussian(),
  simple = pkg.env$simple,
  scores = NA,
  p.f = NA,
  p.simple = pkg.env$simple,
  p.family = binomial(),
  p.scores = NA,
  n.boot = 50,
  ...
)
}
\arguments{
\item{data}{a data frame containing the variables in the model.
This should be the same data used in \code{\link[=init_params]{init_params}}.}

\item{out.mod}{(optional) a regression model that predicts the outcome. NOTE: the model given
must be compatible with the \code{\link[stats:predict]{predict}} generic function.}

\item{p.mod}{(optional) a propensity model that predicts the probability of treatment. NOTE: the model given
must be compatible with the \code{\link[stats:predict]{predict}} generic function.}

\item{f}{(optional) an object of class "formula" that overrides the default parameter}

\item{family}{the family to be used in the general linear model.
By default, this is set to \code{\link[stats:gaussian]{gaussian}}.}

\item{simple}{a boolean indicator to build default formula with interactions.
If true, interactions will be excluded. If false, interactions will be included. By
default, simple is set to false.}

\item{scores}{(optional) use calculated outcome estimates.}

\item{p.f}{(optional) an object of class "formula" that overrides the default formula for the denominator of the IP
weighting function.}

\item{p.simple}{a boolean indicator to build default formula with interactions for the propensity models.
If true, interactions will be excluded. If false, interactions will be included. By
default, simple is set to false.
NOTE: if this is changed, the coefficient for treatment may not accurately represent the average causal effect.}

\item{p.family}{the family to be used in the underlying propensity model.
By default, this is set to \code{\link[stats:gaussian]{binomial}}.}

\item{p.scores}{(optional) use calculated propensity scores.}

\item{n.boot}{an integer value that indicates number of bootstrap iterations to calculate standard error.}

\item{...}{additional arguments that may be passed to the underlying \code{\link[stats:glm]{glm}} model.}
}
\value{
\code{doubly_robust} returns an object of \code{\link[base:class]{class}} "doubly_robust".

The functions \code{print}, \code{summary}, and \code{predict} can be used to interact with
the underlying \code{glm} model.

An object of class \code{"doubly_robust"} is a list containing the following:

 \item{out.call}{the matched call of the outcome model.}
 \item{p.call}{the matched call of the propensity model.}
 \item{out.model}{the underlying outcome model.}
 \item{p.model}{the underlying propensity model.}
 \item{y_hat}{the estimated outcome values.}
 \item{p.scores}{the estimated propensity scores.}
 \item{ATE}{the estimated average treatment effect (risk difference).}
 \item{ATE.summary}{a data frame containing the ATE, SE, and 95\% CI of the ATE. }
 \item{data}{the data frame used to train the model.}
}
\description{
\code{`doubly_robust`} trains both an outcome model and a propensity model to generate predictions
for the outcome and probability of treatment respectively. By default, the model uses
\code{\link[=standardization]{standardization}} and \code{\link[=propensity_scores]{propensity_scores}} to form a
doubly-robust model between standardization and IP weighting. Alternatively, any outcome and treatment
models can be provided instead, but must be compatible with the \code{\link[stats:predict]{predict}} generic function in R.
Since many propensity models may not predict probabilities without additional arguments into the
predict function, the predictions themselves can be given for both the outcome and propensity scores.
}
\examples{
library(causaldata)
data(nhefs)
nhefs.nmv <- nhefs[which(!is.na(nhefs$wt82)), ]
nhefs.nmv$qsmk <- as.factor(nhefs.nmv$qsmk)

confounders <- c(
  "sex", "race", "age", "education", "smokeintensity",
  "smokeyrs", "exercise", "active", "wt71"
)

init_params(wt82_71, qsmk,
  covariates = confounders,
  data = nhefs.nmv
)

# model using all defaults
model <- doubly_robust(data = nhefs.nmv)
summary(model)

# use alternative outcome model
out.mod <- propensity_matching(data = nhefs.nmv)
db.model <- doubly_robust(
  out.mod = out.mod,
  data = nhefs.nmv
)
db.model

# give calculated outcome predictions and give formula for propensity scores
db.model <- doubly_robust(
  scores = predict(out.mod),
  p.f = qsmk ~ sex + race + age,
  data = nhefs.nmv
)
db.model

}
