\name{ensemble.envirem.masterstack}
\alias{ensemble.envirem.masterstack}
\alias{ensemble.envirem.solradstack}
\alias{ensemble.envirem.run}

\title{
Calculate bioclimatic variables via the \code{envirem} package for data.frames.
}

\description{
Function \code{\link[envirem]{generateEnvirem}} uses RasterStack (\code{\link[raster]{stack}}) objects as input and also generates outputs in the same format. The functions described here can be used to generate the bioclimatic variables for \code{data.frames} while using \code{envirem} functions internally. This feature can be useful in situations where models are calibrated with higher resolution data, but where maps will only be generated in lower resolutions, thus avoiding the need to generate the higher resolution envirem layers first.  
}

\usage{
ensemble.envirem.masterstack(
    x,
    precipstack, 
    tmaxstack, tminstack, 
    tmeanstack = NULL,
    envirem3 = TRUE)

ensemble.envirem.solradstack(
    x, solrad,
    envirem3 = TRUE)

ensemble.envirem.run(
    masterstack, solradstack,
    var = "all", ...)
}

\arguments{
  \item{x}{ Point locations provided in 2-column (eg, LON-LAT) format.}
  \item{precipstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly precipitation data.  }
  \item{tmaxstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly maximum temperature data.  }
  \item{tminstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly minimum temperature data.  }
  \item{tmeanstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly average temperature data.  }
  \item{envirem3}{ generate a SpatRaster object (\code{\link[terra]{rast}}) as used by envirem 3.  }
  \item{solrad}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly extrasolar radiation data.  }  
  \item{masterstack}{ RasterStack object (\code{\link[raster]{stack}}) expected to have been created via \code{\link[BiodiversityR]{ensemble.envirem.masterstack}}.  }
  \item{solradstack}{ RasterStack object (\code{\link[raster]{stack}}) expected to have been created via \code{\link[BiodiversityR]{ensemble.envirem.solradstack}}.  }
  \item{var}{ Names of bioclimatic variables to be created; see: \code{\link[envirem]{generateEnvirem}}.  }
  \item{...}{ Other arguments for \code{\link[envirem]{generateEnvirem}}, dealing with the scale of temperature or precipitation data.  }
}

\details{

The objective of these functions is to expand a data.frame of explanatory variables that will be used for calibrating species distribution models with bioclimatic variables that are generated by the envirem package (See details in \code{\link[envirem]{generateEnvirem}}).

It is important that monthly values are sorted sequentially (January - December) as the \code{ensemble.envirem.masterstack} and \code{ensemble.envirem.solradstack} functions expect the inputs to be sorted in this order.

Function \code{ensemble.envirem.solradstack} requires monthly extraterrestrial solar radiation layers at the same resolution as the climatic layers. It is possible, however, to also calculate these values directly for point observation data as shown below in the examples.

}

\value{
Function \code{ensemble.envirem.run} returns a data.frame with bioclimatic variables for each point location. 
}

\author{ Roeland Kindt (World Agroforestry Centre)}

\seealso{
\code{\link[envirem]{generateEnvirem}}, \code{\link[BiodiversityR]{ensemble.calibrate.models}}, \code{\link[BiodiversityR]{ensemble.calibrate.weights}}
}

\references{
Title P.O., Bemmels J.B. 2018. ENVIREM: An expanded set of bioclimatic and topographic variables increases flexibility and improves performance of ecological niche modeling. Ecography 41: 291-307.

Kindt R. 2023. TreeGOER: A database with globally observed environmental ranges for 48,129 tree species. Global Change Biology. 
\doi{10.1111/gcb.16914}
}

\examples{

\dontrun{
# Based on examples in envirem package for envirem::generateEnvirem
# Modified Sep-2023 due to change in function name in envirem

library(terra)
library(envirem)

# Find example rasters
rasterFiles <- list.files(system.file('extdata', package='envirem'), 
                          full.names=TRUE)

precip.files <- rasterFiles[grepl(pattern="prec_", 
                                x=rasterFiles)]
precip.files <- precip.files[c(1, 5:12, 2:4)]
precip.stack <- terra::rast(precip.files)
precip.stack
names(precip.stack)

tmin.files <- rasterFiles[grepl(pattern="tmin_", 
                                x=rasterFiles)]
tmin.files <- tmin.files[c(1, 5:12, 2:4)]
tmin.stack <- terra::rast(tmin.files)
tmin.stack
names(tmin.stack)

tmax.files <- rasterFiles[grepl(pattern="tmax_", 
                                x=rasterFiles)]
tmax.files <- tmax.files[c(1, 5:12, 2:4)]
tmax.stack <- terra::rast(tmax.files)
tmax.stack
names(tmax.stack)

tmean.files <- rasterFiles[grepl(pattern="tmean_", 
                                x=rasterFiles)]
tmean.files <- tmean.files[c(1, 5:12, 2:4)]
tmean.stack <- terra::rast(tmean.files)
tmean.stack
names(tmean.stack)

# Random locations
locs <- dismo::randomPoints(raster::stack(precip.stack[[1]]), n=50)

# Climatic data
master.input <- ensemble.envirem.masterstack(x=locs,
                             precipstack=precip.stack,
                             tmaxstack=tmax.stack,
                             tminstack=tmin.stack,
                             tmeanstack=tmean.stack)

# Calculate solar radiation for 1975
# (Use other midpoint for the 1970-2000 WorldClim 2.1 baseline)
solrad.stack <- ETsolradRasters(precip.stack[[1]], 
                             year = 1975-1950, 
                             outputDir = NULL)

solrad.input <- ensemble.envirem.solradstack(x=locs,
                             solrad=solrad.stack)

# Obtain the envirem bioclimatic data

envirem.data1 <- ensemble.envirem.run(masterstack=master.input,
                             solradstack=solrad.input,
                             tempScale=10)

# Generate all the envirem layers, then extract
# See envirem package for envirem::generateEnvirem

worldclim <- rast(c(precip.files, tmax.files, tmin.files, tmean.files))
names(worldclim)

assignNames(precip = 'prec_##')

# generate all possible envirem variables
envirem.stack <- generateEnvirem(worldclim, solrad.stack, var='all', tempScale = 10)

# set back to defaults
assignNames(reset = TRUE)

envirem.data2 <- extract(envirem.stack, y=locs)

# compare
envirem.data1 - envirem.data2

# Calculate extraterrestrial solar radiation for point observations
solrad1 <- extract(solrad.stack, y=locs)
solrad2 <- array(dim=c(nrow(locs), 12))
for (i in 1:nrow(locs)) {
  lat.i <- locs[i, 2]
  for (m in 1:12) {
    solrad2[i, m] <- envirem:::calcSolRad(year=1975-1950,
                                          lat=lat.i,
                                          month=m)
  }
}

solrad1 - solrad2

}

}

