% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/search.prop.sd.r
\name{search.prop.sd}
\alias{search.prop.sd}
\title{Automated Heuristic Search of a Proposal Standard Deviation for \code{bayes.2S}}
\usage{
search.prop.sd(m, ndraws = 1000, succ.min = 3, acc.bounds.X = c(0.2, 0.25))
}
\arguments{
\item{m}{A model object of class \code{bayes.2S}.}

\item{ndraws}{Starting number of MCMC iterations after which the acceptance rate
is first evaluated. Defaults to 1000.}

\item{succ.min}{The algorithm doubles the number of MCMC draws \code{succ.min} times
(each time the acceptance rate is within \code{acc.bounds.X}), ensuring
stability. Defaults to 3.}

\item{acc.bounds.X}{A numeric vector of length two specifying the lower and upper
bounds for the acceptable acceptance rate. Defaults to \code{c(0.2, 0.25)}.}
}
\value{
A \code{list} with the following elements:
\describe{
\item{\code{prop.sd.X}}{The final (adjusted) proposal standard deviation.}
\item{\code{ac.X}}{The acceptance rate in the last iteration.}
}
}
\description{
The \link{bayes.2S} Gibbs sampler uses a Metropolis step for sampling the
incidence model parameters and requires specifying a standard deviation for
the normal proposal (jumping) distribution. This function uses a heuristic
algorithm to find a proposal distribution standard deviation such that the
Metropolis sampler accepts proposed draws at an acceptance rate within the
user-defined interval (by default around 20--25\%).
}
\details{
Starting from an initial \code{bayes.2S} model object \code{m}, the function
attempts to calibrate the standard deviation of the proposal distribution.
Specifically, it:
\enumerate{
\item Runs an initial update of \code{ndraws} iterations and computes an
acceptance rate.
\item If the acceptance rate lies within \code{acc.bounds.X}, the number
of MCMC draws \code{ndraws} is doubled, and the process repeats.
\item Otherwise, the proposal standard deviation \eqn{\sigma} is adjusted
based on whether the acceptance rate \eqn{p} is below the lower bound \eqn{a}
or above the upper bound \eqn{b} of \code{acc.bounds.X}.
\item The formula for adjustment is:
\deqn{\sigma \leftarrow \sigma \times (1 - \frac{ (a-p)}{a}) \quad\text{if } p < a, \quad
          \sigma \leftarrow \sigma \times (1 + \frac{ (p-b)}{b}) \quad\text{if } p > b.}
}
By default, if the acceptance rate falls within \eqn{[0.2, 0.25]}, that \eqn{\sigma}
is considered acceptable, and the process continues until \code{succ.min} consecutive
successes (doubles) are achieved.
}
\examples{
\donttest{
# Generate data according to Klausch et al. (2025) PIM
set.seed(2025)
dat = gen.dat(kappa = 0.7, n = 1e3, theta = 0.2,
              p = 1, p.discrete = 1,
              beta.X = c(0.2, 0.2), beta.W = c(0.2, 0.2),
              v.min = 20, v.max = 30, mean.rc = 80,
              sigma.X = 0.2, mu.X = 5, dist.X = "weibull",
              prob.r  = 1)

# An initial model fit with a moderate number of ndraws (e.g., 1e3)
mod = bayes.2S(
  Vobs = dat$Vobs, Z.X = dat$Z, Z.W = dat$Z, r = dat$r,
  kappa = 0.7, update.kappa = FALSE, ndraws = 1e3, chains = 2,
  prop.sd.X = 0.005, parallel = TRUE, dist.X = "weibull"
)

# Running the automated search
search.sd <- search.prop.sd(m = mod)
print(search.sd)
}

}
